/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <gtk/gtk.h>
#include <libgnome/libgnome.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <glade/glade.h>
#include "typedefs.h"
#include "catalog.h"
#include "catalog-list.h"
#include "file-utils.h"
#include "main.h"
#include "window.h"


typedef struct {
	gchar     *current_dir;

	GThumbWindow    *window;
	CatalogList *cat_list;

	GladeXML  *gui;
	GtkWidget *dialog;
	GtkWidget *new_catalog_btn;
	GtkWidget *new_dir_btn;
	GtkWidget *ok_btn;
	GtkWidget *cancel_btn;

	/* extra data. */
	union {
		GList     *list;           /* Used by "add to catalog". */
		gchar     *catalog_path;   /* Used by "move to catalog dir". */
	} data;
} DialogData;


/* -- "new catalog" dialog. -- */

/* called when the user enter the name of the catalog to create. */
static void
catalog_string_cb (gchar *string, 
		   gpointer p)
{
	DialogData *data = p;
	gchar *path;
	int fd;
	
	if (string == NULL)
		return;

	path = g_strconcat (data->current_dir,
			    "/",
			    string,
			    CATALOG_EXT,
			    NULL);
	fd = creat (path, 0644);
	close (fd);
	g_free (path);

	/* update the catalog list. */
	catalog_list_change_to (data->cat_list, data->current_dir);
}


/* called when the user click on the "new catalog" button. */
static void
new_catalog_cb (GtkWidget *widget, 
		gpointer p)
{
	DialogData *data = p;
	GtkWidget *dialog;

	dialog = gnome_request_dialog (FALSE,
				       _("Enter the catalog name : "),
				       "",
				       1024,
				       catalog_string_cb,
				       data,
				       GTK_WINDOW (data->dialog));
	gnome_dialog_set_close (GNOME_DIALOG (dialog), TRUE);

	gnome_dialog_run  (GNOME_DIALOG (dialog));
}


/* -- "new catalog directory" dialog. -- */

/* called when the user enter the name of the directory to create. */
static void
dir_string_cb (gchar *string, 
	       gpointer p)
{
	DialogData *data = p;
	gchar *path;

	if (string == NULL)
		return;

	path = g_strconcat (data->current_dir,
			    "/",
			    string,
			    NULL);
	mkdir (path, 0775);
	g_free (path);

	/* update the catalog list. */
	catalog_list_change_to (data->cat_list, data->current_dir);
}


/* called when the user click on the "new directory" button. */
static void
new_dir_cb (GtkWidget *widget, 
	    DialogData *data)
{
	GtkWidget *dialog;

	dialog = gnome_request_dialog (FALSE,
				       _("Enter the directory name : "),
				       "",
				       1024,
				       dir_string_cb,
				       data,
				       GTK_WINDOW (data->dialog));
	gnome_dialog_set_close (GNOME_DIALOG (dialog), TRUE);

	gnome_dialog_run  (GNOME_DIALOG (dialog));
}


/* -- "add to catalog" dialog -- */

/* called when an item of the catalog list is selected. */
static void 
add_to_catalog__catalog_selected_cb (GtkWidget *widget, 
				     gint row, 
				     gint col,
				     GdkEvent *event,
				     gpointer p)
{
	DialogData *data = p;
	gchar *path;

	path = catalog_list_path_from_row (data->cat_list, row);

	if (! path_is_dir (path)) {
		gtk_widget_set_sensitive (data->ok_btn, TRUE);
		g_free (path);
		return;
	} 

	gtk_widget_set_sensitive (data->ok_btn, FALSE);	
	catalog_list_change_to (data->cat_list, path);
	g_free (data->current_dir);
	data->current_dir = path;
}


/* called when the main dialog is closed. */
static void
add_to_catalog__destroy_cb (GtkWidget *widget, 
			    DialogData *data)
{
        gtk_object_unref (GTK_OBJECT (data->gui));
	g_free (data->current_dir);
	catalog_list_free (data->cat_list);
	path_list_free (data->data.list);
	g_free (data);
}


/* called when the "ok" button is clicked. */
static void
add_to_catalog__ok_cb (GtkWidget *widget, 
		       DialogData *data)
{
	gint row;
	gchar *path;
	Catalog *catalog;
	GList *scan;

	row = GPOINTER_TO_INT (GTK_CLIST (data->cat_list->clist)->selection->data);
	path = catalog_list_path_from_row (data->cat_list, row);

	catalog = catalog_new ();
	catalog_load_from_disk (catalog, path);
	scan = data->data.list;
	while (scan) {
		catalog_add_item (catalog, (gchar*) scan->data);
		scan = scan->next;
	}
	catalog_write_to_disk (catalog);
	catalog_free (catalog);

	g_free (path);
	gtk_widget_destroy (data->dialog);
}


/* create the "add to catalog" dialog. */
void
dlg_add_to_catalog (GThumbWindow *window,
		    GList *list)
{
	DialogData *data;
	GtkWidget *list_hbox;

	data = g_new (DialogData, 1);

	data->window = window;
	data->current_dir = get_catalog_full_path (NULL);
	data->cat_list = catalog_list_new ();
	data->data.list = list;

	data->gui = glade_xml_new (GTHUMB_GLADEDIR "/" GLADE_FILE , NULL);
        if (!data->gui) {
                g_warning ("Could not find " GLADE_FILE "\n");
                return;
        }

	/* Get the widgets. */

	data->dialog = glade_xml_get_widget (data->gui, "catalog_dialog");
        list_hbox = glade_xml_get_widget (data->gui, "cat_list_hbox");
        data->new_catalog_btn = glade_xml_get_widget (data->gui, "cat_new_catalog_btn");
        data->new_dir_btn = glade_xml_get_widget (data->gui, "cat_new_dir_btn");
        data->ok_btn = glade_xml_get_widget (data->gui, "cat_ok_btn");
        data->cancel_btn = glade_xml_get_widget (data->gui, "cat_cancel_btn");

	gtk_container_add (GTK_CONTAINER (list_hbox), data->cat_list->root_widget);

	/* Set the signals handlers. */

	gtk_signal_connect (GTK_OBJECT (data->dialog), "destroy",
			    (GtkSignalFunc) add_to_catalog__destroy_cb,
			    data);

	gtk_signal_connect (GTK_OBJECT (data->new_catalog_btn), "clicked",
                            GTK_SIGNAL_FUNC (new_catalog_cb),
                            data);
	gtk_signal_connect (GTK_OBJECT (data->new_dir_btn), "clicked",
                            GTK_SIGNAL_FUNC (new_dir_cb),
                            data);
	gtk_signal_connect (GTK_OBJECT (data->ok_btn), "clicked",
                            GTK_SIGNAL_FUNC (add_to_catalog__ok_cb),
                            data);
	gtk_signal_connect_object (GTK_OBJECT (data->cancel_btn), "clicked",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   GTK_OBJECT (data->dialog));

	gtk_signal_connect (GTK_OBJECT (data->cat_list->clist), 
			    "select_row",
			    (GtkSignalFunc)add_to_catalog__catalog_selected_cb,
			    data);

	/* set data. */
	catalog_list_change_to (data->cat_list, data->current_dir);
	gtk_widget_set_sensitive (data->ok_btn, FALSE);

	/* run dialog. */
	gtk_window_set_transient_for (GTK_WINDOW (data->dialog), GTK_WINDOW (window->app));
	gtk_window_set_modal (GTK_WINDOW (data->dialog), TRUE);
	gtk_widget_show_all (data->dialog);
}


/* -- "move to catalog directory" dialog -- */

static void
move_to_catalog_dir__destroy_cb (GtkWidget *widget, 
				 DialogData *data)
{
        gtk_object_unref (GTK_OBJECT (data->gui));
	g_free (data->current_dir);
	catalog_list_free (data->cat_list);
	g_free (data->data.catalog_path);
	g_free (data);
}


/* called when an item of the catalog list is selected. */
static void 
move_to_catalog_dir__catalog_selected_cb (GtkWidget *widget, 
					  gint row, 
					  gint col,
					  GdkEvent *event,
					  gpointer p)
{
	DialogData *data = p;
	gchar *path;

	path = catalog_list_path_from_row (data->cat_list, row);

	if (! path_is_dir (path)) {
		g_free (path);
		return;
	} 

	catalog_list_change_to (data->cat_list, path);
	g_free (data->current_dir);
	data->current_dir = path;
}


/* called when the "ok" button is clicked. */
static void
move_to_catalog_dir__ok_cb (GtkWidget *widget, 
			    DialogData *data)
{
	gchar *new_path;

	new_path = g_strconcat (data->cat_list->path,
				"/",
				file_name_from_path (data->data.catalog_path),
				NULL);

	file_move (data->data.catalog_path, new_path);
	g_free (new_path);

	all_windows_update_catalog_list ();
	gtk_widget_destroy (data->dialog);
}


/* create the "move to catalog directory" dialog. */
void
dlg_move_to_catalog_directory (GThumbWindow *window,
			       gchar *catalog_path)
{
	DialogData *data;
	GtkWidget *list_hbox;

	data = g_new (DialogData, 1);

	data->window = window;
	data->current_dir = get_catalog_full_path (NULL);
	data->cat_list = catalog_list_new ();
	data->data.catalog_path = catalog_path;

	data->gui = glade_xml_new (GTHUMB_GLADEDIR "/" GLADE_FILE , NULL);
        if (!data->gui) {
                g_warning ("Could not find " GLADE_FILE "\n");
                return;
        }

	/* Get the widgets. */

	data->dialog = glade_xml_get_widget (data->gui, "catalog_dialog");
        list_hbox = glade_xml_get_widget (data->gui, "cat_list_hbox");
        data->new_catalog_btn = glade_xml_get_widget (data->gui, "cat_new_catalog_btn");
        data->new_dir_btn = glade_xml_get_widget (data->gui, "cat_new_dir_btn");
        data->ok_btn = glade_xml_get_widget (data->gui, "cat_ok_btn");
        data->cancel_btn = glade_xml_get_widget (data->gui, "cat_cancel_btn");

	gtk_container_add (GTK_CONTAINER (list_hbox), data->cat_list->root_widget);

	/* Set the signals handlers. */

	gtk_signal_connect (GTK_OBJECT (data->dialog), "destroy",
			    (GtkSignalFunc) move_to_catalog_dir__destroy_cb,
			    data);

	gtk_signal_connect (GTK_OBJECT (data->new_dir_btn), "clicked",
                            GTK_SIGNAL_FUNC (new_dir_cb),
                            data);
	gtk_signal_connect (GTK_OBJECT (data->ok_btn), "clicked",
                            GTK_SIGNAL_FUNC (move_to_catalog_dir__ok_cb),
                            data);
	gtk_signal_connect_object (GTK_OBJECT (data->cancel_btn), "clicked",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   GTK_OBJECT (data->dialog));

	gtk_signal_connect (GTK_OBJECT (data->cat_list->clist), 
			    "select_row",
			    (GtkSignalFunc) move_to_catalog_dir__catalog_selected_cb,
			    data);

	/* set data. */

	catalog_list_change_to (data->cat_list, data->current_dir);
	gtk_widget_set_sensitive (data->new_catalog_btn, FALSE);
	gtk_window_set_title (GTK_WINDOW (data->dialog), _("Move Catalog to..."));

	/* run dialog. */
	gtk_window_set_transient_for (GTK_WINDOW (data->dialog), GTK_WINDOW (window->app));
	gtk_window_set_modal (GTK_WINDOW (data->dialog), TRUE);
	gtk_widget_show_all (data->dialog);
}
