/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <math.h>
#include <libgnomeui/gnome-canvas.h>
#include <libgnomeui/gnome-canvas-util.h>
#include "gnome-canvas-thumb.h"

enum {
	ARG_0,
	ARG_PIXBUF,
	ARG_X,
	ARG_Y,
	ARG_WIDTH,
	ARG_HEIGHT
};


#define DEFAULT_WIDTH  105
#define DEFAULT_HEIGHT 105


static void gnome_canvas_thumb_class_init (GnomeCanvasThumbClass *class);
static void gnome_canvas_thumb_init       (GnomeCanvasThumb      *image);
static void gnome_canvas_thumb_destroy    (GtkObject             *object);
static void gnome_canvas_thumb_set_arg    (GtkObject             *object,
					   GtkArg                *arg,
					   guint                  arg_id);
static void gnome_canvas_thumb_get_arg    (GtkObject             *object,
					   GtkArg                *arg,
					   guint                  arg_id);

static void   gnome_canvas_thumb_update      (GnomeCanvasItem *item, 
					      double *affine, 
					      ArtSVP *clip_path, 
					      int flags);
static void   gnome_canvas_thumb_realize     (GnomeCanvasItem *item);
static void   gnome_canvas_thumb_unrealize   (GnomeCanvasItem *item);
static void   gnome_canvas_thumb_draw        (GnomeCanvasItem *item, 
					      GdkDrawable *drawable,
					      int x, 
					      int y, 
					      int width, 
					      int height);
static double gnome_canvas_thumb_point       (GnomeCanvasItem *item, 
					      double x, 
					      double y,
					      int cx, 
					      int cy, 
					      GnomeCanvasItem **actual_item);
static void   gnome_canvas_thumb_translate   (GnomeCanvasItem *item, 
					      double dx, 
					      double dy);
static void   gnome_canvas_thumb_bounds      (GnomeCanvasItem *item, 
					      double *x1, 
					      double *y1, 
					      double *x2, 
					      double *y2);
static void   gnome_canvas_thumb_render      (GnomeCanvasItem *item, 
					      GnomeCanvasBuf *buf);


static GnomeCanvasItemClass *parent_class;


GtkType
gnome_canvas_thumb_get_type (void)
{
	static GtkType image_type = 0;

	if (!image_type) {
		GtkTypeInfo image_info = {
			"GnomeCanvasThumb",
			sizeof (GnomeCanvasThumb),
			sizeof (GnomeCanvasThumbClass),
			(GtkClassInitFunc) gnome_canvas_thumb_class_init,
			(GtkObjectInitFunc) gnome_canvas_thumb_init,
			NULL, /* reserved_1 */
			NULL, /* reserved_2 */
			(GtkClassInitFunc) NULL
		};

		image_type = gtk_type_unique (gnome_canvas_item_get_type (), 
					      &image_info);
	}

	return image_type;
}


static void
gnome_canvas_thumb_class_init (GnomeCanvasThumbClass *class)
{
	GtkObjectClass *object_class;
	GnomeCanvasItemClass *item_class;

	object_class = (GtkObjectClass *) class;
	item_class = (GnomeCanvasItemClass *) class;

	parent_class = gtk_type_class (gnome_canvas_item_get_type ());

	gtk_object_add_arg_type ("GnomeCanvasThumb::pixbuf", GTK_TYPE_POINTER, GTK_ARG_READWRITE, ARG_PIXBUF);
	gtk_object_add_arg_type ("GnomeCanvasThumb::x", GTK_TYPE_DOUBLE, GTK_ARG_READWRITE, ARG_X);
	gtk_object_add_arg_type ("GnomeCanvasThumb::y", GTK_TYPE_DOUBLE, GTK_ARG_READWRITE, ARG_Y);
	gtk_object_add_arg_type ("GnomeCanvasThumb::width", GTK_TYPE_DOUBLE, GTK_ARG_READWRITE, ARG_WIDTH);
	gtk_object_add_arg_type ("GnomeCanvasThumb::height", GTK_TYPE_DOUBLE, GTK_ARG_READWRITE, ARG_HEIGHT);

	object_class->destroy = gnome_canvas_thumb_destroy;
	object_class->set_arg = gnome_canvas_thumb_set_arg;
	object_class->get_arg = gnome_canvas_thumb_get_arg;

	item_class->update = gnome_canvas_thumb_update;
	item_class->realize = gnome_canvas_thumb_realize;
	item_class->unrealize = gnome_canvas_thumb_unrealize;
	item_class->draw = gnome_canvas_thumb_draw;
	item_class->point = gnome_canvas_thumb_point;
	item_class->translate = gnome_canvas_thumb_translate;
	item_class->bounds = gnome_canvas_thumb_bounds;
	item_class->render = gnome_canvas_thumb_render;
}


static void
gnome_canvas_thumb_init (GnomeCanvasThumb *image)
{
	image->pixbuf = NULL;
	image->x = 0.0;
	image->y = 0.0;
	image->width = DEFAULT_WIDTH;
	image->height = DEFAULT_HEIGHT;
}


static void
free_pixmap_and_mask (GnomeCanvasThumb *image)
{
	if (image->pixmap)
		gdk_pixmap_unref (image->pixmap);
	if (image->mask)
		gdk_bitmap_unref (image->mask);

	image->pixmap = NULL;
	image->mask = NULL;
	image->cwidth = 0;
	image->cheight = 0;
}


static void
gnome_canvas_thumb_destroy (GtkObject *object)
{
	GnomeCanvasThumb *image;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_IS_CANVAS_THUMB (object));

	image = GNOME_CANVAS_THUMB (object);

	free_pixmap_and_mask (image);

	if (image->pixbuf)
		gdk_pixbuf_unref (image->pixbuf);

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


static void 
get_bounds (GnomeCanvasThumb *image, 
	    double *px1, double *py1, 
	    double *px2, double *py2)
{
        GnomeCanvasItem *item;
	double i2c[6];
	ArtDRect i_bbox, c_bbox;

        item = GNOME_CANVAS_ITEM (image);

        i_bbox.x0 = image->x;
        i_bbox.y0 = image->y;
        i_bbox.x1 = image->x + image->width;
        i_bbox.y1 = image->y + image->height;

	gnome_canvas_item_i2c_affine (item, i2c);
	art_drect_affine_transform (&c_bbox, &i_bbox, i2c);

	/* add a fudge factor */
	*px1 = c_bbox.x0 - 1;
	*py1 = c_bbox.y0 - 1;
	*px2 = c_bbox.x1 + 1;
	*py2 = c_bbox.y1 + 1;
}


/* deprecated */
static void
recalc_bounds (GnomeCanvasThumb *image)
{
	GnomeCanvasItem *item;

	item = GNOME_CANVAS_ITEM (image);

	get_bounds (image, &item->x1, &item->y1, &item->x2, &item->y2);

	item->x1 = image->cx;
	item->y1 = image->cy;
	item->x2 = image->cx + image->cwidth;
	item->y2 = image->cy + image->cheight;

	gnome_canvas_group_child_bounds (GNOME_CANVAS_GROUP (item->parent), item);
}


static void
gnome_canvas_thumb_set_arg (GtkObject *object, 
			     GtkArg *arg, 
			     guint arg_id)
{
	GnomeCanvasItem *item;
	GnomeCanvasThumb *image;
	gint update;
	gint calc_bounds;
	GdkPixbuf *pixbuf;

	item = GNOME_CANVAS_ITEM (object);
	image = GNOME_CANVAS_THUMB (object);

	update = FALSE;
	calc_bounds = FALSE;

	switch (arg_id) {
	case ARG_PIXBUF:
		pixbuf = GTK_VALUE_POINTER (*arg);
		if (pixbuf != image->pixbuf) {
			if (image->pixbuf)
				gdk_pixbuf_unref (image->pixbuf);
			image->pixbuf = pixbuf;
			gdk_pixbuf_ref (image->pixbuf);
			
			image->iwidth = gdk_pixbuf_get_width (image->pixbuf);
			image->iheight = gdk_pixbuf_get_height (image->pixbuf);

			free_pixmap_and_mask (image);
			gdk_pixbuf_render_pixmap_and_mask (image->pixbuf, 
							   &(image->pixmap), 
							   &(image->mask), 
							   112 /* FIXME */);
			image->need_recalc = TRUE;
			update = TRUE;
		}
		break;

	case ARG_X:
		image->x = GTK_VALUE_DOUBLE (*arg);
		update = TRUE;
		break;

	case ARG_Y:
		image->y = GTK_VALUE_DOUBLE (*arg);
		update = TRUE;
		break;

	case ARG_WIDTH:
		image->width = GTK_VALUE_DOUBLE (*arg);
		update = TRUE;
		break;

	case ARG_HEIGHT:
		image->height = GTK_VALUE_DOUBLE (*arg);
		update = TRUE;
		break;

	default:
		break;
	}

	if (update)
		gnome_canvas_item_request_update (item);
}


static void
gnome_canvas_thumb_get_arg (GtkObject *object, 
			     GtkArg *arg, 
			     guint arg_id)
{
	GnomeCanvasThumb *image;

	image = GNOME_CANVAS_THUMB (object);

	switch (arg_id) {
	case ARG_PIXBUF:
		GTK_VALUE_POINTER (*arg) = image->pixbuf;
		break;

	case ARG_X:
		GTK_VALUE_DOUBLE (*arg) = image->x;
		break;

	case ARG_Y:
		GTK_VALUE_DOUBLE (*arg) = image->y;
		break;

	case ARG_WIDTH:
		GTK_VALUE_DOUBLE (*arg) = image->width;
		break;

	case ARG_HEIGHT:
		GTK_VALUE_DOUBLE (*arg) = image->height;
		break;

	default:
		arg->type = GTK_TYPE_INVALID;
		break;
	}
}


/* Get's the image bounds expressed as item-relative coordinates. */
static void
get_bounds_item_relative (GnomeCanvasThumb *image, 
			  double *px1, double *py1, 
			  double *px2, double *py2)
{
	GnomeCanvasItem *item;
	double x, y;

	item = GNOME_CANVAS_ITEM (image);

	x = image->x;
	y = image->y;

	*px1 = x;
	*py1 = y;
	*px2 = x + image->width;
	*py2 = y + image->height;
}


static void
gnome_canvas_thumb_update (GnomeCanvasItem *item, 
			    double *affine, 
			    ArtSVP *clip_path, 
			    int flags)
{
	GnomeCanvasThumb *image;
	ArtDRect i_bbox, c_bbox;

	image = GNOME_CANVAS_THUMB (item);

	if (parent_class->update)
		(* parent_class->update) (item, affine, clip_path, flags);

	get_bounds_item_relative (image, 
				  &i_bbox.x0, &i_bbox.y0, 
				  &i_bbox.x1, &i_bbox.y1);
	art_drect_affine_transform (&c_bbox, &i_bbox, affine);

	/* these values only make sense in the non-rotated, non-skewed case */
	image->cx = c_bbox.x0;
	image->cy = c_bbox.y0;

        /* only works for non-rotated, non-skewed transforms */
	image->cwidth = (int) (image->width * affine[0] + 0.5);
	image->cheight = (int) (image->height * affine[3] + 0.5);	

	/* add a fudge factor */
	c_bbox.x0--;
	c_bbox.y0--;
	c_bbox.x1++;
	c_bbox.y1++;

	gnome_canvas_update_bbox (item, c_bbox.x0, c_bbox.y0, c_bbox.x1, c_bbox.y1);
}


static void
gnome_canvas_thumb_realize (GnomeCanvasItem *item)
{
	GnomeCanvasThumb *image;
	guint rgb_color;
	GdkColor c;

	image = GNOME_CANVAS_THUMB (item);

	if (parent_class->realize)
		(* parent_class->realize) (item);

	if (!item->canvas->aa) {
		image->gc = gdk_gc_new (item->canvas->layout.bin_window);
		image->bg_gc = gdk_gc_new (item->canvas->layout.bin_window);
	}

	c.red = c.green = c.blue = 0xcc00;
	rgb_color = ((c.red & 0xff00) << 16 |
		     (c.green & 0xff00) << 8 |
		     (c.blue & 0xff00) |
		     (0xff));
	c.pixel = gnome_canvas_get_color_pixel (item->canvas, rgb_color);
	gdk_gc_set_foreground (image->bg_gc, &c);
}


static void
gnome_canvas_thumb_unrealize (GnomeCanvasItem *item)
{
	GnomeCanvasThumb *image;

	image = GNOME_CANVAS_THUMB (item);

	if (!item->canvas->aa) {
		gdk_gc_unref (image->gc);
		image->gc = NULL;

		gdk_gc_unref (image->bg_gc);
		image->bg_gc = NULL;
	}

	if (parent_class->unrealize)
		(* parent_class->unrealize) (item);
}


static void
recalc_if_needed (GnomeCanvasThumb *image)
{
	if (!image->need_recalc)
		return;

	get_bounds (image, &image->item.x1, &image->item.y1, &image->item.x2, &image->item.y2);

	if (image->gc)
		gdk_gc_set_clip_mask (image->gc, image->mask);

	image->need_recalc = FALSE;
}


static void
gnome_canvas_thumb_draw (GnomeCanvasItem *item, GdkDrawable *drawable,
			  int x, int y, int width, int height)
{
	GnomeCanvasThumb *image;
	double i2w[6], w2c[6], i2c[6];
        int x1, y1, x2, y2;
        ArtPoint i1, i2;
        ArtPoint c1, c2;
	GtkStyle *style;

	image = GNOME_CANVAS_THUMB (item);

	if (!image->pixbuf)
		return;

	recalc_if_needed (image);
	
	gnome_canvas_item_i2w_affine (item, i2w);
        gnome_canvas_w2c_affine (item->canvas, w2c);
        art_affine_multiply (i2c, i2w, w2c);

        i1.x = image->cx;
        i1.y = image->cy;
        i2.x = image->cx + image->cwidth;
        i2.y = image->cy + image->cheight;
        art_affine_point (&c1, &i1, i2c);
        art_affine_point (&c2, &i2, i2c);
        x1 = c1.x - x;
        y1 = c1.y - y;
        x2 = x1 + image->cwidth;
        y2 = y1 + image->cheight;

	style = GTK_WIDGET (item->canvas)->style;
	
	if (image->pixmap) {
		gint ix1, iy1;
		gint ix2, iy2;
		
		ix1 = x1 + (image->cwidth - image->iwidth) / 2 + 1;
		iy1 = y1 + (image->cheight - image->iheight) / 2 + 1;

		/* background. */
		gdk_draw_rectangle (drawable,
				    image->selected ? style->bg_gc[GTK_STATE_SELECTED] : image->bg_gc,
				    TRUE,
				    x1,
				    y1,
				    image->cwidth,
				    iy1 - y1);
		gdk_draw_rectangle (drawable,
				    image->selected ? style->bg_gc[GTK_STATE_SELECTED] : image->bg_gc,
				    TRUE,
				    x1,
				    iy1 + image->iheight - 1,
				    image->cwidth,
				    iy1 - y1);
		gdk_draw_rectangle (drawable,
				    image->selected ? style->bg_gc[GTK_STATE_SELECTED] : image->bg_gc,
				    TRUE,
				    x1,
				    y1,
				    ix1 - x1,
				    image->cheight);
		gdk_draw_rectangle (drawable,
				    image->selected ? style->bg_gc[GTK_STATE_SELECTED] : image->bg_gc,
				    TRUE,
				    ix1 + image->iwidth - 1,
				    y1,
				    ix1 - x1,
				    image->cheight);

		/* image. */
		if (image->mask) 
			gdk_gc_set_clip_origin (image->gc, ix1, iy1);

		gdk_draw_pixmap (drawable,
				 image->gc,
				 image->pixmap,
				 0, 0,
				 ix1,
				 iy1,
				 image->iwidth,
				 image->iheight);

		/* Inner border. */
		ix2 = ix1 + image->iwidth;
		iy2 = iy1 + image->iheight;
		ix1--;
		iy1--;

		gdk_draw_line (drawable,
			       style->black_gc,
			       ix1,
			       iy1,
			       ix2,
			       iy1);
		gdk_draw_line (drawable,
			       style->black_gc,
			       ix1,
			       iy1,
			       ix1,
			       iy2);
		gdk_draw_line (drawable,
			       style->white_gc,
			       ix2,
			       iy1,
			       ix2,
			       iy2);
		gdk_draw_line (drawable,
			       style->white_gc,
			       ix1,
			       iy2,
			       ix2,
			       iy2);
	} else
		gdk_draw_rectangle (drawable,
				    image->selected ? style->bg_gc[GTK_STATE_SELECTED] : image->bg_gc,
				    TRUE,
				    x1,
				    y1,
				    image->cwidth,
				    image->cheight);


	/* Outter border. */
	gdk_draw_line (drawable,
		       style->white_gc,
		       x1,
		       y1,
		       x2,
		       y1);
	gdk_draw_line (drawable,
		       style->white_gc,
		       x1,
		       y1,
		       x1,
		       y2);

	gdk_draw_line (drawable,
		       style->black_gc,
		       x2,
		       y1,
		       x2,
		       y2);
	gdk_draw_line (drawable,
		       style->black_gc,
		       x1,
		       y2,
		       x2,
		       y2);
}


static double
gnome_canvas_thumb_point (GnomeCanvasItem *item, 
			   double x, double y,
			   int cx, int cy, 
			   GnomeCanvasItem **actual_item)
{
	GnomeCanvasThumb *image;
	double x1, y1, x2, y2;
	double dx, dy;

	image = GNOME_CANVAS_THUMB (item);

	*actual_item = item;

	recalc_if_needed (image);

	x1 = image->cx - item->canvas->close_enough;
	y1 = image->cy - item->canvas->close_enough;
	x2 = image->cx + image->cwidth - 1 + item->canvas->close_enough;
	y2 = image->cy + image->cheight - 1 + item->canvas->close_enough;

	if ((cx >= x1) && (cy >= y1) && (cx <= x2) && (cy <= y2)) 
		return 0.0; 

	/* Point is outside rectangle */

	if (x < x1)
		dx = x1 - x;
	else if (x > x2)
		dx = x - x2;
	else
		dx = 0.0;

	if (y < y1)
		dy = y1 - y;
	else if (y > y2)
		dy = y - y2;
	else
		dy = 0.0;

	return sqrt (dx * dx + dy * dy);
}


static void
gnome_canvas_thumb_translate (GnomeCanvasItem *item, double dx, double dy)
{
	GnomeCanvasThumb *image;

	image = GNOME_CANVAS_THUMB (item);

	image->x += dx;
	image->y += dy;

	recalc_bounds (image);
}


static void
gnome_canvas_thumb_bounds (GnomeCanvasItem *item, 
			    double *x1, double *y1, 
			    double *x2, double *y2)
{
	GnomeCanvasThumb *image;

	image = GNOME_CANVAS_THUMB (item);

	*x1 = image->x;
	*y1 = image->y;

	*x2 = *x1 + image->width;
	*y2 = *y1 + image->height;
}


static void
gnome_canvas_thumb_render (GnomeCanvasItem *item, GnomeCanvasBuf *buf)
{
}


void
gnome_canvas_thumb_select (GnomeCanvasThumb *image,
			    gboolean select)
{
	g_return_if_fail (image != NULL);
	g_return_if_fail (GNOME_IS_CANVAS_THUMB (image));

	if (image->selected == select)
		return;

	image->selected = select;
	gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (image));
}
