//*****************************************************************************
//                              CmdNgSpiceOPT.cpp                             *
//                             -------------------                            *
//  Started     : 23/08/2006                                                  *
//  Last Update : 14/01/2008                                                  *
//  Copyright   : (C) 2006 by M.S.Waters                                      *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "ngspice/commands/CmdNgSpiceOPT.hpp"

//*****************************************************************************
// Constructor.

CmdNgSpiceOPT::CmdNgSpiceOPT( void )
{
  bClear( );
}

//*****************************************************************************
// Destructor.

CmdNgSpiceOPT::~CmdNgSpiceOPT( )
{
}

//*****************************************************************************
// Parse the command string.
//
// Eg.s : .OPTIONS NOPAGE
//        .OPTIONS NOPAGE ABSTOL=1.10E-12 BADMOS3 CHGTOL=1.10E-14
//                 DEFAD=1.00E-14 DEFAS=1.00E-14 DEFL=1.10E-04 DEFW=1.10E-04
//                 GMIN=1.10E-12 ITL1=110 ITL2=60 ITL4=20 METHOD=GEAR
//                 PIVREL=0.0020 PIVTOL=1.10E-13 RELTOL=0.0020 TEMP=30.00
//                 TNOM=30.00 TRTOL=8.00E+00 VNTOL=2.00E-06
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpiceOPT::bParse( void )
{
  wxStringTokenizer  ostk1;
  wxString           os1, os2;
  size_t             sz1;
  long               li1;
  double             df1;

  // Clear the object attributes
  os1 = *this;
  bClear( );
  *this = os1;

  // Tokenize the command string
  ostk1.SetString( *this );
  if( ostk1.CountTokens( ) < 2 ) return( FALSE );

  // Check command type
  os1 = ostk1.GetNextToken( ).Left( 8 ).Upper( );
  if( os1 != wxT(".OPTIONS") )   return( FALSE );
  m_osName = wxT("OPTIONS");

  m_bIsOk = TRUE;

  // Extract each parameter value
  while( ostk1.HasMoreTokens( ) )
  {
    // Extract the field name and the associated value
    os1 = ostk1.GetNextToken( );
    os2 = wxT("");
    if( (sz1=os1.find( wxT("=") )) != wxString::npos )
    {
      os2 = os1.Right( os1.Length( )-sz1-1 );
      os1 = os1.Left( sz1 );
      ConvertType::bStrToInt( os2, &li1 );
      ConvertType::bStrToFlt( os2, &df1 );
    }

    // Set the display control value
    if(      os1 == wxT("ABSTOL")  ) m_fABSTOL  = (float) df1;
    else if( os1 == wxT("BADMOS3") ) m_bBADMOS3 = TRUE;
    else if( os1 == wxT("CHGTOL")  ) m_fCHGTOL  = (float) df1;
    else if( os1 == wxT("DEFAD")   ) m_fDEFAD   = (float) df1;
    else if( os1 == wxT("DEFAS")   ) m_fDEFAS   = (float) df1;
    else if( os1 == wxT("DEFL")    ) m_fDEFL    = (float) df1;
    else if( os1 == wxT("DEFW")    ) m_fDEFW    = (float) df1;
    else if( os1 == wxT("GMIN")    ) m_fGMIN    = (float) df1;
    else if( os1 == wxT("ITL1")    ) m_iITL1    = (int)   li1;
    else if( os1 == wxT("ITL2")    ) m_iITL2    = (int)   li1;
    else if( os1 == wxT("ITL4")    ) m_iITL4    = (int)   li1;
    else if( os1 == wxT("METHOD")  ) m_osMETHOD = os2;
    else if( os1 == wxT("PIVREL")  ) m_fPIVREL  = (float) df1;
    else if( os1 == wxT("PIVTOL")  ) m_fPIVTOL  = (float) df1;
    else if( os1 == wxT("RELTOL")  ) m_fRELTOL  = (float) df1;
    else if( os1 == wxT("TEMP")    ) m_fTEMP    = (float) df1;
    else if( os1 == wxT("TNOM")    ) m_fTNOM    = (float) df1;
    else if( os1 == wxT("TRTOL")   ) m_fTRTOL   = (float) df1;
    else if( os1 == wxT("VNTOL")   ) m_fVNTOL   = (float) df1;
    else m_bIsOk = FALSE;
  }

  m_bIsOk = TRUE;

  return( m_bIsOk );
}

//*****************************************************************************
// Format the command string.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpiceOPT::bFormat( void )
{
  wxString  os1;

  m_bIsOk = FALSE;

  os1 = wxT(".OPTIONS NOPAGE");

  if( m_fABSTOL  != NG_ABSTOL )
    os1 << wxT(" ABSTOL=") << wxString::Format( wxT("%.2E"), m_fABSTOL );
  if( m_bBADMOS3 != NG_BADMOS3 )
    os1 << wxT(" BADMOS3");
  if( m_fCHGTOL  != NG_CHGTOL )
    os1 << wxT(" CHGTOL=") << wxString::Format( wxT("%.2E"), m_fCHGTOL );
  if( m_fDEFAD   != NG_DEFAD )
    os1 << wxT(" DEFAD=")  << wxString::Format( wxT("%.2E"), m_fDEFAD );
  if( m_fDEFAS   != NG_DEFAS )
    os1 << wxT(" DEFAS=")  << wxString::Format( wxT("%.2E"), m_fDEFAS );
  if( m_fDEFL    != NG_DEFL )
    os1 << wxT(" DEFL=")   << wxString::Format( wxT("%.2E"), m_fDEFL );
  if( m_fDEFW    != NG_DEFW )
    os1 << wxT(" DEFW=")   << wxString::Format( wxT("%.2E"), m_fDEFW );
  if( m_fGMIN    != NG_GMIN )
    os1 << wxT(" GMIN=")   << wxString::Format( wxT("%.2E"), m_fGMIN );
  if( m_iITL1    != NG_ITL1 )
    os1 << wxT(" ITL1=")   << m_iITL1;
  if( m_iITL2    != NG_ITL2 )
    os1 << wxT(" ITL2=")   << m_iITL2;
  if( m_iITL4    != NG_ITL4 )
    os1 << wxT(" ITL4=")   << m_iITL4;
  if( m_osMETHOD != NG_METHOD )
    os1 << wxT(" METHOD=") << m_osMETHOD.Upper( );
  if( m_fPIVREL  != NG_PIVREL )
    os1 << wxT(" PIVREL=") << wxString::Format( wxT("%.4f"), m_fPIVREL );
  if( m_fPIVTOL  != NG_PIVTOL )
    os1 << wxT(" PIVTOL=") << wxString::Format( wxT("%.2E"), m_fPIVTOL );
  if( m_fRELTOL  != NG_RELTOL )
    os1 << wxT(" RELTOL=") << wxString::Format( wxT("%.4f"), m_fRELTOL );
  if( m_fTEMP    != NG_TEMP )
    os1 << wxT(" TEMP=")   << wxString::Format( wxT("%.2f"), m_fTEMP );
  if( m_fTNOM    != NG_TNOM )
    os1 << wxT(" TNOM=")   << wxString::Format( wxT("%.2f"), m_fTNOM );
  if( m_fTRTOL   != NG_TRTOL )
    os1 << wxT(" TRTOL=")  << wxString::Format( wxT("%.2E"), m_fTRTOL );
  if( m_fVNTOL   != NG_VNTOL )
    os1 << wxT(" VNTOL=")  << wxString::Format( wxT("%.2E"), m_fVNTOL );

  *this = os1;

  m_bIsOk = TRUE;

  return( m_bIsOk );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpiceOPT::bClear( void )
{
  CmdBase::bClear( );

  m_fABSTOL  = NG_ABSTOL;
  m_bBADMOS3 = NG_BADMOS3;
  m_fCHGTOL  = NG_CHGTOL;
  m_fDEFAD   = NG_DEFAD;
  m_fDEFAS   = NG_DEFAS;
  m_fDEFL    = NG_DEFL;
  m_fDEFW    = NG_DEFW;
  m_fGMIN    = NG_GMIN;
  m_iITL1    = NG_ITL1;
  m_iITL2    = NG_ITL2;
  m_iITL4    = NG_ITL4;
  m_osMETHOD = NG_METHOD;
  m_fPIVREL  = NG_PIVREL;
  m_fPIVTOL  = NG_PIVTOL;
  m_fRELTOL  = NG_RELTOL;
  m_fTEMP    = NG_TEMP;
  m_fTNOM    = NG_TNOM;
  m_fTRTOL   = NG_TRTOL;
  m_fVNTOL   = NG_VNTOL;

  return( TRUE );
}

//*****************************************************************************
//                                                                            *
//                                 Test Utility                               *
//                                                                            *
//*****************************************************************************

#ifdef TEST_UTIL

// System include files


// Application includes


// Function prototypes

void  Usage( char * psAppName );

//*****************************************************************************

int  main( int argc, char * argv[ ] )
{
  wxString  osCmd;
  wxString  os1;

  // Validate the argument count passed to the application
  if( argc > 2 )           { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }

  // Process the command line arguments
  os1 = wxConvLibc.cMB2WC( argv[ 1 ] );
  if( argc > 1 )
  {
    if( os1 == wxT("-h") ) { Usage( argv[ 0 ] ); exit( EXIT_SUCCESS ); }
    else                   { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }
  }

  // Display the utility banner
  cout << "\n  Class CmdNgSpiceOPT Test Utility"
       << "\n     Version 1.00 (04/02/2008)\n";

  // Create a NG-SPICE OPTIONS command object
  CmdNgSpiceOPT  oCmd_OPT;

  // Use the following command example to check the formatter and the parser :
  osCmd << wxT(".OPTIONS NOPAGE ABSTOL=1.10E-12 BADMOS3 CHGTOL=1.10E-14 ")
        << wxT("DEFAD=1.00E-14 DEFAS=1.00E-14 DEFL=1.10E-04 DEFW=1.10E-04 ")
        << wxT("GMIN=1.10E-12 ITL1=110 ITL2=60 ITL4=20 METHOD=GEAR ")
        << wxT("PIVREL=0.0020 PIVTOL=1.10E-13 RELTOL=0.0020 TEMP=30.00 ")
        << wxT("TNOM=30.00 TRTOL=8.00E+00 VNTOL=2.00E-06");

  // Set things up for a formatter test
  oCmd_OPT.bClear( );
  oCmd_OPT.m_fABSTOL  = 1.1E-12;
  oCmd_OPT.m_bBADMOS3 = TRUE;
  oCmd_OPT.m_fCHGTOL  = 1.1E-14;
  oCmd_OPT.m_fDEFAD   = 1.0E-14;
  oCmd_OPT.m_fDEFAS   = 1.0E-14;
  oCmd_OPT.m_fDEFL    = 1.1E-04;
  oCmd_OPT.m_fDEFW    = 1.1E-04;
  oCmd_OPT.m_fGMIN    = 1.1E-12;
  oCmd_OPT.m_iITL1    = 110;
  oCmd_OPT.m_iITL2    = 60;
  oCmd_OPT.m_iITL4    = 20;
  oCmd_OPT.m_osMETHOD = wxT("GEAR");
  oCmd_OPT.m_fPIVREL  = 2.0E-03;
  oCmd_OPT.m_fPIVTOL  = 1.1E-13;
  oCmd_OPT.m_fRELTOL  = 2.0E-03;
  oCmd_OPT.m_fTEMP    = 30.0;
  oCmd_OPT.m_fTNOM    = 30.0;
  oCmd_OPT.m_fTRTOL   = 8.0;
  oCmd_OPT.m_fVNTOL   = 2.0E-06;
  cout << "\nRun Formatter     : " << ( oCmd_OPT.bFormat( ) ? "OK" : "FAULT" );
  cout << "\nTest Cmd Format   : " << ( oCmd_OPT == osCmd   ? "OK" : "FAULT" );
  cout << "\nExample Command   : " << osCmd   .mb_str( );
  cout << "\noCmd_OPT Contents : " << oCmd_OPT.mb_str( ) << '\n';

  // Set things up for a parser test
  oCmd_OPT.bClear( );
  oCmd_OPT = osCmd;
  cout << "\nRun Parser        : " << ( oCmd_OPT.bParse( ) ? "OK" : "FAULT" );
  oCmd_OPT.bFormat( );
  cout << "\nTest Cmd Format   : " << ( oCmd_OPT == osCmd  ? "OK" : "FAULT" );
  cout << "\nExample Command   : " << osCmd   .mb_str( );
  cout << "\noCmd_OPT Contents : " << oCmd_OPT.mb_str( ) << '\n';

  cout << '\n';

  exit( EXIT_SUCCESS );
}

//*****************************************************************************

void  Usage( char * psAppName )
{
  cout << "\nUsage   : " << psAppName << " [-OPTIONS]"
       << "\nOptions :"
       << "\n  -h : Print usage (this message)\n";
}

#endif // TEST_UTIL

//*****************************************************************************
