//*****************************************************************************
//                                Simulation.hpp                              *
//                               ----------------                             *
//                                                                            *
//  Description : A class to contain the values required to define a          *
//                particular simulation.                                      *
//  Started     : 23/09/2003                                                  *
//  Last Update : 29/02/2008                                                  *
//  Copyright   : (C) 2003 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#ifndef SIMULATION_HPP
#define SIMULATION_HPP

// System Includes

#include <cfloat>

// wxWindows Includes

#include <wx/wx.h>

// Application Includes

#include "TypeDefs.hpp"
#include "netlist/NetList.hpp"
#include "ngspice/commands/CmdNgSpiceOPT.hpp"
#include "utility/ConvertType.hpp"

// Local Constant Declarations


//*****************************************************************************

class Simulation : public NetList
{
  private:

    // The sweep scale specifier
    enum eSwpType
    {
      eSWP_START = 0,           // Sweep start point
      eSWP_STOP,                // Sweep stop  point
      eSWP_STEP,                // Sweep step size
      eSWP_SCALE,               // Sweep step scale type
      eSWP_SIZE = eSWP_SCALE+1  // Size of array to hold the sweep parameters
    };

    // Attributes to define the simulation
    eSimrType      m_eSimrType;          // Simulator engine type
    eAnaType       m_eAnaType;           // Analysis type
    float          m_fSwp[ eSWP_SIZE ];  // Array to hold sweep parameter values
    bool           m_bPar[ ePAR_LST+1 ]; // Output parameters
    bool           m_bCpx[ eCPX_LST+1 ]; // Output complex parts (AC analysis only)
    float          m_fTempC;             // Analysis temperature in Celcius
    wxArrayString  m_oasTstNodes;        // Array to hold test node names
    wxArrayString  m_oasTstCpnts;        // Array to hold test component names
    wxArrayString  m_oasSimCmds;         // Array to hold simulation commands
    wxString       m_osSigSrc[ 2 ];      // String to hold signal source
    wxString       m_osOPTIONS;          // The OPTIONS command

    // Error message generated by bIsValid( )
    wxString  m_osErrMsg;

    // Function to extract information from the circuit description
    bool  bExtractSimrType( void );
    bool  bExtractSimCmds ( void );
    bool  bExtractSigSrc  ( void );

  public:

    Simulation( void );
   ~Simulation( );

    bool  bClear( void );
    bool  bIsValid( void );

    bool  bLoadFile( const wxString & rosFName=wxT("") );
    bool  bSaveFile( const wxString & rosFName=wxT("") );

    bool  bSetSimrType( eSimrType eSimr );
    bool  bSetAnaType ( eAnaType eAna );
    bool  bSetSwpStart( float fStart );
    bool  bSetSwpStop ( float fStop );
    bool  bSetSwpStep ( float fStep );
    bool  bSetSwpScale( int iScale );
    bool  bSetOutPar  ( eParType ePar, bool bState );
    bool  bSetOutCpx  ( eCpxType eCpx, bool bState );
    bool  bSetTempC   ( float fTempC );
    bool  bSetOPTIONS ( const wxString & rosCmd )
                                      { m_osOPTIONS = rosCmd; return( TRUE ); }

    eSimrType    eGetSimrType( void )        { return( m_eSimrType ); }
    eAnaType     eGetAnaType( void )         { return( m_eAnaType ); }
    float        fGetSwpStart( void )        { return( m_fSwp[ eSWP_START ] ); }
    float        fGetSwpStop ( void )        { return( m_fSwp[ eSWP_STOP  ] ); }
    float        fGetSwpStep ( void )        { return( m_fSwp[ eSWP_STEP  ] ); }
    int          iGetSwpScale( void )        { return( (int) m_fSwp[ eSWP_SCALE ] ); }
    bool         bGetOutPar( eParType ePar ) { return( m_bPar[ ePar ] ); }
    bool         bGetOutCpx( eCpxType eCpx ) { return( m_bCpx[ eCpx ] ); }
    float        fGetTempC   ( void )        { return( m_fTempC ); }
    wxString & rosGetOPTIONS ( void )        { return( m_osOPTIONS ); }

    const wxString & rosGetSwpStart( void );
    const wxString & rosGetSwpStop ( void );
    const wxString & rosGetSwpStep ( void );
    const wxString & rosGetTempC   ( void );

    bool  bAddTstNode( const wxString & rosNode );
    bool  bAddTstCpnt( const wxString & rosCpnt );
    bool  bAddSimCmd ( const wxString & rosCmd );
    bool  bSetSigSrc ( const wxString & rosCpnt );

    const wxArrayString & roasGetTstNodes( void ) { return( m_oasTstNodes );   }
    const wxArrayString & roasGetTstCpnts( void ) { return( m_oasTstCpnts );   }
    const wxArrayString & roasGetSimCmds ( void ) { return( m_oasSimCmds );    }
    const wxString      &  rosGetSigSrc  ( void ) { return( m_osSigSrc[ 0 ] ); }

    void  ClrTstNodes( void ) { m_oasTstNodes.Empty( ); }
    void  ClrTstCpnts( void ) { m_oasTstCpnts.Empty( ); }
    void  ClrSimCmds ( void ) { m_oasSimCmds .Empty( ); }
    void  ClrSigSrc  ( void );

    const wxString & rosGetErrMsg( void ) { return( m_osErrMsg ); }

    void  Print( const char * psPrefix="" );
};

//*****************************************************************************

#endif // SIMULATION_HPP
