#include "msgface.h"
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <ctime>

#define USE_NO_ACK

#ifdef GRUB_UNIX
/* select(), recv() includes */
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#endif

#define HEADER_SIZE 32
#define SETERROR(msg) \
	do { \
		gmsg_set_strerror( grub_info.errstr, msg ); \
	} while (0)

#define RESETERROR() gmsg_set_strerror( grub_info.errstr, "No error" )

static int gmsg_read( SOCKET sock, unsigned int msec,
	char *buf, unsigned int len );
static int gmsg_write( SOCKET sock, unsigned int msec,
	const char *buf, unsigned int len );
static int gmsg_write_header( SOCKET sock, unsigned int msec,
	GMSG_ACK_T send_ackdata, GMSG_SIZE_T send_size );
static int gmsg_read_header( SOCKET sock, unsigned int msec,
	GMSG_ACK_T *recv_ackdata, GMSG_SIZE_T *recv_size );
static void gmsg_set_strerror( char *errstr, const char *msg );

grub_t grub_info = {0};

void gmsg_init( SOCKET s )
{
	memset( &grub_info, 0, sizeof(grub_t) );
	grub_info.is_init = 1;
	grub_info.sock = s;
	grub_info.io_timeout = DEFAULT_TIMEOUT;
}

void gmsg_set_timeout( unsigned int timeout )
{
	grub_info.io_timeout = timeout;
}

void gmsg_uninit()
{
	memset( &grub_info, 0, sizeof(grub_t) );
}

/* ret = GF_SUCCESS, GF_NOINIT, GF_SOCKERR, GF_NOTREADY,
 *       GF_INTRNERR, GF_BADDATA */
/* will mark ready on error */
/* ifdef USE_NO_ACK, ret = GF_SUCCESS, GF_NOINIT */
int gmsg_send_ready()
{
	int ret = GF_SUCCESS;
#ifndef USE_NO_ACK
	int fret;
#endif
	RESETERROR();

	if ( ! grub_info.is_init ) {

		SETERROR("grub_info not initialized");
		return GF_NOINIT;
	}

#ifndef USE_NO_ACK

	/* it's ready for sending */
	if ( ! grub_info.send_notready ) return GF_SUCCESS;

	/* fret = GF_SUCCESS, GF_NOTREADY, GF_INTRNERR */
	fret = gmsg_socket_ready( grub_info.sock, false, 0 );
	if ( fret == GF_NOTREADY ) {

		SETERROR("Timeout while waiting for send ACK");
		return GF_NOTREADY;
	}
	else if ( fret == GF_INTRNERR ) {

		grub_info.send_notready = 0;
		SETERROR("Internal error: select() failed");
		return GF_INTRNERR;
	}
	else {

		/* GF_SUCCESS -- socket has the ACK */

		GMSG_ACK_T send_ackdata;
		GMSG_SIZE_T send_size;

		/* set it to ready even if error occurs */
		grub_info.send_notready = 0;

		/* fret = GF_SUCCESS, GF_SOCKERR, GF_NOTREADY, GF_BADDATA */
		ret = gmsg_read_header( grub_info.sock, grub_info.io_timeout,
			&send_ackdata, &send_size );
		if ( ret == GF_SUCCESS ) {
			if ( grub_info.send_ackdata != send_ackdata ||
				grub_info.send_size != send_size )
			{
				/* incorrect values returned */
				SETERROR("Send ACK returned, "
					"but with incorrect values");
				ret = GF_BADDATA;
			}
		}
		else {

			switch (ret) {
			case GF_SOCKERR:
				SETERROR("Socket input error "
					"while receiving ACK data header");
				break;
			case GF_NOTREADY:
				SETERROR("Timeout while waiting to receive ACK");
				break;
			case GF_BADDATA:
				SETERROR("Invalid data received in the ACK");
				break;
			}
		}
	}
#endif  /* ! USE_NO_ACK */

	return ret;
}

/* ret = GF_SUCCESS, GF_NOINIT, GF_SOCKERR, GF_NOTREADY,
 *       GF_INTRNERR, GF_BADDATA */
/* ifdef USE_NO_ACK, ret = same as above */
int gmsg_send( const char *msg, unsigned int len )
{
	int fret;

	RESETERROR();

	if ( ! grub_info.is_init || ! msg )	{

		SETERROR("grub_info not initialized");
		return GF_NOINIT;
	}

	/* fret = GF_SUCCESS, GF_SOCKERR, GF_NOINIT, GF_NOTREADY,
	 *        GF_INTRNERR, GF_BADDATA */
	fret = gmsg_send_ready();
	if ( fret != GF_SUCCESS )
		return fret;  /* error msg already set */

	if ( grub_info.send_ackdata == 0 )
		srand(time(NULL));

	/* fret = GF_SUCCESS, GF_SOCKERR, GF_NOTREADY */
	fret = gmsg_write_header(
		grub_info.sock,
		grub_info.io_timeout,
		(grub_info.send_ackdata = (GMSG_ACK_T)rand() ),
		(grub_info.send_size = (GMSG_SIZE_T)len )
	);
	if ( fret != GF_SUCCESS ) {

		switch ( fret ) {
		case GF_SOCKERR:
			SETERROR("Socket output error while sending data "
				"header");
			break;
		case GF_NOTREADY:
			SETERROR("Operation timed out while trying to send "
				"data header");
			break;
		}  /* end switch */
		return fret;
	}

	/* fret = GF_SUCCESS, GF_SOCKERR, GF_NOTREADY */
	fret = gmsg_write(
			grub_info.sock,
			grub_info.io_timeout,
			msg,
			len
		);
	switch ( fret ) {
	case GF_SUCCESS:
		/* set it to "not ready" once we send the message.  Next
		 * step in sending is to receive ACK from the other end
		 * first */
		grub_info.send_notready = 1;
		break;
	case GF_SOCKERR:
		SETERROR("Socket output error while sending data");
		break;
	case GF_NOTREADY:
		SETERROR("Operation timed out while trying to send data");
		break;
	}  /* end switch */

	return fret;
}

/* ret = bytes recv > 0, GF_NOINIT, GF_NOTREADY, GF_INTRNERR,
 *       GF_BADDATA, GF_SOCKERR */
int gmsg_recv_available()
{
	int ret = GF_SUCCESS;

	RESETERROR();

	if ( ! grub_info.is_init ) {

		SETERROR("grub_info not initialized");
		return GF_NOINIT;
	}

	if ( ! grub_info.recv_available ) {
		int fret;

		/* fret = GF_SUCCESS, GF_NOTREADY, GF_INTRNERR */
		fret = gmsg_socket_ready( grub_info.sock, false, 0 );
		if ( fret == GF_NOTREADY ) {

			SETERROR("Nothing available: "
				"Timeout while waiting for receive header");
			return GF_NOTREADY;
		}
		else if ( fret == GF_INTRNERR ) {

			SETERROR("Internal error: select() failed");
			return GF_INTRNERR;
		}
		else {

			/* ret = GF_SUCCESS, GF_SOCKERR, GF_NOTREADY,
			 *       GF_BADDATA */
			ret = gmsg_read_header(
				grub_info.sock,
				grub_info.io_timeout,
				&grub_info.recv_ackdata,
				&grub_info.recv_size
			);
			if ( ret == GF_SUCCESS ) {

				if ( grub_info.recv_size > 0 )
					grub_info.recv_available = 1;
				else {

					/* data must have at least one byte */

					SETERROR("Zero byte data size received"
						"in the header: "
						"data must be at least 1 byte");
					ret = GF_BADDATA;
				}
			}
			else {

				switch (ret) {
				case GF_SOCKERR:
					SETERROR("Socket input error while "
						"receiving a data header");
					break;
				case GF_NOTREADY:
					SETERROR("Timeout while waiting "
						"to receive a data header");
					break;
				case GF_BADDATA:
					SETERROR("Invalid data received "
						"in the data header");
					break;
				} /* end switch */
			}
		}
	}

	return ret == GF_SUCCESS ? grub_info.recv_size : ret;
}

/* ret = GF_SUCCESS, GF_NOINIT, GF_NOTREADY, GF_INTRNERR,
 *       GF_BADDATA, GF_SOCKERR */
/* ifdef USE_NO_ACK, ret = same as above */
/* buf must be at least return value of gmsg_recv_available() + 1 long */
/* will store NULL at end of string */
int gmsg_recv( char *buf )
{
	int ret = GF_SUCCESS;

	RESETERROR();

	if ( ! grub_info.is_init || ! buf ) {

		SETERROR("grub_info not initialized");
		return GF_NOINIT;
	}

	/* ret = bytes recv, GF_NOINIT, GF_NOTREADY, GF_INTRNERR,
	 *       GF_BADDATA */
	ret = gmsg_recv_available();  /* error msg already set if error */
	if ( ret > 0 ) {
		int fret;

		/* fret = GF_SUCCESS, GF_SOCKERR, GF_NOTREADY */
		fret = gmsg_read(
			grub_info.sock,
			grub_info.io_timeout,
			buf,
			(unsigned int)ret
		);
		if ( fret == GF_SUCCESS )
			buf[ret] = '\0';

		ret = fret;


		if ( ret == GF_SUCCESS ) {

#ifndef USE_NO_ACK

			/* send header as acknowledgement */

			/* ret = GF_SUCCESS, GF_SOCKERR, GF_NOTREADY */
			ret = gmsg_write_header(
				grub_info.sock,
				grub_info.io_timeout,
				grub_info.recv_ackdata,
				grub_info.recv_size
			);
			if ( ret != GF_SUCCESS ) {

				switch ( fret ) {
				case GF_SOCKERR:
					SETERROR("Socket output error "
						"while sending ACK header");
					break;
				case GF_NOTREADY:
					SETERROR("Operation timed out while "
						"trying to send ACK header");
					break;
				}  /* end switch */
			}
#endif  /* ! USE_NO_ACK */
		}
		else {

			/* gmsg_read() returned error */

			switch (ret) {
			case GF_SOCKERR:
				SETERROR("Socket input error "
					"while receiving reading data");
				break;
			case GF_NOTREADY:
				SETERROR("Timeout while waiting to receive data");
				break;
			}
		}
	}

	grub_info.recv_available = 0;

	return ret;
}

const char *gmsg_strerror()
{
	return grub_info.errstr;
}

// static functions
// they do not access directly the grub_info struct variable

/* write_sock should be true for write, false for read; msec is timeout
 * in miliseconds, 0 to see if socket is immediately available
 * ret = GF_SUCCESS, GF_NOTREADY, GF_INTRNERR
 */
int gmsg_socket_ready( SOCKET sock, bool write_sock,
	unsigned int msec )
{
	int ret = GF_SUCCESS;
	int fret, fdp1;
	fd_set the_set;
	fd_set *read_set_ptr = (fd_set *)0, *write_set_ptr = (fd_set *)0;
	struct timeval timeout;

	if ( write_sock )
		write_set_ptr = &the_set;
	else
		read_set_ptr = &the_set;

	FD_ZERO( &the_set );
	FD_SET( sock, &the_set );
	fdp1 = sock + 1;

	timeout.tv_sec = msec / 1000;
	timeout.tv_usec = msec * 1000;

	fret = select( fdp1, read_set_ptr, write_set_ptr, (fd_set *)0,
		&timeout );
	if ( fret > 0 ) {

		/* sock is ready for input/output */
		ret = GF_SUCCESS;
	}
	else if ( fret == 0 ) {

		/* timeout occured -- socket not ready */
		ret = GF_NOTREADY;
	}
	else
		ret = GF_INTRNERR;  /* select() failed */

	return ret;
}

/* ret = GF_SUCCESS, GF_SOCKERR, GF_NOTREADY */
static int gmsg_read( SOCKET sock, unsigned int msec,
	char *buf, unsigned int len )
{
	unsigned int nread = 0;

	while ( nread < len ) {
		int fret;

		/* fret = GF_SUCCESS, GF_NOTREADY, GF_INTRNERR */
		fret = gmsg_socket_ready( sock, false, msec );
		if ( fret == GF_NOTREADY )
			return GF_NOTREADY;
		else if ( fret != GF_SUCCESS )
			return GF_SOCKERR;

		fret = recv( sock, buf + nread, len - nread, 0 );
		if ( fret <= 0 )
			return GF_SOCKERR;

		nread += fret;
	}

	return GF_SUCCESS;
}

/* ret = GF_SUCCESS, GF_SOCKERR, GF_NOTREADY */
static int gmsg_write( SOCKET sock, unsigned int msec,
	const char *buf, unsigned int len )
{
	unsigned int nwrite = 0;

	while ( nwrite < len ) {
		int fret;

		/* fret = GF_SUCCESS, GF_NOTREADY, GF_INTRNERR */
		fret = gmsg_socket_ready( sock, true, msec );
		if ( fret == GF_NOTREADY )
			return GF_NOTREADY;
		else if ( fret != GF_SUCCESS )
			return GF_SOCKERR;

		fret = send( sock, buf + nwrite, len - nwrite, 0 );
		if ( fret <= 0 )
			return GF_SOCKERR;

		nwrite += fret;
	}

	return GF_SUCCESS;
}

/* ret = GF_SUCCESS, GF_SOCKERR, GF_NOTREADY */
static int gmsg_write_header( SOCKET sock, unsigned int msec,
	GMSG_ACK_T send_ackdata, GMSG_SIZE_T send_size )
{
	char hdr_block[HEADER_SIZE + 1];
	int str_len;

	sprintf( hdr_block, "%u %u", send_ackdata, send_size );
	str_len = strlen(hdr_block);
	memset( hdr_block + str_len, ' ', HEADER_SIZE - str_len );
	hdr_block[HEADER_SIZE] = '\0';

	return gmsg_write( sock, msec, hdr_block, HEADER_SIZE );
}

/* ret = GF_SUCCESS, GF_SOCKERR, GF_NOTREADY, GF_BADDATA */
static int gmsg_read_header( SOCKET sock, unsigned int msec,
	GMSG_ACK_T *recv_ackdata, GMSG_SIZE_T *recv_size )
{
	char hdr_block[HEADER_SIZE + 1];
	int ret;

	/* ret = GF_SUCCESS, GF_SOCKERR, GF_NOTREADY */
	ret = gmsg_read( sock, msec, hdr_block, HEADER_SIZE );
	if ( ret == GF_SUCCESS ) {

		hdr_block[HEADER_SIZE] = '\0';
		if ( sscanf( hdr_block, "%u %u", recv_ackdata, recv_size )
			!= 2 )
		{
			ret = GF_BADDATA;
		}
	}

	return ret;
}

static void gmsg_set_strerror( char *errstr, const char *msg )
{
	strncpy( errstr, msg, GMSG_ERRBUFSIZE );
	errstr[GMSG_ERRBUFSIZE - 1] = '\0';
}
