#include "XmlWrapper.h"
#include <cstdio>
#include <cstring>
#include <cctype>

void xmlw_startElement(void *ptr, const char *name, const char **atts);
void xmlw_endElement(void *ptr, const char *name );
void xmlw_dataHandler(void *ptr, const char *data, int data_len );

XmlWrapper::XmlWrapper() : page(0), page_len(0), doc_depth(0), tagmsgs_len(0),
	indent_char('\t'),
#ifdef HAVE_LIBEXPAT
	xml_parser(0),
#endif
	error_msg(0), error_line(-1)
{
	resetData();
	resetTags();
	resetMsg();
}

XmlWrapper::~XmlWrapper()
{
	if ( page ) delete [] page;
	resetMsg();
#ifdef HAVE_LIBEXPAT
	if ( xml_parser ) XML_ParserFree(xml_parser);
#endif
	if ( error_msg ) delete [] error_msg;
}

void XmlWrapper::setIndentChar( char new_char )
{
	indent_char = new_char;
}

const char *XmlWrapper::getData()
{
	return page;
}

void XmlWrapper::resetData()
{
	if ( page ) delete [] page;

	page = new char[1];
	*page = '\0';

	page_len = 0;
	doc_depth = 0;

	resetTags();
	resetMsg();
}

const char *XmlWrapper::lastErrorMsg()
{
	return error_msg;
}

int XmlWrapper::lastErrorLineNumber()
{
	return error_line;
}

// creating an XML message

void XmlWrapper::addSimpleHeader()
{
	appendString("<?xml version=\"1.0\"?>\n\n");
}

void XmlWrapper::openElement( const char *el_name )
{
	appendIndent();
	appendString("<");
	appendString(el_name);
	appendString(">\n");
	doc_depth++;
}

void XmlWrapper::closeElement( const char *el_name )
{
	doc_depth--;
	appendIndent();
	appendString("</");
	appendString(el_name);
	appendString(">\n");
}

void XmlWrapper::addElementString( const char *el_name, const char *str )
{
	appendTagPair( el_name, str );
}

void XmlWrapper::addElementInt( const char *el_name, int value )
{
	char buf[100];

	sprintf( buf, "%d", value );
	appendTagPair( el_name, buf );
}

void XmlWrapper::addElementDouble( const char *el_name, double value )
{
	char buf[100];

	sprintf( buf, "%lf", value );
	appendTagPair( el_name, buf );
}

// parsing an XML message

// return 0, -1 on XML file error
// call lastErrorMsg() and lastErrorLineNumber() to get the reason on error
int XmlWrapper::parseData( const char *data )
{
	int ret = 0;

	resetData();

	/* will return -1 if no support from expat */
	ASSERT_HAVE_EXPAT();

#ifdef HAVE_LIBEXPAT
	if ( xml_parser )
		XML_ParserFree(xml_parser);

	xml_parser = XML_ParserCreate(NULL);
	XML_SetUserData( xml_parser, (void *)this );
	XML_SetElementHandler( xml_parser, xmlw_startElement, xmlw_endElement );
	XML_SetCharacterDataHandler( xml_parser, xmlw_dataHandler );
	appendString(data);

	// parse the XML here
	if ( XML_Parse( xml_parser, data, strlen(data), 1 ) == 0 ) {

		error_msg = my_strdup(
			XML_ErrorString(XML_GetErrorCode(xml_parser)) );
		error_line = XML_GetCurrentLineNumber(xml_parser);
		ret = -1;
	}

	XML_ParserFree(xml_parser);
	xml_parser = 0;
#endif  /* HAVE_LIBEXPAT */

	return ret;
}

// return 0, -1 some overflow
int XmlWrapper::parseDataResults( char **& tag_names, char **& tag_name_values,
	int& msgs_count )
{
	return getMsg( tag_names, tag_name_values, msgs_count );
}

// ret = 1 (found), 0 (not found)
int XmlWrapper::parsedValueString( const char *tag_name, int index, char *& value )
{
	int i, j;
	int ret = 0;

	value = 0;
	for ( i = 0, j = 0; i < tagmsgs_len; i++ ) {

		if ( strcmp( tag_name, tagmsgs.tag_name[i] ) == 0 ) {

			if ( j == index ) {

				value = tagmsgs.value[i];
				ret = 1;
				break;
			}
			else
				j++;
		}
	}

	return ret;
}

// ret = 1 (found), 0 (not found), -1 (bad format)
int XmlWrapper::parsedValueInteger( const char *tag_name, int index, int& value )
{
	int ret = 0;
	char *buf;

	// ret = 1, 0
	ret = parsedValueString( tag_name, index, buf );
	if ( ret && sscanf( buf, "%d", &value ) != 1 )
		ret = -1;

	return ret;
}

// ret = 1 (found), 0 (not found), -1 (bad format)
int XmlWrapper::parsedValueDouble( const char *tag_name, int index, double& value )
{
	int ret = 0;
	char *buf;

	// ret = 1, 0
	ret = parsedValueString( tag_name, index, buf );
	if ( ret && sscanf( buf, "%lf", &value ) != 1 )
		ret = -1;

	return ret;
}

// private

void XmlWrapper::appendString( const char *str )
{
	unsigned int i;

	if ( ! page )
		resetData();

	int add_len = strlen( str );
	char *ptr = 0;

	ptr = new char[page_len + add_len + 1];

	for ( i = 0; i < page_len; i++ )
		ptr[i] = page[i];

	for ( i = page_len; i < page_len + add_len; i++ )
		ptr[i] = str[i - page_len];

	ptr[page_len + add_len] = '\0';

	delete [] page;
	page = ptr;
	page_len += add_len;
}

void XmlWrapper::appendTagPair( const char *el_name, const char *str )
{
	appendIndent();
	appendString("<");
	appendString(el_name);
	appendString(">");
	appendString(str);
	appendString("</");
	appendString(el_name);
	appendString(">\n");
}

void XmlWrapper::resetTags()
{
	tag_list[0] = '\0';
}

const char *XmlWrapper::getTags()
{
	return tag_list;
}

void XmlWrapper::pushTag( const char *tag_name )
{
	int len = strlen(tag_list);

	if ( len + strlen(tag_name) + 2 > TAG_LIST_SIZE ) {

		// tag list gets too big
		resetTags();
	}
	else {

		strcat(tag_list, ".");
		strcat(tag_list, tag_name);
	}
}

void XmlWrapper::popTag()
{
	char *last_tag;

	last_tag = strrchr( tag_list, '.' );
	if ( last_tag )
		*last_tag = '\0';
}

void XmlWrapper::resetMsg()
{
	int i;

	for ( i = 0; i < tagmsgs_len; i++ ) {

		if ( tagmsgs.tag_name[i] )
			delete [] tagmsgs.tag_name[i];

		if ( tagmsgs.value[i] )
			delete [] tagmsgs.value[i];
	}

	tagmsgs_len = 0;
}

void XmlWrapper::appendMsg( const char *tag_name, const char *value )
{
	// don't append if array is to be overflown
	if ( tagmsgs_len < TAGMSGS_SIZE - 1 && tag_name && value ) {

		tagmsgs.tag_name[tagmsgs_len] = my_strdup( tag_name );
		tagmsgs.value[tagmsgs_len] = my_strdup( value );
		tagmsgs_len++;
	}
}

void XmlWrapper::appendIndent()
{
	char indent_buf[100];

	if ( indent_char != '\0' ) {

		memset( indent_buf, indent_char, doc_depth < 99 ? doc_depth : 99 );
		indent_buf[doc_depth < 99 ? doc_depth : 99] = '\0';

		appendString( indent_buf );
	}
}

// ret = 0 on success, -1 on overflow
int XmlWrapper::getMsg( char **& tag_names, char **& tag_name_values,
	int& msgs_count )
{
	if ( tagmsgs_len > 0 ) {

		tag_names = tagmsgs.tag_name;
		tag_name_values = tagmsgs.value;
	}
	msgs_count = tagmsgs_len;

	return tagmsgs_len < TAGMSGS_SIZE - 1 ? 0 : -1;
}

void XmlWrapper::callbackStartElement( const char *name )
{
	pushTag( name );
}

void XmlWrapper::callbackDataHandler( const char *data )
{
	appendMsg( getTags(), data );
}

void XmlWrapper::callbackEndElement()
{
	popTag();
}

// like strdup, but using new instead of malloc
char *XmlWrapper::my_strdup( const char *str )
{
	int len;
	char *new_str;

	len = strlen(str);

	new_str = new char[len + 1];
	strcpy( new_str, str );

	return new_str;
}

// static functions

void xmlw_startElement(void *ptr, const char *name, const char **atts)
{
	((XmlWrapper *)ptr)->callbackStartElement( name );
}

void xmlw_endElement(void *ptr, const char *name )
{
	((XmlWrapper *)ptr)->callbackEndElement();
}

void xmlw_dataHandler(void *ptr, const char *data, int data_len )
{
	char *buf;
	const char *start;

	if ( !data || data_len < 1 ) return;

	buf = new char[data_len + 1];

	memcpy( buf, data, data_len );
	buf[data_len] = '\0';

	start = buf;
	while ( *start && isspace(*start) )
		start++;

	if ( *start ) {
		char *end = buf + data_len - 1;

		while ( end > start && isspace(*end) )
			end--;

		*(end + 1) = '\0';
	}

	if ( *start )
		((XmlWrapper *)ptr)->callbackDataHandler( start );

	delete [] buf;
}
