/*
 * GQview
 * (C) 2001 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "window.h"

#include "filelist.h"
#include "image.h"
#include "img-main.h"
#include "menu.h"
#include "ui_menu.h"
#include "ui_tabcomp.h"

#include "icons/tools.xpm"


#define MAINWINDOW_DEF_WIDTH 500
#define MAINWINDOW_DEF_HEIGHT 400
#define TOOLWINDOW_DEF_WIDTH 224
#define TOOLWINDOW_DEF_HEIGHT 450
#define LIST_DEFAULT_WIDTH -1
#define LIST_DEFAULT_HEIGHT 100

static GtkWidget *add_label(gchar *text, GtkWidget *box, gint start, gint size, gint expand);
static void info_area_create(GtkWidget *vbox);

static void toolwindow_create(void);
static void toolwindow_destroy(void);
static void toolwindow_show(void);

static void create_tools(GtkAccelGroup *accel_grp, GtkTooltips *tooltips);
static gint mainwindow_delete_cb(GtkWidget *widget, gpointer data);

/*
 *-----------------------------------------------------------------------------
 * information widget creation routines (private)
 *-----------------------------------------------------------------------------
 */ 

static void sort_menu_cb(GtkWidget *widget, gpointer data)
{
	SortType type;
	GtkWidget *button;

	type = (SortType)GPOINTER_TO_INT(data);

	if (type == file_sort_method) return;

	button = submenu_item_get_data(widget);
	if (button) gtk_label_set_text(GTK_LABEL(GTK_BIN(button)->child), sort_type_get_text(type));

	file_sort_method = type;
	filelist_set_sort_method(file_sort_method, FALSE);
}

static void sort_menu_ascend_cb(GtkWidget *widget, gpointer data)
{
	file_sort_ascending = (!file_sort_ascending);
	filelist_set_sort_method(file_sort_method, TRUE);
}

static void sort_menu_hide_cb(GtkWidget *widget, gpointer data)
{
	/* destroy the menu */
	gtk_widget_unref(GTK_WIDGET(data));
}

static void sort_menu_press_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *button = data;
	GtkWidget *menu;
	GtkWidget *item;

	menu = submenu_add_sort(NULL, sort_menu_cb, button, FALSE, FALSE);

	/* ascending option */
	menu_item_add_divider(menu);
	item = gtk_check_menu_item_new_with_label(_("Ascending"));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), file_sort_ascending);

	gtk_signal_connect(GTK_OBJECT(item), "activate", (GtkSignalFunc)sort_menu_ascend_cb, NULL);
	gtk_menu_append(GTK_MENU(menu), item);
	gtk_widget_show(item);

	gtk_signal_connect(GTK_OBJECT(menu), "selection_done", sort_menu_hide_cb, menu);
	gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 0, gdk_time_get());
}

static GtkWidget *sort_menu_create(SortType method)
{
	GtkWidget *option;

	option = gtk_button_new_with_label(sort_type_get_text(method));
	gtk_signal_connect(GTK_OBJECT(option), "clicked", sort_menu_press_cb, option);
	gtk_button_set_relief(GTK_BUTTON(option), GTK_RELIEF_NONE);

	return option;
}

static GtkWidget *add_label(gchar *text, GtkWidget *box, gint start, gint size, gint expand)
{
	GtkWidget *label;
	GtkWidget *frame;

	frame = gtk_frame_new (NULL);
	if (size)
		gtk_widget_set_usize (frame, size, -1);
	gtk_frame_set_shadow_type(GTK_FRAME(frame),GTK_SHADOW_IN);
	if (start)
		gtk_box_pack_start(GTK_BOX(box), frame, expand, expand, 0);
	else
		gtk_box_pack_end(GTK_BOX(box), frame, expand, expand, 0);
	gtk_widget_show(frame);

	label = gtk_label_new(text);
	gtk_container_add (GTK_CONTAINER (frame), label);
	gtk_widget_show(label);

	return label;
}

static void info_area_create(GtkWidget *vbox)
{
	gchar *status_text;
	gchar *details_text;
	gchar *zoom_text;
	gchar *buf;
	GtkWidget *sort_menu;

	if (info_status)
		{
		gtk_label_get(GTK_LABEL(info_status), &buf);
		status_text = g_strdup(buf);

		gtk_label_get(GTK_LABEL(info_details), &buf);
		details_text = g_strdup(buf);

		gtk_label_get(GTK_LABEL(info_zoom), &buf);
		zoom_text = g_strdup(buf);
		}
	else
		{
		status_text = g_strdup("");
		details_text = g_strdup("GQview");
		zoom_text = g_strdup(":");
		}

	if (info_box)
		{
		gtk_widget_destroy(info_box);
		info_box = NULL;
		}

	if (vbox)
		{
		GtkWidget *hbox;
		hbox = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
		gtk_widget_show(hbox);
			
		info_progress_bar = gtk_progress_bar_new();
		gtk_widget_set_usize(info_progress_bar,150,-1);
		gtk_box_pack_start (GTK_BOX (hbox), info_progress_bar, TRUE, TRUE, 0);
		gtk_widget_show(info_progress_bar);

		sort_menu = sort_menu_create(file_sort_method);
		gtk_box_pack_start (GTK_BOX (hbox), sort_menu, FALSE, FALSE, 0);
		gtk_widget_show(sort_menu);

		info_status = add_label(status_text, vbox, TRUE, 0, FALSE);

		hbox = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
		gtk_widget_show(hbox);

		info_details = add_label(details_text, hbox, TRUE, 0, TRUE);
		info_zoom = add_label(zoom_text, hbox, FALSE, 48, FALSE);
		}
	else
		{
		info_box = gtk_hbox_new(FALSE, 0);

		info_progress_bar = gtk_progress_bar_new();
		gtk_widget_set_usize(info_progress_bar,150,-1);
		gtk_box_pack_start (GTK_BOX (info_box), info_progress_bar, FALSE, FALSE, 0);
		gtk_widget_show(info_progress_bar);

		sort_menu = sort_menu_create(file_sort_method);
		gtk_box_pack_start (GTK_BOX (info_box), sort_menu, FALSE, FALSE, 0);
		gtk_widget_show(sort_menu);

		info_status = add_label(status_text, info_box, TRUE, 0, TRUE);
		info_details = add_label(details_text, info_box, TRUE, 0, TRUE);
		info_zoom = add_label(zoom_text, info_box, FALSE, 48, FALSE);

		gtk_widget_show(info_box);
		}

	main_image_set_labels(info_details, info_zoom);

	g_free(status_text);
	g_free(details_text);
	g_free(zoom_text);
}

/*
 *-----------------------------------------------------------------------------
 * tool window create/show/hide routines (private)
 *-----------------------------------------------------------------------------
 */ 

static gint toolwindow_delete_cb(GtkWidget *widget, gpointer data)
{
	toolwindow_float();
	return TRUE;
}

static void toolwindow_create(void)
{
	GtkWidget *vbox;

	toolwindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_signal_connect(GTK_OBJECT(toolwindow), "delete_event", (GtkSignalFunc)toolwindow_delete_cb, NULL);
	gtk_window_set_policy(GTK_WINDOW(toolwindow), TRUE, TRUE, FALSE);
	gtk_window_set_title (GTK_WINDOW(toolwindow), _("GQview Tools"));
	gtk_window_set_wmclass(GTK_WINDOW(toolwindow), "tools", "GQview");
	gtk_container_border_width(GTK_CONTAINER(toolwindow), 0);
	gtk_window_add_accel_group(GTK_WINDOW(toolwindow), mainwindow_accel_grp);

	window_set_icon(toolwindow, (const gchar **)tools_xpm, NULL);

	gtk_signal_connect(GTK_OBJECT(toolwindow), "key_press_event", GTK_SIGNAL_FUNC(key_press_cb), NULL);

	if (save_window_positions)
		{
		gtk_window_set_default_size (GTK_WINDOW(toolwindow), float_window_w, float_window_h);
		gtk_widget_set_uposition(toolwindow, float_window_x, float_window_y);
		}
	else
		{
		gtk_window_set_default_size (GTK_WINDOW(toolwindow), TOOLWINDOW_DEF_WIDTH, TOOLWINDOW_DEF_HEIGHT);
		}

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(toolwindow), vbox);

	gtk_widget_realize(toolwindow);
	gtk_widget_realize(vbox);

	if (tool_vbox->parent)
		{
		gtk_widget_ref(tool_vbox);
		gtk_container_remove(GTK_CONTAINER(tool_vbox->parent), tool_vbox);
		gtk_box_pack_start(GTK_BOX(vbox), tool_vbox, TRUE, TRUE, 0);
		gtk_widget_unref(tool_vbox);
		}
	else
		{
		gtk_box_pack_start(GTK_BOX(vbox), tool_vbox, TRUE, TRUE, 0);
		}

	info_area_create(vbox);

	gtk_widget_show(vbox);

	image_top_window_set_sync(normal_image, TRUE);
}

static void toolwindow_destroy(void)
{
	if (toolwindow && GTK_WIDGET_VISIBLE(toolwindow))
		{
		gdk_window_get_root_origin(toolwindow->window, &float_window_x, &float_window_y);
		gdk_window_get_size(toolwindow->window, &float_window_w, &float_window_h);
		}

	info_area_create(NULL);

	gtk_widget_ref(tool_vbox);
	gtk_container_remove(GTK_CONTAINER(tool_vbox->parent), tool_vbox);
	gtk_box_pack_start(GTK_BOX(mainwindow_hbox), tool_vbox, FALSE, FALSE, 0);
	gtk_widget_unref(tool_vbox);

	gtk_box_pack_start(GTK_BOX(mainwindow_vbox), info_box, FALSE, FALSE, 0);
	gtk_widget_destroy(toolwindow);
	toolwindow = NULL;

	image_top_window_set_sync(normal_image, FALSE);
}

static void toolwindow_show(void)
{
	gtk_widget_show(toolwindow);
}

/*
 *-----------------------------------------------------------------------------
 * tool window hide/float routines (public)
 *-----------------------------------------------------------------------------
 */ 

void toolwindow_float(void)
{
	if (toolwindow)
		{
		if (GTK_WIDGET_VISIBLE(toolwindow))
			{
			toolwindow_destroy();
			tools_float = FALSE;
			tools_hidden = FALSE;
			}
		else
			{
			toolwindow_show();
			tools_float = TRUE;
			tools_hidden = FALSE;
			}
		}
	else
		{
		toolwindow_create();
		toolwindow_show();
		tools_float = TRUE;
		tools_hidden = FALSE;
		}
}

void toolwindow_hide(void)
{
	if (toolwindow)
		{
		if (GTK_WIDGET_VISIBLE(toolwindow))
			{
			gdk_window_get_root_origin(toolwindow->window, &float_window_x, &float_window_y);
			gdk_window_get_size(toolwindow->window, &float_window_w, &float_window_h);
			main_window_vdivider_pos = dir_clist->allocation.height;
			gtk_widget_hide(toolwindow);
			tools_hidden = TRUE;
			}
		else
			{
			if (tools_float)
				toolwindow_show();
			else
				toolwindow_destroy();
			tools_hidden = FALSE;
			}
		}
	else
		{
		main_window_vdivider_pos = dir_clist->allocation.height;
		toolwindow_create();
		tools_hidden = TRUE;
		}
}

static void mainwindow_update_cb(ImageWindow *imd, gpointer data)
{
	gchar *text;
	gchar *b;

	text = image_zoom_get_as_text(imd);
	gtk_label_set_text(GTK_LABEL(info_zoom), text);
	g_free(text);

	b = text_from_size(imd->size);

	if (imd->unknown)
                {
		text = g_strdup_printf(_("( ? x ? ) %s bytes"), b);
                }
        else
                {
		text = g_strdup_printf(_("( %d x %d ) %s bytes"), imd->image_width, imd->image_height, b);
                }

	gtk_label_set(GTK_LABEL(info_details), text);

	g_free(b);
	g_free(text);
}

/*
 *-----------------------------------------------------------------------------
 * main window setup
 *-----------------------------------------------------------------------------
 */ 

static void create_tools(GtkAccelGroup *accel_grp, GtkTooltips *tooltips)
{
	GtkWidget *menu_bar;
	GtkWidget *button_bar;
	GtkWidget *vpaned;
	GtkWidget *scrolled;
	GtkWidget *tabcomp;
	GtkWidget *popwin;

	tool_vbox = gtk_vbox_new(FALSE, 0);

	menu_bar = create_menu_bar(accel_grp);
	gtk_box_pack_start (GTK_BOX(tool_vbox), menu_bar,FALSE,FALSE,0);
	gtk_widget_show(menu_bar);

	button_bar = create_button_bar(tooltips);
	gtk_box_pack_start (GTK_BOX(tool_vbox), button_bar,FALSE,FALSE,0);
	gtk_widget_show(button_bar);

	/* path entry */
	tabcomp = tab_completion_new_with_history(&path_entry, NULL, "path_list", 32,
						  path_entry_cb, NULL);
	tab_completion_add_tab_func(path_entry, path_entry_tab_cb, NULL);

	/* connect to hide signal of popup of the combo list, after hide it will sync the path */
	popwin = gtk_widget_get_toplevel(GTK_COMBO(tabcomp)->list);
	gtk_signal_connect_after(GTK_OBJECT(popwin), "hide",
				 GTK_SIGNAL_FUNC(path_entry_changed_cb), NULL);

	gtk_box_pack_start (GTK_BOX (tool_vbox), tabcomp, FALSE, FALSE, 0);
	gtk_widget_show (tabcomp);

        /* history button */
	history_menu = gtk_option_menu_new ();
	gtk_box_pack_start (GTK_BOX (tool_vbox), history_menu, FALSE, FALSE, 0);
	gtk_widget_show (history_menu);

	vpaned = gtk_vpaned_new ();
	gtk_paned_handle_size (GTK_PANED(vpaned), 10);
	gtk_paned_gutter_size (GTK_PANED(vpaned), 10);
	gtk_box_pack_start (GTK_BOX (tool_vbox), vpaned, TRUE, TRUE, 0);
	gtk_widget_show (vpaned);

	/* dir list */
	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
				GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_paned_add1 (GTK_PANED(vpaned), scrolled);
	gtk_widget_show(scrolled);

	dir_clist=gtk_clist_new(1);
	gtk_signal_connect (GTK_OBJECT (dir_clist), "button_press_event",(GtkSignalFunc) dir_press_cb, NULL);
	gtk_signal_connect (GTK_OBJECT (dir_clist), "select_row",(GtkSignalFunc) dir_select_cb, NULL);
	gtk_clist_column_titles_passive (GTK_CLIST (dir_clist)); 
	gtk_container_add (GTK_CONTAINER (scrolled), dir_clist);
	gtk_widget_show(dir_clist);

	/* file list */
	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
				GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_paned_add2 (GTK_PANED(vpaned), scrolled);
	gtk_widget_show(scrolled);

	file_clist=gtk_clist_new(3);
	gtk_clist_set_column_auto_resize(GTK_CLIST(file_clist), 1, TRUE);
	gtk_clist_set_column_justification(GTK_CLIST(file_clist), 1, GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_auto_resize(GTK_CLIST(file_clist), 2, TRUE);
	gtk_clist_set_column_justification(GTK_CLIST(file_clist), 2, GTK_JUSTIFY_RIGHT);
	gtk_clist_set_selection_mode(GTK_CLIST(file_clist), GTK_SELECTION_EXTENDED);
	gtk_signal_connect (GTK_OBJECT (file_clist), "button_press_event",(GtkSignalFunc) file_press_cb, NULL);
	gtk_signal_connect (GTK_OBJECT (file_clist), "select_row",(GtkSignalFunc) file_select_cb, NULL);
	gtk_signal_connect_after (GTK_OBJECT (file_clist), "unselect_row",(GtkSignalFunc) file_unselect_cb, NULL);
	gtk_clist_column_titles_passive (GTK_CLIST (file_clist)); 
	gtk_container_add (GTK_CONTAINER (scrolled), file_clist);
	gtk_widget_show(file_clist);

	if (save_window_positions)
                gtk_paned_set_position(GTK_PANED(vpaned), main_window_vdivider_pos);
        else
                gtk_paned_set_position(GTK_PANED(vpaned), LIST_DEFAULT_HEIGHT);

	gtk_widget_show(tool_vbox);
}

static gint mainwindow_delete_cb(GtkWidget *widget, gpointer data)
{
	exit_gqview();
	return TRUE;
}

void create_main_window(void)
{
	GtkWidget *image_window;
	GtkTooltips *tooltips;
	GdkColormap *colormap;
	GdkPixbuf *pixbuf;
	static GdkColor tooltip_color = { 0, 0xffff, 0xf9f9, 0xcbcb }; /*255 249 203*/

	mainwindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_policy(GTK_WINDOW(mainwindow), TRUE, TRUE, FALSE);

	gtk_signal_connect (GTK_OBJECT (mainwindow), "delete_event", (GtkSignalFunc)mainwindow_delete_cb, NULL);

	gtk_window_set_title(GTK_WINDOW (mainwindow), "GQview");
	gtk_window_set_wmclass(GTK_WINDOW (mainwindow), "gqview", "GQview");
	gtk_container_border_width (GTK_CONTAINER (mainwindow), 0);

	window_set_icon(mainwindow, NULL, NULL);

	gtk_signal_connect(GTK_OBJECT(mainwindow), "key_press_event", GTK_SIGNAL_FUNC(key_press_cb), NULL);

	if (save_window_positions)
		{
		gtk_window_set_default_size (GTK_WINDOW(mainwindow), main_window_w, main_window_h);
		gtk_widget_set_uposition(mainwindow, main_window_x, main_window_y);
		}
	else
		{
		gtk_window_set_default_size (GTK_WINDOW(mainwindow), MAINWINDOW_DEF_WIDTH, MAINWINDOW_DEF_HEIGHT);
		}

	gtk_widget_realize(mainwindow);

	mainwindow_accel_grp = gtk_accel_group_new ();
	gtk_window_add_accel_group(GTK_WINDOW(mainwindow),mainwindow_accel_grp);

	tooltips = gtk_tooltips_new();
	colormap = gdk_window_get_colormap (mainwindow->window);
	gdk_color_alloc (colormap, &tooltip_color);
	gtk_tooltips_set_colors(tooltips, &tooltip_color, &mainwindow->style->fg[GTK_STATE_NORMAL]);

	create_tools(mainwindow_accel_grp, tooltips);

	image_window = main_image_create();
	image_attach_window(normal_image, mainwindow, NULL, " - GQview", FALSE);

	pixbuf = gdk_pixbuf_new_from_xpm_data(logo);
	image_change_pixbuf(normal_image, pixbuf, 1.0);
	gdk_pixbuf_unref(pixbuf);

	image_set_update_func(normal_image, mainwindow_update_cb, NULL);

	mainwindow_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add (GTK_CONTAINER (mainwindow), mainwindow_vbox);
	gtk_widget_show(mainwindow_vbox);

	mainwindow_hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(mainwindow_vbox), mainwindow_hbox, TRUE, TRUE, 0);
	gtk_widget_show(mainwindow_hbox);

	if (tools_float || tools_hidden)
		{
		toolwindow_create();
		if (!tools_hidden)
			{
			toolwindow_show();
			}
		}
	else
		{
		info_area_create(NULL);
		gtk_box_pack_start(GTK_BOX(mainwindow_hbox), tool_vbox, FALSE, FALSE, 0);
		gtk_box_pack_start(GTK_BOX(mainwindow_vbox), info_box, FALSE, FALSE, 0);
		}

	gtk_box_pack_end(GTK_BOX(mainwindow_hbox), image_window, TRUE, TRUE, 0);
	gtk_widget_show_all(image_window);

	gtk_widget_show(mainwindow);
}





