# -*- python -*-

### Copyright (C) 2005 Peter Williams <pwil3058@bigpond.net.au>

### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; version 2 of the License only.

### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.

### You should have received a copy of the GNU General Public License
### along with this program; if not, write to the Free Software
### Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# This file provides access to "quilt" functionality required by "gquilt"

import gquilt_utils, os.path, os, gquilt_const, gquilt_tool, gquilt_pfuns, re, errno

class quilt_commands(gquilt_utils.cmd_list):
    def __init__(self):
        gquilt_utils.cmd_list.__init__(self)
    def _get_commands(self):
        res, sop, eop = gquilt_utils.run_cmd("quilt")
        if sop == "":
            return None
        lines = sop.splitlines()
        index = 2
        cmds = []
        while True:
            lcmds = lines[index].split()
            if len(lcmds) > 0:
                cmds += lcmds
            else:
                cmds.sort()
                return cmds
            index += 1

# Find the value of a specified quilt environment variable.  These can be set
# in as environment variables or in /etc/quilt.quiltrc, ~/.quiltrc or the file
# specified by the QUILTRC environment variable (if they exist).  Only one of
# the files will be read with precedence in the reverse of the order listed.
# Any variable set in the file that is read will override the value for the
# same variable that has been set as an environment variable.
# So we need to do the same so we know what quilt will see.
_default_env_var_value = { 'QUILT_PATCHES': 'patches', 'QUILT_SERIES': 'series' }

def _get_quilt_env_var_value(vname, default_ok=False):
    if default_ok:
        try:
            val = _default_env_var_value[vname]
        except:
            val = None
    else:
        val = None
    try:
        quiltrc = os.environ['QUILTRC']
    except:
        for fname in [os.path.expanduser("~/.quiltrc"), "/etc/quilt.quiltrc"]:
            if os.path.exists(fname):
                quiltrc = fname
                break
            else:
                quiltrc = None
    if quiltrc:
        regex = re.compile("^\w*%s=(.*)$" % vname)
        quiltrc_f = open(quiltrc, 'r')
        for line in quiltrc_f.readlines():
            m = regex.match(line.strip())
            if m:
                val = m.group(1)
                break
        quiltrc_f.close()
    else:
        try:
            val = os.environ[vname]
        except:
            pass
    return val

def find_patchfns():
    binloc = gquilt_utils.which("quilt")
    if binloc is None:
        return None
    bi = binloc.find("bin" + os.path.sep + "quilt")
    if bi == -1:
        return None
    res = os.path.join(binloc[0:bi], "share", "quilt", "scripts", "patchfns")
    if not os.path.exists(res) or not os.path.isfile(res):
        return None
    return res

patchfns = find_patchfns()
if patchfns is None:
    qbsfe = None
else:
    qbsfe = 'bash -c ". ' + os.path.join(os.environ['GQUILT_LIB_DIR'], "qbsfe" + os.extsep + "sh") + '" gquilt ' + patchfns

def internal(cmd, input=None):
    if qbsfe is None:
        raise "quilt_specific_error", "Couldn't find patchfns"
    return gquilt_utils.run_cmd(" ".join([qbsfe, cmd]), input)

# Some useful private quilt functions (that shouldn't fail)
def _patch_file_name(patch):
    res, so, se = internal(" ".join(["patch_file_name", patch]))
    if res != 0:
        raise "quilt_specific_error", se
    return so[:-1]

def _patch_basename(patchfile):
    res, so, se = internal(" ".join(["basename", patchfile]))
    if res != 0:
        raise "quilt_specific_error", se
    return so[:-1]

def _patch_backup_dir(patch):
    if _get_quilt_env_var_value("QUILT_PATCHES_PREFIX"):
        patch = _patch_basename(patch)
    res, so, se = internal(" ".join(["backup_file_name", patch, "x"]))
    if res != 0:
        raise "quilt_specific_error", se
    return so[:-3]

def _patch_backup_file(patch, filename):
    if _get_quilt_env_var_value("QUILT_PATCHES_PREFIX"):
        patch = _patch_basename(patch)
    res, so, se = internal(" ".join(["backup_file_name", patch, filename]))
    if res != 0:
        raise "quilt_specific_error", se
    return so[:-1]

def _convert_pop_res(res, so, se):
    if res != gquilt_const.OK:
        if re.search("needs to be refreshed first", so + se):
            res = gquilt_const.ERROR_REFRESH
        elif re.search("\(refresh it or enforce with -f\)", so + se):
            res = gquilt_const.ERROR_FORCE_OR_REFRESH
        else:
            res = gquilt_const.ERROR
    return (res, so, se)

def _convert_push_res(res, so, se):
    if res != gquilt_const.OK:
        if re.search("\(forced; needs refresh\)", so + se):
            res = gquilt_const.OK
        elif re.search("needs to be refreshed first", so + se):
            res = gquilt_const.ERROR_REFRESH
        elif re.search("\(enforce with -f\)", so + se):
            res = gquilt_const.ERROR_FORCE
        else:
            res = gquilt_const.ERROR
    return (res, so, se)

def _convert_import_res(res, so, se):
    if res != gquilt_const.OK:
        if re.search("Replace with -f", so + se):
            res = gquilt_const.ERROR_FORCE
    return (res, so, se)

def _quilt_version():
    res, so, se = gquilt_utils.run_cmd("quilt --version")
    if res != 0:
        return None
    return so

def _quilt_version_ge(rmajor, rminor):
    vstr = _quilt_version()
    if not vstr:
        return False
    vstrs = vstr.split(".")
    amajor = int(vstrs[0])
    if  amajor > rmajor:
        return True
    elif amajor < rmajor:
        return False
    else:
        return int(vstrs[1]) >= rminor

def _patch_dir(dir=None):
    pd = _get_quilt_env_var_value('QUILT_PATCHES', default_ok=True)
    if dir is None:
        dir = os.getcwd()
    return os.path.join(dir, pd)
    
def _series_file(dir=None):
    sn = _get_quilt_env_var_value('QUILT_SERIES', default_ok=True)
    return os.path.join(_patch_dir(dir), sn)
    
# Now implement the tool interface for quilt
class interface(gquilt_tool.interface):
    def __init__(self):
        gquilt_tool.interface.__init__(self, "quilt")
    def is_available(self):
        if _quilt_version() is None:
            return False
        return True
    def requires(self):
        return "\"quilt\" <http://savannah.nongnu.org/projects/quilt>"
    def new_playground(self, dir=None):
        patch_dir = _patch_dir(dir)
        if not os.path.exists(patch_dir):
            try:
                os.makedirs(patch_dir)
            except os.error, value:
               return (gquilt_const.ERROR, "", value[1])
        return gquilt_utils.run_cmd("touch " + _series_file(dir))
    def is_playground(self, dir=None):
        root = self.get_playground_root(dir)
        if not root:
            return False
        return os.path.exists(_series_file(root))
    def is_patch_applied(self, patch):
        res, so, se = internal("is_applied " + patch)
        return res == 0
    def patch_file_name(self, patch):
        if _get_quilt_env_var_value("QUILT_PATCHES_PREFIX"):
            return patch
        return _patch_file_name(patch)
    def top_patch(self):
        res, so, se = gquilt_utils.run_cmd("quilt top")
        if res == 0 or (se.strip() == "" and so.strip() == ""):
            return so.strip()
        elif (res == 512 or res == 2) and so.strip() == "":
            return ""
        else:
            raise "quilt_specific_error", se
    def next_patch(self):
        res, so, se = gquilt_utils.run_cmd("quilt next")
        if res == 0 or (se.strip() == "" and so.strip() == ""):
            return so.strip()
        elif (res == 512 or res == 2) and so.strip() == "":
            return ""
        else:
            raise "quilt_specific_error", se
    def base_patch(self):
        res, so, se = gquilt_utils.run_cmd("hg qapplied")
        if res == 0 and so:
            lines = so.splitlines(false)
            return lines[0]
        else:
            return ""
    def last_patch_in_series(self):
        res, op, err = gquilt_utils.run_cmd("quilt series")
        if res != 0:
            raise "quilt_specific_error", se
        return op.splitlines()[-1]
    def count_ok_meld(selfself, count):
        return count == 1
    def display_files_diff_in_viewer(self, viewer, files, patch=None):
        difffld = "--diff=" + viewer
        if patch is None:
            pid = os.spawnlp(os.P_NOWAIT, "quilt", "quilt", "diff", difffld, files[0])
        else:
            pid = os.spawnlp(os.P_NOWAIT, "quilt", "quilt", "diff", difffld, "-P", patch, files[0])
    def get_playground_root(self, dir=None):
        pd = _get_quilt_env_var_value('QUILT_PATCHES', default_ok=True)
        if not dir:
            dir = os.getcwd()
        root = dir
        while True:
            apd = os.path.join(root, pd)
            if os.path.exists(apd) and os.path.isdir(apd):
                return root
            newroot = os.path.dirname(root)
            if root == newroot:
                break
            root = newroot
        return None
    def get_patch_status(self, patch):
        if not self.is_patch_applied(patch):
            return (gquilt_const.OK, gquilt_const.NOT_APPLIED, "")
        patch_fn = _patch_file_name(patch)
        patch_dirn = _patch_backup_dir(patch)
        if (not os.path.exists(patch_fn)) or os.path.exists(patch_dirn + "~refresh"):
            needs_refresh = True
        else:
            timestamp_fn = os.path.join(patch_dirn + ".timestamp")
            if not os.path.exists(timestamp_fn):
                needs_refresh = True
            else:
                timestamp = os.stat(timestamp_fn).st_mtime
                if timestamp > os.stat(patch_fn).st_mtime:
                    needs_refresh = True
                else:
                    needs_refresh = False
                    res, files, err = gquilt_utils.run_cmd("quilt files " + patch)
                    for f in files.splitlines():
                        if timestamp < os.stat(f).st_mtime:
                            needs_refresh = True
                            break
        res, tp, se = self.get_top_patch()
        if tp == patch:
            if needs_refresh:
                return (gquilt_const.OK, gquilt_const.TOP_PATCH_NEEDS_REFRESH, "")
            else:
                return (gquilt_const.OK, gquilt_const.TOP_PATCH, "")
        else:
            if needs_refresh:
                return (gquilt_const.OK, gquilt_const.APPLIED_NEEDS_REFRESH, "")
            else:
                return (gquilt_const.OK, gquilt_const.APPLIED, "")
    def get_series(self):
        res, op, err = gquilt_utils.run_cmd("quilt series -v")
        if res != 0:
            return (gquilt_const.ERROR, (None, None), err)
        series = []
        for line in op.splitlines():
            if line[0] == " ":
                series.append((line[2:], gquilt_const.NOT_APPLIED))
            else:
                pn = line[2:]
                #res, status, err = self.get_patch_status(pn)
                #series.append((pn, status))
                series.append((pn, gquilt_const.APPLIED))
        # if the patches are in a non standard directory and patch names
        # aren't being prefixed with that name then let the client know so
        # that they can display the name of the patch directory in the header
        if not _get_quilt_env_var_value("QUILT_PATCHES_PREFIX"):
            patch_dir = _get_quilt_env_var_value('QUILT_PATCHES', default_ok=False)
            if patch_dir == _default_env_var_value['QUILT_PATCHES']:
                patch_dir = None
        else:
            patch_dir = None
        return (gquilt_const.OK, (patch_dir, series), "")
    def get_diff(self, filelist=[], patch=None):
        if patch is None:
            cmd = "quilt diff"
        else:
            cmd = "quilt diff -P " + patch
        res, so, se = gquilt_utils.run_cmd(" ".join([cmd, " ".join(filelist)]))
        if res != 0:
            if not self.is_patch_applied(patch):
                res, diff = gquilt_pfuns.get_patch_diff_lines(self.patch_file_name(patch))
                if res:
                    return (gquilt_const.OK, os.linesep.join(diff), [])
            return (gquilt_const.ERROR, so, se)
        return (res, so, se)
    def get_combined_diff(self, start_patch=None, end_patch=None):
        cmd = "quilt diff --sort --combine "
        if start_patch is None:
            cmd += "-"
        else:
            cmd += start_patch
        if end_patch is not None:
            cms += " -P " + end_patch
        res, so, se = gquilt_utils.run_cmd(cmd)
        if res != 0:
            return (gquilt_const.ERROR, so, se)
        return (res, so, se)
    def get_patch_files(self, patch=None, withstatus=True):
        if self.top_patch() == "":
            return (gquilt_const.OK, "", "")
        cmd = "quilt files"
        if withstatus:
            cmd += " -v"
        if patch is not None:
            cmd += " " + patch
        res, so, se = gquilt_utils.run_cmd(cmd)
        if res != 0:
            if patch and not self.is_patch_applied(patch):
                patchfile = self.patch_file_name(patch)
                ok, filelist = gquilt_pfuns.get_patch_files(patchfile, withstatus)
                if ok:
                    return (gquilt_const.OK, filelist, "")
                else:
                    return (gquilt_const.ERROR, "", filelist)
            return (gquilt_const.ERROR, so, se)
        if withstatus:
            filelist = []
            for line in so.splitlines():
                if line[0] == "+":
                    filelist.append((line[2:], gquilt_const.ADDED))
                elif line[0] == "-":
                    filelist.append((line[2:], gquilt_const.DELETED))
                else:
                    filelist.append((line[2:], gquilt_const.EXTANT))
        else:
            filelist = so.splitlines()
        return (res, filelist, se)
    def do_rename_patch(self, console, patch, newname):
        if quilt_commands().has_cmd("rename"):
            cmd = "quilt rename "
            if patch is not None:
                if _quilt_version_ge(0, 43):
                    cmd += "-P " + patch + " "
                else:
                    cmd += "-p " + patch + " "
            cmd += newname
            res, so, se = gquilt_tool.exec_console_cmd(console, cmd)
            if res != 0:
                return (gquilt_const.ERROR, so, se)
            else:
                return (gquilt_const.OK, so, se)
        else:
            newname_pfn = _patch_file_name(newname)
            newname_backup_dir =  _patch_backup_dir(newname)
            if os.path.exists(newname_pfn) or os.path.exists(newname_backup_dir):
                return (gquilt_const.ERROR, "", 'Patch rename failed. "' + newname + '" is already being used')
            if patch is None:
                res, patch, se = self.get_top_patch()
                if res != gquilt_const.OK:
                    return (res, patch, se)
            patchname_pfn = _patch_file_name(patch)
            if self.is_patch_applied(patch):
                res, so, se = internal(" ".join(["rename_in_db",  patch, newname]))
                if res != 0:
                    return (gquilt_const.ERROR, so, se)
                os.rename(_patch_backup_dir(patch), newname_backup_dir)
                if os.path.exists(patchname_pfn):
                    os.rename(patchname_pfn, newname_pfn)
            else:
                if os.path.exists(patchname_pfn):
                    os.rename(patchname_pfn, newname_pfn)
            res, so, se = internal(" ".join(["rename_in_series", patch, newname]))
            if res != 0:
                return (gquilt_const.ERROR, so, se)
            else:
                return (gquilt_const.OK, so, se)

    def do_pop_patch(self, console, force=False):
        cmd = "quilt pop"
        if force:
            cmd += " -f"
        res, so, se = gquilt_tool.exec_console_cmd(console, cmd)
        return _convert_pop_res(res, so, se)
    def do_push_patch(self, console, force=False):
        cmd = "quilt push"
        if force:
            cmd += " -f"
        res, so, se = gquilt_tool.exec_console_cmd(console, cmd)
        return _convert_push_res(res, so, se)
    def do_pop_to_patch(self, console, patch=None):
        cmd = "quilt pop"
        if patch is not None:
            if patch is "":
                cmd += " -a"
            else:
                cmd += " " + patch
        res, so, se = gquilt_tool.exec_console_cmd(console, cmd)
        return _convert_pop_res(res, so, se)
    def do_push_to_patch(self, console, patch=None):
        cmd = "quilt push"
        if patch is not None:
            if patch is "":
                cmd += " -a"
            else:
                cmd += " " + patch
        res, so, se = gquilt_tool.exec_console_cmd(console, cmd)
        return _convert_push_res(res, so, se)
    def do_refresh_patch(self, console, patch=None, force=False):
        cmd = "quilt refresh"
        if force:
            cmd += " -f"
        if patch is not None:
            cmd = " ".join([cmd, patch])
        res, so, se = gquilt_tool.exec_console_cmd(console, cmd)
        if res != gquilt_const.OK:
            if re.search("Enforce refresh with -f", so + se):
                res = gquilt_const.ERROR_FORCE
            else:
                res = gquilt_const.ERROR
        return (res, so, se)
    def do_create_new_patch(self, console, name, force=False):
        if force:
            return (gquilt_const.ERROR, "", "\"quilt\" cannot force \"new\"")
        return gquilt_tool.exec_console_cmd(console, " ".join(["quilt", "new", name]))
    def do_import_patch(self, console, filename, patchname=None, force=False):
        if patchname:
            cmd = "quilt import -P %s " % patchname
        else:
            cmd = "quilt import "
        if force:
            cmd += "-f -d n "
        res, so, se = gquilt_tool.exec_console_cmd(console, cmd + filename)
        return _convert_import_res(res, so, se)
    def do_finish_patch(sefl, console, patch):
        return (gquilt_const.ERROR, "", '"quilt" does not have this feature')
    def do_merge_patch(self, console, patch):
        filename = _patch_file_name(patch)
        res, so, se = self.do_merge_patch_file(console, filename)
        if res == 0:
            self.do_delete_patch(console, patch)
        return (res, so, se)
    def do_merge_patch_file(self, console, filename):
        return gquilt_tool.exec_console_cmd(console, " < ".join(["quilt fold", filename]))
    def do_delete_patch(self, console, patch):
        return gquilt_tool.exec_console_cmd(console, " ".join(["quilt", "delete", patch]))
    def do_add_files_to_patch(self, console, filelist):
        cmd = "quilt add"
        # quilt will screw up semi silently if you try to add directories to a patch
        for f in filelist:
            if os.path.isdir(f):
                return (gquilt_const.ERROR, "", f + ' is a directory.\n "quilt" does not handle adding directories to patches\n')
        return gquilt_tool.exec_console_cmd(console, " ".join([cmd, " ".join(filelist)]))
    def do_remove_files_from_patch(self, console, filelist, patch=None):
        if quilt_commands().has_cmd("remove"):
            if patch == None:
                cmd = "quilt remove"
            else:
                if _quilt_version_ge(0, 43):
                    cmd = "quilt remove -P " + patch
                else:
                    cmd = "quilt remove -p " + patch
            # quilt will screw up semi silently if you try to remove directories to a patch
            for f in filelist:
                if os.path.isdir(f):
                    return (gquilt_const.ERROR, "", f + ' is a directory.\n "quilt" does not handle removing directories from patches\n')
            return gquilt_tool.exec_console_cmd(console, " ".join([cmd, " ".join(filelist)]))
        else:
            return (gquilt_const.ERROR, "", '"quilt" does not handle removing files from patches\n')
    def do_revert_files_in_patch(self, console, filelist, patch=None):
        if quilt_commands().has_cmd("revert"):
            if patch == None:
                cmd = "quilt revert"
            else:
                cmd = "quilt revert -P " + patch
            # quilt will screw up semi silently if you try to revert directories in a patch
            for f in filelist:
                if os.path.isdir(f):
                    return (gquilt_const.ERROR, "", f + ' is a directory.'
                            + os.linesep + '"quilt" does not handle reversion for directories in patches' + os.linesep)
            return gquilt_tool.exec_console_cmd(console, " ".join([cmd, " ".join(filelist)]))
        else:
            return (gquilt_const.ERROR, "", '"quilt" does not handle reverting changes to in from patches\n')
    def do_exec_tool_cmd(self, console, cmd):
        return gquilt_tool.exec_console_cmd(console, " ".join(["quilt", cmd]))
