# -*- python -*-

### Copyright (C) 2005 Peter Williams <pwil3058@bigpond.net.au>

### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; version 2 of the License only.

### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.

### You should have received a copy of the GNU General Public License
### along with this program; if not, write to the Free Software
### Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# This file provides access to "quilt" functionality required by "gquilt"

import gquilt_utils, os.path, os, gquilt_const, gquilt_tool, gquilt_pfuns, re, errno

def _hg_command_avail(cmd):
    res, so, se = gquilt_utils.run_cmd("hg help %s" % cmd)
    return res == 0

def _convert_pop_res(res, so, se):
    if res != gquilt_const.OK:
        if re.search("local changes found, refresh first", so + se):
            res = gquilt_const.ERROR_REFRESH
        else:
            res = gquilt_const.ERROR
    return (res, so, se)

def _convert_push_res(res, so, se):
    if res != gquilt_const.OK:
        if re.search("local changes found, refresh first", so + se):
            res = gquilt_const.ERROR_FORCE_OR_REFRESH
        else:
            res = gquilt_const.ERROR
    return (res, so, se)

def _convert_copy_res(res, so, se):
    if re.search("not overwriting - file exists", so + se):
        res = gquilt_const.ERROR_FORCE
    elif res != gquilt_const.OK:
        res = gquilt_const.ERROR
    return (res, so, se)

def _convert_import_res(res, so, se):
    if res != gquilt_const.OK:
        if re.search("already exists", so + se):
            res = gquilt_const.ERROR_FORCE
    return (res, so, se)

def _hg_dir(dir=None):
    if dir is None:
        dir = os.getcwd()
    return os.path.join(dir, ".hg")

def _patch_dir(dir=None):
    if dir is None:
        dir = os.getcwd()
    return os.path.join(dir, ".hg", "patches")
    
def _patch_file_name(patch):
    return os.path.join(_patch_dir(), patch)

def _numeric_str_sort(a, b):
    return cmp(int(a), int(b))

# Now implement the tool interface for mq
class interface(gquilt_tool.interface):
    def __init__(self):
        gquilt_tool.interface.__init__(self, "mq")
    def is_available(self):
        return _hg_command_avail("mq")
    def requires(self):
        return "\"mercurial\" with \"mq\" extension enabled <http://www.selenic.com/mercurial/>"
    def _no_patches_applied(self):
        return self.top_patch() == ""
    def _patch_parent_revs(self, patch):
        if not self.is_patch_applied(patch) and patch != "qbase":
            return (gquilt_const.ERROR, "", "Patch %s is not applied." % patch)
        res, so, se = gquilt_utils.run_cmd("hg parent --template \"{rev}" + os.linesep + "\" -r " + patch)
        if res != 0:
            return (res, so, se)
        elif so == "":
            return (res, ["0"], se)
        plist = so.splitlines()
        plist.sort(_numeric_str_sort)
        return (res, plist, se)
    def _patch_most_recent_parent_rev(self, patch):
        res, plist, se = self._patch_parent_revs(patch)
        if res != 0:
            return (res, plist, se)
        return (res, plist[-1], se)
    def new_playground(self, dir=None):
        if os.path.exists(_patch_dir(dir)):
            return (gquilt_const.OK, "", "")
        hgdir = _hg_dir(dir)
        if not os.path.exists(_hg_dir(dir)):
            cmd = "hg init"
            if dir is not None:
                cmd += " " + dir
            res, so, se = gquilt_utils.run_cmd(cmd)
            if res != gquilt_const.OK:
                return (res, so, se)
        cmd = "hg qinit"
        if dir is not None:
            olddir = os.getcwd()
            os.chdir(dir)
        res, so, se = gquilt_utils.run_cmd(cmd)
        if dir is not None:
            os.chdir(olddir)
        return (res, so, se)
    def is_playground(self, dir=None):
        root = self.get_playground_root(dir)
        if not root:
            return False
        pdir = _patch_dir(root)
        return os.path.exists(pdir) and os.path.isdir(pdir)
    def is_patch_applied(self, patch):
        res, so, se = gquilt_utils.run_cmd("hg qapplied")
        return res == 0 and patch in so.splitlines()
    def patch_file_name(self, patch):
        return _patch_file_name(patch)
    def top_patch(self):
        res, so, se = gquilt_utils.run_cmd("hg qtop")
        if res == 0:
            return so.strip()
        else:
            return ""
    def next_patch(self):
        res, so, se = gquilt_utils.run_cmd("hg qnext")
        if res == 0:
            return so.strip()
        else:
            return ""
    def base_patch(self):
        res, so, se = gquilt_utils.run_cmd("hg qapplied")
        if res == 0 and so:
            lines = so.splitlines(False)
            return lines[0]
        else:
            return ""
    def last_patch_in_series(self):
        res, op, err = gquilt_utils.run_cmd("hg qseries")
        if res != 0:
            raise "mq_specific_error", se
        return op.splitlines()[-1]
    def count_ok_meld(selfself, count):
        return _hg_command_avail("extdiff")
    def has_add_files(self):
        return True
    def has_finish_patch(self):
        return True
    def display_files_diff_in_viewer(self, viewer, files, patch=None):
        top = self.top_patch()
        if not top:
            return
        if not patch:
            patch = top
        elif not self.is_patch_applied(patch):
            return
        res, prev, se = self._patch_most_recent_parent_rev(patch)
        if res != gquilt_const.OK:
            raise "mq_specific_error", se
        pats = (os.P_NOWAIT, "hg", "hg", "extdiff", "--program=" + viewer)
        if patch is top:
            pats += ("--rev=" + prev,)
        else:
            pats += ("--rev=" + prev, "--rev=" + patch)
        pats += tuple(files)
        pid = apply(os.spawnlp, pats)
    def get_playground_root(self, dir=None):
        if dir:
            old_dir = os.getcwd()
            os.chdir(dir)
        res, root, err = gquilt_utils.run_cmd("hg root")
        if dir:
            os.chdir(old_dir)
        if res == 0:
            return root.strip()
        else:
            return None
    def get_patch_status(self, patch):
        if not patch or patch == self.top_patch():
            return (gquilt_const.OK, gquilt_const.TOP_PATCH, "")
        elif self.is_patch_applied(patch):
            return (gquilt_const.OK, gquilt_const.APPLIED, "")
        else:
            return (gquilt_const.OK, gquilt_const.NOT_APPLIED, "")
    def get_series(self):
        series = []
        for cmd, tag in [("qapplied", gquilt_const.APPLIED), ("qunapplied", gquilt_const.NOT_APPLIED)]:
            res, op, err = gquilt_utils.run_cmd("hg " + cmd)
            if res != 0:
                raise "mq_specific_error", se
            for line in op.splitlines():
                series.append((line, tag))
        return (gquilt_const.OK, ("", series), "")
    def get_diff(self, filelist=[], patch=None):
        if not patch or patch == self.top_patch():
            cmd = "hg qdiff"
        elif not self.is_patch_applied(patch):
            res, diff = gquilt_pfuns.get_patch_diff_lines(self.patch_file_name(patch))
            if res:
                return (gquilt_const.OK, os.linesep.join(diff), [])
            return (gquilt_const.ERROR, "", "Patch %s is not applied." % patch)
        else:
            res, prev, se = self._patch_most_recent_parent_rev(patch)
            if res != gquilt_const.OK:
                raise "mq_specific_error", se
            cmd = "hg diff -r " + prev + " -r " + patch
        res, so, se = gquilt_utils.run_cmd(" ".join([cmd, " ".join(filelist)]))
        if res != 0:
            return (gquilt_const.ERROR, so, se)
        return (res, so, se)
    def get_combined_diff(self, start_patch=None, end_patch=None):
        if self._no_patches_applied():
            return (gquilt_const.ERROR, "", "No patches applied")
        cmd = "hg diff --rev "
        if start_patch is None:
            res, prev, se = self._patch_most_recent_parent_rev("qbase")
        else:
            res, prev, se = self._patch_most_recent_parent_rev(start_patch)
        if res != gquilt_const.OK:
            return (res, prev, se)
        cmd += prev
        if end_patch is not None and end_patch != self.top_patch():
            cms += " -rev " + end_patch
        res, so, se = gquilt_utils.run_cmd(cmd)
        if res != 0:
            return (gquilt_const.ERROR, so, se)
        return (res, so, se)
    def get_patch_files(self, patch=None, withstatus=True):
        if self._no_patches_applied():
            return (gquilt_const.OK, "", "")
        cmd = "hg status -m -a -d"
        if not withstatus:
            cmd += " -n"
        top = self.top_patch()
        if patch is None:
            patch = top
        res, prev, se = self._patch_most_recent_parent_rev(patch)
        if res != gquilt_const.OK:
            if patch and not self.is_patch_applied(patch):
                patchfile = self.patch_file_name(patch)
                ok, filelist = gquilt_pfuns.get_patch_files(patchfile, withstatus)
                if ok:
                    return (gquilt_const.OK, filelist, "")
                else:
                    return (gquilt_const.ERROR, "", filelist)
            return (res, prev, se)
        cmd += " --rev " + prev
        if patch != top:
            cmd += " --rev " + patch
        res, so, se = gquilt_utils.run_cmd(cmd)
        if res != 0:
            return (gquilt_const.ERROR, so, se)
        if withstatus:
            filelist = []
            for line in so.splitlines():
                file = line[2:]
                if line[0] == "A":
                    filelist.append((file, gquilt_const.ADDED))
                elif line[0] == "!":
                    filelist.append((file, gquilt_const.DELETED))
                else:
                    filelist.append((file, gquilt_const.EXTANT))
        else:
            filelist = so.splitlines()
        return (res, filelist, se)
    def get_description_is_finish_ready(self, patch):
        pfn = self.patch_file_name(patch)
        res, pf_descr_lines = gquilt_pfuns.get_patch_descr_lines(pfn)
        pf_descr = os.linesep.join(pf_descr_lines).strip()
        if not pf_descr:
            return False
        res, rep_descr, sout = gquilt_utils.run_cmd('hg log --template "{desc}" --rev %s' % patch)
        if pf_descr != rep_descr.strip():
            top = self.top_patch()
            if top == patch:
                gquilt_utils.run_cmd('hg qrefresh')
            else:
                # let's hope the user doesn't mind having the top patch refreshed
                gquilt_utils.run_cmd('hg qrefresh')
                gquilt_utils.run_cmd('hg qgoto %s' % patch)
                gquilt_utils.run_cmd('hg qrefresh')
                gquilt_utils.run_cmd('hg qgoto %s' % top)
        return True
    def do_rename_patch(self, console, patch, newname):
        cmd = "hg qrename "
        if patch is not None:
            cmd += patch + " "
        cmd += newname
        res, so, se = gquilt_tool.exec_console_cmd(console, cmd)
        if res != 0:
            return (gquilt_const.ERROR, so, se)
        else:
            return (gquilt_const.OK, so, se)

    def do_pop_patch(self, console, force=False):
        cmd = "hg qpop"
        if force:
            cmd += " -f"
        res, so, se = gquilt_tool.exec_console_cmd(console, cmd)
        return _convert_pop_res(res, so, se)
    def do_push_patch(self, console, force=False):
        cmd = "hg qpush"
        if force:
            cmd += " -f"
        res, so, se = gquilt_tool.exec_console_cmd(console, cmd)
        return _convert_push_res(res, so, se)
    def do_pop_to_patch(self, console, patch=None):
        cmd = "hg qpop"
        if patch is not None:
            if patch is "":
                cmd += " -a"
            else:
                cmd += " " + patch
        res, so, se = gquilt_tool.exec_console_cmd(console, cmd)
        return _convert_pop_res(res, so, se)
    def do_push_to_patch(self, console, patch=None):
        cmd = "hg qpush"
        if patch is not None:
            if patch is "":
                cmd += " -a"
            else:
                cmd += " " + patch
        res, so, se = gquilt_tool.exec_console_cmd(console, cmd)
        return _convert_push_res(res, so, se)
    def do_refresh_patch(self, console, patch=None, force=False):
        cmd = "hg qrefresh"
        if force:
            return (gquilt_const.OK, "", "mq has no concept of forced refresh")
        if patch is not None and patch is not self.top_patch():
            return (gquilt_const.OK, "", "mq will not refresh non top patches")
        return gquilt_tool.exec_console_cmd(console, cmd)
    def do_create_new_patch(self, console, name, force=False):
        cmd = "hg qnew"
        if force:
            cmd += " -f"
        res, so, se = gquilt_tool.exec_console_cmd(console, " ".join([cmd, name]))
        return _convert_push_res(res, so, se)
    def do_import_patch(self, console, filename, patchname=None, force=False):
        if patchname:
            cmd = "hg qimport -n %s " % patchname
        else:
            cmd = "hg qimport "
            patchname = os.path.basename(filename)
        if self.is_patch_applied(patchname):
            return (gquilt_const.ERROR, "", "Patch %s is already applied!" % patchname)
        if force:
            cmd += "-f "
        res, so, se = gquilt_tool.exec_console_cmd(console, cmd + filename)
        return _convert_import_res(res, so, se)
    def do_finish_patch(self, console, patch):
        return gquilt_tool.exec_console_cmd(console, " ".join(["hg", "qfinish", patch]))
    def do_merge_patch(self, console, patch):
        return gquilt_tool.exec_console_cmd(console, " ".join(["hg", "qfold", patch]))
    def do_merge_patch_file(self, console, filename):
        res, so, se = gquilt_tool.exec_console_cmd(console, " < ".join(["patch -p1 --dry-run", filename]))
        if res != 0:
            return (gquilt_const.ERROR, so, se)
        res, so, se = gquilt_utils.run_cmd("hg status -u")
        old_unknowns = so.splitlines()
        pres, pso, pse = gquilt_tool.exec_console_cmd(console, " < ".join(["patch -p1", filename]))
        if pres != 0:
            return (pres, pso, pse)
        res, so, se = gquilt_utils.run_cmd("hg status -u")
        for f in so.splitlines():
            if f not in old_unknowns:
                console.exec_cmd_with_alert("hg add " + f[2:])
        return (pres, pso, pse)
    def do_delete_patch(self, console, patch):
        return gquilt_tool.exec_console_cmd(console, " ".join(["hg", "qdelete", patch]))
    def _get_untracked_or_ignored_files(self, filelist):
        res, so, se = gquilt_utils.run_cmd(" ".join(["hg status -uin"] + filelist))
        if not res:
            return (res, so, se)
        newfilelist = []
        for line in so.splitlines(False):
            newfilelist.append(line)
        return (res, newfilelist, se)
    def do_add_files_to_patch(self, console, filelist):
        if not self.top_patch():
            return (gquilt_const.ERROR, "", "No patches applied")
        res, trimmedlist, se = self._get_untracked_or_ignored_files(filelist)
        if res:
            return (res, trimmedlist, se)
        if not trimmedlist:
            return (gquilt_const.OK, "", "")
        return gquilt_tool.exec_console_cmd(console, " ".join(["hg add", trimmedlist]))
    def do_copy_file(self, console, file, dest, force=False):
        cmd = "hg copy "
        if force:
            cmd += "-f "
        cmd += " ".join([file, dest])
        res, so, se = gquilt_tool.exec_console_cmd(console, cmd)
        return _convert_copy_res(res, so, se)
    def do_move_file(self, console, file, dest, force=False):
        cmd = "hg rename "
        if force:
            cmd += "-f "
        cmd += " ".join([file, dest])
        res, so, se = gquilt_tool.exec_console_cmd(console, cmd)
        return _convert_copy_res(res, so, se)
    def do_remove_files_from_patch(self, console, filelist, patch=None):
        top = self.top_patch()
        if patch and patch != top:
            return (gquilt_const.ERROR, "", "mq will not modify non top patches")
        if not top:
            return (gquilt_const.ERROR, "", "No patches applied")
        res, prev, se = self._patch_most_recent_parent_rev(top)
        if res != gquilt_const.OK:
            return (res, prev, se)
        files = " ".join(filelist)
        console.exec_cmd_with_alert("hg qrefresh " + files)
        cmd = " ".join(["hg revert --rev", prev, files])
        res, so, se = gquilt_tool.exec_console_cmd(console, cmd)
        return (res, so, se)
    def do_revert_files_in_patch(self, console, filelist, patch=None):
        top = self.top_patch()
        if patch and patch != top:
            return (gquilt_const.ERROR, "", "mq will not modify non top patches")
        if not top:
            return (gquilt_const.ERROR, "", "No patches applied")
        cmd = " ".join(["hg revert", " ".join(filelist)])
        res, so, se = gquilt_tool.exec_console_cmd(console, cmd)
        return (res, so, se)
    def do_exec_tool_cmd(self, console, cmd):
        return gquilt_tool.exec_console_cmd(console, " ".join(["hg", cmd]))
    def get_author_name_and_email(self):
        res, uiusername, serr = gquilt_utils.run_cmd("hg showconfig ui.username")
        if res == 0 and uiusername:
            return uiusername.strip()
        else:
            return gquilt_tool.interface.get_author_name_and_email(self)

