# -*- python -*-

### Copyright (C) 2005 Peter Williams <pwil3058@bigpond.net.au>

### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; version 2 of the License only.

### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.

### You should have received a copy of the GNU General Public License
### along with this program; if not, write to the Free Software
### Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import gtk, os, pango, gquilt_utils, time

def _read_fs_entity_name(flag, prompt, startdir, ok_button_label, sugg=""):
    dialog = gtk.FileChooserDialog(prompt, None, flag,
        (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL, ok_button_label, gtk.RESPONSE_OK))
    dialog.set_default_response(gtk.RESPONSE_OK)
    dialog.set_position(gtk.WIN_POS_MOUSE)
    if flag == gtk.FILE_CHOOSER_ACTION_SAVE:
        dialog.set_current_name(sugg)
    if startdir != None:
        dialog.set_current_folder(startdir)
    response = dialog.run()
    if response == gtk.RESPONSE_OK:
        res = (True, dialog.get_filename())
    elif response == gtk.RESPONSE_CANCEL:
        res = (False, None)
    else:
        res = (True, os.getcwd())
    dialog.destroy()
    return res

def read_new_file_name(prompt="New file ...", startdir=None, ok_button_label=gtk.STOCK_OK, sugg=""):
    return _read_fs_entity_name(gtk.FILE_CHOOSER_ACTION_SAVE, prompt, startdir, ok_button_label, sugg)

def read_file_name(prompt="File ...", startdir=None, ok_button_label=gtk.STOCK_OPEN):
    return _read_fs_entity_name(gtk.FILE_CHOOSER_ACTION_OPEN, prompt, startdir, ok_button_label)

def read_directory_name(prompt="Directory ...", startdir=None, ok_button_label=gtk.STOCK_OPEN):
    return _read_fs_entity_name(gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER, prompt, startdir, ok_button_label)

class _info_dialog(gtk.Dialog):
    def __init__(self, title=None, parent=None, flags=0, buttons=()):
        gtk.Dialog.__init__(self, title, parent, flags, buttons)
        if parent is None:
            self.set_position(gtk.WIN_POS_MOUSE)
        else:
            self.set_position(gtk.WIN_POS_CENTER_ON_PARENT)
        hbox = gtk.HBox()
        self.vbox.add(hbox)
        hbox.show()
        self.image = gtk.Image()
        self.image.set_from_stock(gtk.STOCK_DIALOG_INFO, gtk.ICON_SIZE_DIALOG)
        hbox.pack_start(self.image, expand=False)
        self.image.show()
        sw = gtk.ScrolledWindow()
        sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        sw.show()
        hbox.add(sw)
        self.tv = gtk.TextView()
        self.tv.set_cursor_visible(False)
        self.tv.set_editable(False)
        self.tv.set_size_request(320, 80)
        self.tv.show()
        sw.add(self.tv)
    def set_message(self, msg):
        self.tv.get_buffer().set_text(msg)

class info_dialog(_info_dialog):
    def __init__(self, msg=None, parent=None):
        _info_dialog.__init__(self, "FYI", parent,
                              gtk.DIALOG_DESTROY_WITH_PARENT,
                              (gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE))
        if msg != None:
            self.set_message(msg)
    def inform(self):
        res = self.run()
        self.destroy()

FORCE = 1
REFRESH_AND_RETRY = 2
SKIP = 3
CANCEL = gtk.RESPONSE_CANCEL

class info_force_dialog(_info_dialog):
    def __init__(self, msg=None, parent=None):
        _info_dialog.__init__(self, "FYI", parent,
                              gtk.DIALOG_DESTROY_WITH_PARENT,
                              (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                               "_Force", FORCE))
        if msg != None:
            self.set_message(msg)
    def ask(self):
        res = self.run()
        self.destroy()
        return res == FORCE

class info_force_or_skip_dialog(_info_dialog):
    def __init__(self, msg=None, parent=None):
        _info_dialog.__init__(self, "FYI", parent,
                              gtk.DIALOG_DESTROY_WITH_PARENT,
                              (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                               "_Skip", SKIP,
                               "_Force", FORCE))
        if msg != None:
            self.set_message(msg)
    def ask(self):
        res = self.run()
        self.destroy()
        return res

class info_refresh_dialog(_info_dialog):
    def __init__(self, msg=None, parent=None):
        _info_dialog.__init__(self, "FYI", parent,
                              gtk.DIALOG_DESTROY_WITH_PARENT,
                              (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                               "_Refresh and Retry", REFRESH_AND_RETRY))
        if msg != None:
            self.set_message(msg)
    def ask(self):
        res = self.run()
        self.destroy()
        return res == REFRESH_AND_RETRY

class info_force_or_refresh_dialog(_info_dialog):
    def __init__(self, msg=None, parent=None):
        _info_dialog.__init__(self, "FYI", parent,
                              gtk.DIALOG_DESTROY_WITH_PARENT,
                              (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                              "_Force", FORCE,
                               "_Refresh and Retry", REFRESH_AND_RETRY))
        if msg != None:
            self.set_message(msg)
    def ask(self):
        res = self.run()
        self.destroy()
        return res

class info_ok_dialog(_info_dialog):
    def __init__(self, msg=None, parent=None):
        _info_dialog.__init__(self, "FYI", parent,
                              gtk.DIALOG_DESTROY_WITH_PARENT,
                              (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                               gtk.STOCK_OK, gtk.RESPONSE_OK))
        if msg != None:
            self.set_message(msg)
    def ask(self):
        res = self.run()
        self.destroy()
        return res == gtk.RESPONSE_OK

class info_yes_no_dialog(_info_dialog):
    def __init__(self, msg=None, parent=None):
        _info_dialog.__init__(self, "FYI", parent,
                              gtk.DIALOG_DESTROY_WITH_PARENT,
                              (gtk.STOCK_NO, gtk.RESPONSE_NO,
                               gtk.STOCK_YES, gtk.RESPONSE_YES))
        if msg != None:
            self.set_message(msg)
    def ask(self):
        res = self.run()
        self.destroy()
        return res == gtk.RESPONSE_YES

class text_entry_dialog(gtk.Dialog):
    def __init__(self, prompt="Enter text:", parent=None):
        gtk.Dialog.__init__(self, "Text Entry", parent,
                            gtk.DIALOG_DESTROY_WITH_PARENT,
                            (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                             gtk.STOCK_OK, gtk.RESPONSE_OK)
                           )
        if parent is None:
            self.set_position(gtk.WIN_POS_MOUSE)
        else:
            self.set_position(gtk.WIN_POS_CENTER_ON_PARENT)
        hbox = gtk.HBox()
        self.vbox.add(hbox)
        hbox.show()
        self.prompt = gtk.Label(prompt)
        hbox.pack_start(self.prompt, expand=False)
        self.prompt.show()
        self.te = gtk.Entry()
        self.te.connect("activate", self._enter)
        self.te.show()
        hbox.add(self.te)
    def _enter(self, obj):
        if len(obj.get_text()) > 0:
            self.response(gtk.RESPONSE_OK)
        else:
            self.response(gtk.RESPONSE_CANCEL)
    def read_text(self):
        res = self.run()
        str = self.te.get_text()
        self.destroy()
        return (res == gtk.RESPONSE_OK and len(str) > 0, str)
    def set_text(self, text):
        self.te.set_text(text)

class select_from_list_dialog(gtk.Dialog):
    def __init__(self, list=[], title="Select from list", parent=None):
        gtk.Dialog.__init__(self, title, parent,
                            gtk.DIALOG_DESTROY_WITH_PARENT,
                            (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                             gtk.STOCK_OK, gtk.RESPONSE_OK)
                           )
        if parent is None:
            self.set_position(gtk.WIN_POS_MOUSE)
        else:
            self.set_position(gtk.WIN_POS_CENTER_ON_PARENT)
        self.cbox = gtk.combo_box_new_text()
        for i in list:
            self.cbox.append_text(i)
        self.vbox.add(self.cbox)
        self.cbox.show()
    def make_selection(self):
        res = self.run()
        str = self.cbox.get_active_text()
        self.destroy()
        return (res == gtk.RESPONSE_OK and str is not None, str)

class update_dialog(gtk.Dialog):
    def __init__(self, title):
        gtk.Dialog.__init__(self, title, None,
            gtk.DIALOG_DESTROY_WITH_PARENT, ("_Update", 1, gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE))
        self.connect("response", self._process_response)
        self.connect("close", self._process_close)
    def do_update(self):
        assert 0, 'do_update() must be provided in child class!!!'
    def _process_response(self, dialog, response):
        if response == 1:
            self.do_update()
        else:
            self.destroy()
    def _process_close(self, dialog):
        self.destroy()
    def display(self):
        self.show()

class update_save_dialog(gtk.Dialog):
    def __init__(self, title):
        gtk.Dialog.__init__(self, title, None,
            gtk.DIALOG_DESTROY_WITH_PARENT,
            ("_Update", 1, "_Save", 2, "Save _As", 3,
            gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE))
        self.connect("response", self._process_response)
        self.connect("close", self._process_close)
        self.filename = None
        self.set_response_sensitive(2, False)
    def do_update(self):
        assert 0, 'do_update() must be provided in child class!!!'
    def do_write_to_file(self, filename):
        assert 0, 'do_write_to_file() must be provided in child class!!!'
    def _process_response(self, dialog, response):
        if response == 1:
            self.do_update()
        elif response == 2:
            if not self.do_write_to_file(self.filename):
                info_dialog("Save failed!" ).inform()
        elif response == 3:
            res, filename = read_new_file_name("Save as ...", self.filename, "_Save")
            if not res:
                return
            if self.do_write_to_file(filename):
                self.filename = filename
                self.set_response_sensitive(2, True)
            else:
                info_dialog("Save as \"" + filename + "\" failed!" ).inform()
        else:
            self.destroy()
    def _process_close(self, dialog):
        self.destroy()
    def display(self):
        self.show()

class _file_treestore(gtk.TreeStore):
    def __init__(self):
        gtk.TreeStore.__init__(self, str, "gboolean", int, str, str)
    def is_dir(self, iter):
        return self.get_value(iter, 1)
    def fs_path(self, iter):
        if iter is None:
            return None
        parent = self.iter_parent(iter)
        me = self.get_value(iter, 0)
        if parent is None:
            return me
        else:
            return self.fs_path(parent) + os.sep + me
    def _find_dir(self, lpath, iter):
        while iter != None:
            if not self.is_dir(iter):
                return None
            if self.get_value(iter, 0) == lpath[0]:
                if len(lpath) == 1:
                    return iter
                return self._find_dir(lpath[1:], self.iter_children(iter))
            iter = self.iter_next(iter)
        return iter
    def find_dir(self, path):
        iter = self.get_iter_first()
        lpath = path.split(os.sep)
        return self._find_dir(lpath, iter)
    def find_file_in_dir(self, fname, iter):
        if iter is None:
            iter = self.get_iter_first()
        else:
            iter = self.iter_children(iter)
        while iter != None:
            if not self.is_dir(iter) and self.get_value(iter, 0) == fname:
                break
            iter = self.iter_next(iter)
        return iter
    def find_file(self, path):
        dirpath, fname = os.path.split(path)
        if dirpath == "":
            return self.find_file_in_dir(fname, None)
        dir = self.find_dir(dirpath)
        if dir is None:
            return None
        return self.find_file_in_dir(fname, dir)
    def _get_file_paths(self, iter, list):
        while iter != None:
            if not self.is_dir(iter):
                list.append(self.fs_path(iter))
            else:
                ci = self.iter_children(iter)
                if ci != None:
                    self._get_file_paths(ci, list)
            iter = self.iter_next(iter)
    def get_file_paths(self):
        list = []
        self._get_file_paths(self.get_iter_first(), list)
        return list
    def recursive_remove(self, iter):
        ci = self.iter_children(iter)
        if ci != None:
            while self.remove(ci):
                pass
        return self.remove(iter)

class scrolled_treeview_with_popup(gtk.Frame):
    def __init__(self, store, view_title=None, tooltips=None, frame_title=None):
        gtk.Frame.__init__(self)
        self.store = store
        self.view = gtk.TreeView(self.store)
        sw = gtk.ScrolledWindow()
        sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        sw.add(self.view)
        self.add(sw)
        self.show_all()
        self.menu = gtk.Menu()
        self.view.connect("button_press_event", self._handle_button_press)
        self.set_tooltips(tooltips)
        self._selection_mi_list = []
        self._unique_selection_mi_list = []
        self._conditional_mi_list = []
    def set_tooltips(self, tooltips):
        self.tooltips = tooltips
    def _popup_ok(self):
        return True
    def _handle_button_press(self, widget, event):
        if event.type == gtk.gdk.BUTTON_PRESS:
            if event.button == 3:
                if not self._popup_ok():
                    return False
                sel_sz = self.view.get_selection().count_selected_rows()
                unique = sel_sz == 1
                selmade = sel_sz > 0
                for mi in self._selection_mi_list:
                    mi.set_sensitive(selmade)
                for mi in self._unique_selection_mi_list:
                    mi.set_sensitive(unique)
                for mit in self._conditional_mi_list:
                    mit[0].set_sensitive(mit[1](mit[2]))
                self.menu.popup(None, None, None, event.button, event.time)
                return True
            elif event.button == 2:
                self.view.get_selection().unselect_all()
                return True
        elif event.type == gtk.gdk._2BUTTON_PRESS:
            if event.button == 1:
                self._handle_double_click()
        return False
    def _handle_double_click(self):
        pass
    def append_menu_item(self, label, tooltip, stock_id, action, arg=None):
        mi = gtk.ImageMenuItem(label)
        if stock_id is not None:
            im = gtk.Image()
            im.set_from_stock(stock_id, gtk.ICON_SIZE_MENU)
            mi.set_image(im)
        if (self.tooltips is not None) and (tooltip is not None):
            self.tooltips.set_tip(mi, tooltip)
        mi.show()
        if arg is None:
            mi.connect("activate", action)
        else:
            mi.connect("activate", action, arg)
        self.menu.append(mi)
        return mi
    def append_selection_menu_item(self, label, tooltip, stock_id, action, arg=None):
        mi = self.append_menu_item(label, tooltip, stock_id, action, arg)
        self._selection_mi_list.append(mi)
    def append_unique_selection_menu_item(self, label, tooltip, stock_id, action, arg=None):
        mi = self.append_menu_item(label, tooltip, stock_id, action, arg)
        self._unique_selection_mi_list.append(mi)
    def append_conditional_menu_item(self, label, tooltip, stock_id, action, cond, arg=None, condarg=None):
        mi = self.append_menu_item(label, tooltip, stock_id, action, arg)
        self._conditional_mi_list.append((mi, cond, condarg))
    def _show_busy(self):
        self.window.set_cursor(gtk.gdk.Cursor(gtk.gdk.WATCH))
        while gtk.events_pending():
            gtk.main_iteration()
    def _unshow_busy(self):
        self.window.set_cursor(None)

class file_tree(scrolled_treeview_with_popup):
    def __init__(self, view_title="Files", tooltips=None, frame_title=None):
        scrolled_treeview_with_popup.__init__(self, _file_treestore(), view_title, tooltips, frame_title)
        text_cell = gtk.CellRendererText()
        icon_cell = gtk.CellRendererPixbuf()
        tvcolumn = gtk.TreeViewColumn(view_title)
        tvcolumn.pack_start(icon_cell, False)
        tvcolumn.pack_start(text_cell)
        tvcolumn.set_attributes(text_cell, text=0, style=2, foreground=3)
        tvcolumn.set_attributes(icon_cell, stock_id=4)
        self.view.append_column(tvcolumn)
        self.console = None
    def set_console(self, console):
        self.console = console
    def _get_selected_files_cb(self, store, path, iter, list):
        list.append(store.fs_path(iter))
    def get_selected_files(self):
        res = []
        self.view.get_selection().selected_foreach(self._get_selected_files_cb, res)
        return res

class dir_file_tree(file_tree):
    def _on_row_expanded(self, view, iter, path):
        store = view.get_model()
        ci = store.iter_children(iter)
        if store.get_value(ci, 0) is None:
            # This was the place holder row put in unpopulated directories
            # to make them display properly so we need to remove it
            store.remove(ci)
            self._populate(self.store.fs_path(iter), iter)
        else:
            self._update_dir(self.store.fs_path(iter), iter)
        view.expand_row(path, False)
    def __init__(self, view_title="Directory Files", root=None, tooltips=None, frame_title=None):
        file_tree.__init__(self, view_title, tooltips, frame_title)
        self.root = root
        self.show_hidden = False
        self.view.connect("row-expanded", self._on_row_expanded)
        if root != None:
            self.set_root(root)
    def _dir_contents(self, dir):
        hdirs, dirs, hfiles, files = gquilt_utils.dir_contents(dir)
        if self.show_hidden:
            return (hdirs + dirs, hfiles + files)
        else:
            return (dirs, files)
        pass
    def _populate(self, dir, parent=None, style=pango.STYLE_NORMAL, foreground="black"):
        dirs, files = self._dir_contents(dir)
        dirs.sort()
        for d in dirs:
            iter = self.store.append(parent, (d, True, style, foreground, gtk.STOCK_DIRECTORY))
            self.store.append(iter)
        files.sort()
        for f in files:
            iter = self.store.append(parent, (f, False, style, foreground, gtk.STOCK_FILE))
    def _update_dir(self, dir, parent=None, style=pango.STYLE_NORMAL, foreground="black"):
        dirs, files = self._dir_contents(dir)
        dirs.sort()
        files.sort()
        if parent is None:
            ci = self.store.get_iter_first()
        else:
            ci = self.store.iter_children(parent)
        dead_entries = []
        for d in dirs:
            while (ci is not None) and self.store.is_dir(ci) and (self.store.get_value(ci, 0) < d):
                dead_entries.append(ci)
                ci = self.store.iter_next(ci)
            if ci is None:
                iter = self.store.append(parent, (d, True, style, foreground, gtk.STOCK_DIRECTORY))
                self.store.append(iter)
                continue
            name = self.store.get_value(ci, 0)
            if (not self.store.is_dir(ci)) or (name > d):
                iter = self.store.insert_before(parent, ci, (d, True, style, foreground, gtk.STOCK_DIRECTORY))
                self.store.append(iter)
                continue
            if self.view.row_expanded(self.store.get_path(ci)):
                self._update_dir(dir + os.sep + name, ci, style, foreground)
            ci = self.store.iter_next(ci)
        while (ci is not None) and self.store.is_dir(ci):
            dead_entries.append(ci)
            ci = self.store.iter_next(ci)
        for f in files:
            while (ci is not None) and (self.store.get_value(ci, 0) < f):
                dead_entries.append(ci)
                ci = self.store.iter_next(ci)
            if ci is None:
                iter = self.store.append(parent, (f, False, style, foreground, gtk.STOCK_FILE))
                continue
            if self.store.get_value(ci, 0) > f:
                iter = self.store.insert_before(parent, ci, (f, False, style, foreground, gtk.STOCK_FILE))
                continue
            ci = self.store.iter_next(ci)
        while ci is not None:
            dead_entries.append(ci)
            ci = self.store.iter_next(ci)
        for e in dead_entries:
            self.store.recursive_remove(e)
    def set_root(self, root=None):
        if root is None:
            self.root = os.getcwd()
        else:
            self.root = root
        self.store.clear()
        self._populate(self.root)
    def update(self, iter=None):
        if iter is None:
            self._update_dir(".", None)
        else:
            path = self.store.fs_path(iter)
            if not os.path.exists(path):
                self.store.recursive_remove(iter)
            elif os.path.isdir(path):
                self._update_dir(path, iter)

class path_file_tree(file_tree):
    def __init__(self, view_title="Path Files", tooltips=None, frame_title=None):
        file_tree.__init__(self, view_title, tooltips, frame_title)
    def _find_or_insert_dir(self, parent, name, style=pango.STYLE_NORMAL, foreground="black"):
        if parent is None:
            iter = self.store.get_iter_first()
        else:
            iter = self.store.iter_children(parent)
        if iter is None:
            return (False, self.store.append(parent, (name, True, style, foreground, gtk.STOCK_DIRECTORY)))
        while self.store.is_dir(iter) and self.store.get_value(iter, 0) < name:
            next = self.store.iter_next(iter)
            if next is None or not self.store.get_value(next, 1):
                return (False, self.store.insert_after(parent, iter, (name, True, style, foreground, gtk.STOCK_DIRECTORY)))
            iter = next
        if self.store.get_value(iter, 0) == name:
            return (True, iter)
        return (False, self.store.insert_before(parent, iter, (name, True, style, foreground, gtk.STOCK_DIRECTORY)))
    def find_or_insert_dir(self, path, foreground="black"):
        if path == "":
            return (False, None)
        if os.path.exists(path):
            style=pango.STYLE_NORMAL
        else:
            style=pango.STYLE_ITALIC
        iter = None
        for name in path.split(os.sep):
            found, iter = self._find_or_insert_dir(iter, name, style, foreground)
        return (found, iter)
    def _find_or_insert_file(self, parent, name, style=pango.STYLE_NORMAL, foreground="black"):
        if parent is None:
            iter = self.store.get_iter_first()
        else:
            iter = self.store.iter_children(parent)
        if iter is None:
            return (False, self.store.append(parent, (name, False, style, foreground, gtk.STOCK_FILE)))
        while self.store.is_dir(iter):
            next = self.store.iter_next(iter)
            if next is None:
                return (False, self.store.insert_after(parent, iter, (name, False, style, foreground, gtk.STOCK_FILE)))
            iter = next
        while self.store.get_value(iter, 0) < name:
            next = self.store.iter_next(iter)
            if next is None:
                return (False, self.store.insert_after(parent, iter, (name, False, style, foreground, gtk.STOCK_FILE)))
            iter = next
        if self.store.get_value(iter, 0) == name:
            return (True, iter)
        return (False, self.store.insert_before(parent, iter, (name, False, style, foreground, gtk.STOCK_FILE)))
    def find_or_insert_file(self, path, foreground="black"):
        if os.path.exists(path):
            style=pango.STYLE_NORMAL
        else:
            style=pango.STYLE_ITALIC
        dir, name = os.path.split(path)
        found, iter = self.find_or_insert_dir(dir)
        return self._find_or_insert_file(iter, name, style, foreground)
    def delete_file(self, path):
        file_iter = self.store.find_file(path)
        if file_iter is None:
            return
        parent_iter = self.store.iter_parent(file_iter)
        self.store.recursive_remove(file_iter)
        while parent_iter is not None:
            ci = self.store.iter_children(parent_iter)
            if ci is None:
                pi = parent_iter
                parent_iter = self.store.iter_parent(pi)
                self.store.remove(pi)
            else:
                break

class text_list(scrolled_treeview_with_popup):
    def __init__(self, view_title="List", tooltips=None, frame_title=None):
        scrolled_treeview_with_popup.__init__(self, gtk.ListStore(str, int, str, str), view_title, tooltips, frame_title)
        text_cell = gtk.CellRendererText()
        icon_cell = gtk.CellRendererPixbuf()
        tvcolumn = gtk.TreeViewColumn(view_title)
        tvcolumn.pack_start(icon_cell, False)
        tvcolumn.pack_start(text_cell)
        tvcolumn.set_attributes(text_cell, text=0, style=1, foreground=2)
        tvcolumn.set_attributes(icon_cell, stock_id=3)
        self.view.append_column(tvcolumn)
    def get_selected_text(self):
        model, iter = self.view.get_selection().get_selected()
        if iter is None:
            return None
        else:
            return model.get_value(iter, 0)
    def append(self, text, style=pango.STYLE_NORMAL, foreground="black", stock_id=None):
        self.store.append([text, style, foreground, stock_id])

class console(gtk.Frame):
    def __init__(self, frame_title=None):
        gtk.Frame.__init__(self)
        self.view = gtk.TextView()
        self.text = self.view.get_buffer()
        self.dat_tag = self.text.create_tag("DAT", weight=pango.WEIGHT_BOLD)
        self.stderr_tag = self.text.create_tag("STDERR", foreground="red")
        self.stdout_tag = self.text.create_tag("STDOUT", foreground="black")
        self.cmd_tag = self.text.create_tag("CMD", foreground="black")
        self.eobuf = self.text.create_mark("eobuf", self.text.get_end_iter(), False)
        sw = gtk.ScrolledWindow()
        sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        sw.add(self.view)
        self.add(sw)
        self.view.set_cursor_visible(False)
        self.view.set_editable(False)
        self.show_all()
    def _append_to_console_tagged(self, msg, tag):
        self.text.insert_with_tags(self.text.get_end_iter(), msg, tag)
        self.view.scroll_to_mark(self.eobuf, 0.001)
    def append_dat(self, suffix=""):
        self._append_to_console_tagged(time.strftime("%Y-%m-%d %H:%M:%S") + suffix, self.dat_tag)
    def echo_cmd(self, cmd):
        self.append_dat(": ")
        self._append_to_console_tagged(cmd, self.cmd_tag)
    def append_stdout(self, msg):
        self._append_to_console_tagged(msg, self.stdout_tag)
    def append_stderr(self, msg):
        self._append_to_console_tagged(msg, self.stderr_tag)
    def append_newline(self):
        self._append_to_console_tagged("\n", self.stdout_tag)
    def log_cmd(self, cmd, so, se):
        self.echo_cmd(cmd)
        if so is not None:
            self.append_stdout(so)
        if se is not None:
            self.append_stderr(se)
    def exec_cmd(self, cmd, args=""):
        if args is not None and args != "":
            cmd += " " + args
        self.echo_cmd(cmd)
        self.append_newline()
        res, op, err = gquilt_utils.run_cmd(cmd)
        self.append_stdout(op)
        self.append_stderr(err)
        if res != 0:
            if len(err) > 0:
                info_dialog(err).inform()
            else:
                info_dialog(op).inform()
    def exec_cmd_with_force(self, cmd, args="", force_option="-f"):
        if args is None or args == "":
            rcmd = cmd
        else:
            rcmd = cmd + " " + args
            force_option += " " + args
        self.echo_cmd(rcmd)
        self.append_newline()
        res, op, err = gquilt_utils.run_cmd(rcmd)
        self.append_stdout(op)
        self.append_stderr(err)
        if res != 0:
            if len(err) > 0:
                emsg = err
            else:
                emsg = op
            if info_force_dialog(emsg).ask():
                self.exec_cmd(cmd, force_option)

class scrolled_diff_text(gtk.Frame):
    def __init__(self, frame_title=None):
        gtk.Frame.__init__(self)
        self.view = gtk.TextView()
        self.text = self.view.get_buffer()
        self.index_tag = self.text.create_tag("INDEX", weight=pango.WEIGHT_BOLD, foreground="#0000AA", family="monospace")
        self.sep_tag = self.text.create_tag("SEP", weight=pango.WEIGHT_BOLD, foreground="#0000AA", family="monospace")
        self.minus_tag = self.text.create_tag("MINUS", foreground="#AA0000", family="monospace")
        self.lab_tag = self.text.create_tag("LAB", foreground="#AA0000", family="monospace")
        self.plus_tag = self.text.create_tag("PLUS", foreground="#006600", family="monospace")
        self.star_tag = self.text.create_tag("STAR", foreground="#006600", family="monospace")
        self.rab_tag = self.text.create_tag("RAB", foreground="#006600", family="monospace")
        self.change_tag = self.text.create_tag("CHANGED", foreground="#AA6600", family="monospace")
        self.stats_tag = self.text.create_tag("STATS", foreground="#AA00AA", family="monospace")
        self.func_tag = self.text.create_tag("FUNC", foreground="#00AAAA", family="monospace")
        self.unchanged_tag = self.text.create_tag("UNCHANGED", foreground="black", family="monospace")
        self.view.set_cursor_visible(False)
        self.view.set_editable(False)
        sw = gtk.ScrolledWindow()
        sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        sw.add(self.view)
        self.add(sw)
        self.show_all()
    def _append_tagged_text(self, text, tag):
        self.text.insert_with_tags(self.text.get_end_iter(), text, tag)
    def _append_patch_line(self, line):
        fc = line[0]
        if fc == " ":
            self._append_tagged_text(line, self.unchanged_tag)
        elif fc == "+":
            self._append_tagged_text(line, self.plus_tag)
        elif fc == "-":
            self._append_tagged_text(line, self.minus_tag)
        elif fc == "!":
            self._append_tagged_text(line, self.change_tag)
        elif fc == "@":
            i = line.find("@@", 2)
            if i == -1:
                self._append_tagged_text(line, self.stats_tag)
            else:
                self._append_tagged_text(line[:i+2], self.stats_tag)
                self._append_tagged_text(line[i+2:], self.func_tag)
                pass
        elif fc == "=":
            self._append_tagged_text(line, self.sep_tag)
        elif fc == "*":
            self._append_tagged_text(line, self.star_tag)
        elif fc == "<":
            self._append_tagged_text(line, self.lab_tag)
        elif fc == ">":
            self._append_tagged_text(line, self.rab_tag)
        else:
            self._append_tagged_text(line, self.index_tag)
    def set_contents(self, text):
        self.text.set_text("")
        for line in text.splitlines():
            self._append_patch_line(line + "\n")
    def save_to_file(self, filename):
        try:
            file = open(filename, 'w')
        except IOError:
            return False
        text = self.text.get_text(self.text.get_start_iter(), self.text.get_end_iter())
        file.write(text)
        file.close()
        return True
    
