/*
 * (SLIK) SimpLIstic sKin functions
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#include "ui2_includes.h"
#include "ui2_typedefs.h"
#include "ui2_skin.h"

#include <math.h>

#include "ui2_button.h"
#include "ui2_display.h"
#include "ui2_main.h"
#include "ui2_util.h"
#include "ui2_widget.h"
#include "ui_pixbuf_ops.h"


/*
 *-------------
 * background
 *-------------
 */

static void skin_back_setup(SkinData *skin, UIData *ui)
{
	GdkPixbuf *s;
	GdkPixbuf *d;
	gint w, h;

	if (!skin->real_overlay) return;
	if (skin->overlay && !skin->sizeable && !skin->has_border &&
	    skin->width == gdk_pixbuf_get_width(skin->overlay) &&
	    skin->height == gdk_pixbuf_get_height(skin->overlay) ) return;

	if (skin->overlay) gdk_pixbuf_unref(skin->overlay);
	skin->overlay = gdk_pixbuf_new(GDK_COLORSPACE_RGB, gdk_pixbuf_get_has_alpha(skin->real_overlay), 8,
				       skin->width, skin->height);

	s = skin->real_overlay;
	w = gdk_pixbuf_get_width(s);
	h = gdk_pixbuf_get_height(s);

	d = skin->overlay;

	if (skin->has_border)
		{
		gint n, m;
		/* corners */
		pixbuf_copy_area(s, 0, 0,
				 d, 0, 0,
				 skin->border_left, skin->border_top, TRUE);
		pixbuf_copy_area(s, w - skin->border_right, 0,
				 d, skin->width - skin->border_right, 0,
				 skin->border_right, skin->border_top, TRUE);
		pixbuf_copy_area(s, 0, h - skin->border_bottom,
				 d, 0, skin->height - skin->border_bottom,
				 skin->border_left, skin->border_bottom, TRUE);
		pixbuf_copy_area(s, w - skin->border_right, h - skin->border_bottom,
				 d, skin->width - skin->border_right, skin->height - skin->border_bottom,
				 skin->border_right, skin->border_bottom, TRUE);

		/* sides */
		n = skin->border_top + skin->border_bottom;	/* vert borders tot. */
		m = skin->border_left + skin->border_right;	/* horz borders tot. */

		pixbuf_copy_fill(s, 0, skin->border_top, skin->border_left, h - n,
				 d, 0, skin->border_top, skin->border_left, skin->height - n,
				 skin->border_left_stretch, TRUE);
		pixbuf_copy_fill(s, w - skin->border_right, skin->border_top, skin->border_right, h - n,
				 d, skin->width - skin->border_right, skin->border_top, skin->border_right, skin->height - n,
				 skin->border_right_stretch, TRUE);

		pixbuf_copy_fill(s, skin->border_left, 0, w - m, skin->border_top,
				 d, skin->border_left, 0, skin->width - m, skin->border_top,
				 skin->border_top_stretch, TRUE);
		pixbuf_copy_fill(s, skin->border_left, h - skin->border_bottom, w - m, skin->border_bottom,
				 d, skin->border_left, skin->height - skin->border_bottom, skin->width - m, skin->border_bottom,
				 skin->border_bottom_stretch, TRUE);

		/* center field */
		pixbuf_copy_fill(s, skin->border_left, skin->border_top, w - m, h - n,
				 d, skin->border_left, skin->border_top, skin->width - m, skin->height - n,
				 skin->stretch, TRUE);
		}
	else
		{
		pixbuf_copy_fill(s, 0, 0, w, h,
				 d, 0, 0, skin->width, skin->height,
				 skin->stretch, TRUE);
		}

	if (skin->mask_buffer) gdk_bitmap_unref(skin->mask_buffer);
	skin->mask_buffer = NULL;
	if (skin->mask)
		{
		gdk_pixbuf_render_pixmap_and_mask(skin->mask, NULL, &skin->mask_buffer, 128);
		}
	else
		{
		gdk_pixbuf_render_pixmap_and_mask(skin->overlay, NULL, &skin->mask_buffer, 1);
		}
}

/*
 *-------------
 * skin side keys
 *-------------
 */

WidgetData *skin_register_widget(SkinData *skin, const gchar *key, const gchar *text_id, WidgetType type, gpointer widget)
{
	WidgetData *wd;

	if (key == NULL)
		{
		printf("Attempt to register skin widget with NULL key!\n");
		return NULL;
		}
	if (debug_mode) printf("skin registering widget \"%s\" (%d)\n", key, type);
	wd = ui_widget_new(key, text_id, type, widget);
	skin->widget_list = g_list_append(skin->widget_list, wd);

	return wd;
}

void skin_register_free_all(SkinData *skin)
{
	while(skin->widget_list)
		{
		WidgetData *wd = skin->widget_list->data;
		skin->widget_list = g_list_remove(skin->widget_list, wd);

		ui_widget_free(wd);
		}
}

WidgetData *skin_widget_get_by_key(SkinData *skin, const gchar *key, WidgetType type)
{
	GList *work;

	if (!key) return NULL;

	work = skin->widget_list;
	while(work)
		{
		WidgetData *wd = work->data;

		if (wd->type == type && strcmp(key, wd->key) == 0) return wd;

		work = work->next;
		}
	return NULL;
}

WidgetData *skin_widget_get_by_text_id(SkinData *skin, const gchar *text_id)
{
	GList *work;

	if (!text_id) return NULL;

	work = skin->widget_list;
	while(work)
		{
		WidgetData *wd = work->data;

		if (wd->text_id && strcmp(text_id, wd->text_id) == 0) return wd;

		work = work->next;
		}
	return NULL;
}

static gint skin_widget_for_each_do(UIData *ui, const gchar *key, WidgetType type,
				    void (*func)(WidgetData *wd, gpointer data, GdkPixbuf *pb, UIData *ui),
				    gpointer data)
{
	gint count = 0;
	GList *work;

	if (!ui->skin) return 0;

	work = ui->skin->widget_list;
	while (work)
		{
		WidgetData *wd = work->data;

		if (wd->type == type && wd->in_bounds && !wd->hidden && strcmp(key, wd->key) == 0)
			{
			func(wd, data, ui->skin->pixbuf, ui);
			count++;
			}

		work = work->next;
		}

	return count;
}

gint skin_widget_for_each_key(UIData *ui, const gchar *key, WidgetType type,
			      void (*func)(WidgetData *wd, gpointer data, GdkPixbuf *pb, UIData *ui),
			      gpointer data)
{
	gint count;
	GList *work;
	UIData *parent;

	if (!func || !key) return 0;

	if (ui->parent)
		{
		parent = ui->parent;
		}
	else
		{
		parent = ui;
		}

	count = skin_widget_for_each_do(ui, key, type, func, data);

	/* obey private keys */
	if (ui_registered_key_is_private(ui, key, type))
		{
		return count;
		}

	/*
	 * Children keys override the parent's,
	 * so if at least one hit, do not propogate up to parent.
	 */
	if (parent != ui && count > 0) return count;

	count += skin_widget_for_each_do(parent, key, type, func, data);

	work = parent->children;
	while (work)
		{
		UIData *child;

		child = work->data;

		if (child != ui)
			{
			count += skin_widget_for_each_do(child, key, type, func, data);
			}

		work = work->next;
		}

	/* editor updates (only parent can have an editor) */
	if (parent->edit)
		{
		skin_widget_for_each_key(parent->edit, key, type, func, data);
		}

	return count;
}

static gint skin_widget_for_each_key_simple(SkinData *skin, const gchar *key, WidgetType type,
					    void (*func)(WidgetData *wd, gpointer data),
					    gpointer data)
{
	gint count = 0;
	GList *work;

	if (!func || !key) return 0;

	work = skin->widget_list;
	while(work)
		{
		WidgetData *wd = work->data;

		if (!key || (wd->type == type && strcmp(key, wd->key) == 0))
			{
			func(wd, data);
			count++;
			}

		work = work->next;
		}

	return count;
}

/*
 *-------------
 * skin stuff
 *-------------
 */

SkinData *skin_new(void)
{
	SkinData *skin;

	skin = g_new0(SkinData, 1);
	skin->widget_list = NULL;
	skin->transparent = FALSE;
	skin->ui = NULL;

	skin->background_filename = NULL;
	skin->background_mask_filename = NULL;
	skin->focus_filename = NULL;

	skin->focus_overlay = NULL;
	skin->focus_stretch = FALSE;
	skin->focus_has_border = FALSE;
	skin->focus_border_left = 0;
	skin->focus_border_right = 0;
	skin->focus_border_top = 0;
	skin->focus_border_bottom = 0;
	skin->focus_anchor_right = FALSE;
	skin->focus_anchor_bottom = FALSE;

	skin->focus_box_filled = FALSE;
	skin->focus_box_alpha = 128;
	skin->focus_box_r = 255;
	skin->focus_box_g = 0;
	skin->focus_box_b = 0;

	return skin;
}

void skin_free(SkinData *skin)
{
	if (!skin) return;

	while(skin->widget_list)
		{
		WidgetData *wd = skin->widget_list->data;
		skin->widget_list = g_list_remove(skin->widget_list, wd);
		ui_widget_free(wd);
		}

	if (skin->overlay) gdk_pixbuf_unref(skin->overlay);
	if (skin->pixbuf) gdk_pixbuf_unref(skin->pixbuf);
	if (skin->buffer) gdk_pixmap_unref(skin->buffer);
	if (skin->mask) gdk_pixbuf_unref(skin->mask);
	if (skin->mask_buffer) gdk_bitmap_unref(skin->mask_buffer);

	if (skin->real_overlay) gdk_pixbuf_unref(skin->real_overlay);

	if (skin->focus_overlay) gdk_pixbuf_unref(skin->focus_overlay);

	g_free(skin->background_filename);
	g_free(skin->background_mask_filename);
	g_free(skin->focus_filename);

	g_free(skin);
}

GdkPixmap *skin_get_buffer(SkinData *skin)
{
	return skin->buffer;
}

GdkPixbuf *skin_get_pixbuf(SkinData *skin)
{
	return skin->pixbuf;
}

UIData *skin_get_ui(SkinData *skin)
{
	return skin->ui;
}

static void skin_sync_back_on_widgets(SkinData *skin, UIData *ui)
{
	GList *work;

	work = skin->widget_list;
	while(work)
		{
		WidgetData *wd = work->data;
		work = work->next;
		ui_widget_sync_back(ui, wd);
		}
}

void skin_sync_back(SkinData *skin, UIData *ui)
{
	GdkPixbuf *pb;

	if (!skin || !skin->pixbuf) return;

	pb = skin->pixbuf;

	if (ui->back_func &&
	    (gdk_pixbuf_get_has_alpha(skin->overlay) || slik_transparency_force) &&
	    ui->back_func(ui, pb, ui->back_data))
		{
		if (slik_transparency_force)
			{
			pixbuf_copy_area_alpha(skin->overlay, 0, 0, pb, 0, 0, skin->width, skin->height,
					       slik_transparency_force_a);
			}
		else
			{
			pixbuf_copy_area_alpha(skin->overlay, 0, 0, pb, 0, 0, skin->width, skin->height, 255);
			}
		}
	else if (skin->transparent || slik_transparency_force)
		{
		gint x, y;

		if (ui->window)
			{
			gdk_window_get_position(ui->window->window, &x, &y);
			}
		else
			{
			gdk_window_get_position(ui->display->window, &x, &y);
			}

		util_pixbuf_fill_from_root_window(pb, x, y, FALSE);

		if (slik_transparency_force)
			{
			pixbuf_copy_area_alpha(skin->overlay, 0, 0, pb, 0, 0, skin->width, skin->height,
					       slik_transparency_force_a);
			}
		else
			{
			pixbuf_copy_area_alpha(skin->overlay, 0, 0, pb, 0, 0, skin->width, skin->height, 255);
			}
		}
	else
		{
		pixbuf_copy_area(skin->overlay, 0, 0, pb, 0, 0, skin->width, skin->height, FALSE);
		}

	skin_sync_back_on_widgets(skin, ui);
}

void skin_set_underlay(SkinData *skin, UIData *ui, GdkPixbuf *pb)
{
	if (!skin || !skin->pixbuf || !pb) return;

	pixbuf_copy_area(pb, 0, 0, skin->pixbuf, 0, 0, skin->width, skin->height, FALSE);
	pixbuf_copy_area_alpha(skin->overlay, 0, 0, skin->pixbuf, 0, 0, skin->width, skin->height, 255);

	skin_sync_back_on_widgets(skin, ui);
}

void skin_sync_buffer(SkinData *skin, UIData *ui)
{
	if (!skin) return;

	skin_back_setup(skin, ui);

	if (!GTK_WIDGET_REALIZED(ui->display)) gtk_widget_realize(ui->display);

	if (skin->buffer) gdk_pixmap_unref(skin->buffer);
	skin->buffer = gdk_pixmap_new(ui->display->window, skin->width, skin->height, -1);

	if (skin->pixbuf) gdk_pixbuf_unref(skin->pixbuf);
	skin->pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8, skin->width, skin->height);

	skin_sync_back(skin, ui);
}

void skin_debug_print_registered_keys(SkinData *skin)
{
	GList *work;

	if (!skin)
		{
		printf("no skin\n");
		return;
		}

	printf("SKIN widget keys (%3d):\n", g_list_length(skin->widget_list));
	printf("-[key]----------------[in bounds]-[type]-----[text id]--[data]-\n");

	work = skin->widget_list;
	while(work)
		{
		WidgetData *wd;
		const gchar *data;

		wd = work->data;
		work = work->next;

		data = ui_widget_get_data(wd, "data");

		printf("%-30s %1d   %-10s %-10s %s\n", wd->key, wd->in_bounds, ui_widget_type_to_text(wd->type), (wd->text_id) ? wd->text_id : "", (data) ? data : "");
		}
}

void skin_widget_check_bounds(SkinData *skin, WidgetData *wd)
{
	gint x, y, w, h;

	if (wd->od && wd->od->is_visible &&
	    ui_widget_get_geometry(wd, &x, &y, &w, &h))
		{
		if (x < 0 || y < 0 ||
		    x + w > skin->width || y + h > skin->height)
			{
			wd->in_bounds = FALSE;
			}
		else
			{
			wd->in_bounds = TRUE;
			}
		}
}

void skin_check_bounds(SkinData *skin)
{
	GList *work;

	if (!skin) return;

	work = skin->widget_list;
	while (work)
		{
		WidgetData *wd = work->data;
		skin_widget_check_bounds(skin, wd);
		work = work->next;
		}
}

/* this also takes on the work of calling set_size */
static void skin_move_anchored(SkinData *skin, gint move_w, gint move_h)
{
	GList *work;

	work = skin->widget_list;
	while(work)
		{
		WidgetData *wd = work->data;
		gint x, y;

		if ((wd->anchor_right || wd->anchor_bottom) &&
		    ui_widget_get_geometry(wd, &x, &y, NULL, NULL))
			{
			if (wd->anchor_right) x += move_w;
			if (wd->anchor_bottom) y += move_h;
			ui_widget_set_coord(skin->ui, wd, x, y, FALSE);
			}
		ui_widget_set_size(skin->ui, wd, move_w, move_h, FALSE);

		work = work->next;
		}
}

void skin_resize(UIData *ui, gint w, gint h)
{
	gint x, y;
	gint dw, dh;

	if (w > SKIN_SIZE_MAX) w = SKIN_SIZE_MAX;
	if (h > SKIN_SIZE_MAX) h = SKIN_SIZE_MAX;
	if (w < SKIN_SIZE_MIN) w = SKIN_SIZE_MIN;
	if (h < SKIN_SIZE_MIN) h = SKIN_SIZE_MIN;

	if (w == ui->skin->width && h == ui->skin->height) return;
	if (w < 4 || h < 4) return;

	x = y = 0;

	if (ui->window) gdk_window_get_position(ui->window->window, &x, &y);

	dw = w - ui->skin->width;
	dh = h - ui->skin->height;

	ui->skin->width = w;
	ui->skin->height = h;

	skin_move_anchored(ui->skin, dw, dh);

	ui_display_sync_all(ui);

	if (ui->window) gdk_window_move(ui->window->window, x, y);
}

/*
 *-------------
 * skin widget init (after setting to a ui)
 *-------------
 */

void skin_widgets_init(SkinData *skin, UIData *ui)
{
	GList *work;

	if (!skin || !ui) return;

	work = skin->widget_list;
	while (work)
		{
		WidgetData *wd = work->data;

		if (wd->od && wd->od->func_init)
			{
			wd->od->func_init(wd->widget, wd->key, ui);
			}
		
		work = work->next;
		}
}

/*
 *-------------
 * skin resizing button handling
 *-------------
 */

static void (*skin_size_func_motion)(gpointer, const gchar *, gint, gint, GdkPixbuf *, UIData *) = NULL;

static void skin_resize_real_motion(gpointer widget, const gchar *key, gint x, gint y, GdkPixbuf *pb, UIData *ui)
{
	ButtonData *button = widget;

	if (button->active && ui->skin->sizeable && ui->skin->real_overlay)
		{
		SkinData *skin = ui->skin;
		gint mx, my;
		gint nw, nh;
		gint w, h;

		mx = x - ui->press_x;
		my = y - ui->press_y;

		/* sanity check */
		if (skin->width_inc < 1) skin->width_inc = 1;
		if (skin->height_inc < 1) skin->height_inc = 1;

		/* lock to increments */
		mx = (gint)floor((float)mx / skin->width_inc + 0.5) * skin->width_inc;
		my = (gint)floor((float)my / skin->height_inc + 0.5) * skin->height_inc;

		/* compute new size */
		nw = skin->width + mx;
		nh = skin->height + my;

		/* clamp to valid range */
		if (nw < skin->width_min) nw = skin->width_min;
		if (nw > skin->width_max) nw = skin->width_max;
		if (nh < skin->height_min) nh = skin->height_min;
		if (nh > skin->height_max) nh = skin->height_max;

		w = nw - skin->width;
		h = nh - skin->height;

		if ((w != 0 || h != 0) &&
		    button->x + w >= 0 &&
		    button->y + h >= 0)
			{
			/* resize it */
			skin_resize(ui, nw, nh);

			/* remember last event */
			ui->press_x += w;
			ui->press_y += h;
			}
		}

	/* pass it on to the button (at this time not necessary,
	 * but the button motion API may change in the future */
	if (skin_size_func_motion) skin_size_func_motion(widget, key, x, y, ui->skin->pixbuf, ui);
}

static void skin_resize_setup_cb(WidgetData *wd, gpointer data)
{
	static WidgetObjectData *od = NULL;

	/* since this intercepts func_motion, we need to copy the original button od,
	 * set our own func_motion, and set the WidgetData to use the copy
	 */
	if (!od)
		{
		od = ui_widget_object_copy(ui_widget_object_by_type(button_type_id()));
		skin_size_func_motion = od->func_motion;
		od->func_motion = skin_resize_real_motion;
		}

	if (od && wd->od)
		{
		wd->od = od;

		/* this button always anchors bottom, right */
		wd->anchor_right = TRUE;
		wd->anchor_bottom = TRUE;
		}
}

/* this overrides things like the skin resize button */
void skin_finalize(SkinData *skin)
{
	skin_widget_for_each_key_simple(skin, "skin_size", button_type_id(),
					skin_resize_setup_cb, skin);
}

