/*  
  Copyright 2002, Andreas Rottmann

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public  License
  along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

// Much of this code was taken from the PostgreSQL DB terminal psql.
// See README for more information

#include <config.h>

#include <getopt.h>
#include <unistd.h>

#include <stdio.h>

#include <readline/readline.h>
#include <readline/history.h>

#include <iomanip>

#include "gql++/statement.h"

#include "app.h"
#include "print.h"

namespace GQLShell
{

using namespace std;
using namespace GQL;

static struct option long_options[] =
{
  { "verbose", optional_argument, NULL, 'v' },
  { "version", no_argument, NULL, 'V' },
  { "help", no_argument, NULL, 'h' },
  { "list-drivers", no_argument, NULL, 'l' },
  { "username", required_argument, NULL, 'U' },
  { "password", optional_argument, NULL, 'W' },
  { NULL, 0, NULL, 0 }
};

enum Command
{
  CMD_LIST_DRIVERS,
  CMD_HELP,
  CMD_VERSION,
  CMD_EXEC,
  CMD_DEFAULT
};

namespace 
{

char *getline(FILE *stdin);
char *getline(const std::string& prompt);

}

App::App(int argc, char *argv[]) throw (CmdLineError, DBOpenError)
    : rs_printer_(tbl_printer_)
{
  int c, option_idx;
  string pass;
  
  flags_ = 0;
  cmd_ = CMD_DEFAULT;
  conn_ = 0;
  lineno_ = 0;
  fout_ = stdout;
  verbosity_ = 0;

  prompt_[0] = "%/%R%# ";
  prompt_[1] = "%/%R%# ";
  prompt_[2] = ">> ";
  
  if (isatty(fileno(stdin)) && isatty(fileno(stdout)))
    flags_ |= F_INTERACTIVE;
  
  while ((c = getopt_long(argc, argv, "hv::lU:W::", long_options, 
                          &option_idx)) != -1)
  {
    switch(c)
    {
      case 'v':
        if (optarg)
        {
          char *tail;
          verbosity_ = strtoul(optarg, &tail, 10);
          if (tail == optarg || *tail != '\0')
          {
            throw CmdLineError("invalid verbosity specification");
          }
        }
        else
          verbosity_++;
        break;
      case 'h':
        cmd_ = CMD_HELP;
        break;
      case 'l':
        cmd_ = CMD_LIST_DRIVERS;
        break;
      case 'c':
        cmd_ = CMD_EXEC;
        break;
      case 'U':
        username_ = optarg;
        break;
      case 'W':
        if (optarg)
          pass = optarg;
        else
          flags_ |= F_PROMPT_PASSWD;
        break;
      case 'V':
        cmd_ = CMD_VERSION;
        break;
    }
  }

  static int log_verbosities[] = {
    Yehia::ErrorHandler::LOG_ERROR,
    Yehia::ErrorHandler::LOG_CRITICAL,
    Yehia::ErrorHandler::LOG_WARNING,
    Yehia::ErrorHandler::LOG_MESSAGE,
    Yehia::ErrorHandler::LOG_INFO,
    Yehia::ErrorHandler::LOG_DEBUG
  };
  
  if (verbosity_ >= 0 && verbosity_ <= 5)
    log_verbosity_ = log_verbosities[verbosity_];
  else
    log_verbosity_ = verbosity_ > 0 ? Yehia::ErrorHandler::LOG_DEBUG : -1;

  dm_.plugin_manager().log.connect(SigC::slot(*this, &App::logger));
  
  if (cmd_ == CMD_DEFAULT)
  {
    if (argc - optind != 1)
      throw CmdLineError("gql-shell must be called with exactly one "
                         "non-option argument.");
  
    url_ = argv[optind];
    if (url_.substr(0, 4) != "gql:")
      url_.insert(0, "gql:");
  
    if (prompt_passwd())
    {
      pass = simple_prompt("Password: ", false);
    }
    try
    { 
      conn_ = dm_.get_connection(url_, username_ , pass);
    }
    catch (const SQLException& e)
    {
      throw DBOpenError(url_, e.what());
    }
  }
}

App::~App()
{
  delete conn_;
}

void App::list_drivers(std::ostream& os)
{
  Yehia::PluginNode::const_iterator it;
  std::string::size_type max_width = 0;
  
  // get longest id/plugin name
  for (it = dm_.get_drivers().begin(); it != dm_.get_drivers().end(); ++it)
    if ((*it).name().size() > max_width)
      max_width = (*it).name().size();
  
  os << "Available drivers:" << endl;

  for (it = dm_.get_drivers().begin(); it != dm_.get_drivers().end(); ++it)
  {
    dm_.register_driver((*it).name());
    
    Driver *driver = dm_.get_driver((*it).name());
    if (driver)
      os << "  " << setw(max_width) << driver->get_id().c_str()
         << ": " << driver->get_name() << endl;
    else
      os << "  " << setw(max_width) << (*it).name().c_str()
         << ": " << "No information available" << endl;
  }
}

int App::run()
{
  using namespace std;

  switch (cmd_)
  {
    case CMD_LIST_DRIVERS:
      list_drivers();
      return EXIT_SUCCESS;
    case CMD_HELP:
      usage();
      return EXIT_SUCCESS;
    case CMD_VERSION:
      cout << "GQL-Shell " << VERSION << endl;
      cout << "Copyright (c) 2002-2003 Andreas Rottmann" << endl;
      return EXIT_SUCCESS;
  }
  
  if (interactive())
  {
    cout << "Welcome to gql-shell, the interactive SQL terminal." << endl
         << endl
         << "Type:  \\copyright for distribution terms" << endl
         << "\\h for help with SQL commands" << endl
         << "\\? for help on internal slash commands" << endl
         << "\\g or terminate with semicolon to execute query" << endl
         << "\\q to quit" << endl
         << endl;
  }

  char *cp_line;
  char in_quote = 0; // == 0 for no in_quote
  bool in_xcomment = false; // in extended comment
  bool success, die_on_error = false;
  int paren_level = 0;
  int eof_count = 0;
  int query_start;
  int exit_code = EXIT_SUCCESS;
  CmdStatus cmd_status = CMD_STATUS_UNKNOWN;
  PromptType prompt_type;
  FILE *source = stdin;
  string query, prev_query;

  while (1)
  {
    if (cmd_status == CMD_STATUS_NEWEDIT)
    {
      // just returned from editing the line? then just copy to the
      // input buffer
      cp_line = g_strdup(query.c_str());
      query = "";
      
      // reset parsing state since we are rescanning whole line
      in_xcomment = false;
      in_quote = 0;
      paren_level = 0;
      cmd_status = CMD_STATUS_UNKNOWN;
    }
    else
    {
      if (interactive())
      {
        if (in_quote == '\'')
          prompt_type = PROMPT_SINGLEQUOTE;
        else if (in_quote == '"')
          prompt_type = PROMPT_DOUBLEQUOTE;
        else if (in_xcomment)
          prompt_type = PROMPT_COMMENT;
        else if (paren_level > 0)
          prompt_type = PROMPT_PAREN;
        else if (query.length() > 0)
          prompt_type = PROMPT_CONTINUE;
        else
          prompt_type = PROMPT_READY;
        
        cp_line = getline(get_prompt(prompt_type));
      }
      else
        cp_line = getline(source);
    }
    
    // No more input.  Time to quit, or \i done.
    if (cp_line == NULL)
    {
      if (interactive())
      {
        // TODO: implement bash-like IGNOREEOF feature
        if (quiet())
          putc('\n', stdout);
        else
          puts("\\q");
        break;
      }
      else
        // non-interactive
        break; 
    }
    
    lineno_++;
    
    // nothing left on cp_line? then ignore
    if (cp_line[0] == '\n' && !in_quote)
    {
      free(cp_line);
      continue;
    }
    
    if (variable_isset("ECHO") && interactive() && 
        get_variable("ECHO") == "all")
      puts(cp_line);
    fflush(stdout);
    
    query_start = 0;
    
    // we append a newline to always have a 'next' char.
    string line = string(cp_line) + "\n";
    free(cp_line);
    
    
    // parse line, looking for command separators
    success = true;
    for (string::size_type i = 0; i < line.length() - 1; i++)
    {
      bool was_bslash = (i > 0 && line[i - 1] == '\\');
      int bslash_count = 0;
      
      if (was_bslash)
        bslash_count++;
      
      if (in_quote)
      {
        if (line[i] == in_quote && bslash_count % 2 == 0)
          in_quote = 0;
      }
      else if (in_xcomment)
      {
        if (line[i] == '*' && line[i + 1] == '/')
        {
          in_xcomment = false;
          i++;
        }
      }
      // start of extended comment?
      else if (line[i] == '/' && line[i + 1] == '*')
      {
        in_xcomment = true;
        i++;
      }
      // start of quote
      else if (!was_bslash && line[i] == '\'' || line[i] == '"')
        in_quote = line[i];
      // single-line comment? truncate line
      else if ((line[i] == '-' && line[i + 1] == '-') ||
               (line[i] == '/' && line[i + 1] == '/'))
      {
        line.erase(i, line.length() - i - 1);
        break;
      }
      // count nested parentheses
      else if (line[i] == '(')
      {
        paren_level++;
      }
      else if (line[i] == ')')
        paren_level--;
      // TODO: implement variable substitution (:varname)
      else if (false)
      {
      }
      // semicolon? then send query
      else if (line[i] == ';' && !was_bslash && paren_level == 0)
      {
        // is there anything else on the line?
        if (line.find_first_not_of(" \t\n\r", query_start) != string::npos)
        {
          // insert a a cosmetic newline, if this is not the first
          // line in the buffer
          if (query.length() > 0)
            query += '\n';
          
          // append the line to the query buffer
          query += line.substr(query_start, i - query_start);
        }
        success = execute_query(query);
        cmd_status =  success ? CMD_STATUS_SEND : CMD_STATUS_ERROR;
        prev_query = query;
        query = "";
        query_start = i + 1;
      }
      // if you have a burning need to send a semicolon or colon to
      // the backend ...
      else if (was_bslash && (line[i] == ';' || line[i] == ':'))
      {
        // remove the backslash
        line.erase(i, 1);
      }
      // backslash command
      else if (was_bslash)
      {
        string::size_type end_of_cmd = 0;
        
        paren_level = 0;
        // is there anything else on the line for the command?
        if (line.find_first_not_of(" \t\n\r") != string::npos)
        {
          // insert a a cosmetic newline, if this is not the first
          // line in the buffer
          if (query.length() > 0)
            query += '\n';
          // append the line to the query buffer
          query += line.substr(query_start, i - query_start - 1);
        }
        cmd_status = handle_slash_cmd(line.substr(i), 
                                      query.length() > 0 ? query : prev_query,
                                      end_of_cmd);
        success = cmd_status != CMD_STATUS_ERROR;
        if ((cmd_status == CMD_STATUS_SEND || 
             cmd_status == CMD_STATUS_NEWEDIT) && 
            query.length() == 0)
        {
          // copy previous buffer to current for handling
          query = prev_query;
        }
        if (cmd_status == CMD_STATUS_SEND)
        {
          success = execute_query(query);
          query_start = i + 1;
          prev_query = query;
          query = "";
        }
        // process anything left after the backslash command
        i += end_of_cmd;
        query_start = i;
      }
      if (!success && die_on_error)
        break;
    }                      // for (line)
    
    if (cmd_status == CMD_STATUS_TERMINATE)
    {
      exit_code = EXIT_SUCCESS;
      break;
    }
    
    //  Put the rest of the line in the query buffer
    if (in_quote || 
        line.find_first_not_of(" \t\n\r", query_start) != string::npos)
    {
      if (query.length() > 0)
        query += '\n';
      query += line.substr(query_start);
    }
  }
  
  return exit_code;
}

bool App::execute_query(const std::string& query)
{
  using namespace GQL;
  
  //fprintf(fout_, "executing: %s\n", query.c_str());
  bool success = true;
  Statement *stmt = 0;
  ResultSet *rs = 0;
  SQLObject *obj = 0;
  
  if (!conn_)
  {
    error("You are currently not connected to a database.\n");
    return false;
  }
  
  try 
  {
    stmt = conn_->create_statement();
    rs = stmt->execute_query(query);
    if (rs)
      rs_printer_.print(rs, fout_);
  }
  catch (const SQLException& e)
  {
    error("%s\n", e.get_message());
    success = false;
  }
  
  if (obj) delete obj;
  if (stmt) delete stmt;

  return success;
}

void App::error(const char *fmt, ...)
{
  va_list args;

  fflush(stdout);
  if (fout_ != stdout)
    fflush(fout_);
  
  if (!infile_.empty())
    fprintf(stderr, "%s:%s:%u: ", progname_.c_str(), infile_.c_str(), lineno_);
  
  va_start(args, fmt);
  vfprintf(stderr, fmt, args);
  va_end(args);
}

bool App::logger(int level, const string& msg)
{
  if (level <= log_verbosity_)
    error("LOG[%d]: %s\n", level, msg.c_str());

  return true;
}

namespace 
{

char *getline(FILE *source)
{
  int bufsz = 32, strsz = 0;
  char *buf = (char *)malloc(bufsz);
  int c;
  
  while ((c = getc(source)) != EOF)
  {
    if (bufsz < strsz + 1)
    {
      bufsz <<= 1;
      buf = (char *)realloc(buf, bufsz);
    }
    if (buf == NULL)
      throw std::bad_alloc();
    
    if (c == '\n')
    {
      buf[strsz] = '\0';
      break;
    }
    else
      buf[strsz] = c;

    bufsz++;
  }
  return buf;
}

char *getline(const std::string& prompt)
{
  char *result = readline(prompt.c_str());

  if (result && result[0] != '\0')
    add_history(result);

  return result;
}

}

}
