/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  satlist-single-aos.c:  Helper functions accompaning satlist.c

  Copyright (C)  2002-2003   Alexandru Csete.

  Authors:  Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#include <gnome.h>
#include <gconf/gconf-client.h>
#include <libgnomeui/gnome-window-icon.h>
#include <math.h>
#include "defaults.h"
#include "qth.h"
#include "satlog.h"
#include "util.h"
#include "fileio.h"
#include "satdata.h"
#include "satplot.h"
#include "satprint.h"
#include "sataos.h"   /**** FIXME: should be included here! */
#include "satlist-single-aos.h"

#ifdef HAVE_CONFIG_H
#  include "../config.h"
#endif


extern GtkWidget *app;
extern qth_struc qth;    /* defined in qth.c */
extern GConfClient *client; /* main.c */

/* private function prototypes */
static void satlist_single_aos_print_cb (GtkWidget *, gpointer);
static void satlist_single_aos_plot_cb (GtkWidget *, gpointer);
static void satlist_single_aos_save_cb (GtkWidget *, gpointer);
static void satlist_single_aos_save_save (GtkFileSelection *, gpointer);
static void satlist_single_aos_close_cb (GtkWidget *, gpointer);



void satlist_single_aos_cb (GtkWidget *button, gpointer clist)
{
	/* This function is called when the user clicks on the
	   "Next AOS" button. It pops up a dialog showing the
	   details about the upcoming pass.
	*/
  	GtkWidget *dialog,*swin,*list;
	GList *selection = ((GtkCList *) clist)->selection;
/*	GDate *epoch;*/
	sat_t *sat;
	guint index,n,i,row,step,orbit,bit,bitmask,listwidth=0;
	gchar **vbuff,*buff,**detlist;
	gchar *titles[9] = { N_(" Time (UTC) "),
			     N_(" Az "), N_(" El "), N_(" Lat "), N_(" Lon "),
			     N_(" Alt "), N_(" Range "), N_(" Vel "), N_(" Footprint ")	};

	if (selection) {
		row = GPOINTER_TO_UINT (selection->data);
		index = GPOINTER_TO_UINT (gtk_clist_get_row_data (GTK_CLIST (clist), row));
		sat = satdata_get_sat (index);

		switch ( sat->status ) {
		case SAT_STATUS_DECAYED:
			satlog_log (SAT_LOG_MESSAGE, _("Selected satellite has decayed (no AOS)."));
			gnome_app_message (GNOME_APP (app), _("This satellite has DECAYED!"));
			break;
		case SAT_STATUS_GEOSTAT:
			satlog_log (SAT_LOG_MESSAGE, _("Selected satellite is geostationary."));
			gnome_app_message (GNOME_APP (app), _("This satellite is GEOSTATIONARY!"));
			break;
		case SAT_STATUS_NOAOS:
			satlog_log (SAT_LOG_MESSAGE, _("Selected satellite has no AOS."));
			gnome_app_message (GNOME_APP (app), _("This satellite NEVER reaches AOS!"));
			break;
		default:
			/* get time resolution */
			step = gconf_client_get_int (client, AOS_DETAIL_RES_PATH, NULL) ?
			       gconf_client_get_int (client, AOS_DETAIL_RES_PATH, NULL) :
				       AOS_DETAIL_DEF_RES;
			
			detlist = aos_detailed_list (*sat, step);
			/* get orbit number and number of datalines */
			vbuff = g_strsplit (detlist[0], ";", 2);
			orbit = (guint) g_strtod (vbuff[0], NULL);
			n = (guint) g_strtod (vbuff[1], NULL);
			g_strfreev (vbuff);
			/* create a clist widget */
			list = gtk_clist_new_with_titles (9, titles);
			gtk_clist_column_titles_passive (GTK_CLIST (list));
			/* Add data to clist */
			for ( i=1; i<n; i++ ) {
				vbuff = g_strsplit (detlist[i], ";", 9);
				gtk_clist_append (GTK_CLIST (list), vbuff);
				g_strfreev (vbuff);
			}
			/* adjust the width of the columns */
			gtk_clist_set_column_width ( GTK_CLIST(list), 0,
						     gtk_clist_optimal_column_width (GTK_CLIST (list),0));
			gtk_clist_set_column_resizeable (GTK_CLIST (list), 0, FALSE );

			/* get bitmask */
			bitmask = gconf_client_get_int (client, AOS_DETAIL_MASK_PATH, NULL) ? 
				  gconf_client_get_int (client, AOS_DETAIL_MASK_PATH, NULL) :
					  AOS_DETAIL_DEF_MASK;

			for (i=1,bit=1; i<9; i++,bit=bit<<1) {
				if (bitmask & bit) {
					gtk_clist_set_column_width ( GTK_CLIST(list), i,
								     gtk_clist_optimal_column_width(GTK_CLIST(list),i));
					gtk_clist_set_column_resizeable (GTK_CLIST (list), i, FALSE );
					listwidth += gtk_clist_optimal_column_width (GTK_CLIST (list),i);
				}
				else
					gtk_clist_set_column_visibility (GTK_CLIST (list), i, FALSE );
			}
			/* try to fix the height and width ... */
			gtk_object_set (GTK_OBJECT(list), "height", 200, "width", listwidth+200, NULL);

			swin = gtk_scrolled_window_new ( NULL, NULL );
			gtk_scrolled_window_set_policy ( GTK_SCROLLED_WINDOW (swin),
							 GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
			gtk_container_add (GTK_CONTAINER (swin), list );

			/* create the dialog window */
			buff = g_strconcat (_("Upcoming pass for "), sat->name, NULL);
			dialog = gnome_dialog_new (buff, NULL);
			g_free (buff);
			/* window icon */
			buff = g_strconcat (PACKAGE_PIXMAPS_DIR, "/icons/stock_timer.png", NULL);
			gnome_window_icon_set_from_file (GTK_WINDOW (dialog), buff);
			g_free (buff);

			gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (dialog),
								_("Save"),
								GNOME_STOCK_PIXMAP_SAVE_AS);
			gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (dialog),
								_("Print"),
								GNOME_STOCK_PIXMAP_PRINT);
			gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (dialog),
								_("Plot"),
								GNOME_STOCK_PIXMAP_BOOK_GREEN);
			gnome_dialog_append_button (GNOME_DIALOG (dialog), GNOME_STOCK_BUTTON_CANCEL);

			/* connect signals */
			gnome_dialog_button_connect (GNOME_DIALOG (dialog), 0,
						     GTK_SIGNAL_FUNC (satlist_single_aos_save_cb),
						     dialog); 
			gnome_dialog_button_connect (GNOME_DIALOG (dialog), 1,
						     GTK_SIGNAL_FUNC (satlist_single_aos_print_cb),
						     dialog);
			gnome_dialog_button_connect (GNOME_DIALOG(dialog), 2,
						     GTK_SIGNAL_FUNC(satlist_single_aos_plot_cb),
						     dialog);
			gnome_dialog_button_connect (GNOME_DIALOG (dialog), 3,
						     GTK_SIGNAL_FUNC (satlist_single_aos_close_cb),
						     dialog);
			
			/* disable unavailable features */
			//			gnome_dialog_set_sensitive (GNOME_DIALOG (dialog), 0, FALSE);
			/* "Close" is the default button */
			gnome_dialog_set_default (GNOME_DIALOG (dialog), 3);

			/* This will (?) automatically free the memory occupied by slist,
			   when the dialog is closed. */
			gtk_object_set_data_full (GTK_OBJECT (dialog), "list", detlist, free_vector_notify_cb);
			/* Also add satellite index - needed for list title when printing from
			   several open dialogs. */
			gtk_object_set_data (GTK_OBJECT (dialog), "index", GUINT_TO_POINTER (index));
			gtk_object_set_data (GTK_OBJECT (dialog), "size", GUINT_TO_POINTER (n));
			gtk_object_set_data (GTK_OBJECT (dialog), "satname", sat->name);
			gtk_object_set_data (GTK_OBJECT (dialog), "orbit", GUINT_TO_POINTER (orbit));

			gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), swin,
					    TRUE, TRUE , 0);
			gtk_widget_show_all (dialog);

			break; /* well... */
		}

	}
	else
		gnome_app_error (GNOME_APP (app), _("Please select a satellite!"));
}


static void
satlist_single_aos_print_cb (GtkWidget *widget, gpointer data)
{
	guint index,n;
	sat_t *sat;
	gchar **detlist;

	detlist = gtk_object_get_data (GTK_OBJECT (data), "list" );
	n = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (data), "size") );
	index = GPOINTER_TO_UINT ( gtk_object_get_data (GTK_OBJECT (data), "index") );
	sat = satdata_get_sat (index);

	sat_print_aos_det (sat, detlist, n);
}


static void
satlist_single_aos_plot_cb (GtkWidget *widget, gpointer data)
{
	guint index,n;
	sat_t *sat;
	gchar **detlist;
	gchar *message;

	detlist = gtk_object_get_data (GTK_OBJECT (data), "list" );
	n = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (data), "size") );
	index = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (data), "index"));
	sat = satdata_get_sat (index);

	/* log entry */
	message = g_strdup_printf ("%s called with %s", __FUNCTION__, sat->name);
	satlog_log (SAT_LOG_INFO, message);
	g_free (message);

	sat_plot_aos_track (sat, detlist, n);

}



static void
satlist_single_aos_save_cb (GtkWidget *widget, gpointer data )
{
	/* This function is called when the user clicks on the
	   "save detailed list" button. It prompts for a
	   filename and, if user presses OK, calls a function
	   which in turn will call the general data file saving
	   routine, whith the correct parameters. If the user
	   presses cancel, nothing will happen.
	*/
	GtkWidget *filesel;

	filesel = gtk_file_selection_new (_("Select a filename"));
	/* set default path if exists */
	if (gconf_client_get_string (client, AOS_DETAIL_SAVE_PATH, NULL) )
		gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel),
						 gnome_config_get_string (AOS_DETAIL_SAVE_PATH) );
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC (satlist_single_aos_save_save), data);
                             
	/* Ensure that the dialog box is destroyed when the user clicks a button. */
	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->ok_button),
				   "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   (gpointer) filesel);
	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->cancel_button),
				   "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   (gpointer) filesel);
	gtk_widget_show (filesel);

}


static void
satlist_single_aos_save_save (GtkFileSelection *fsel, gpointer data)
{
	/* This function is called by the above function, and
	   it calls the general data file saving function in
	   fileio.c . "data" is a pointer to the GnomeDialog
	   from which the callbacks were called and has various
	   data attached to it.
	   *** FIXME ***
	   The saved data will only contain the time, azimuth,
	   elevation, latitude, longitude and altitude (like
	   the printout), regardless of the user preferences.
	*/
	gchar *fname,**slist,*header;
	guint n=0;

	/* get the data to be saved */
	slist = gtk_object_get_data (GTK_OBJECT (data), "list");
	n = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (data), "size"));
/*	orbit = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (data), "orbit"));*/
	fname = gtk_file_selection_get_filename (GTK_FILE_SELECTION(fsel));
	if ( !g_file_test (fname, G_FILE_TEST_ISDIR) ) {
		/* create header */
		header = g_strdup_printf (_("Detailed pass information for %s orbit %d\n"\
					    "%s %.2f%c %.2f%c\n"\
					    "--------------------------------------------------------------------\n"\
					    " Time (UTC)              Az        El       Lat       Lon     Alt\n"\
					    "--------------------------------------------------------------------\n"),
					  (gchar *)gtk_object_get_data (GTK_OBJECT (data), "satname"),
					  GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (data), "orbit")), 
					  qth.name,
					  fabs (qth.lat), (qth.lat < 0) ? 'S' : 'N',
					  fabs (qth.lon), (qth.lon < 0) ? 'E' : 'W');

		fileio_save_vector (fname, header, NULL, slist, 6, n, 1, ";", "   ");
		g_free (header);
		/* save path */
		gconf_client_set_string (client, AOS_DETAIL_SAVE_PATH, fname, NULL);
	}
}



static void
satlist_single_aos_close_cb (GtkWidget *widget, gpointer data)
{
	/* This function is caaled when the user clicks on
	   the "Close" button in the AOS/LOS predictions
	   dialog (verbose). It frees allocated vector and
	   closes the dialog.
	*/
	gnome_dialog_close (GNOME_DIALOG (data));
}


void satlist_single_aos_from_multi (sat_t *sat, gdouble aos)
{
	/* This function is called when the user clicks on the
	   "Next AOS" button. It pops up a dialog showing the
	   details about the upcoming pass.
	*/
  	GtkWidget *dialog,*swin,*list;
/*	GList *selection = ((GtkCList *) clist)->selection;*/
/*	GDate *epoch;*/
/*	sat_t *sat;*/
	guint n,i,step,orbit,bit,bitmask,listwidth=0;
	gchar **vbuff,*buff,**detlist;
	gchar *titles[9] = { N_(" Time (UTC) "),
			     N_(" Az "), N_(" El "), N_(" Lat "), N_(" Lon "),
			     N_(" Alt "), N_(" Range "), N_(" Vel "), N_(" Footprint ")	};

	if (sat) {
		switch ( sat->status ) {
		case SAT_STATUS_DECAYED:
			satlog_log (SAT_LOG_MESSAGE, _("Selected satellite has decayed (no AOS)."));
			gnome_app_message (GNOME_APP (app), _("This satellite has DECAYED!"));
			break;
		case SAT_STATUS_GEOSTAT:
			satlog_log (SAT_LOG_MESSAGE, _("Selected satellite is geostationary."));
			gnome_app_message (GNOME_APP (app), _("This satellite is GEOSTATIONARY!"));
			break;
		case SAT_STATUS_NOAOS:
			satlog_log (SAT_LOG_MESSAGE, _("Selected satellite has no AOS."));
			gnome_app_message (GNOME_APP (app), _("This satellite NEVER reaches AOS!"));
			break;
		default:
			/* get time resolution */
			step = gconf_client_get_int (client, AOS_DETAIL_RES_PATH, NULL) ?
			       gconf_client_get_int (client, AOS_DETAIL_RES_PATH, NULL) :
				       AOS_DETAIL_DEF_RES;
			
			detlist = aos_detailed_list_new (*sat, step, aos);
			/* get orbit number and number of datalines */
			vbuff = g_strsplit (detlist[0], ";", 2);
			orbit = (guint) g_strtod (vbuff[0], NULL);
			n = (guint) g_strtod (vbuff[1], NULL);
			g_strfreev (vbuff);
			/* create a clist widget */
			list = gtk_clist_new_with_titles (9, titles);
			gtk_clist_column_titles_passive (GTK_CLIST (list));
			/* Add data to clist */
			for ( i=1; i<n; i++ ) {
				vbuff = g_strsplit (detlist[i], ";", 9);
				gtk_clist_append (GTK_CLIST (list), vbuff);
				g_strfreev (vbuff);
			}
			/* adjust the width of the columns */
			gtk_clist_set_column_width ( GTK_CLIST(list), 0,
						     gtk_clist_optimal_column_width (GTK_CLIST (list),0));
			gtk_clist_set_column_resizeable (GTK_CLIST (list), 0, FALSE );

			/* get bitmask */
			bitmask = gconf_client_get_int (client, AOS_DETAIL_MASK_PATH, NULL) ? 
				  gconf_client_get_int (client, AOS_DETAIL_MASK_PATH, NULL) :
					  AOS_DETAIL_DEF_MASK;

			for (i=1,bit=1; i<9; i++,bit=bit<<1) {
				if (bitmask & bit) {
					gtk_clist_set_column_width ( GTK_CLIST(list), i,
								     gtk_clist_optimal_column_width(GTK_CLIST(list),i));
					gtk_clist_set_column_resizeable (GTK_CLIST (list), i, FALSE );
					listwidth += gtk_clist_optimal_column_width (GTK_CLIST (list),i);
				}
				else
					gtk_clist_set_column_visibility (GTK_CLIST (list), i, FALSE );
			}
			/* try to fix the height and width ... */
			gtk_object_set (GTK_OBJECT(list), "height", 200, "width", listwidth+200, NULL);

			swin = gtk_scrolled_window_new ( NULL, NULL );
			gtk_scrolled_window_set_policy ( GTK_SCROLLED_WINDOW (swin),
							 GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
			gtk_container_add (GTK_CONTAINER (swin), list );

			/* create the dialog window */
			buff = g_strdup_printf (_("Orbit %d for %s"), orbit, sat->name);
			dialog = gnome_dialog_new (buff, NULL);
			g_free (buff);
			/* window icon */
			buff = g_strconcat (PACKAGE_PIXMAPS_DIR, "/icons/stock_timer.png", NULL);
			gnome_window_icon_set_from_file (GTK_WINDOW (dialog), buff);
			g_free (buff);

			gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (dialog),
								_("Save"),
								GNOME_STOCK_PIXMAP_SAVE_AS);
			gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (dialog),
								_("Print"),
								GNOME_STOCK_PIXMAP_PRINT);
			gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (dialog),
								_("Plot"),
								GNOME_STOCK_PIXMAP_BOOK_GREEN);
			gnome_dialog_append_button (GNOME_DIALOG (dialog), GNOME_STOCK_BUTTON_CANCEL);

			/* connect signals */
			gnome_dialog_button_connect (GNOME_DIALOG (dialog), 0,
						     GTK_SIGNAL_FUNC (satlist_single_aos_save_cb),
						     dialog); 
			gnome_dialog_button_connect (GNOME_DIALOG (dialog), 1,
						     GTK_SIGNAL_FUNC (satlist_single_aos_print_cb),
						     dialog);
			gnome_dialog_button_connect (GNOME_DIALOG(dialog), 2,
						     GTK_SIGNAL_FUNC(satlist_single_aos_plot_cb),
						     dialog);
			gnome_dialog_button_connect (GNOME_DIALOG (dialog), 3,
						     GTK_SIGNAL_FUNC (satlist_single_aos_close_cb),
						     dialog);
			
			/* disable unavailable features */
			//			gnome_dialog_set_sensitive (GNOME_DIALOG (dialog), 0, FALSE);
			/* "Close" is the default button */
			gnome_dialog_set_default (GNOME_DIALOG (dialog), 3);

			/* This will (?) automatically free the memory occupied by slist,
			   when the dialog is closed. */
			gtk_object_set_data_full (GTK_OBJECT (dialog), "list", detlist, free_vector_notify_cb);
			/* Also add satellite index - needed for list title when printing from
			   several open dialogs. */
			gtk_object_set_data (GTK_OBJECT (dialog), "index",
					     GINT_TO_POINTER (satdata_get_sat_index (0, sat->name)));
			gtk_object_set_data (GTK_OBJECT (dialog), "size", GUINT_TO_POINTER (n));
			gtk_object_set_data (GTK_OBJECT (dialog), "satname", sat->name);
			gtk_object_set_data (GTK_OBJECT (dialog), "orbit", GUINT_TO_POINTER (orbit));

			gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), swin,
					    TRUE, TRUE , 0);
			gtk_widget_show_all (dialog);

			break; /* well... */
		}

	}
	else
		gnome_app_error (GNOME_APP (app), _("Please select a satellite!"));
}
