/*
  Gnome-o-Phone - A program for internet telephony
  Copyright (C) 1999  Roland Dreier
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  $Id: gphone.c 1.25 Fri, 10 Dec 1999 17:29:43 -0600 dreier $
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <pthread.h>
#include <popt.h>

#include "gphone.h"
#include "gphone-lib.h"
#include "talk.h"
#include "listen.h"
#include "connection.h"
#include "thread.h"
#include "ui.h"
#include "sound.h"

int Local_Port = GPHONE_DEFAULT_PORT;
int Remote_Port = GPHONE_DEFAULT_PORT;

char *Call_Name = NULL;         /* name of computer to call */
char *Test_File = NULL;

static int Half_Flag = 0;
static int Eight_Flag = 0;
static int IXJ_Flag = 0;
static int Verbose_Flag = 0;

struct poptOption optionsTable[] = {
  { "remote-port", 'r', POPT_ARG_INT, &Remote_Port, 0,
    "Set RTP port to connect to on remote computer", "port" },
  { "local-port", 'l', POPT_ARG_INT, &Local_Port, 0,
    "Set RTP port to listen on", "port" },
  { "call", 'c', POPT_ARG_STRING, &Call_Name, 0,
    "Call to another computer", "hostname" },
  { "half-duplex", 'h', POPT_ARG_NONE, &Half_Flag, 0,
    "Talk half-duplex", NULL},
  { "eight-bit", '8', POPT_ARG_NONE, &Eight_Flag, 0,
    "Use eight bit sound", NULL },
  { "verbose", 'v', POPT_ARG_NONE, &Verbose_Flag, 0,
    "Print informational messages to the console", NULL },
  { "ixj", '\0', POPT_ARG_NONE, &IXJ_Flag, 0,
    "Use Internet PhoneJACK", NULL },
  { "testfile", 't', POPT_ARG_STRING, &Test_File, 0,
    "Send raw sound data from a file (for testing)", "filename" },
  POPT_AUTOHELP
  { NULL, 0, 0, NULL, 0 }
};

static void
signal_handler(int signum)
{
  switch (signum) {
  case SIGINT:
#ifdef HAVE_LIBSLANG
    term_cleanup();
#endif
    exit(0);
    break;
  }
}

static void
null_log_function(const gchar *log_domain, GLogLevelFlags log_level,
                  const gchar *message, gpointer user_data)
{
  return;
}

int
main(int argc, char *argv[])
{
  char rc;
  poptContext optCon;
  struct sigaction sa;
  int rtn;
  int sock;
  pthread_t ui_thread, talk_thread, listen_thread;

  g_thread_init(NULL);
  gtk_init(&argc, &argv);

  optCon = poptGetContext(NULL, argc, argv, optionsTable, 0);

  rc = poptGetNextOpt(optCon);

  memset(&sa, 0, sizeof(sa));
  sa.sa_handler = signal_handler;

  if (sigaction(SIGINT, &sa, NULL)) {
    gphone_perror_exit("*** main : sigaction", 2);
  }

  sound_init();

  if (Half_Flag) {
    set_sound_duplex(HALF_DUPLEX);
  } else {
    set_sound_duplex(FULL_DUPLEX);
  }
  set_sound_eight_bit(Eight_Flag);

  if (IXJ_Flag) {
    set_sound_device(IXJ_DEVICE);
  } else {
    set_sound_device(OSS_DEVICE);
  }

  if (!Verbose_Flag) {
    g_log_set_handler("GPHONE", G_LOG_LEVEL_DEBUG | G_LOG_LEVEL_INFO,
                      null_log_function, NULL);
  }

  connection_init(Local_Port);
  talk_init();
  listen_init();
  ui_init();

  set_status(STAT_IDLE);
  set_mic_level(0);
  set_spk_level(0);

  if ((rtn = pthread_create(&ui_thread, NULL, run_ui, NULL))) {
    g_error("pthread_create ui_thread: %s\n", strerror(rtn));
    exit(1);
  }
  if ((rtn = pthread_create(&talk_thread, NULL, talk_side,
                            (void *) &sock))) {
    fprintf(stderr, "pthread_create talk_thread: %s\n", strerror(rtn));
    exit(1);
  }
  if ((rtn = pthread_create(&listen_thread, NULL, listen_side,
                            (void *) &sock))) {
    fprintf(stderr, "pthread_create listen_thread: %s\n", strerror(rtn));
    exit(1);
  }
  
  if (Call_Name != NULL) {
    Request req;

    set_status(STAT_CALL);

    req = g_malloc(sizeof *req);

    req->type = REQUEST_CALL;
    req->data_len = strlen(Call_Name) + 1;
    req->data = g_strdup(Call_Name);

    talk_add_request(req);
  } else {
    set_status(STAT_WAIT);
  }

  pthread_join(ui_thread, NULL);
  g_log("GPHONE", G_LOG_LEVEL_DEBUG,
        "... quitting ui");
  pthread_join(talk_thread, NULL);
  g_log("GPHONE", G_LOG_LEVEL_DEBUG,
        "... quitting talk");
  pthread_join(listen_thread, NULL);
  g_log("GPHONE", G_LOG_LEVEL_DEBUG,
        "... quitting listen");

  close(sock);
  return(0);
}



/*
 * Local variables:
 *  compile-command: "make -k gphone"
 * End:
 */
