#include "gperiodic.h"
extern struct table_entry table[];

/**
 * isValidAtomicNumber - returns true/false based on weather
 * the given atomic number is valid (corresponds to a valid 
 * data entry for the corresponding element).
 *
 * @param int atomicNumber
 * @return true/false
 *
 */
int isValidAtomicNumber(int atomicNumber)
{
  return atomicNumber > 0 && atomicNumber <= 118;
}

/**
 * gpdata_get_element_name - returns element's name.  Also verifies
 * validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
const char* gpdata_get_element_name(int atomicNumber)
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].info[NAME];
  }
  return NULL;
}

/**
 * gpdata_get_element_symbol - returns element's symbol.Also verifies
 * validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
const char* gpdata_get_element_symbol(int atomicNumber)
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].info[SYMBOL];
  }
  return NULL;
}

/**
 * gpdata_get_element_weight - returns element's weight.  Also verifies
 * validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
const char* gpdata_get_element_weight(int atomicNumber)
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].info[WEIGHT];
  }
  return NULL;
}
/**
 * gpdata_get_element_density - returns element's density.  Also verifies
 * validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
const char* gpdata_get_element_density(int atomicNumber)
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].info[DENSITY];
  }
  return NULL;
}
/**
 * gpdata_get_element_melting_point - returns element's melting point.
 * Also verifies validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
const char* gpdata_get_element_melting_point(int atomicNumber)
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].info[MELTING];
  }
  return NULL;
}

/**
 * gpdata_get_element_boiling_point - returns element's boiling point.
 * Also verifies validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
const char* gpdata_get_element_boiling_point(int atomicNumber)
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].info[BOILING];
  }
  return NULL;
}
/**
 * gpdata_get_element_atomic_radius - returns element's atomic radius.
 * Also verifies validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
const char* gpdata_get_element_atomic_radius(int atomicNumber)
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].info[ATOMIC_RADIUS];
  }
  return NULL;
}
/**
 * gpdata_get_element_covalent_radius - returns element's covalent radius.
 * Also verifies validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
const char* gpdata_get_element_covalent_radius(int atomicNumber)
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].info[COVALENT_RADIUS];
  }
  return NULL;
}
/**
 * gpdata_get_element_ionic_radius - returns element's ionic radius.
 * Also verifies validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
const char* gpdata_get_element_ionic_radius(int atomicNumber)
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].info[IONIC_RADIUS];
  }
  return NULL;
}
/**
 * gpdata_get_element_atomic_volume - returns element's atomic volume.
 * Also verifies validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
const char* gpdata_get_element_atomic_volume(int atomicNumber)
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].info[ATOMIC_VOLUME];
  }
  return NULL;
}
/**
 * gpdata_get_element_specific_heat - returns element's specific heat.
 * Also verifies validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
const char* gpdata_get_element_specific_heat(int atomicNumber)
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].info[SPECIFIC_HEAT];
  }
  return NULL;
}
/**
 * gpdata_get_element_specific_heat - returns element's fuusion heat.
 * Also verifies validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
const char* gpdata_get_element_fusion_heat(int atomicNumber)
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].info[FUSION_HEAT];
  }
  return NULL;
}
/**
 * gpdata_get_element_evaporation_heat - returns element's evaporation heat.
 * Also verifies validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
const char* gpdata_get_element_evaporation_heat(int atomicNumber)
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].info[EVAPORATION_HEAT];
  }
  return NULL;
}

/**
 * gpdata_get_element_evaporation_heat - returns element's termal conductivity.
 * Also verifies validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
const char* gpdata_get_element_termal_conductivity(int atomicNumber)
{
  if(isValidAtomicNumber(atomicNumber)) {
    printf("%s(%d) returning TERMAL_CONDUCTIVITY[%d] for %d : %s\n",
      __FILE__,__LINE__,TERMAL_CONDUCTIVITY,atomicNumber,table[atomicNumber-1].info[TERMAL_CONDUCTIVITY]);
    return table[atomicNumber-1].info[TERMAL_CONDUCTIVITY];
  }
  return NULL;
}
/**
 * gpdata_get_element_pauling - returns element's pauling negativity number.
 * Also verifies validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
const char* gpdata_get_element_pauling(int atomicNumber)
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].info[PAULING];
  }
  return NULL;
}
/**
 * gpdata_get_element_energy - returns element's first ionizing energy.
 * Also verifies validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
const char* gpdata_get_element_energy(int atomicNumber)
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].info[ENERGY];
  }
  return NULL;
}
/**
 * gpdata_get_element_oxidation - returns element's oxidation state.
 * Also verifies validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
const char* gpdata_get_element_oxidation(int atomicNumber)
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].info[OXIDATION];
  }
  return NULL;
}
/**
 * gpdata_get_element_configuration - returns element's electronic configuration.
 * Also verifies validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
const char* gpdata_get_element_configuration(int atomicNumber)
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].info[CONFIGURATION];
  }
  return NULL;
}

/**
 * gpdata_get_element_x_position - returns element's x position for the grid.
 * Also verifies validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
int gpdata_get_element_x_position(int atomicNumber)
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].x;
  }
  return 0;
}

/**
 * gpdata_get_element_y_position - returns element's y position for the grid.
 * Also verifies validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
int gpdata_get_element_y_position(int atomicNumber)
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].y;
  }
  return 0;
}


/**
 * gpdata_get_atomic_number_for_symbol - searches (linear search)
 * for the atomic number of the given symbol.
 *
 * @param const char* symbol
 * @return int - atomic number
 * 
 */
int gpdata_get_atomic_number_for_symbol(const char*sym)
{
  int i;
  for(i = 0; i < 118; ++i) {
    if(!strcasecmp(table[i].info[SYMBOL],sym)) {
      return i+1;
    }
  }
  return 0;
}

/**
 * gpdata_get_element_pixel_depth - returns the pixeldeph field
 * Also verifies validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
int gpdata_get_element_pixel_depth(int atomicNumber) 
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].color.pixel;
  }
  return 0;
}

/**
 * gpdata_get_element_red - returns the red field
 * Also verifies validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
int gpdata_get_element_red(int atomicNumber) 
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].color.red;
  }
  return 0;
}

/**
 * gpdata_get_element_blue - returns the blue field
 * Also verifies validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
int gpdata_get_element_blue(int atomicNumber) 
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].color.blue;
  }
  return 0;
}

/**
 * gpdata_get_element_green - returns the green field
 * Also verifies validity of the passed atomic number.
 *
 * @param atomicNumber
 * @return const char*
 * 
 */
int gpdata_get_element_green(int atomicNumber) 
{
  if(isValidAtomicNumber(atomicNumber)) {
    return table[atomicNumber-1].color.green;
  }
  return 0;
}
