// Copyright 2012-2015 Oliver Eilhard. All rights reserved.
// Use of this source code is governed by a MIT-license.
// See http://olivere.mit-license.org/license.txt for details.

package elastic

// SumAggregation is a single-value metrics aggregation that sums up
// numeric values that are extracted from the aggregated documents.
// These values can be extracted either from specific numeric fields
// in the documents, or be generated by a provided script.
// See: http://www.elasticsearch.org/guide/en/elasticsearch/reference/current/search-aggregations-metrics-sum-aggregation.html
type SumAggregation struct {
	field           string
	script          string
	scriptFile      string
	lang            string
	format          string
	params          map[string]interface{}
	subAggregations map[string]Aggregation
}

func NewSumAggregation() SumAggregation {
	a := SumAggregation{
		params:          make(map[string]interface{}),
		subAggregations: make(map[string]Aggregation),
	}
	return a
}

func (a SumAggregation) Field(field string) SumAggregation {
	a.field = field
	return a
}

func (a SumAggregation) Script(script string) SumAggregation {
	a.script = script
	return a
}

func (a SumAggregation) ScriptFile(scriptFile string) SumAggregation {
	a.scriptFile = scriptFile
	return a
}

func (a SumAggregation) Lang(lang string) SumAggregation {
	a.lang = lang
	return a
}

func (a SumAggregation) Format(format string) SumAggregation {
	a.format = format
	return a
}

func (a SumAggregation) Param(name string, value interface{}) SumAggregation {
	a.params[name] = value
	return a
}

func (a SumAggregation) SubAggregation(name string, subAggregation Aggregation) SumAggregation {
	a.subAggregations[name] = subAggregation
	return a
}

func (a SumAggregation) Source() interface{} {
	// Example:
	//	{
	//    "aggs" : {
	//      "intraday_return" : { "sum" : { "field" : "change" } }
	//    }
	//	}
	// This method returns only the { "sum" : { "field" : "change" } } part.

	source := make(map[string]interface{})
	opts := make(map[string]interface{})
	source["sum"] = opts

	// ValuesSourceAggregationBuilder
	if a.field != "" {
		opts["field"] = a.field
	}
	if a.script != "" {
		opts["script"] = a.script
	}
	if a.scriptFile != "" {
		opts["script_file"] = a.scriptFile
	}
	if a.lang != "" {
		opts["lang"] = a.lang
	}
	if a.format != "" {
		opts["format"] = a.format
	}
	if len(a.params) > 0 {
		opts["params"] = a.params
	}

	// AggregationBuilder (SubAggregations)
	if len(a.subAggregations) > 0 {
		aggsMap := make(map[string]interface{})
		source["aggregations"] = aggsMap
		for name, aggregate := range a.subAggregations {
			aggsMap[name] = aggregate.Source()
		}
	}

	return source
}
