/*
 *
 * Copyright 2019 gRPC authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package grpclb

import (
	"encoding/json"
	"errors"
	"fmt"
	"reflect"
	"strings"
	"testing"

	"google.golang.org/grpc/serviceconfig"
)

func Test_Parse(t *testing.T) {
	tests := []struct {
		name    string
		s       string
		want    serviceconfig.LoadBalancingConfig
		wantErr error
	}{
		{
			name:    "empty",
			s:       "",
			want:    nil,
			wantErr: errors.New("unexpected end of JSON input"),
		},
		{
			name: "success1",
			s:    `{"childPolicy":[{"pick_first":{}}]}`,
			want: &grpclbServiceConfig{
				ChildPolicy: &[]map[string]json.RawMessage{
					{"pick_first": json.RawMessage("{}")},
				},
			},
		},
		{
			name: "success2",
			s:    `{"childPolicy":[{"round_robin":{}},{"pick_first":{}}]}`,
			want: &grpclbServiceConfig{
				ChildPolicy: &[]map[string]json.RawMessage{
					{"round_robin": json.RawMessage("{}")},
					{"pick_first": json.RawMessage("{}")},
				},
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got, err := (&lbBuilder{}).ParseConfig(json.RawMessage(tt.s)); !reflect.DeepEqual(got, tt.want) || !strings.Contains(fmt.Sprint(err), fmt.Sprint(tt.wantErr)) {
				t.Errorf("parseFullServiceConfig() = %+v, %+v, want %+v, <contains %q>", got, err, tt.want, tt.wantErr)
			}
		})
	}
}

func Test_childIsPickFirst(t *testing.T) {
	tests := []struct {
		name string
		s    string
		want bool
	}{
		{
			name: "pickfirst_only",
			s:    `{"childPolicy":[{"pick_first":{}}]}`,
			want: true,
		},
		{
			name: "pickfirst_before_rr",
			s:    `{"childPolicy":[{"pick_first":{}},{"round_robin":{}}]}`,
			want: true,
		},
		{
			name: "rr_before_pickfirst",
			s:    `{"childPolicy":[{"round_robin":{}},{"pick_first":{}}]}`,
			want: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			gc, err := (&lbBuilder{}).ParseConfig(json.RawMessage(tt.s))
			if err != nil {
				t.Fatalf("Parse(%v) = _, %v; want _, nil", tt.s, err)
			}
			if got := childIsPickFirst(gc.(*grpclbServiceConfig)); got != tt.want {
				t.Errorf("childIsPickFirst() = %v, want %v", got, tt.want)
			}
		})
	}
}
