#include <config.h>

#include <gnome.h>
#include <glade/glade.h>
#include <gconf/gconf-client.h>
#include <bonobo.h>

#include "goat.h"
#include "goats-applet.h"
#include "goats-applet-preferences.h"
#include "goats-applet-callbacks.h"

void draw_pixmap(GoatsApplet *goats, GdkPixbuf *pixbuf)
{
  GdkPixbuf *scaled;

  scaled  = gdk_pixbuf_scale_simple(pixbuf, 
				    goats->panel_size-2, 
				    goats->panel_size-2, 
				    GDK_INTERP_BILINEAR);

  gtk_image_set_from_pixbuf(GTK_IMAGE(goats->pixmap), scaled);
  g_object_unref(scaled);
}

void panel_resize_cb(GtkWidget *w, gint size,
		     GoatsApplet *goats)

     /* Resize the icon if the panel changes size */
{
  goats->panel_size = size;
 
  draw_pixmap(goats, goats->pixbuf_unlit);
}

static void new_note(GoatsApplet *goats)
{
  gchar *name, *key;

  name = g_strdup_printf("applet_goat_%i", g_random_int());
  key = g_strdup_printf("/apps/goats/herd/%s/alive", name);
  
  gconf_client_set_bool(goats->gconf_client, key,
			TRUE, NULL);
  g_free(key);
  g_free(name);
}

gint applet_key_press_cb(GtkWidget *widget, GdkEventKey *event,
			 GoatsApplet *goats)
{
  switch (event->keyval) {
  case GDK_KP_Enter:
  case GDK_ISO_Enter:
  case GDK_3270_Enter:
  case GDK_Return:
  case GDK_space:
  case GDK_KP_Space:
    new_note(goats);
    return TRUE;
  default:
    break;
  }
  
  return FALSE;
}

gint applet_pressed_cb(GtkWidget *w, GdkEventButton *event,
		       GoatsApplet *goats)

     /* Left click creates new note */
{
  gchar *name, *key;

  if (event->button == 1) {
    new_note(goats);
  }

  return FALSE;
}

gint applet_prelight_cb(GtkWidget *w, GdkEvent *event,
			GoatsApplet *goats)

     /* Prelight the icon on cursor entry */
{
  draw_pixmap(goats, goats->pixbuf_lit);

  return FALSE;
}

gint applet_prelight_off_cb(GtkWidget *w, GdkEvent *event,
			    GoatsApplet *goats)

     /* Remove prelighting the icon on cursor leaving */
{
  draw_pixmap(goats, goats->pixbuf_unlit);

  return FALSE;
}

void my_goats_property(const gchar *key_name, gboolean value,
		       GoatsApplet *goats)

     /* Change the property given by key_name to value
	for all notes */
{
  gint i;
  gchar *name, *full_key;
  GSList *goat_list;

  goat_list = gconf_client_all_dirs(goats->gconf_client, "/apps/goats/herd", NULL);

  for (i=0; i<g_slist_length(goat_list); i++) {
    name = g_slist_nth_data(goat_list, i);
    full_key = g_strdup_printf("%s/%s",
			       name, key_name);
    gconf_client_set_bool(goats->gconf_client, full_key,
			  value, NULL);
    g_free(full_key);
    g_free(name);
  }

  g_slist_free(goat_list);
}

void show_summary_cb(BonoboUIComponent *uic, gpointer goats, const gchar *verbname)
{
  GtkWidget *w;

  w = glade_xml_get_widget(((GoatsApplet *)goats)->summary_xml, 
			   "summary");
  gtk_window_present(GTK_WINDOW(w));
}

void delete_all_cb(BonoboUIComponent *uic, gpointer goats, const gchar *verbname)
{
  gboolean confirm;

  confirm = goats_confirm("Are you sure you want to delete all your notes?", 
			  GTK_STOCK_DELETE);

  if (confirm) my_goats_property("alive", FALSE, (GoatsApplet *)goats);
}

void raise_all_cb(BonoboUIComponent *uic, gpointer goats, const gchar *verbname)
{
  my_goats_property("raised", TRUE, (GoatsApplet *)goats);
}

void lower_all_cb(BonoboUIComponent *uic, gpointer goats, const gchar *verbname)
{
  my_goats_property("lowered", TRUE, (GoatsApplet *)goats);
}

void hide_all_cb(BonoboUIComponent *uic, gpointer goats, const gchar *verbname)
{
  my_goats_property("hidden", TRUE, (GoatsApplet *)goats);
}

void show_all_cb(BonoboUIComponent *uic, gpointer goats, const gchar *verbname)
{
  my_goats_property("hidden", FALSE, (GoatsApplet *)goats);
}

void shade_all_cb(BonoboUIComponent *uic, gpointer goats, const gchar *verbname)
{
  my_goats_property("shaded", TRUE, (GoatsApplet *)goats);
}

void unshade_all_cb(BonoboUIComponent *uic, gpointer goats, const gchar *verbname)
{
  my_goats_property("shaded", FALSE, (GoatsApplet *)goats);
}

void show_preferences_cb(BonoboUIComponent *uic, gpointer goats, const gchar *verbname)

     /* show the preferences dialog */
{
  GtkWidget *w;
  GoatsApplet *goats_app;

  goats_app = (GoatsApplet *)goats;

  w = glade_xml_get_widget(goats_app->prefs_xml, "preferences");
  gtk_window_present(GTK_WINDOW(w));
}

void show_about_cb(BonoboUIComponent *uic, gpointer goats, const gchar *verbname)

     /* show the about dialog */
{
  GtkWidget *about;
  GdkPixbuf *logo;
  gchar *logo_file;

  const gchar *authors[] = {
    "M Craig <martin@menudo.freeserve.co.uk> (main author)",
    "Ishikawa Yuichiro",
    "Tomokazu Matsumaru",
    NULL
  };

  const gchar *documenters[] = {
    "M Craig",
    NULL
  };

  gchar *translators = _("This version of Goats is untranslated");

  logo_file = g_strdup_printf("%sgoats_logo.png",GOATS_DIR);
  logo = gdk_pixbuf_new_from_file(logo_file, NULL);
  g_free(logo_file);
  about = gnome_about_new ("Goats", VERSION,
                           "Copyright 2002 M Craig",
			   _("A sticky goat program for Gnome 2"),
			   authors,
			   documenters, (const char *)translators,
			   logo);
  gtk_window_set_icon(GTK_WINDOW(about), ((GoatsApplet *)goats)->pixbuf_lit);
  g_object_unref(logo);
  gtk_widget_show(about);
}

void goats_client_defaults_changed_cb(GConfClient *client, 
				      guint id,
				      GConfEntry *entry, 
				      GoatsApplet *goats)

     /* Update the background colour samples in the 
	preferences dialog */
{
  goats_client_preferences_update_swatches(goats); /* FIXME */
}

void goats_client_herd_changed_cb(GConfClient *client, 
				  guint id,
				  GConfEntry *entry, 
				  GoatsApplet *goats)
     
     /* Update the background colour samples in the 
	preferences dialog */
{
  gint length, end, start;
  gchar *name, *pref, *goat;
  gboolean new;

  if (entry->value == NULL) return;

  length = strlen(entry->key);
  pref = entry->key + (length - 5) * sizeof(char);

  if (strcmp(pref, "alive") == 0) {
    new = gconf_value_get_bool(entry->value);
    if (new) {
      end = length - 7;
      for (start=end; start>=0; start--) if (entry->key[start] == '/') break;
      if (start > 0) {
	start++;
	goat = entry->key + start * sizeof(char);
	name = g_strndup(goat, end - start + 1);
	goat_new(name, TRUE);
	g_free(name);
      }
    }
  }
}

void show_help_cb(BonoboUIComponent *uic, gpointer goats, 
		  const gchar *verbname)

     /* display the help file */
{
  GError *err;

  err = NULL;
  gnome_help_display ("goats", "toc",
                      &err);
 
  if (err) {
    goats_error("show_help_cb", err->message);
    g_error_free(err);
  }
}
