/* -*-C-*-
   Josh Pieper, (c) 2000
   Robert Munafo, (c) 2001

   This file is distributed under the GPL, see file COPYING for details */

/* level-1 system headers */
#include <stdio.h>
#include <stdlib.h>
#include "sh_sys_types.h"
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <netdb.h>

#if defined(HAVE_FCNTL)
# include <fcntl.h>
#endif

#include <errno.h>
#include <string.h>
#include <string.h>
#include <pthread.h>
#include <ctype.h>

/* level-2 system headers */
#include "sv_regex.h"

/* local headers */
#include "blacklist.h"
#include "cache.h"
#include "conf.h"
#include "connection.h"
#include "gnut.h"
#include "hash.h"
#include "host.h"
#include "http.h"
#include "lib.h"
#include "net.h"
#include "qry.h"
#include "share.h"
#include "threads.h"
#include "transfer.h"
#include "ui.h"

#define SAFE_CLOSE(x) if (x>=0) { shutdown(x,2); close_s(x); x=-1; }

int num_outgoing = 0;
int num_incoming = 0;
int num_downloads = 0;
int num_uploads = 0;
int count_downloads = 0;
int count_uploads = 0;
int num_searches = 0;

int gh_did_receive;

gnutella_packet *current_query_packet;
GnutSearch      *current_searches;

void fre_bip(bad_ip **x, int bugnum)
{
  yfre((void **) x, bugnum);
}

void fre_gs(GnutSearch **x, int bugnum)
{
  yfre((void **) x, bugnum);
}

void fre_ia(incoming_arg **x, int bugnum)
{
  yfre((void **) x, bugnum);
}

void fre_pa(push_arg **x, int bugnum)
{
  yfre((void **) x, bugnum);
}

GnutSearch *gnut_search_add(char *guid, char *query)
{
  GnutSearch *t = 0;
  GnutSearch *n = 0;
  int         i = 0;

  if (!guid || !query) {
    return 0;
  }

  if (gnut_search_find_by_query(query)) {
    printf(UI_TH_DUP_SEARCH, query);
    return 0;
  }
  n = ymaloc(sizeof(GnutSearch), 420);
  if (!n) {
    return 0;
  } else {
    n->guid = ymaloc(16, 421);
    /* copy the GUID we were given */
    for (i=0; i<14; i++) {
      n->guid[i] = guid[i];
    }
    n->guid[14] = 0;
    n->guid[15] = 0;
    n->query = ystdup(query, 459);
    n->responses = 0;
    n->next = 0;
  }

  if (!current_searches) {
    current_searches = n;
  } else {
    for (t=current_searches; t->next; t=t->next) { }
    t->next = n;
  }
  return n;
}

void gnut_search_remove(GnutSearch *s)
{
  GnutSearch *t = 0;
  
  if (!s) {
    return;
  }

  if (current_searches==s) {
    current_searches = s->next;
  } else {
    for (t=current_searches; t; t=t->next) {
      if (t->next==s) {
	t->next = s->next;
	break;
      }
    }
  }

  fre_str(&(s->query), 101);
  fre_str(&(s->guid), 102);
  fre_gs(&s, 103);
  return;
}

void gnut_search_foreach(void (*cb)(GnutSearch *search,
                                    void       *data),
                         void  *data)
{
  GnutSearch *t = 0;
  GnutSearch *next = 0;
  
  for (t=current_searches; t; t=next) {
    next = t->next;
    cb(t,data);
  }
  
  return;
}

GnutSearch *gnut_search_find(char (*cb)(GnutSearch *search, void *data),
                             void  *data)
{
  GnutSearch *t;
  GnutSearch *r;
  GnutSearch *next;

  t = r = next = 0;

  for (t = current_searches; t; t=next) {
    next = t->next;
    if (cb(t,data)) {
      r = t;
      break;
    }
  }
  
  return r;
}

char by_guid_cb(GnutSearch *s, void *data)
{
  char r = 0;

  if (!memcmp(s->guid, data, 14)) {
    r = 1;
  }
  return r;
}

GnutSearch *gnut_search_find_by_guid(char *guid)
{
  return gnut_search_find(by_guid_cb, guid);
}

char by_query_cb(GnutSearch *s, void *data)
{
  char r;

  r = 0;
  if (!strcasecmp(s->query, data)) {
    r = 1;
  }
  return r;
}

GnutSearch *gnut_search_find_by_query(char *query)
{
  return gnut_search_find(by_query_cb, query);
}

int gnut_search_count()
{
  GnutSearch *t = 0;
  int c = 0;

  for (t=current_searches; t; t=t->next) {
    c++;
  }
  return c;
}

int gnut_threads_num_downloads()
{
  return num_downloads;
}

int gnut_threads_num_uploads()
{
  return num_uploads;
}

int gnut_threads_count_downloads()
{
  return count_downloads;
}

int gnut_threads_count_uploads()
{
  return count_uploads;
}

int gnut_threads_num_incoming()
{
  return num_incoming;
}

int gnut_threads_num_outgoing()
{
  return num_outgoing;
}

/* 0.4.28.c02 This used to be the first half of transfer_cleanup */
int transfer_close(gnut_transfer *gt, int state)
{
  dqi(0x016b);
  GD_S(2, "transfer.close IN\n");

  if ((gt->sock) >= 0) {
    shutdown(gt->sock, 2);
    close_s(gt->sock);
    gt->sock = -1;
  }

  gt->gt_state = state;

  GD_S(2, "transfer.close OUT\n");
  return 0;
}

/* 0.4.28.c02 This used to be the second half of transfer_cleanup */
int transfer_dealloc(gnut_transfer *gt)
{
  if (!conf_get_int("auto_clear")) {
    GD_S(2, "transfer.dealloc waiting for external clear\n");
    while (gt->gt_state != STATE_CLEANABLE) {
      sleep(1);
    }
  }

  GD_S(2, "transfer.dealloc done waiting\n");

  gnut_xfer_delete(gt);

  GD_S(2, "transfer.dealloc OUT\n");
  return 0;
}

#define BMSG_LEN 300 /* 0.4.28.c23 */

int send_welcome_message(int sock, char *message)
{
  int ret;
  char b[BMSG_LEN+1]; /* 0.4.28.c23 */

  GD_S(3, "send.welcome_message entering\n");

  b[BMSG_LEN] = 0; /* 0.4.28.c23 (and following lines) */
  strncpy(b, conf_get_str("virtual_network_name"), BMSG_LEN-1);
  strcatlim(b, " CONNECT/", BMSG_LEN);
  strcatlim(b, message, BMSG_LEN);
  strcatlim(b, "\n\n", BMSG_LEN);

  do {
    ret = write(sock, b, strlen(b)); /* 0.4.28.c23 */
  } while (errno==EINTR && ret<0);

  if (ret<0) {
    GD_PERROR(4, "send.welcome_message");
  }

  return ret;
}

int read_welcome_message(int sock, char *hname)
{
  char buf[BMSG_LEN+1]; /* 0.4.28.c23 */
  char msg1[BMSG_LEN+1]; /* 0.4.28.c23 */
  char msg2[BMSG_LEN+1]; /* 0.4.28.c23 */
  int i;
  int flag=0;

  dqi(0x0189);

  i = read_line(sock, buf, BMSG_LEN);
  GD_S(3, "read_welcome.message read_line returned: "); GD_I(3, i);
  GD_S(3, "\n");

  if (i <= 0) {
    return -1;
  }

  if (gc_debug_opts & 4) {
    /* Print up to 100 bytes of the message */
    printf("%s rwm: ", hname);
    for(i=0; (i < BMSG_LEN) && (buf[i]); i++) {
      printf("%c", buf[i]);
    }
    printf("\n");
  }

  dqi(0x0187);

  msg1[BMSG_LEN] = 0; /* 0.4.28.c23 (and following lines) */
  strncpy(msg1, conf_get_str("virtual_network_name"), BMSG_LEN-1);
  strcatlim(msg1, " OK", BMSG_LEN);

  msg2[BMSG_LEN] = 0; /* 0.4.28.c23 (and following lines) */
  strncpy(msg2, conf_get_str("virtual_network_name"), BMSG_LEN-1);
  strcatlim(msg2, " PASSWORD REQUIRED", BMSG_LEN);

  dqi(0x0188);

  if (strncmp(buf, msg1, strlen(msg1)) == 0) { /* 0.4.28.c23 */
    flag = 1;
  } else if (strncmp(buf, msg2, strlen(msg2)) == 0) { /* 0.4.28.c23 */
    flag = 2;
  } else {
    return -2;
  }

  /* This read appears to be for discarding extra bytes written to us after
     the OK or PASSWORD REQUIRED line. I'm not sure why it's needed. */
  read(sock, buf, sizeof(buf));

  return flag;
}

int incoming_gnutella(int sock, struct sockaddr_in *sin)
{
  char *buf;
  int ret;
  gcs *gc;

  dqi(0x0131);

  GD_S(3, "incoming.gnutella entering\n");

  /* next we need to read the second line of the welcome */
  buf = (char *) ymaloc(BMSG_LEN+1, 312); /* 0.4.28.c23 */
  ret = read_line(sock, buf, BMSG_LEN); /* 0.4.28.c23 */
  GD_S(3, "incoming.gnutella read_line returned "); GD_I(3, ret);
  GD_S(3, "\n");

  if (ret <= 0) {
    GD_PERROR(1,"incoming.gnutella, read_line");
    fre_str(&buf, 104);
    close_s(sock);
    return -1;
  }

  /* if authentication is turned on, we must now ask for the 
   * pass phrase */
  if (conf_get_int("password_required")) {
    buf[BMSG_LEN] = 0; /* 0.4.28.c23 (and following lines) */
    strncpy(buf, conf_get_str("virtual_network_name"), BMSG_LEN);
    strcatlim(buf, " PASSWORD REQUIRED", BMSG_LEN);

    write(sock, buf, strlen(buf));

    /* there should be a one line response, consisting of the
     * pass phrase... if it doesn't match, then we abort, otherwise
     * allow the connection */
    ret = read_line(sock,buf,100);
    gnut_strstrip(buf);
    GD_S(1, "incoming.gnutella \""); GD_S(1, buf);
    GD_S(1, "\" attempted as a password\n");
    if ((ret <= 0) || (strcmp(buf, conf_get_str("password")) != 0)) {
      GD_PERROR(1,"incoming.gnutella, read_line2");
      fre_str(&buf, 105);
      close_s(sock);
      return -1;
    }
  }

  dqi(0x0132);
  /* ok we've received it alright, now we can send our reply... */

  if (++num_incoming > conf_get_int("max_incoming")) {
    if (conf_get_int("redirect")) {
      /* drop oldest connection to make room for new */
      gnut_connection_kill_oldest();
    } else {
      num_incoming--;
      /* we have too many incoming! */
      fre_str(&buf, 106);
      close_s(sock);
      return 0;
    }
  }

  dqi(0x0133);
  /* first we need to enter ourselves into the connection list */
  gc = gnut_connection_new();

  dqi(0x0134);
  gc->gc_sock = sock;
  gc->ctype = TYPE_INCOMING;
  gc->cstate = STATE_NEGOTIATING;

  memcpy(&(gc->ip.a.s_addr), &sin->sin_addr.s_addr, 4);
  gc->port = ntohs(sin->sin_port);

  dqi(0x0135);
  buf[BMSG_LEN] = 0; /* 0.4.28.c23 (and following lines) */
  strncpy(buf, conf_get_str("virtual_network_name"), BMSG_LEN);
  strcatlim(buf, " OK\n\n", BMSG_LEN);

  ret = write(sock, buf, strlen(buf));

  if (ret < 0) {
    GD_PERROR(0, "incoming.gnutella, write");
    fre_str(&buf, 107);
    gnut_connection_delete(gc);
    num_incoming--;
    close_s(sock);
    return -1;
  }

  gc->cstate = STATE_CONNECTED;

  ret = connection_loop(gc);
  dqi(0x003a);
  GD_S(1, "incoming.gnutella connection_loop returned: "); GD_I(1, ret);
  GD_S(1, "\n");

  gc->cstate = STATE_CLEANABLE;

  num_incoming--;

  close_s(gc->gc_sock);
  fre_str(&buf, 108);

  gnut_connection_delete(gc);

  GD_S(3, "incoming.gnutella returning success\n");
  return 0;
}

/* Removed the unused parse_http_request code that was here (an almost
   identical routine in http.c is the one that was actually being
   used) */
#if 0
int parse_range(char *buf, int size, int *rangemin, int *rangemax)
{
  int i;
  int num,num2;
  char *ptr;
  
  for (i=6;i<strlen(buf) && !G_ISDIGIT(buf[i]) && buf[i]!='-';i++) {
  }
  
  if (i==strlen(buf)) {
    return 0;
  }
  
  if (buf[i]=='-') {
    num=atoi(&buf[i+1]);
    *rangemin=size-num;
    *rangemax=size;
    return 1;
  }
  
  num=(int) strtol(&buf[i],&ptr,10);
  
  if ((*ptr)!='-') {
    return 0;
  }
  
  ptr++;
  
  if (!G_ISDIGIT(*ptr)) {
    *rangemin = num;
    *rangemax = size;
    return 1;
  }
  
  num2 = (int) strtol(ptr, 0, 10);
  
  *rangemin = num;
  *rangemax = num2;
  return 1;
}
#endif

int incoming_transfer_error(int sock, int err, char *message)
{
  /* used to be 404. and "File not found" */
  char a[100];

  sprintf(a, "HTTP %i ", err);
  writestr(sock, a);
  writestr(sock, message);
  writestr(sock, "\r\n\r\n");
  writestr(sock, "There was an error processing your request\r\n");
  writestr(sock, message);
  writestr(sock, "\r\n\r\n");
  
  return 0;
}

int incoming_transfer_handle(gnut_transfer *gt, struct sockaddr_in *sin,
  char *request_line)
{
  int ret;
  http_request_t *req;

  GD_S(1, "incoming_transfer_handle entering\n");

  gt->gt_state = STATE_NEGOTIATING;
  memcpy(gt->ip, &sin->sin_addr.s_addr, 4);
  gt->gt_port = ntohs(sin->sin_port);
  gt->rate_limit = 0;

  req = http_request_new(gt->sock);
  ret = http_request_parse(req, request_line);
  if (!ret) {
    ret = http_request_read_header(req);
  }
  /* This is not a good request, we want nothing to do with it... shut
     it down now. */
  if (ret) {
    /* shutdown(gt->sock, 2); */ /* This is done by transfer.cleanup */

    transfer_close(gt, STATE_ERROR); /* 0.4.28.c02 */
    transfer_dealloc(gt);
    return -1;
  }

  GD_S(1, "incoming_transfer_handle Got HTTP request, processing\n");

  /* Check for standard Gnutella file GET request */
  if (strncmp(req->url_file, "/get/", 5) == 0) {
    ret = gnut_http_serve_file(gt, req);
    http_request_delete(req);
    num_uploads--;
    if (!ret) {
      count_uploads++;
    }

    transfer_close(gt, (ret==0) ? STATE_DEAD : STATE_ERROR); /* 0.4.28.c02 */
    transfer_dealloc(gt);
    return ret;
  }

#ifdef GNUT_HTTP_SEARCH
  /* gnut also supports "/" and "/search/...", which provide the HTTP
     gateway interface. */
  else if ((strcmp(req->url_file, "/") == 0) && conf_get_int("html_enable")) {
    GD_S(2, "incoming_transfer_handle handling file list...\n");
    ret = gnut_http_file_list(gt, req);
    http_request_delete(req);
    num_uploads--;

    transfer_close(gt, (ret==0) ? STATE_DEAD : STATE_ERROR); /* 0.4.28.c02 */
    transfer_dealloc(gt);
    return ret;
  }
#endif

#ifdef GNUT_HTTP_FILE_LIST
  /* Handle a search request. */
  else if ((strncmp(req->url_file, "/search/", 8) == 0)
	   && conf_get_int("html_enable")) {
    GD_S(2, "incoming_transfer_handle handling search form...\n");
    ret = gnut_http_file_search(gt, req);
    http_request_delete(req);
    num_uploads--;

    transfer_close(gt, (ret==0) ? STATE_DEAD : STATE_ERROR); /* 0.4.28.c02 */
    transfer_dealloc(gt);
    return ret;
  }
#endif

  GD_S(1, "incoming_transfer_handle removing http request\n");
  http_request_delete(req);
  num_uploads--;

  transfer_close(gt, STATE_ERROR); /* 0.4.28.c02 */
  transfer_dealloc(gt);
  
  return ret;
}

int incoming_transfer(int sock, struct sockaddr_in *sin, char *request_line)
{
  gnut_transfer *gt;

  dqi(0x005a);

  GD_S(1, "incoming_transfer entering num_uploads:"); GD_I(1, num_uploads);
  GD_S(1, "\n");

  if (++num_uploads > gc_max_uploads) {
    num_uploads--;
    incoming_transfer_error(sock, 503, "Upload limit reached"); /* tagok */
    write(sock, "The maximum number of uploads at this location has been reached.\r\n", 66);
    shutdown(sock,2);
    close_s(sock);
    return 0;
  }

  GD_S(3, "incoming_transfer INCOMING TRANSFER!!!\n");

  gt = gnut_xfer_new();
  gt->sock = sock;
  gt->type = TYPE_OUTGOING;

  return incoming_transfer_handle(gt, sin, request_line);  
}

/* After a file transfer connection has been established and the initial
   handshaking (HTTP get header, etc.) are handled, this routine is called.
   The data transfer is passed up to gnut_xfer_loop and this routine
   handles the errors and retry decision logic. */
int gtdh_1(gnut_transfer *a_gt, int startp, int overlap,
	   char ** overlap_buf, query_resp *a_qrp1, int *rtr_en1,
	   int *tried_push)
{
  int wh;
  int ret;
  int dest_cach;
  query_resp * a_qrp4;

  dest_cach = a_qrp1->qr_dcache;

  a_gt->gt_state = STATE_CONNECTED;
  a_gt->gt_bytes = startp;

  if (gc_debug_opts & 16) {
    printf("lseek %d", startp);
  }
  wh = lseek(a_gt->fsock, startp, SEEK_SET);
  if (gc_debug_opts & 16) {
    printf(" returned %d\n", wh);
  }

  count_downloads++;

  dqi(0x0060);
  ret = gnut_xfer_loop(a_gt, overlap, *overlap_buf, dest_cach);
  dqi(0x0061);

  if (*overlap_buf) {
    fre_str(overlap_buf, 137);
  }

  dqi(0x0179);

  /* do not create incomplete cache files, and don't leave *any*
     incomplete files that are less than the cutoff set by the user */
  if (ret != 0) {
    if (dest_cach) {
      if (gc_verbose & 4) {
	printf(UI_TH_CA_FAILED, a_gt->fname);
      }
      unlink(a_gt->fname);
    } else if (a_gt->gt_bytes < conf_get_int("leave_turds")) {
      dqi(0x017c);
      unlink(".gnut_last_turd");
      dqi(0x017d);
      rename(a_gt->fname, ".gnut_last_turd");
    }
  }

  dqi(0x017a);

  if (ret == 0) {
    /* Successful download */
    if (rtr_en1) {
      if (*rtr_en1) {
	if (gc_debug_opts & 8) {
	  printf("retry disable 03 (success, general)\n");
	  printf("  %d.%d.%d.%d", (int)(a_gt->ip[0]), (int)(a_gt->ip[1]),
		 (int)(a_gt->ip[2]), (int)(a_gt->ip[3]));
	  printf("  %s\n", a_gt->fname);
	}
	*rtr_en1 = 0;
      }
    }

    /* since we succeeded, let's change the name to what it originally was
       intended to be */
    if (strlen(a_gt->fname) > 5) {
      char *c;
      char *c2;

      c = &a_gt->fname[strlen(a_gt->fname)-5];
      GD_S(1, "gtdh1 c="); GD_S(1, c); GD_S(1, "\n");
      if (strncmp(c, ".gnut", 5)==0) {
	GD_S(1, "gtdh1 attempt rename file...\n");
	c2 = ystdup(a_gt->fname, 457);
	*c = 0;
	rename(c2, a_gt->fname);
	fre_str(&c2, 557);
	/* %%% Here's where we would test for a renaming error. any
	   attempted fix should include proper handling of the
	   extension, if any. */
      }
    }

    if (a_gt->type == TYPE_RECV_PUSH) {
      GD_S(1, "gtdh1 finishing off query\n");
      /* we succeeded, so we can finish off the query struc */
      if (rtr_en1) {
	if (*rtr_en1) {
	  if (gc_debug_opts & 8) {
	    printf("retry disable 04 (success, push)\n");
	    printf("  %d.%d.%d.%d", (int)(a_gt->ip[0]), (int)(a_gt->ip[1]),
		   (int)(a_gt->ip[2]), (int)(a_gt->ip[3]));
	    printf("  %s\n", a_gt->fname);
	  }
	  *rtr_en1 = 0;
	}
      }
    }
  } else {
    /* Incomplete transfer. */
    int allow_push;

    dqi(0x017e);

    /* Figure out if we should push. At last count, there were a total
       of seven requirements that must be met! */
    allow_push = 1;

    /* No push if the state was changed (which means we were stopped by
       the user) */
    if (a_gt->gt_state != STATE_CONNECTED) {
      allow_push = 0;
      if (rtr_en1) {
	if (*rtr_en1) {
	  *rtr_en1 = 0;
	  if (gc_debug_opts & 8) {
	    printf("retry disable 13 (by 'stop' command)\n");
	    printf("  %d.%d.%d.%d", (int)(a_gt->ip[0]), (int)(a_gt->ip[1]),
		   (int)(a_gt->ip[2]), (int)(a_gt->ip[3]));
	    printf("  %s\n", a_gt->fname);
	  }
	}
      }
    }

    dqi(0x0178);

    /* No push if download retry is disabled */
    if (conf_get_int("auto_download_retry") == 0) {
      if (rtr_en1) {
	if (*rtr_en1) {
	  if (gc_debug_opts & 8) {
	    printf("retry disable 05 (by user setting, lost)\n");
	    printf("  %d.%d.%d.%d", (int)(a_gt->ip[0]), (int)(a_gt->ip[1]),
		   (int)(a_gt->ip[2]), (int)(a_gt->ip[3]));
	    printf("  %s\n", a_gt->fname);
	  }
	  *rtr_en1 = 0;
	}
	allow_push = 0;
      }
    }

    /* If we've already sent a push, we don't send another */
    if (tried_push) {
      if(*tried_push) {
	allow_push = 0;
      }
    }

    /* The user can completely disallow push retries */
    if (conf_get_int("retry_push") <= 0) {
      allow_push = 0;
    }

    /* Check to see if user selected rfc-1597 compliant push retries,
       and if so, push only if the host would be able to connect to us */
    dqi(0x0100);
    if ((conf_get_int("retry_push") == 1)
	&& (!(host_ok_for_push(a_qrp1->qr_ip)))) {
      allow_push = 0;
    }

    dqi(0x0103);
    /* If this transfer was a direct (non-push) and we got a
       significant number of bytes, then we know a direct download is
       a workable way to get the file, and therefore we don't have to
       do the push thang */
    if ((a_gt->type != TYPE_RECV_PUSH) && (a_gt->gt_bytes >=512)) {
      allow_push = 0;
    }

    /* We don't push for cache downloads, because the cache is supposed
       to produce minimal load on the network. */
    if (dest_cach) {
      allow_push = 0;
    }

    if (allow_push) {
      /* If this was a push connection, we put the push request back
         on the list to allow another future one to come in. If this
         wasn't a push connection, we need to create a new push record
         and send the push request packet, and we also in that case
         need to clone the qreply so the push thread can delete its
         qreply when it gets terminated and we can delete ours,
         without stepping on each other. */

      dqi(0x0104);

      if (a_gt->type != TYPE_RECV_PUSH) {
	gnutella_packet *gpa;

	if (gc_verbose & 1) {
	  dqi(0x0101);
	  printf(UI_TH_DL_STPUSH, a_qrp1->qr_ndata);
	}

	dqi(0x0102);
	gpa = gp_push_make(conf_get_str("mac"), conf_get_int("ttl"),
			   a_qrp1->qr_guid, a_qrp1->qr_ref, net_local_ip(),
			   conf_get_int("local_port"));

	send_to_all(gpa);

	fre_v(&(gpa->data), 138);
	fre_gpa(&gpa, 139);

	if (tried_push) {
	  *tried_push = 1;
	}
      }

      GD_S(1, "gtdh1 adding query back to list\n");

      /* Clone the qreply because our caller deletes the one we have here */
      dqi(0x0112);
      a_qrp4 = query_copy(a_qrp1, 548);
      dqi(0x0113);
      query_push_add(a_qrp4);
    }
  }

  dqi(0x0105);

  if (gc_transferlog) {
    time_t tim;
    struct tm *ltim;
    tim = time(0);
    ltim = localtime(&tim);

    if ((ret == 0) || (a_gt->gt_bytes >= conf_get_int("leave_turds"))) {
      fprintf(gc_transferlog, "%d%02d%02d.%02d%02d%02d"
	      "\tr\t%i.%i.%i.%i:%i\t%u\t%u\t%u\t%s\n",
	      1900+ltim->tm_year, 1+ltim->tm_mon, ltim->tm_mday,
	      ltim->tm_hour, ltim->tm_min, ltim->tm_sec,
	      a_gt->ip[0], a_gt->ip[1], a_gt->ip[2], a_gt->ip[3],
	      a_gt->gt_port, a_gt->gt_bytes, a_gt->gt_total,
	      a_gt->rate_bytes >> 4L, a_gt->fname);
    }
  }
  GD_S(4, "gtdh1 decreasing downloads\n");

  return ret;
}

int gnut_threads_dl_handle(query_resp *a_qrp1, gnut_transfer *a_gt,
			   int *rtr_en2, int *tried_push)
{
  char buf[2048];
  int startp;
  char *dpath;
  char *request;
  char *shortname;
  int ret;
  char *dest_dir;
  int dest_cach;
  struct stat ss;
#if !defined(HAVE_FLOCK) && defined(HAVE_FCNTL)
  struct flock lock;
#endif
  int overlap;
  char *overlap_buf;

  dqi(0x005d);

  startp = 0;
  dest_cach = a_qrp1->qr_dcache;
  a_gt->gt_dest_cache = dest_cach;
  if (dest_cach) {
    int cl1;

    dqi(0x006b);
    cl1 = cache_lock();
    if (cl1 == 0) {
      /* Cache was already locked elsewhere, so we should forget this
         transfer */
      /* %%% clean */
      transfer_close(a_gt, STATE_ERROR); /* 0.4.28.c02 */
      /* %%% return */
      return -5;
    }
  }
  dqi(0x0072);

  /* !!HERE!! is where the entry point needs to be for push connections... */
  a_gt->gt_state = STATE_NEGOTIATING;
  dqi(0x0073);
  dql((uint32) a_qrp1);

  /* first we're going to check to see if the file already exists,
   * if so, then we'll formulate a partial request and use it instead */
  shortname = strrchr(a_qrp1->qr_ndata, '/');
  dqi(0x0074);

  if (shortname) {
    dqi(0x006c);
    shortname++;
  } else {
    dqi(0x006d);
    shortname = a_qrp1->qr_ndata;
  }
  dqi(0x006e);
  dest_dir = (dest_cach ? "cache_path" : "download_path");

  if (conf_get_str(dest_dir)) {
    dqi(0x0065);
    a_gt->fname = (char *) ymaloc(strlen(conf_get_str(dest_dir))
				  + a_qrp1->qr_nlen + 250, /* tagok */
				  316);
    a_gt->fname[0] = 0;
    dpath = expand_path(conf_get_str(dest_dir));
    if (strlen(dpath) > 1) {
      strcpy(a_gt->fname, dpath);
      if (dpath[strlen(dpath)-1] != path_slash[0]) {
	strcat(a_gt->fname, path_slash);
      }
    }
    strcat(a_gt->fname, shortname);
    strcpy(buf, a_gt->fname);
    strcat(a_gt->fname, ".gnut");
    fre_str(&dpath, 134);
  } else {
    dqi(0x0066);
    if (dest_cach) {
      dqi(0x005e);
      GD_S(1, "gnut_thr_dl_handle no cache dir set.\n");
      cache_unlock();

      /* %%% clean */
      transfer_close(a_gt, STATE_ERROR); /* 0.4.28.c02 */
      /* %%% return */
      return -1;
    }
    a_gt->fname = (char *) ymaloc(strlen(shortname)+7, 315);
    strcpy(a_gt->fname, shortname);
    strcpy(buf, a_gt->fname);
    strcat(a_gt->fname, ".gnut");
  }

  dqi(0x0067);

  /* first we're going to see if the un-munged name exists, if so
   * then always make the download fail, because purportedly a good file
   * already exists */
  GD_S(1, "gnut_thr_dl_handle checking if file \""); GD_S(1, buf);
  GD_S(1, "\" exists\n");

  ret = stat(buf, &ss);
  if (ret != -1) {
    /* woah! this file already exists, we shouldn't download it */
    if (gc_verbose & 1) {
      printf(UI_TH_FILE_EXISTS, buf);
    }

    if (dest_cach) {
      cache_unlock();
    }

    if (rtr_en2) {
      if (*rtr_en2) {
	if (gc_debug_opts & 8) {
	  printf("retry disable 01 (finished transfer already exists)\n");
	  printf("  %d.%d.%d.%d", (int)(a_gt->ip[0]), (int)(a_gt->ip[1]),
		 (int)(a_gt->ip[2]), (int)(a_gt->ip[3]));
	  printf("  %s\n", a_gt->fname);
	}
        *rtr_en2 = 0;
      }
    }

    /* %%% clean */
    transfer_close(a_gt, STATE_ERROR); /* 0.4.28.c02 */
    /* %%% return */
    return -3;
  }  

  dqi(0x0068);

  GD_S(1, "gnut_thr_dl_handle testing file \""); GD_S(1, a_gt->fname);
  GD_S(1, "\" for size\n");

  overlap = 0;
  overlap_buf = 0;

  a_gt->fsock = open(a_gt->fname, O_RDONLY);
  if (a_gt->fsock >= 0) {
    dqi(0x0069);
    GD_S(1, "gnut_thr_dl_handle file exists!\n");
    /* the file exists, so we'll find out its size... */
    startp = lseek(a_gt->fsock, 0, SEEK_END);
    if (gc_debug_opts & 16) {
      printf("End of existing file is %d\n", startp);
    }
    GD_S(1, "gnut_thr_dl_handle start="); GD_I(1, startp); GD_S(1, "\n");
    if (startp < 0) {
      startp = 0;
    } else if (startp >512) {
      dqi(0x006a);
      overlap_buf = ymaloc(512, 317);
      if (overlap_buf) {
	overlap = 512;
	startp -= overlap; /* %%% overlap checking isn't implemented
			      yet, we just blindly overlap always. This
			      eliminates error-text turds but not
			      variant content errors. Still need to
			      merge file I/O */
      }
    } else {
      startp = 0;
    }

    close_f(a_gt->fsock);
  }

  GD_S(3, "gnut_thr_dl_handle checked for existing, start="); GD_I(3, startp);
  GD_S(3, "\n");

  /* now that we're connected, we can send our request message.... */
  dqi(0x005f);
  request = (char *) ymaloc(a_qrp1->qr_nlen + 2049, 318);

  /* Write out the initial GET */
  dqi(0x00ff);
  sprintf(request, "GET /get/%u/%s HTTP/1.0\r\n",
	  a_qrp1->qr_ref,
	  a_qrp1->qr_ndata);
  write(a_gt->sock, request, strlen(request));

  /* Write out the user-specified headers for HTTP client emulation */
  {
    char *c;

    sprintf(request, "%s", conf_get_str("http_client_headers"));
    /* Do backslash escapes. Only one is provided -- '\n' becomes an HTTP
     * standard line-break CRLF. All other '\' sequences are left untouched.
     * conveniently, we can turn '\n' into CRLF without having to shift
     * the rest of the string */
    for(c = request; *c;) {
      if (*c == '\\') {
        if (*(c+1) == 'n') {
	  *c++ = '\r';
	  *c++ = '\n';
	}
      }
      if (*c) {
	c++;
      }
    }
    if (gc_debug_opts & 4) {
      printf("gdh02 Sending HTTP client header:\n%s", request);
    }
    write(a_gt->sock, request, strlen(request));
  }

  /* Write out the range header, if any */
  if (startp == 0) {
    /* We want the whole file -- no range necessary */
  } else {
    sprintf(request, "Range: bytes=%i-\r\n", startp);
  }
  write(a_gt->sock, request, strlen(request));

  /* Write out the final crlf */
  sprintf(request, "\r\n");
  write(a_gt->sock, request, strlen(request));

  GD_S(3,"gnut_thr_dl_handle connect message sent\n");

  /* now we look at the http response and make sure that the
   * correct number of newlines is returned.... */
  while (1) {
    ret = read_line(a_gt->sock, buf, sizeof(buf));
    if (gc_debug_opts & 4) {
      printf("gdh01 %s", buf);
    }

    if ((ret <= 0) || strstr(buf, "HTTP 4")) {
      if (dest_cach) {
	cache_unlock();
      }
      fre_str(&request, 135);

      if (strstr(buf, "HTTP 4")) {
        if (rtr_en2) {
	  if (*rtr_en2) {
	    if (gc_debug_opts & 8) {
	      printf("retry disable 11 (HTTP 4xx response)\n");
	      printf("  %d.%d.%d.%d", (int)(a_gt->ip[0]), (int)(a_gt->ip[1]),
		     (int)(a_gt->ip[2]), (int)(a_gt->ip[3]));
	      printf("  %s\n", a_gt->fname);
	    }
	    *rtr_en2 = 0;
	  }
	}
      }
      if (overlap_buf) {
	fre_str(&overlap_buf, 136);
      }

      /* %%% clean */
      transfer_close(a_gt, STATE_ERROR); /* 0.4.28.c02 */
      /* %%% return */
      return -4;
    }
    if (strlen(buf) <= 2) {
      break;
    }
  }

  GD_S(3, "gnut_thr_dl_handle read http headers...\n");

  /* We used to open with O_WRONLY | O_CREAT | O_APPEND, but once overlap
   * is implemented we'll need to be able to read (in order to verify that
   * it's the same file) and in order to reset the pointer we cannot
   * have O_APPEND. */
  a_gt->fsock = open(a_gt->fname, O_RDWR | O_CREAT, OCTAL(6,6,6));

  if (a_gt->fsock < 0) {
    fprintf(stderr, UI_TH_CANT_WRITE, a_gt->fname);
  }

  /* This file-locking is necessary because the user might try to download
   * multiple files with exactly the same name. The effect of the locking
   * is to allow only one such download at a time. The first one is allowed,
   * and all subsequent ones fail. */

  ret = 0;
#ifdef HAVE_FLOCK
  ret = flock(a_gt->fsock,LOCK_EX | LOCK_NB);
#elif defined(HAVE_FCNTL)
  lock.l_type = F_WRLCK;
  lock.l_whence = SEEK_SET;
  lock.l_start = 0;
  lock.l_len = 0;
  ret = fcntl(a_gt->fsock, F_SETLK, &lock);
#endif

  if ((a_gt->fsock > 0) && (ret >= 0)) {
    /* Do transfer and deal with results including retry enable and push */
    ret = gtdh_1(a_gt, startp, overlap, &overlap_buf, a_qrp1, rtr_en2,
		 tried_push);
  } else if (a_gt->fsock > 0) {
    if (gc_verbose & 1) {
      printf(UI_TH_NO_DUPE, a_gt->fname);
    }
    close_f(a_gt->fsock);
  }

  dqi(0x0106);
  GD_S(3, "gnut_thr_dl_handle closing up transfer\n");
  fre_str(&request, 140);

  if (dest_cach) {
    cache_timestamp();
    rescan_cache();
    cache_unlock();
  }

  /* %%% clean */
  transfer_close(a_gt, STATE_DEAD); /* 0.4.28.c02 */

  GD_S(1, "gnut_thr_dl_handle returning "); GD_I(1, ret); GD_S(1, "\n");
  /* %%% return */
  return ret;
}

/* This routine is for incoming PUSH connections, which are downloads
   we requested. It does the special stuff that needs to be done for
   an incoming push connection and then calls gnut_threads_dl.handle
   to do the actual file transfer */
int incoming_push(int sock, struct sockaddr_in *sin, char *request)
{
  int ret;
  int i;
  char buf[256];
  uchar guid[16];
  uint32 ref;
  uint16 port;
  char *ptr;
  query_resp *a_qrp;  
  gnut_transfer *gt;
  
  dqi(0x0059);

  GD_S(1, "inc_push entering sock="); GD_I(1, sock); GD_S(1, " request=");
  GD_S(1, request); GD_S(1, "\n");

  ret = 1;
  while ((ret > 0) || (errno == EINTR)) {
    dqi(0x0064);

    GD_S(1, "inc_push calling readline\n");
    ret = read_line(sock, buf, 256); /* tagok */

    GD_S(1, "inc_push recvd: "); GD_S(1, buf); GD_S(1, "\n");
    if (strlen(buf) < 3) {
      break;
    }
  }

  /* we need to create the appropriate transfer and query struct,
   * then pass them on to the standard downloader. */

  /* Get the file refnum */
  ref = atoi(&request[4]);

  ptr = strchr(request, ':');
  if (ptr == 0) {
    /* No ":" in GIV line */
    dqi(0x0071);
    GD_S(0, "inc_push AGGG!\n");
    return 0;
  }

  /* Convert GUID from 32 hex digits to 16 bytes */  
  ptr++;
  for (i=0; i<16; i++) {
    strncpy(buf, &ptr[i*2], 2);
    buf[2] = 0;
    guid[i] = strtol(buf, 0, 16);
  }
  GD_S(2, "inc_push ref="); GD_I(2, ref); GD_S(2, " GUID=");
  GD_02X(2, guid[0]); GD_02X(2, guid[1]); GD_02X(2, guid[2]);
  GD_02X(2, guid[3]); GD_S(2, "...\n");
  GD_S(1, "inc_push calling query find...\n");

  a_qrp = query_push_find(guid, ref); /* no qr new */
  if (a_qrp == 0) {
    /* Usually this happens because a remote host got two copies of
     * a PUSH packet that we sent and has initiated a second connection. */
    dqi(0x005b);

    GD_S(1, "inc_push push request not in our list: "); GD_S(1, request);
    GD_S(1, "\n");
    close_s(sock);
    return 0;
  }

  gt = gnut_xfer_new();
  gt->sock = sock;
  gt->type = TYPE_RECV_PUSH;
  gt->gt_state = STATE_NEGOTIATING;
  gt->gt_total = a_qrp->qr_size;
  memcpy(gt->ip, a_qrp->qr_ip, 4);

  port = GET_LEU16(a_qrp->qr_port_le, 33);

  gt->gt_port = port;
  gt->rate_limit = 0;
  gt->gt_dest_cache = a_qrp->qr_dcache;

  /* Here there was some disabled (#if 0) code that tested
     max_downloads and went to sleep until max_downloads was no longer
     exceeded. The reason we can't do this on a push is because the
     other end will drop the push connection if we wait more than a
     few seconds to start the download, and if we let that happen,
     then what was the point of sending out the push in the first
     place? To see the code, look in version 0.4.27 or earlier. */

  num_downloads++;

  dqi(0x0070);
  ret = gnut_threads_dl_handle(a_qrp, gt, 0, 0);
  dqi(0x0111);

  transfer_dealloc(gt); /* 0.4.28.c02 */

  /* Get rid of the qreply. If it was returned to the push list,
     it will have been copied first */
  query_kill(&a_qrp, 550);  /* qr-del */
  dqi(0x0126);

  GD_S(1, "inc_push gnut_thr_dl_handle returned: "); GD_I(1, ret);
  GD_S(1, "\n");
  num_downloads--;
  
  return 0;
}

/* void *gnut_threads_incoming(void *arg)
 *
 * main thread for handling incoming connections, socket is passed via the
 * pointer, it needs to be fr.ee'ed.  First it determines whether this is
 * an upload, or gnutella connection, then calls the appropriate handler
 * for each. */
void *gnut_threads_incoming(void *arg)
{
  int sock;
  char buf1[256];
  char buf2[BMSG_LEN+1]; /* 0.4.28.c23 */
  int ret;
  incoming_arg *ia;

  dq_start("IN", 0x004d);

  GD_S(2,"gnut_threads_incoming entering\n");
  
  ia = arg;
  sock = ia->s;  
  
  fcntl(sock, F_SETFL, 0);

  ret = read_line(sock, buf1, sizeof(buf1));
  GD_S(1, "gnut_threads_incoming read_line returned: "); GD_I(1, ret);
  GD_S(1, "\n"); GD_S(1, "the initial line was: "); GD_S(1, buf1);
  GD_S(1, "\n");
  
  if (ret <= 0) {
    dqi(0x0052);

    close_s(sock);
    fre_ia(&ia, 111);

    dq_end(0);
  }

  /* This is now considered a successful connection for the purposes of
     the must_push EQHD flag */
  gh_did_receive = 1; /* 0.4.28.c03 */

  buf2[BMSG_LEN] = 0; /* 0.4.28.c23 (and following lines) */
  strncpy(buf2, conf_get_str("virtual_network_name"), BMSG_LEN);
  strcatlim(buf2, " CONNECT", BMSG_LEN);

  if (strncmp(buf1, buf2, strlen(buf2))==0) { /* 0.4.28.c23 */
    /* this is an incoming gnutella connection */
    dqi(0x0053);
    if (!gnut_blacklist_allow(BLACKLIST_TYPE_INCOMING, 0, &(ia->sin.sin_addr),
			      ia->sin.sin_port, 0)) {
      close_s(sock);
    } else {
      incoming_gnutella(sock, &ia->sin);
    }
  } else if (strncmp(buf1, "GET ", 4)==0) {
    /* this is an incoming http connection */
    dqi(0x0054);
    if (gc_debug_opts & 4) {
      printf("gth01 %s", buf1);
    }

    if (!gnut_blacklist_allow(BLACKLIST_TYPE_HTTP, 0, &(ia->sin.sin_addr),
			      ia->sin.sin_port, 0)) {
      dqi(0x0055);
      GD_S(1, "gnut_threads_incoming Blacklisted\n");
      close_s(sock);
    } else {
      incoming_transfer(sock, &ia->sin, buf1);
    }
  } else if (strncmp(buf1, "GIV ", 4)==0) {
    /* this is a push connection */
    dqi(0x0056);
    if (gc_debug_opts & 4) {
      printf("gth01 %s", buf1);
    }

    incoming_push(sock, &ia->sin, buf1);
  } else {
    dqi(0x0057);
    if (gc_debug_opts & 4) {
      printf("gth01 %s", buf1);
    }

    close_s(sock);
  }
  fre_ia(&ia, 112);

  dqi(0x0058);

  GD_S(3,"gnut_threads_incoming returning\n");

  dq_end(0);
  return 0;
}

Gnut_List *push_crcs = 0;

/* 0.4.28.c01 Moved mutex to gnut.c */

Gnut_List *bad_push_ips = 0;

/* Search for a bad IP in the bad push IPs list. Must be called inside
 * mutex lock on the push lists. */
time_t check_push_ip(uint32 tip)
{
  Gnut_List *gl;
  bad_ip *bip;
  time_t rv;

  gl = bad_push_ips;
  rv = 0;
  while(gl) {
    bip = (bad_ip *) (gl->data);
    if (bip) {
      if (bip->ip == tip) {
	rv = bip->when;
	gl = 0;
      }
    }
    if (gl) {
      gl = gl->next;
    }
  }

  return rv;
}

/* Remove a bad IP from the bad push IPs list. Must be called inside mutex
 * lock on the push lists */
void remove_push_ip(uint32 tip)
{
  Gnut_List *gl;
  bad_ip *bip;

  gl = bad_push_ips;
  while(gl) {
    bip = (bad_ip *) (gl->data);
    if (bip) {
      if (bip->ip == tip) {
	bad_push_ips = gnut_list_remove(bad_push_ips, bip, 6);
	fre_bip(&bip, 113);
	gl = 0;
      }
    }
    if (gl) {
      gl = gl->next;
    }
  }
}

/* Throw a bad IP address in jail, for a specified number of seconds.
 * Must be called inside push mutex lock. */
void add_push_ip(uint32 tip, uint32 sentence)
{
  bad_ip *bip;

  bip = (bad_ip *) ymaloc(sizeof(bad_ip), 313);
  bip->ip = tip;
  bip->when = time(0) + sentence;
  bad_push_ips = gnut_list_prepend(bad_push_ips, (void *) bip);
}

/* this is passed a push_arg struc
 *
 * it is called from our program in response to a push packet,
 * so it needs to contact the foreign host, do the negotiating,
 * then send the file
 *
 * There are a bunch of different reasons why push requests fail,
 * and most of these can be used to test subsequent push requests
 * to see if they should be attempted.
 *
 *  - Often, multiple identical push requests will arrive over
 *    different GnutellaNet links. This results from other
 *    clients not routing the push packets properly. To
 *    avoid making multiple simultaneous connections, each push
 *    request must be checked to see if it is identical to a
 *    request already in progress.
 *  - Sometimes a duplicate request arrives after the first request
 *    has completed. In this case, the connect works but the host
 *    at the other end does not issue a GET command. These are put
 *    on the "bad list" for 10 seconds.
 *  - I have seen push requests for IP addresses (such as
 *    100.100.100.100 and 123.45.67.89) that do not work and
 *    are clearly intended to be spam. These do not respond to
 *    connects at all. gnut cannot identify a bad IP without
 *    trying to connect, but if a connect fails the IP is put on
 *    the "bad list" for 10 minutes.
 */
void *gnut_threads_push(void *arg)
{
  push_arg *pa;
  gnut_transfer *gt;
  uint16 port;
  char buf_ip[256];
  char buf2[256];
  int ret;
  char *lname; /* leaf name */
  char *request;
  int i;
  struct sockaddr_in sin;
  int reject;
  unsigned long crc1;
  unsigned long ip1;

  dq_start("PU", 0x004e);

  GD_S(1, "gnut_threads_push entering\n");
  pa = arg;

  sprintf(buf_ip, "%i.%i.%i.%i", pa->ip[0], pa->ip[1], pa->ip[2], pa->ip[3]);

  lname = pa->si->path;
#ifndef WIN32
  lname = strrchr(pa->si->path, '/');
  if (lname) {
    lname++;
  } else {
    lname = pa->si->path;
  }
#endif

  /* Calculate CRC for this PUSH request */
  crc1 = crc32_string(lname); ip1 = 0;
  for(i=0; i<4; i++) {
    crc32_add8(((uint32 *) (&crc1)), pa->ip[i]);
    ip1 = (ip1 << 8L) | ((unsigned) pa->ip[i]);
  }

  if (gc_debug_opts & 1) {
    printf("PUSH %lu %s %s\n", crc1, buf_ip, lname);
  }

  /* Check for various reasons why we would reject the PUSH request */
  reject = 0;
  if (!(host_ok_for_get(pa->ip))) {
    /* We cannot reach the IP address in the PUSH request. */
    if (gc_debug_opts & 1) {
      printf("PUSH %lu IP is in inaccessible VPN, ignoring.\n", crc1);
    }
    GD_S(1, "gnut_threads_push reject PUSH from inaccessible VPN address\n");
    reject = 1;
  } else if (num_uploads >= gc_max_uploads) {
    /* We have enough uploads already going on. We don't queue PUSH
     * requests. */
    if (gc_debug_opts & 1) {
      printf("PUSH %lu upload limit exceeded, ignoring PUSH.\n", crc1);
    }
    GD_S(1, "gnut_threads_push too many uploads!\n");
    reject = 1;
  } else {
    /* Check for duplicate PUSH request already in progress. */
	
    /* NOTE: The check (searching the list) and insert must be done together
     * as an atomic operation */
    pthread_mutex_lock(&push_crc_mutex);
	
    /* Check PUSH's currently in progress, then check previous known
     * bad IPs */
    if (gnut_list_seek(push_crcs, (void *) crc1)) {
      /* Yup, it's a duplicate */
      if (gc_debug_opts & 1) {
	printf("PUSH %lu already in progress through another thread.\n", crc1);
      }
      GD_S(1, "gnut_threads_push ignoring duplicate PUSH\n");
      reject = 1;
    } else {
      int32 whenbad;
      int32 now;

      now = (int32) time(0);
      whenbad = (int32) check_push_ip(ip1);
      if (whenbad) {
	if (whenbad > now) {
	  /* This IP is blacklisted and its sentence has not yet
           * been served */
	  if (gc_debug_opts & 1) {
	    printf("PUSH %lu bad push IP, ignoring request.\n", crc1);
	  }
	  GD_S(1, "gnut_threads_push ignoring bad push IP\n");
	  reject = 1;
	} else {
	  /* This IP was bad, but it has served its time and may
	   * now go fr.ee */
	  if (gc_debug_opts & 1) {
	    printf("PUSH %lu IP was bad, but it's been a while.\n", crc1);
	  }
	  remove_push_ip(ip1);
	}
      } else {
	/* IP isn't in bad list, so we're okay. */
      }
    }

    if (reject == 0) {
      /* join the active push list */
      push_crcs = gnut_list_prepend(push_crcs, (void *) crc1);
    }

    pthread_mutex_unlock(&push_crc_mutex);
  }

  if (reject) {
    /* Forget push request, clean up */
    fre_str(&(pa->si->path), 114);
    fre_str(&(pa->si->lc_path), 115);
    fre_str(&(pa->si->fpath), 116);
    fre_si(&(pa->si), 117);
    fre_pa(&pa, 118);
    dq_end(0);
  }

  num_uploads++;
  
  gt = gnut_xfer_new();
  
  gt->type = TYPE_SEND_PUSH;
  gt->gt_state = STATE_CONNECTING;
  gt->substate = 1;
  gt->gt_total = pa->si->size;
  gt->fname = ystdup(pa->si->path, 461);
  memcpy(gt->ip, pa->ip, 4);

  port = GET_LEU16(pa->port_le, 34);

  gt->gt_port = port;
  gt->rate_limit = 0;
  
  GD_S(1, "gnut_threads_push created new transfer struct\n");

  gt->sock = make_connection /* In net.c */
    (buf_ip, gt->gt_port, gt->ip, 1, MT_FOR_PUSH, gt, 0);
  if (gt->sock < 0) {
    if (gc_debug_opts & 1) {
      printf("PUSH %lu make_conn failed.\n", crc1);
    }

    pthread_mutex_lock(&push_crc_mutex);
    push_crcs = gnut_list_remove(push_crcs, (void *) crc1, 7);
    add_push_ip(ip1, 600); /* 10 minutes on the no-no list tagok */
    pthread_mutex_unlock(&push_crc_mutex);

    GD_S(1, "gnut_threads_push couldn't connect to "); GD_S(1, buf_ip);
    GD_S(1, ":"); GD_I(1, port); GD_S(1, " make_conn returned: ");
    GD_I(1, gt->sock); GD_S(1, "\n");
    close_s(gt->sock);
    fre_str(&(pa->si->path), 119);
    fre_str(&(pa->si->lc_path), 120);
    fre_str(&(pa->si->fpath), 121);
    fre_si(&(pa->si), 122);
    fre_pa(&pa, 123);
    num_uploads--;

    transfer_close(gt, STATE_ERROR); /* 0.4.28.c02 */
    transfer_dealloc(gt);

    dq_end(0);
  }

  memcpy(&sin.sin_addr.s_addr, gt->ip, 4);
  sin.sin_port = gt->gt_port;

  GD_S(1, "gnut_threads_push made connection to "); GD_S(1, buf_ip);
  GD_S(1, ":"); GD_I(1, port); GD_S(1, "\n");

  gt->gt_state = STATE_NEGOTIATING;

  request = (char *) ymaloc(4096, 314);

  sprintf(request, "GIV %u:", pa->si->ref);

  for(i=0; i<16; i++) {
    sprintf(buf2, "%02X", (uchar) (conf_get_str("guid"))[i]);
    strcat(request, buf2);
  }

  sprintf(buf2, "/%s\n\n", lname);

  strcat(request, buf2);

  write(gt->sock, request, strlen(request));

  GD_S(1, "gnut_threads_push connect message sent:\n"); GD_S(1, request);
  GD_S(1, "\n");

  /* now we look at the http response and make sure that the
   * correct number of newlines is returned.... */
  while (1) {
    GD_S(1, "gnut_threads_push trying to read line\n");
    ret = read_line(gt->sock, buf2, sizeof(buf2));
    if (ret <= 0) {
      if (gc_debug_opts & 1) {
	printf("PUSH %lu remote did not send 'GET', closing.\n", crc1);
      }

      pthread_mutex_lock(&push_crc_mutex);
      push_crcs = gnut_list_remove(push_crcs, (void *) crc1, 8);
      add_push_ip(ip1, 10); /* only 10 seconds, because it's probably just
			     * a duplicate and satisfied past push request */
      pthread_mutex_unlock(&push_crc_mutex);

      GD_S(1, "gnut_threads_push fr""ee'ing request\n");
      fre_str(&request, 124);
      GD_S(1, "gnut_threads_push closing sock\n");
      close_s(gt->sock);
      GD_S(1, "gnut_threads_push fr""ee'ing pa->si->path\n");
      fre_str(&(pa->si->path), 125);
      GD_S(1, "gnut_threads_push fr""ee'ing pa->si\n");
      fre_si(&(pa->si), 126);
      GD_S(1, "gnut_threads_push fr""ee'ing pa\n");
      fre_pa(&pa, 127);
      num_uploads--;

      transfer_close(gt, STATE_ERROR); /* 0.4.28.c02 */
      transfer_dealloc(gt);

      dq_end(0);
    }
    GD_S(1, "gnut_threads_push received: "); GD_S(1, buf2);
    if (strncmp(buf2, "GET ", 4)==0) {
      break;
    }
  }

  if (gc_debug_opts & 1) {
    printf("PUSH %lu data transfer begun\n", crc1);
  }

  /* now that we've gotten the GET, we can treat this as any normal
   * file upload, so we'll pass it on to the normal upload handler... */
  ret = incoming_transfer_handle(gt, &sin, buf2);

  GD_S(2,"gnut_threads_push ITH returned, now cleaning up\n");

  if (gc_debug_opts & 1) {
    printf("PUSH %lu data transfer done, returned %d\n", crc1, ret);
  }

  fre_str(&request, 128);

  GD_S(2, "gnut_threads_push request fr""ee'ed and sock closed\n");

  fre_str(&(pa->si->path), 129);
  fre_str(&(pa->si->lc_path), 130);
  fre_str(&(pa->si->fpath), 131);
  fre_si(&(pa->si), 132);
  fre_pa(&pa, 133);

  pthread_mutex_lock(&push_crc_mutex);
  push_crcs = gnut_list_remove(push_crcs, (void *) crc1, 9);
  pthread_mutex_unlock(&push_crc_mutex);

  GD_S(2, "gnut_threads_push push_arg struc fr""ee'ed\n");

  dq_end(0);
  return 0;
}    

/* gnut_th.dl sets up a connection for downloading a file. */
void *gnut_th_dl(gnut_transfer *gt, query_resp *a_qrp2, int *rtr_en3,
		 int *tried_push, int sl)
{
  gnutella_packet *gpa;
  char buf[100];
  int ret;

  GD_S(2, "gnut_th.dl entering\n");

  dqi(0x0108);
  if (a_qrp2->qr_dcache) {
    int cl2;

    dqi(0x0109);
    /* check if we've reached the limit for number of simultaneous cache
       downloads */
    cl2 = cache_lock();
    if (cl2 == 0) {
      /* yeah, we hit the limit */
      if (rtr_en3) {
	if (*rtr_en3) {
	  if (gc_debug_opts & 8) {
	    printf("retry disable 06 (mutex cache)\n");
	    printf("  %d.%d.%d.%d", (int)(a_qrp2->qr_ip[0]),
		   (int)(a_qrp2->qr_ip[1]), (int)(a_qrp2->qr_ip[2]),
		   (int)(a_qrp2->qr_ip[3]));
	    printf("  %s\n", a_qrp2->qr_ndata);
	  }
	  *rtr_en3 = 0;
	}
      }
      /* %%% return */
      return 0;
    }

    dqi(0x0128);

    /* empty enough space in the cache */
    if (cache_shrink(a_qrp2->qr_size) < 0) {
      GD_S(2, "gnut_th.dl aborting download - couldn't fr""ee enough cache space.\n");
      if (rtr_en3) {
	if (*rtr_en3) {
	  if (gc_debug_opts & 8) {
	    printf("retry disable 07 (cache full)\n");
	    printf("  %d.%d.%d.%d", (int)(a_qrp2->qr_ip[0]),
		   (int)(a_qrp2->qr_ip[1]), (int)(a_qrp2->qr_ip[2]),
		   (int)(a_qrp2->qr_ip[3]));
	    printf("  %s\n", a_qrp2->qr_ndata);
	  }
	  *rtr_en3 = 0;
	}
      }
      /* %%% return */
      return 0;
    }
  }

  /* 0.4.28.c02 Moved gt alloc and filling in fields to gnut_threads.dl */

  dqi(0x0110);

  gt->gt_state = STATE_RETRY_WAIT; /* 0.4.28.c02 */

  /* Sleep, if requested by our caller */
  if (sl) {
    if (gc_debug_opts & 8) {
      printf(UI_TH_DL_RETRY, sl);
    }
    GD_S(1, "gnut_th.dl retry-sleep "); GD_I(1, sl); GD_S(1, "\n");
    while ((sl > 0) && (gt->gt_state == STATE_RETRY_WAIT)) {
      sleep(1);
      sl--;
    }
  }

  dqi(0x014a);
  /* See if we got stopped while sleeping */
  if (gt->gt_state != STATE_RETRY_WAIT) {
    if (rtr_en3) {
      if (*rtr_en3) {
	if (gc_debug_opts & 8) {
	  printf("retry disable 08 (stopped in retry-sleep)\n");
	    printf("  %d.%d.%d.%d", (int)(gt->ip[0]), (int)(gt->ip[1]),
		   (int)(gt->ip[2]), (int)(gt->ip[3]));
	  printf("  %s\n", gt->fname);
	}
	*rtr_en3 = 0;
      }
    }
    GD_S(3, "gnut_th.dl state was changed while we waited for retry\n");
    /* %%% clean */
    transfer_close(gt, STATE_DEAD); /* 0.4.28.c02 */
    /* %%% return */
    return 0;
  }

  dqi(0x014b);
  gt->gt_state = STATE_CONNECTING;
  gt->substate = 3;

  if (++num_downloads > conf_get_int("max_downloads")) {
    /* enter queued state... */
    gt->substate = 2;
    GD_S(1, "gnut_th.dl I'm queued\n");
    num_downloads--;

    if (gc_verbose & 1) {
      printf(UI_TH_DL_QUEUED, gt->fname);
    }
    gt->gt_state = STATE_QUEUED;
    while ((num_downloads >= conf_get_int("max_downloads")) 
	   && (gt->gt_state == STATE_QUEUED)) {
      sleep(1);
    }
    /* Now check to see if the transfer was killed while we were
       sleeping */
    if (gt->gt_state != STATE_QUEUED) {
      if (rtr_en3) {
	if (*rtr_en3) {
	  if (gc_debug_opts & 8) {
	    printf("retry disable 12 (stopped while queued)\n");
	    printf("  %d.%d.%d.%d", (int)(gt->ip[0]), (int)(gt->ip[1]),
		   (int)(gt->ip[2]), (int)(gt->ip[3]));
	    printf("  %s\n", gt->fname);
	  }
	  *rtr_en3 = 0;
	}
      }
      /* %%% clean */
      transfer_close(gt, STATE_DEAD); /* 0.4.28.c02 */
      /* %%% return */
      return 0;
    }
    GD_S(1, "gnut_th.dl and now I'm not!\n");
    num_downloads++;
    gt->gt_state = STATE_CONNECTING;
    gt->substate = 3;
  }

  dqi(0x0149);
  sprintf(buf, "%i.%i.%i.%i", a_qrp2->qr_ip[0], a_qrp2->qr_ip[1],
	  a_qrp2->qr_ip[2], a_qrp2->qr_ip[3]);

  gt->sock = make_connection  /* In net.c */
    (buf, gt->gt_port, gt->ip, 1, MT_FOR_DOWNLOAD, gt, rtr_en3);

  dqi(0x014c);

  if (gt->sock < 0) {
    int allow_push;

    dqi(0x015e);
    allow_push = 1;

    /* If there is no retry, then we don't push */
    if (conf_get_int("auto_download_retry") == 0) {
      if (rtr_en3) {
	if (*rtr_en3) {
	  if (gc_debug_opts & 8) {
	    printf("retry disable 09 (by user setting, blocked)\n");
	    printf("  %d.%d.%d.%d", (int)(gt->ip[0]), (int)(gt->ip[1]),
		   (int)(gt->ip[2]), (int)(gt->ip[3]));
	    printf("  %s\n", gt->fname);
	  }
	  *rtr_en3 = 0;
	}
	allow_push = 0;
      }
    }

    /* If we've already tried push once, we don't try again */
    if (tried_push) {
      if (*tried_push) {
	allow_push = 0;
      }
    }

    /* Check to see if user completely disallowed push retries */
    if (conf_get_int("retry_push") == 0) {
      allow_push = 0;
    }

    dqi(0x015f);
    /* Check to see if user selected rfc-1597 compliant push retries */
    if ((conf_get_int("retry_push") == 1)
	&& (!(host_ok_for_push(a_qrp2->qr_ip)))) {
      allow_push = 0;
    }

    if (a_qrp2->qr_dcache) {
      allow_push = 0;
    }

    num_downloads--;

    /* If it's for the cache, we only try once per file. The idea is to
       have the cache be friendly to the hosts -- the cache is less important,
       so it shouldn't generate repeated connections to other servers.
       For similar reasons, the cache doesn't generate push requests */
    if (a_qrp2->qr_dcache) {
      dqi(0x016b);
      if (rtr_en3) {
	if (*rtr_en3) {
	  if (gc_debug_opts & 8) {
	    printf("retry disable 10 (dest cache)\n");
	    printf("  %d.%d.%d.%d", (int)(gt->ip[0]), (int)(gt->ip[1]),
		   (int)(gt->ip[2]), (int)(gt->ip[3]));
	    printf("  %s\n", gt->fname);
	  }
	  *rtr_en3 = 0;
	}
      }
      cache_unlock();
      /* %%% clean */
      transfer_close(gt, STATE_ERROR); /* 0.4.28.c02 */
    /* %%% return */
      return 0;
    }

    dqi(0x0160);
    GD_S(1, "gnut_th.dl couldn't connect to "); GD_S(1, buf); GD_S(1, ":");
    GD_I(1, gt->gt_port); GD_S(1, " make_conn returned: "); GD_I(1, gt->sock);
    GD_S(1, "\n");

    /* %%% clean */
    transfer_close(gt, STATE_ERROR); /* 0.4.28.c02 */

    dqi(0x0161);
    if (allow_push) {
      query_resp *b_qrp2;

      /* what I'm going to do now is send out a push request for the file
       * we just couldn't get.... maybe it will work on occasion */

      if (gc_verbose & 1) {
	printf(UI_TH_DL_STPUSH, a_qrp2->qr_ndata);
      }

      if (tried_push) {
	*tried_push = 1;
      }

      gpa = gp_push_make(conf_get_str("mac"), conf_get_int("ttl"),
		         a_qrp2->qr_guid, a_qrp2->qr_ref, net_local_ip(),
		         conf_get_int("local_port"));

      /* Clone the qreply because our caller deletes the one we have here */
      b_qrp2 = query_copy(a_qrp2, 519);
      query_push_add(b_qrp2);  

      send_to_all(gpa);
      fre_v(&(gpa->data), 141);
      fre_gpa(&gpa, 142);
    }

    dqi(0x0162);
    /* %%% return */
    return 0;
    /* end of handling failed connect */
  }

  /* connect succeeded, so now we proceed with handshaking and actual file
     transfer! */
  dqi(0x0163);

  GD_S(3, "gnut_th.dl made connection to "); GD_S(3, buf); GD_S(3, ":");
  GD_I(3, gt->gt_port); GD_S(3, "\n");

  dqi(0x0164);
  fre_str(&(gt->fname), 143);
  dqi(0x006f);
  ret = gnut_threads_dl_handle(a_qrp2, gt, rtr_en3, tried_push);

  num_downloads--;
  /* %%% return */
  return 0;
}

void *gnut_threads_dl(void *arg)
{
  gnut_transfer *gt;
  query_resp *a_qrp3;
  int retry_enable;
  int tried_push;
  int sl;
  uint16 port;

  dq_start("DL", 0x004f);
  dqp(arg);

  a_qrp3 = arg;
  retry_enable = a_qrp3->qr_retry;
  tried_push = 0;

  sl = 0;
  dqi(0x0107);

  /* 0.4.28.c02 Moved this code here from gnut_th.dl() */
  /* Put ourselves in the transfers list */
  gt = gnut_xfer_new();
  gt->type = TYPE_INCOMING;
  gt->gt_state = STATE_RETRY_WAIT;
  gt->gt_total = a_qrp3->qr_size;
  memcpy(gt->ip, a_qrp3->qr_ip, 4);

  port = GET_LEU16(a_qrp3->qr_port_le, 35);

  gt->gt_port = port;
  gt->rate_limit = 0;
  gt->fname = ystdup(a_qrp3->qr_ndata, 458);
  gt->gt_dest_cache = a_qrp3->qr_dcache;

  GD_S(3, "gnut_threads.dl created & setup xfer struc 1\n");
  gnut_th_dl(gt, a_qrp3, &retry_enable, &tried_push, sl);

  sl = 8;
  while (conf_get_int("auto_download_retry") && retry_enable) {
    /* Exponential backoff algorithm */
    if (sl < conf_get_int("auto_download_retry")) {
      /* They set a positive integer */
      sl = ((sl * 5) / 4) + 1;
    } else {
      /* They set a negative integer, or we've reached the limit */
      sl = conf_get_int("auto_download_retry");
      /* Check for negative */
      if (sl < 0) {
	sl = 0 - sl;
      }
    }

    /* Okay, try that download again! */

    /* 0.4.28.c02 reset the fields that need to be reset for a new download
       attempt */
    gt->gt_state = STATE_RETRY_WAIT;
    gt->gt_bytes = 0;
    gt->rate_bytes = 0;
    gt->rate_incr_time = time(0);
    gt->substate = 0;

    GD_S(3, "gnut_threads.dl created & setup xfer struc 1\n");
    gnut_th_dl(gt, a_qrp3, &retry_enable, &tried_push, sl);
  }

  transfer_dealloc(gt); /* 0.4.28.c02 */

  dqi(0x016c);
  query_kill(&a_qrp3, 144); /* qr-del */

  dq_end(0);
  return 0;
}

/* void *gnut_threads_outgoing(void *arg)
 *
 * This is the main thread for handling an outgoing GnutellaNet connection.
 * It does the connecting, negotiates with the remote servant, transfers
 * the packets and cleans up after itself.
 * arg is a char * containing the name or ip address of the host to
 * connect to */
void *gnut_threads_outgoing(void *a)
{
  char *hname;
  gcs *mgc;
  char *ptr;
  int ret;
  uchar cport_le[2];
  struct hostent *he;

  hname = a;

  dq_start("OU", 0x0050);

  GD_S(1, "gnut_thr_outgoing entering hname="); GD_S(1, hname); GD_S(1, "\n");

  num_outgoing++;

  mgc = gnut_connection_new();
  dqi(0x0115);

  mgc->ctype = TYPE_OUTGOING;
  mgc->cstate = STATE_CONNECTING;

  gnut_strdelimit(hname, ":", ' ');

  ptr = strchr(hname, ' ');
  if (ptr) {
    *(ptr++) = 0;
    mgc->port = atoi(ptr);
  } else {
    mgc->port = 6346;
  }
  
  dqi(0x0116);
  PUT_LEU16(cport_le, mgc->port, 53);

  /* inet_aton(hname, (struct in_addr *) mgc->ip); */
  he = gethostbyname(hname);
  if (he == 0) {
    mgc->cstate = STATE_CLEANABLE;
    gnut_connection_delete(mgc);
    fre_str(&hname, 146);
    num_outgoing--;

    dq_end(0); /* exit the thread */
  }
  memcpy(mgc->ip.b, he->h_addr_list[0], 4);

  dqi(0x0117);
  mgc->gc_sock = make_connection  /* In net.c */
    (hname, mgc->port, mgc->ip.b, 0, MT_FOR_GNET, 0, 0);
  dqi(0x0118);

  if  (mgc->gc_sock<0) {
    GD_S(1, "gnut_thr_outgoing couldn't connect to: "); GD_S(1, hname);
    GD_S(1, ":"); GD_I(1, mgc->port); GD_S(1, "\n");
    mgc->cstate = STATE_CLEANABLE;

    dqi(0x012a);
    host_remove(mgc->ip.b, cport_le);

    dqi(0x012b);
    gnut_connection_delete(mgc);
    dqi(0x0170);
    fre_str(&hname, 147);
    num_outgoing--;

    dq_end(0); /* exit the thread */
  }

  dqi(0x012c);
  mgc->cstate = STATE_NEGOTIATING;

  ret = send_welcome_message(mgc->gc_sock, "0.4");

  dqi(0x012e);

  if (ret<0) {
    GD_S(1, "gnut_thr_outgoing couldn't send message\n");
    close_s(mgc->gc_sock);

    host_remove(mgc->ip.b, cport_le);

    gnut_connection_delete(mgc);
    GD_S(1, "gnut_thr_outgoing fr""ee'ing hname\n");
    fre_str(&hname, 148);
    GD_S(1, "gnut_thr_outgoing done fr""ee'ing hname\n");
    num_outgoing--;

    dq_end(0); /* exit the thread */
  }

  dqi(0x012f);
  ret = read_welcome_message(mgc->gc_sock, hname);
  if (conf_get_int("password_only") && (ret == 1)) { /* 0.4.28.c23 */
    /* password-only is set, but this connection isn't asking for a
       password, so we reject it as if it had the wrong welcome message
       string. */
    dqi(0x0185);
    ret = -3;
  } else if (ret == 2) {
    char buf[256];

    dqi(0x0186);
    /* a password was requested, we'll try sending the
     * one in our config */
    sprintf(buf, "%s\n", conf_get_str("password"));

    write(mgc->gc_sock, buf, strlen(buf));

    /* now we need to repeat the read for a welcome message */
    ret = read_welcome_message(mgc->gc_sock, hname);
    if (ret < 0) {
      if (strlen(conf_get_str("password"))) {
	printf(UI_TH_INCOR_PASS);
      }
    }
  }

  if (ret < 0) {
    dqi(0x0130);

    if (ret == -3) { /* 0.4.28.c23 */
      GD_S(1, "gnut_thr_outgoing no non-password connections\n");
    } else {
      GD_S(1, "gnut_thr_outgoing couldn't read welcome message\n");
    }
    close_s(mgc->gc_sock);

    host_remove(mgc->ip.b, cport_le);
    gnut_connection_delete(mgc);
    GD_S(1, "gnut_thr_outgoing fr""ee'ing hname\n");
    if (hname) {
      fre_str(&hname, 149); /* %%% this is protected by "if (hname)"
             because it has generated double-fr.ee errors and I don't
             know why. The block comes from ymaloc 269. */
    }
    GD_S(1, "gnut_thr_outgoing done fr""ee'ing hname\n");
    num_outgoing--;

    dq_end(0); /* exit the thread */
  }
  
  dqi(0x012d);
  mgc->cstate = STATE_CONNECTED;

  /* Read and write packets */
  ret = connection_loop(mgc);
  dqi(0x003b);

  mgc->cstate = STATE_CLEANABLE;
  num_outgoing--;
  close_s(mgc->gc_sock);
  host_remove(mgc->ip.b, cport_le);
  dqi(0x0172);

  gnut_connection_delete(mgc);
  dqi(0x0173);
  fre_str(&hname, 150);

  dq_end(0);
  return 0;
}

/* 0.4.28.c01 Moved mutexes_init() to gnut.c */
