/* -*- C++ -*-
   XGStreamContext - Drawing context using the XR Library.

   Copyright (C) 1995 Free Software Foundation, Inc.

   Written by:  Adam Fedor <fedor@boulder.colorado.edu>
   Date: Nov 1995
   
   This file is part of the GNU Objective C User Interface Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
   */

#include "gnustep/xgps/XGStreamContext.h"
#include "gnustep/xgps/XGGState.h"
#include "gnustep/xgps/XGContextPrivate.h"
#include "SharedX/xrtools.h"
#include <Foundation/NSArray.h>
#include <Foundation/NSDictionary.h>
#include <Foundation/NSData.h>
#include <Foundation/NSValue.h>
#include <Foundation/NSString.h>
#include <Foundation/NSUserDefaults.h>
#include "string.h"


#define XDPY (((RContext *)context)->dpy)
#define XSCR (((RContext *)context)->screen_number)

@interface XGStreamContext (Private)

- (void) output: (const char*)s;

@end

@implementation XGStreamContext 

+ (Display*) currentXDisplay
{
  return NULL;
}

- (void) destroyContext;
{
  if (gstream)
    fclose(gstream);
  [super destroyContext];
}

- initWithContextInfo: (NSDictionary *)info
{
  NSZone *zone =  GSObjCZone(self);

  /* Initialize lists and stacks */
  opstack = [[NSMutableArray allocWithZone: zone] initWithCapacity: 2];
  gstack = [[NSMutableArray allocWithZone: zone] initWithCapacity: 2];
  glist  = [[NSMutableArray allocWithZone: zone] initWithCapacity: 32];
  
  /* gstate index of 0 indicates no gstate, thus we put a dummy object at 0.
     Then create a default gstate */
  [glist addObject: [[NSObject allocWithZone: zone] init]];
  //gstate = [[XGGState allocWithZone: zone] initWithDrawContext: self];
  //[glist addObject: gstate];

  //XDPY = 0;

  if (info && [info objectForKey: @"NSOutputFile"])
    {
      NSString *path = [info objectForKey: @"NSOutputFile"];
      gstream = fopen([path fileSystemRepresentation], "w");
      if (!gstream)
        {
	  NSDebugLLog(@"XGContext", @"%@: Could not open printer file %@",
		      DPSinvalidfileaccess, path);
	  return nil;
	}
    }
  else
    {
      NSDebugLLog(@"XGContext", @"%@: No stream file specified",
		  DPSconfigurationerror);
      return nil;
    }

  [super initWithContextInfo: info];
  return self;
}

- (BOOL)isDrawingToScreen
{
  return NO;
}

- (void) dealloc
{
  RELEASE(opstack);
  RELEASE(gstack);
  RELEASE(glist);
  [super dealloc];
}

- (void *) xrContext
{
  return context;
}

- (XGGState *) xrCurrentGState
{
  return gstate;
}

- (XGGState *) xrGStateWithIdent: (int)gst
{
  return [glist objectAtIndex:gst];
}

- (Region)viewclipRegion
{
  return 0;
}

- (void) setViewclipRegion: (Region)region;
{
}

/* Override superclass methods */
- (Display *) xDisplay
{
  return NULL;
}

- (void) setXDisplay: (Display *)xdisplay
{
}

- (Window) xDisplayRootWindow
{
  return 0;
}

- (Window) xAppRootWindow
{
  return 0;
}

@end

@implementation XGStreamContext (Ops)

/* ----------------------------------------------------------------------- */
/* Color operations */
/* ----------------------------------------------------------------------- */
- (void)DPScurrentcmykcolor: (float *) c : (float *) m : (float *) y : (float *) k 
{
}

- (void)DPSsetcmykcolor: (float)c : (float)m : (float)y : (float)k 
{
  fprintf(gstream, "%g %g %g %g setcmykcolor\n", c, m, y, k);
}

/* ----------------------------------------------------------------------- */
/* Data operations */
/* ----------------------------------------------------------------------- */
- (void)DPSclear 
{
  fprintf(gstream, "clear\n");
}

- (void)DPScleartomark 
{
  fprintf(gstream, "cleartomark\n");
}

- (void)DPScopy: (int)n 
{
  fprintf(gstream, "%d copy\n", n);
}

- (void)DPScount: (int *)n 
{
}

- (void)DPScounttomark: (int *)n 
{
}

- (void)DPSdup 
{
  fprintf(gstream, "dup\n");
}

- (void)DPSexch 
{
  fprintf(gstream, "exch\n");
}

- (void)DPSexecstack 
{
  fprintf(gstream, "execstack\n");
}

- (void)DPSget 
{
  fprintf(gstream, "get\n");
}

- (void)DPSindex: (int)i 
{
  fprintf(gstream, "%d index\n", i);
}

- (void)DPSmark 
{
  fprintf(gstream, "mark\n");
}

- (void)DPSmatrix 
{
  fprintf(gstream, "matrix\n");
}

- (void)DPSnull 
{
  fprintf(gstream, "null\n");
}

- (void)DPSpop 
{
  fprintf(gstream, "pop\n");
}

- (void)DPSput 
{
  fprintf(gstream, "put\n");
}

- (void)DPSroll: (int)n : (int)j 
{
  fprintf(gstream, "%d %d roll\n", n, j);
}

/* ----------------------------------------------------------------------- */
/* Font operations */
/* ----------------------------------------------------------------------- */
- (void)DPSFontDirectory
{
  fprintf(gstream, "FontDirectory\n");
}

- (void)DPSISOLatin1Encoding
{
  fprintf(gstream, "ISOLatin1Encoding\n");
}

- (void)DPSSharedFontDirectory
{
  fprintf(gstream, "SharedFontDirectory\n");
}

- (void)DPSStandardEncoding
{
  fprintf(gstream, "StandardEncoding\n");
}

- (void)DPScachestatus: (int *) bsize : (int *) bmax : (int *) msize 
{
}

- (void)DPScurrentcacheparams
{
  fprintf(gstream, "currentcacheparams\n");
}

- (void)DPScurrentfont
{
  fprintf(gstream, "currentfont\n");
}

- (void)DPSdefinefont
{
  fprintf(gstream, "definefont\n");
}

- (void)DPSfindfont: (const char *) name 
{
  fprintf(gstream, "/%s findfont", name);
}

- (void)DPSmakefont
{
  fprintf(gstream, "makefont\n");
}

- (void)DPSscalefont: (float)size 
{
  fprintf(gstream, "%f scalefont\n", size);
}

- (void)DPSselectfont: (const char *)name : (float)scale 
{
  fprintf(gstream, "/%s %f selectfont\n", name, scale);
}

- (void)DPSsetcachedevice: (float)wx : (float)wy : (float)llx : (float)lly : (float)urx : (float)ury 
{
  fprintf(gstream, "%f %f %f %f %f %f setcachedevice\n", 
	  wx, wy, llx, lly, urx, ury);
}

- (void)DPSsetcachelimit: (float)n 
{
  fprintf(gstream, "%f setcachelimit\n", n);
}

- (void)DPSsetcacheparams
{
  fprintf(gstream, "setcacheparams\n");
}

- (void)DPSsetcharwidth: (float)wx : (float)wy 
{
  fprintf(gstream, "%f %f setcharwidth\n", wx, wy);
}

- (void)DPSsetfont: (int)f 
{
  fprintf(gstream, "%d setfont\n", f);
}

- (void)DPSundefinefont: (const char *) name 
{
  fprintf(gstream, "/%s undefinefont\n", name);
}

- (void) setFont: (NSFont*) font
{
  const float *matrix = [font matrix];
  const float *m;
  fprintf(gstream, "/%s findfont ", [[font fontName] cString]);

  if ([[self focusView] isFlipped])
    {
      float invmatrix[6];
      
      memcpy(invmatrix, matrix, sizeof(invmatrix));
      invmatrix[3] = -invmatrix[3];
      m = invmatrix;
    }
  else
      m = matrix;

  fprintf(gstream, "[%g %g %g %g %g %g] ", 
	  m[0], m[1], m[2], m[3], m[4], m[5]);
  fprintf(gstream, " makefont setfont\n");
}

/* ----------------------------------------------------------------------- */
/* Gstate operations */
/* ----------------------------------------------------------------------- */
- (void)DPSconcat: (const float *)m 
{
  fprintf(gstream, "[%g %g %g %g %g %g] concat\n", 
	  m[0], m[1], m[2], m[3], m[4], m[5]);
}

- (void)DPScurrentdash
{
  fprintf(gstream, "currentdash\n");
}

- (void)DPScurrentflat: (float *) flatness 
{
}

- (void)DPScurrentgray: (float *) gray 
{
}

- (void)DPScurrentgstate: (int)gst 
{
}

- (void)DPScurrenthalftone
{
}

- (void)DPScurrenthalftonephase: (float *) x : (float *) y 
{
}

- (void)DPScurrenthsbcolor: (float *) h : (float *) s : (float *) b 
{
}

- (void)DPScurrentlinecap: (int *) linecap 
{
}

- (void)DPScurrentlinejoin: (int *) linejoin 
{
}

- (void)DPScurrentlinewidth: (float *) width 
{
}

- (void)DPScurrentmatrix
{
}

- (void)DPScurrentmiterlimit: (float *) limit 
{
}

- (void)DPScurrentpoint: (float *) x : (float *) y 
{
}

- (void)DPScurrentrgbcolor: (float *) r : (float *) g : (float *) b 
{
}

- (void)DPScurrentscreen
{
}

- (void)DPScurrentstrokeadjust: (int *) b 
{
}

- (void)DPScurrenttransfer
{
}

- (void)DPSdefaultmatrix
{
  fprintf(gstream, "defaultmatrix\n");
}

- (void)DPSgrestore
{
  fprintf(gstream, "grestore\n");
}

- (void)DPSgrestoreall
{
  fprintf(gstream, "grestoreall\n");
}

- (void)DPSgsave
{
  fprintf(gstream, "gsave\n");
}

- (void)DPSgstate
{
  fprintf(gstream, "gstate\n");
}

- (void)DPSinitgraphics
{
  fprintf(gstream, "initgraphics\n");
}

- (void)DPSinitmatrix
{
  fprintf(gstream, "initmatrix\n");
}

- (void)DPSrotate: (float)angle 
{
  fprintf(gstream, "%g rotate\n", angle);
}

- (void)DPSscale: (float)x : (float)y 
{
  fprintf(gstream, "%f %f scale\n", x, y);
}

- (void)DPSsetdash: (const float *)pat : (int)size : (float)offset 
{
  int i;
  fprintf(gstream, "[");
  for (i = 0; i < size; i++)
    fprintf(gstream, "%f ", pat[i]);
  fprintf(gstream, "] %g setdash\n", offset);
}

- (void)DPSsetflat: (float)flatness 
{
  fprintf(gstream, "%f setflat\n", flatness);
}

- (void)DPSsetgray: (float)gray 
{
    fprintf(gstream, "%f setgray\n", gray);
}

- (void)DPSsetgstate: (int)gst 
{
// Commented out as a hack, as the stored state for a window or view
// will not be defined in the current context.
//  fprintf(gstream, "%d setgstate\n", gst);
}

- (void)DPSsethalftone
{
  fprintf(gstream, "sethalftone\n");
}

- (void)DPSsethalftonephase: (float)x : (float)y 
{
  fprintf(gstream, "%f %f sethalftonephase\n", x, y);
}

- (void)DPSsethsbcolor: (float)h : (float)s : (float)b 
{
  fprintf(gstream, "%f %f %f sethsbcolor\n", h, s, b);
}

- (void)DPSsetlinecap: (int)linecap 
{
  fprintf(gstream, "%d setlinecap\n", linecap);
}

- (void)DPSsetlinejoin: (int)linejoin 
{
  fprintf(gstream, "%d setlinejoin\n", linejoin);
}

- (void)DPSsetlinewidth: (float)width 
{
  fprintf(gstream, "%g setlinewidth\n", width);
}

- (void)DPSsetmatrix
{
  fprintf(gstream, "setmatrix\n");
}

- (void)DPSsetmiterlimit: (float)limit 
{
  fprintf(gstream, "%f setmiterlimit\n", limit);
}

- (void)DPSsetrgbcolor: (float)r : (float)g : (float)b 
{
  fprintf(gstream, "%g %g %g setrgbcolor\n", r, g, b);
}

- (void)DPSsetscreen
{
  fprintf(gstream, "setscreen\n");
}

- (void)DPSsetstrokeadjust: (int)b 
{
  fprintf(gstream, "%d setstrokeadjust\n", b);
}

- (void)DPSsettransfer
{
  fprintf(gstream, "settransfer\n");
}

- (void)DPStranslate: (float)x : (float)y 
{
  fprintf(gstream, "%g %g translate\n", x, y);
}

/* ----------------------------------------------------------------------- */
/* Matrix operations */
/* ----------------------------------------------------------------------- */
- (void)DPSconcatmatrix
{
  fprintf(gstream, "concatmatrix\n");
}

- (void)DPSdtransform: (float)x1 : (float)y1 : (float *) x2 : (float *) y2 
{
}

- (void)DPSidentmatrix
{
  fprintf(gstream, "identmatrix\n");
}

- (void)DPSidtransform: (float)x1 : (float)y1 : (float *) x2 : (float *) y2 
{
}

- (void)DPSinvertmatrix
{
  fprintf(gstream, "invertmatrix\n");
}

- (void)DPSitransform: (float)x1 : (float)y1 : (float *) x2 : (float *) y2 
{
}

- (void)DPStransform: (float)x1 : (float)y1 : (float *) x2 : (float *) y2 
{
  fprintf(gstream, "%f %f transform\n", x1, y1);
  *x2 = 0;
  *y2 = 0;
}

/* ----------------------------------------------------------------------- */
/* Paint operations */
/* ----------------------------------------------------------------------- */
- (void)DPSashow: (float)x : (float)y : (const char *) s 
{
  fprintf(gstream, "%g %g (", x, y);
  [self output: s];
  fprintf(gstream, ") ashow\n");
}

- (void)DPSawidthshow: (float)cx : (float)cy : (int)c : (float)ax : (float)ay : (const char *) s 
{
}

- (void)DPScopypage
{
  fprintf(gstream, "copypage\n");
}

- (void)DPSeofill
{
  fprintf(gstream, "eofill\n");
}

- (void)DPSerasepage
{
  fprintf(gstream, "erasepage\n");
}

- (void)DPSfill
{
  fprintf(gstream, "fill\n");
}

- (void)DPSimage
{
  fprintf(gstream, "image\n");
}

- (void)DPSimagemask
{
  fprintf(gstream, "imagemask\n");
}

- (void)DPSkshow: (const char *) s 
{
}

- (void)DPSrectfill: (float)x : (float)y : (float)w : (float)h 
{
  fprintf(gstream, "%g %g %g %g rectfill\n", x, y, w, h);
}

- (void)DPSrectstroke: (float)x : (float)y : (float)w : (float)h 
{
  fprintf(gstream, "%g %g %g %g rectstroke\n", x, y, w, h);
}

- (void)DPSshow: (const char *) s 
{
  fprintf(gstream, "(");
  [self output: s];
  fprintf(gstream, ") show\n");
}

- (void)DPSshowpage
{
  fprintf(gstream, "showpage\n");
}

- (void)DPSstroke
{
  fprintf(gstream, "stroke\n");
}

- (void)DPSstrokepath
{
  fprintf(gstream, "strokepath\n");
}

- (void)DPSwidthshow: (float)x : (float)y : (int)c : (const char *) s 
{
  fprintf(gstream, "%g %g %d (", x, y, c);
  [self output: s];
  fprintf(gstream, ") widthshow\n");
}

- (void)DPSxshow: (const char *) s : (const float *)numarray : (int)size 
{
}

- (void)DPSxyshow: (const char *) s : (const float *)numarray : (int)size 
{
}

- (void)DPSyshow: (const char *) s : (const float *)numarray : (int)size 
{
}

/* ----------------------------------------------------------------------- */
/* Path operations */
/* ----------------------------------------------------------------------- */
- (void)DPSarc: (float)x : (float)y : (float)r : (float)angle1 : (float)angle2 
{
  fprintf(gstream, "%g %g %g %g %g arc\n", x, y, r, angle1, angle2);
}

- (void)DPSarcn: (float)x : (float)y : (float)r : (float)angle1 : (float)angle2 
{
  fprintf(gstream, "%g %g %g %g %g arcn\n", x, y, r, angle1, angle2);
}

- (void)DPSarct: (float)x1 : (float)y1 : (float)x2 : (float)y2 : (float)r 
{
  fprintf(gstream, "%g %g %g %g %g arct\n", x1, y1, x2, y2, r);
}

- (void)DPSarcto: (float)x1 : (float)y1 : (float)x2 : (float)y2 : (float)r : (float *) xt1 : (float *) yt1 : (float *) xt2 : (float *) yt2 
{
}

- (void)DPScharpath: (const char *) s : (int)b 
{
  fprintf(gstream, "(");
  [self output: s];
  fprintf(gstream, ") %d charpath\n", b);
}

- (void)DPSclip
{
  fprintf(gstream, "clip\n");
}

- (void)DPSclippath
{
  fprintf(gstream, "clippath\n");
}

- (void)DPSclosepath
{
  fprintf(gstream, "closepath\n");
}

- (void)DPScurveto: (float)x1 : (float)y1 : (float)x2 : (float)y2 : (float)x3 : (float)y3 
{
  fprintf(gstream, "%g %g %g %g %g %g curveto\n", x1, y1, x2, y2, x3, y3);
}

- (void)DPSeoclip
{
  fprintf(gstream, "eoclip\n");
}

- (void)DPSeoviewclip
{
  fprintf(gstream, "eoviewclip\n");
}

- (void)DPSflattenpath
{
  fprintf(gstream, "flattenpath\n");
}

- (void)DPSinitclip
{
  fprintf(gstream, "initclip\n");
}

- (void)DPSinitviewclip
{
  fprintf(gstream, "initviewclip\n");
}

- (void)DPSlineto: (float)x : (float)y 
{
  fprintf(gstream, "%g %g lineto\n", x, y);
}

- (void)DPSmoveto: (float)x : (float)y 
{
  fprintf(gstream, "%g %g moveto\n", x, y);
}

- (void)DPSnewpath
{
  fprintf(gstream, "newpath\n");
}

- (void)DPSpathbbox: (float *) llx : (float *) lly : (float *) urx : (float *) ury 
{
}

- (void)DPSpathforall
{
  fprintf(gstream, "pathforall\n");
}

- (void)DPSrcurveto: (float)x1 : (float)y1 : (float)x2 : (float)y2 : (float)x3 : (float)y3 
{
  fprintf(gstream, "%g %g %g %g %g %g rcurveto\n", x1, y1, x2, y2, x3, y3);
}

- (void)DPSrectclip: (float)x : (float)y : (float)w : (float)h 
{
  fprintf(gstream, "%g %g %g %g rectclip\n", x, y, w, h);
}

- (void)DPSrectviewclip: (float)x : (float)y : (float)w : (float)h 
{
  fprintf(gstream, "%g %g %g %g rectviewclip\n", x, y, w, h);
}

- (void)DPSreversepath
{
  fprintf(gstream, "reversepath\n");
}

- (void)DPSrlineto: (float)x : (float)y 
{
  fprintf(gstream, "%g %g rlineto\n", x, y);
}

- (void)DPSrmoveto: (float)x : (float)y 
{
  fprintf(gstream, "%g %g rmoveto\n", x, y);
}

- (void)DPSsetbbox: (float)llx : (float)lly : (float)urx : (float)ury 
{
  fprintf(gstream, "%g %g %g %g setbbox\n", llx, lly, urx, ury);
}

- (void)DPSsetucacheparams
{
}

- (void)DPSviewclip
{
}

- (void)DPSviewclippath
{
}

/*-------------------------------------------------------------------------*/
/* Graphics Extension Ops */
/*-------------------------------------------------------------------------*/
- (void) DPScomposite: (float)x : (float)y : (float)w : (float)h : (int)gstateNum : (float)dx : (float)dy : (int)op
{
}

- (void) DPScompositerect: (float)x : (float)y : (float)w : (float)h : (int)op
{
}

- (void) DPSdissolve: (float)x : (float)y : (float)w : (float)h : (int)gstateNum
 : (float)dx : (float)dy : (float)delta
{
}

- (void) DPSreadimage
{
}

- (void) DPSsetalpha: (float)a
{
}

- (void) DPScurrentalpha: (float *)alpha
{
}

/*-------------------------------------------------------------------------*/
/* Window Extension Ops */
/*-------------------------------------------------------------------------*/
- (void) DPSflushwindowrect: (float) x : (float) y : (float) w : (float) h : (int) win ;
{
  // FIXME: This should rather not be called from window, 
  // when not drawing on the screen
}

/* ----------------------------------------------------------------------- */
/* X operations */
/* ----------------------------------------------------------------------- */

/* ----------------------------------------------------------------------- */
/* Client functions */
/* ----------------------------------------------------------------------- */
- (void) DPSPrintf: (char *)fmt : (va_list)args
{
  vfprintf(gstream, fmt, args);
}

- (void) DPSWriteData: (char *)buf : (unsigned int)count
{
  /* Not sure here. Should we translate to ASCII if it's not
     already? */
}

@end

static char *hexdigits = "0123456789abcdef";

void
writeHex(FILE *gstream, const unsigned char *data, int count)
{
  int i;
  for (i = 0; i < count; i++)
    {
      fprintf(gstream, "%c%c", hexdigits[(int)(data[0]/16)],
	      hexdigits[(data[0] % 16)]);
      if (i && i % 40 == 0)
	fprintf(gstream, "\n");
    }
}

@implementation XGStreamContext (Graphics)

- (void) NSDrawBitmap: (NSRect) rect : (int) pixelsWide : (int) pixelsHigh
		     : (int) bitsPerSample : (int) samplesPerPixel 
		     : (int) bitsPerPixel : (int) bytesPerRow : (BOOL) isPlanar
		     : (BOOL) hasAlpha : (NSString *) colorSpaceName
		     : (const unsigned char *const [5]) data
{
  int bytes;
  NSSize scale;

  scale = NSMakeSize(NSWidth(rect) / pixelsWide, 
		     NSHeight(rect) / pixelsHigh);
  /* Save scaling */
  fprintf(gstream, "matrix\ncurrentmatrix\n");
  fprintf(gstream, "%f %f translate %f %f scale\n", 
	  NSMinX(rect), NSMinY(rect), scale.width, scale.height);

  if (bitsPerSample == 0)
    bitsPerSample = 8;
  bytes = 
    (bitsPerSample * pixelsWide * pixelsHigh + 7) / 8;
  if (bytes * samplesPerPixel != bytesPerRow * pixelsHigh) 
    {
      NSLog(@"Image Rendering Error: Dodgy bytesPerRow value %d", bytesPerRow);
      NSLog(@"   pixelsHigh=%d, bytes=%d, samplesPerPixel=%d",
	    bytesPerRow, pixelsHigh, bytes);
      return;
    }

  if(samplesPerPixel > 1) 
    {
      if(isPlanar || hasAlpha) 
	{
	  if(bitsPerSample != 8) 
	    {
	      NSLog(@"Image format conversion not supported for bps!=8");
	      return;
	    }
	}
      fprintf(gstream, "%d %d %d [%d 0 0 -%d 0 %d]\n",
	      pixelsWide, pixelsHigh, bitsPerSample, pixelsWide,
	      pixelsHigh, pixelsHigh);
      fprintf(gstream, "{currentfile %d string readhexstring pop}\n",
	      bytesPerRow);
      fprintf(gstream, "false %d colorimage\n", 
	      hasAlpha?(samplesPerPixel-1):samplesPerPixel);
    } 
  else
    {
      fprintf(gstream, "%d %d %d [%d 0 0 -%d 0 %d]\n",
	      pixelsWide, pixelsHigh, bitsPerSample, pixelsWide,
	      pixelsHigh, pixelsHigh);
      fprintf(gstream, "currentfile image\n");
    }
  
  // The context is now waiting for data on its standard input
  if(isPlanar || hasAlpha) 
    {
      // We need to do a format conversion.
      // We do this on the fly, sending data to the context as soon as
      // it is computed.
      int i, j, spp, alpha;
      unsigned char val;
      if(hasAlpha)
	spp = samplesPerPixel - 1;
      else
	spp = samplesPerPixel;
      
      for(j=0; j<bytes; j++) 
	{
	  if(hasAlpha) 
	    {
	      if(isPlanar)
		alpha = data[spp][j];
	      else
		alpha = data[0][spp+j*samplesPerPixel];
	    }
	  for (i = 0; i < spp; i++) 
	    {
	      if(isPlanar)
		val = data[i][j];
	      else
		val = data[0][i+j*samplesPerPixel];
	      if(hasAlpha)
		val = 255 - ((255-val)*(long)alpha)/255;
	      writeHex(gstream, &val, 1);
	    }
	  if (j && j % 40 == 0)
	    fprintf(gstream, "\n");
	}
      fprintf(gstream, "\n");
    } 
  else 
    {
      // The data is already in the format the context expects it in
      writeHex(gstream, data[0], bytes*samplesPerPixel);
    }

  /* Restore original scaling */
  fprintf(gstream, "setmatrix\n");
}

@end

@implementation XGStreamContext (Private)

- (void) output: (const char*)s
{
  const char *t = s;

  while (*t)
    {
      switch (*t)
      {
	case '(':
	    fputs("\\(", gstream);
	    break;
	case ')':
	    fputs("\\)", gstream);
	    break;
	default:
	    fputc(*t, gstream);
	    break;
      }
      t++;
    }
}

@end
