/*
**  GNUMail.m
**
**  Copyright (c) 2001, 2002, 2003
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "GNUMail.h"

#include "AboutPanelController.h"
#include "AddressBookController.h"
#include "BounceWindowController.h"
#include "ConsoleWindowController.h"
#include "EditWindowController.h"
#include "ExtendedMenuItem.h"
#include "ExtendedWindow.h"
#include "Filter.h"
#include "FilterManager.h"
#include "FindWindowController.h"
#include "FolderNode.h"
#include "GNUMail+Extensions.h"
#include "GNUMail+TaskManager.h"
#include "GNUMail/GNUMailBundle.h"
#include "Constants.h"
#include "MailWindowController.h"
#include "MailboxManagerCache.h"
#include "MailboxManagerController.h"
#include "MessageViewWindowController.h"
#include "MimeTypeManager.h"
#include "MimeType.h"
#include "PreferencesWindowController.h"
#include "Utilities.h"
#include "MessageComposition.h"
#include "STScriptingSupport.h"

#include <Pantomime/Constants.h>
#include <Pantomime/Flags.h>
#include <Pantomime/InternetAddress.h>
#include <Pantomime/IMAPCacheManager.h>
#include <Pantomime/IMAPFolder.h>
#include <Pantomime/IMAPStore.h>
#include <Pantomime/LocalFolder.h>
#include <Pantomime/LocalStore.h>
#include <Pantomime/Message.h>
#include <Pantomime/MimeMultipart.h>
#include <Pantomime/MimeUtility.h>
#include <Pantomime/Store.h>
#include <Pantomime/URLName.h>


// GNUMail.app's home page -- where to find more details/information
#define APPLICATION_URL	@"http://www.collaboration-world.com/"

// GNUMail.app's download page -- where to download from
#define DOWNLOAD_URL	@"http://www.collaboration-world.com/cgi-bin/collaboration-world/project/release.cgi?pid=2"

// GNUMail.app's Version Property Page
#define PROPERTY_URL	@"http://www.collaboration-world.com/versionlist.xml"

static NSMutableArray *allEditWindows = nil;
static NSMutableArray *allMailWindows = nil;
static NSMutableArray *allBundles;

static NSString *currentWorkingPath = nil;

static id lastAddressTakerWindowOnTop = nil;
static id lastMailWindowOnTop = nil;
static id requestLastMailWindowOnTop = nil;
static BOOL doneInit = NO;


//
//
//
@implementation GNUMail

- (id) init
{
  self = [super init];

  messageUsingTextEncoding = nil;

#ifdef MACOSX
  _messageCompositions = [[NSMutableArray alloc] init];
#endif
  return self;
}


//
//
//
- (void) dealloc
{
  TEST_RELEASE(messageUsingTextEncoding);
#ifdef MACOSX
  RELEASE(dock);
  RELEASE(_messageCompositions);
#endif  

  [super dealloc];
}

//
// action methods
//
- (IBAction) applyManualFilter: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] ) 
    {
      MailWindowController *aMailWindowController;
      FilterManager *aFilterManager;
      Folder *aFolder;

      NSArray *selectedMessages;
      int i, aTag, nbOfTransferredMessages, aType;
      
      aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      selectedMessages = [aMailWindowController selectedMessages];

      if ( !selectedMessages || [selectedMessages count] == 0 )
	{
	  NSBeep();
	  return;
	}

      aTag = [sender tag];
      nbOfTransferredMessages = 0;

      aFolder = [aMailWindowController folder];

      // If we are in the Sent, we consider ONLY outgoing filters. 
      // Otherwise, we always consider ONLY incoming filters.
      if ( [Utilities stringValueOfURLName: [Utilities stringValueOfURLNameFromFolder: aFolder]  
		      isEqualTo: @"SENTFOLDERNAME"] )
	{
	  aType = TYPE_OUTGOING;
	}
      else
	{
	  aType = TYPE_INCOMING;
	}

      aFilterManager = [FilterManager singleInstance];
      
      for (i = 0; i < [selectedMessages count]; i++)
	{
	  Message *aMessage;
	  Filter *aFilter;

	  aMessage = [selectedMessages objectAtIndex: i];
	  
	  // If we have selected ALL our filters...
	  if ( aTag < 0 )
	    {
	      aFilter = [aFilterManager matchedFilterForMessage: aMessage
					type: aType];
	    }
	  else
	    {
	      aFilter = [aFilterManager filterAtIndex: aTag];
	    }

	  // We verify if our filter matches our message.
	  if ( aFilter && [aFilterManager matchExistsForFilter: aFilter
					  message: aMessage] )
	    {
	      NSString *aFolderName;
	      Store *aStore;

	      aFolderName = nil;
	      aStore = nil;
	      
	      // FIXME - this part of the code is WRONG
	      // We get the right folder name
	      if ( [aFilter action] == TRANSFER_TO_FOLDER )
		{
		  URLName *theURLName;

     		  theURLName = [[URLName alloc] initWithString: [aFilter actionFolderName]
						path: [[NSUserDefaults standardUserDefaults] 
							objectForKey: @"LOCALMAILDIR"]];
		  AUTORELEASE(theURLName);
		  
		  
		  aFolderName = [theURLName foldername];
		  aStore = [[MailboxManagerController singleInstance] storeForURLName: theURLName];
		  
		  if ( !aStore )
		    {
		      NSRunAlertPanel(_(@"Error!"),
				      _(@"An error occured while attempting to obtain the message store for target filter."),
				      _(@"OK"),
				      NULL,
				      NULL);
		      aStore = nil;
		      aFolderName = nil;
		    }
		  
		  // We verify that our target folder isn't the one that is currently open (and last on top)
		  // If so, we do nothing.
		  if ( aFolderName && aStore )
		    {
		      if ( [aFolderName isEqualToString: [[aMailWindowController folder] name]] &&
			   aStore == [[aMailWindowController folder] store] )
			{
			  aStore = nil;
			  aFolderName = nil;
			}
		    }

		  // We verify that our folder hasn't been deleted!
		  // If it was deleted, we do nothing. 
		  if ( aFolderName && aStore )
		    {
		      if ( ![[MailboxManagerController singleInstance] folderNameExists: aFolderName
								       forStore: aStore] )
			{
			  aStore = nil;
			  aFolderName = nil;
			}
		    }
		}
	      // We verify if the operation is DELETE. If so, we transfer the message to the trash folder.
	      // FIXME, must support IMAP folder
	      else if ( [aFilter action] == DELETE )
		{      
		  URLName *theURLName;
		  
		  // FIXME
		  theURLName = [[URLName alloc] initWithString: [[NSUserDefaults standardUserDefaults] 
								  objectForKey: @"TRASHFOLDERNAME"]
						path: [[NSUserDefaults standardUserDefaults] 
							objectForKey: @"LOCALMAILDIR"]];
		  AUTORELEASE(theURLName);
		  
		  aFolderName = [theURLName foldername];
		  aStore = [[MailboxManagerController singleInstance] storeForURLName: theURLName];
		  
		  if ( !aStore )
		    {
		      NSRunAlertPanel(_(@"Error!"),
				      _(@"An error occured while attempting to obtain the message store for target filter."),
				      _(@"OK"),
				      NULL,
				      NULL);
		      aStore = nil;
		      aFolderName = nil;
		    }
		}
	      else
		{
		  aStore = nil;
		  aFolderName = nil;
		}

	      // We got a valid folder name
	      if ( aFolderName && aStore )
		{
		  if ( [[MailboxManagerController singleInstance] transferMessageFromRawSource: [aMessage rawSource]
								  flags: AUTORELEASE([[aMessage flags] copy])
								  toFolderWithName: aFolderName
								  orToFolder: nil
								  store: aStore] )
		    {
		      Flags *theFlags;

		      theFlags = [[aMessage flags] copy];
		      [theFlags add: DELETED];
		      [aMessage setFlags: theFlags];
		      RELEASE(theFlags);

		      nbOfTransferredMessages++;
		    }
		  else
		    {
		      NSDebugLog(@"Transfer failed for message to folder %@", aFolderName);
		    }
		}
	      else
		{
		  NSDebugLog(@"Folder name is nil.");
		}
	    } // if ( aFilter && [aFilterManager ...] )
	} // for (...)
      
      if ( nbOfTransferredMessages > 0 )
	{
	  // We force the reload of our cache in Folder
	  [[aMailWindowController folder] updateCache];
	  [aMailWindowController dataViewShouldReloadData];
	}
    }
  else
    {
      NSBeep();
    }
}


#ifdef MACOSX
//
//
//
- (void) awakeFromNib
{
  dock = [[NSMenu alloc] init];
  [dock setAutoenablesItems: NO];
}


//
//
//
- (NSMenu *) applicationDockMenu: (NSApplication *) sender
{
  return dock; 
}

#endif


//
//
//
- (IBAction) bounceMessage: (id) sender
{
  NSDictionary *allAccounts;

  allAccounts = [[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"];

  // We first verify if we have at least one transport agent defined
  if ( !allAccounts || [allAccounts count] == 0 )
    {
      NSRunAlertPanel(_(@"Error!"),
		      _(@"You must have at least one transport agent defined.\nSee Preferences -> Account."),
		      _(@"OK"), // default
		      NULL,     // alternate
		      NULL);
      return;
    }

  if ( [GNUMail lastMailWindowOnTop] ) 
    {
      id aWindowController;
      Message *theMessage;
      
      aWindowController = [[GNUMail lastMailWindowOnTop] delegate];
      theMessage = [aWindowController selectedMessage];
      
      if ( theMessage )
	{
	  BounceWindowController *bounceWindowController;
	  
	  bounceWindowController = [[BounceWindowController alloc] initWithWindowNibName: @"BounceWindow"];
	  [bounceWindowController setMessage: theMessage];
	  [bounceWindowController showWindow: self];
	}
      else
	{
	  NSBeep();
	}
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) changeTextEncoding: (id) sender
{
  NSWindow *aWindow;

  aWindow = [NSApp keyWindow];

  if ( !aWindow )
    {
      NSBeep();
      return;
    }
  else
    {
      id aWindowController;

      aWindowController = [aWindow windowController];

      //
      // We are working with a MailWindowController / MessageViewWindowController
      //
      if ( [aWindowController isKindOfClass: [MailWindowController class]] ||
	   [aWindowController isKindOfClass: [MessageViewWindowController class]] )
	{
	  Message *theMessage;
	  
	  if ( [aWindowController isKindOfClass: [MailWindowController class]] )
	    {
	      theMessage = [aWindowController selectedMessage];
	    }
	  else
	    {
	      theMessage = [aWindowController message];
	    }

	  if ( theMessage )
	    {
	      NSString *theCharset;
	      id aDataView;

	      aDataView = [aWindowController dataView];
	      theCharset = [theMessage charset];

	      // We get the selected charset
	      if ( [sender tag] >= 0 )
		{
		  NSAutoreleasePool *pool;
		  NSData *theRawSource;
	      
		  pool = [[NSAutoreleasePool alloc] init];
		  theRawSource = [theMessage rawSource];    
		  theCharset = [[[MimeUtility allCharsets] allKeysForObject: [sender title]] objectAtIndex: 0];
	      
		  // We release our previous initialized message
		  TEST_RELEASE(messageUsingTextEncoding);
	      
		  // We re-init using an other charset
		  messageUsingTextEncoding = [[Message alloc] initWithData: theRawSource
							      charset: theCharset];
	      
		  // We release our local pool
		  RELEASE(pool);
		}
	      else
		{
		  messageUsingTextEncoding = theMessage;
		  RETAIN(messageUsingTextEncoding);
		}
	  
	      // We show the new message
	      [Utilities showMessage: messageUsingTextEncoding
			 target: [aWindowController textView]
			 showAllHeaders: [aWindowController showAllHeaders]];

	      // We set the new headers of the message and we refresh the selected row in our data view
	      [theMessage setHeaders: [messageUsingTextEncoding allHeaders]];
	      [theMessage setDefaultCharset: theCharset];
	      [aDataView setNeedsDisplayInRect: [aDataView rectOfRow: [aDataView selectedRow]]];		
	    }
	  else
	    {
	      NSBeep();
	    }
	}
      //
      // We are working with an EditWindowController
      //
      else if ( [aWindowController isKindOfClass: [EditWindowController class]] )
	{
	  [aWindowController setCharset: [sender title]];
	}
      //
      // The rest, just beep!
      //
      else
	{
	  NSBeep();
	}
    }
}


//
// Handles the 'check for update' request
//
- (IBAction) checkForUpdates: (id) sender
{
  NSString *msg, *error;
  
  msg = nil;
  error = nil;

  NS_DURING
    {
      if ( ![self _checkForUpdate] )
        {
          msg = [NSString stringWithFormat: _(@"There is no new version of %@ available."),
			  [[NSProcessInfo processInfo] processName]];
          error = @"";
        }
    }
  NS_HANDLER
    {
      NSDebugLog(@"Software update exception: %@", localException);      
      msg = _(@"Unable to check for new software.");
      error = [NSString stringWithFormat: _(@"Check failed due to the following reason:\n%@"),
			[localException reason]];
    }
  NS_ENDHANDLER {}
  
  if ( msg )
    {
      NSRunInformationalAlertPanel(msg,
				   error,
				   _(@"OK"),
				   NULL,
				   NULL);
    }
  
  return;
}


//
//
//
- (IBAction) close: (id) sender
{
  if ( [NSApp keyWindow] )
    {
      [[NSApp keyWindow] performClose: sender];
    }
}


//
//
//
- (IBAction) compactMailbox: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] ) 
    {
      MailWindowController *aMailWindowController;
      int choice;
      
      aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      choice = NSAlertDefaultReturn;

      if ( ![[NSUserDefaults standardUserDefaults] objectForKey: @"PROMPT_BEFORE_COMPACT"] ||
	   [[NSUserDefaults standardUserDefaults] boolForKey: @"PROMPT_BEFORE_COMPACT"] )
	{
	  choice = NSRunAlertPanel(_(@"Compact..."),
				   _(@"Compacting a mailbox will permanently remove deleted messages.\nDo you want to continue?"),
				   _(@"Compact"),  // default
				   _(@"Cancel"),   // alternate
				   NULL);
	}
      
      if (choice == NSAlertDefaultReturn)
	{
	  [[ConsoleWindowController singleInstance] addConsoleMessage: [NSString stringWithFormat: _(@"Compacting mailbox, please wait..."),
										 [[aMailWindowController folder] name]]];	  
	  [[aMailWindowController folder] expunge: NO];
	  [aMailWindowController dataViewShouldReloadData];
	  [aMailWindowController updateStatusLabel];

	  [[ConsoleWindowController singleInstance] addConsoleMessage: [NSString stringWithFormat: _(@"Done compacting mailbox %@."),
										 [[aMailWindowController folder] name]]];
	}
    }
  else
    {
      NSBeep();
    }
}


//
// This method is used to compose a new message, with an empty content.
//
- (IBAction) composeMessage: (id) sender
{
  EditWindowController *editWindowController;
  Message *aMessage;
  
  // We create an empty message
  aMessage = [[Message alloc] init];
  
  // We create our controller and we show the window
  editWindowController = [[EditWindowController alloc] initWithWindowNibName: @"EditWindow"];

  if ( editWindowController )
    {
      id aMailWindowController;

      [[editWindowController window] setTitle: _(@"New message...")];
      [editWindowController setMessage: aMessage];
      [editWindowController setShowCc: NO];
            
      // We try to guess the best account to use
      aMailWindowController = [GNUMail lastMailWindowOnTop];
      
      if ( aMailWindowController )
	{
	  [editWindowController setAccountName: [Utilities accountNameForFolder: [[aMailWindowController windowController] folder]]];
	}

      [editWindowController showWindow: self];
    }

  RELEASE(aMessage);
}



//
//
//
- (IBAction) copy: (id) sender
{
  NSPasteboard *pboard;
  int i, count;

  
  pboard = [NSPasteboard generalPasteboard];
  
  if ( [[[NSApp keyWindow] delegate] isKindOfClass: [MailWindowController class]] && [GNUMail lastMailWindowOnTop] )
    {
      //
      // MessagePboardType
      // 
      MailWindowController *aMailWindowController;
      NSArray *anArray;
      Message *aMessage;
      NSMutableArray *propertyList;
      NSString *firstSubject;
      
      aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      anArray = [aMailWindowController selectedMessages];
      count = [anArray count];
      firstSubject = nil;

      NSDebugLog(@"copy: %i message(s)", count);

      if ( count )
	{
	  propertyList = [[NSMutableArray alloc] initWithCapacity: count];
      
	  // By default, we assume that our copy operation was sucessful
	  lastCopyOpWasSuccessful = YES;

	  for (i = (count - 1); i >= 0; i--)
	    {
	      // For each implicated message, insert its raw representation
	      // in a dictionary with two keys: 'flags' and 'message'
	      
	      NSMutableDictionary *aDictionary = [[NSMutableDictionary alloc] initWithCapacity: 2];
	      NSData *flags, *rawSource;
	      
	      aMessage = (Message *)[anArray objectAtIndex: i];
	      
	      if (firstSubject == nil)
		{
		  firstSubject = [aMessage subject];
		}

	      flags = [NSArchiver archivedDataWithRootObject: [aMessage flags]];
	      [aDictionary setObject: flags forKey: @"flags"];
	      
	      rawSource = [NSData dataWithData: [aMessage rawSource]];
	      
	      if ( rawSource )
		{
		  [aDictionary setObject: rawSource forKey: @"message"];
		  [propertyList addObject: aDictionary];
		}
	      else
		{
		  lastCopyOpWasSuccessful = NO;
		}
	      
	      RELEASE(aDictionary);
	    }
	  
	  [pboard declareTypes: [NSArray arrayWithObjects: NSStringPboardType, MessagePboardType, nil] 
		  owner: self];
	  [pboard setPropertyList: propertyList 
		  forType: MessagePboardType];
	  if (firstSubject != nil)
	    {
	      [pboard setString: firstSubject
		      forType: NSStringPboardType];
	  }
	  RELEASE(propertyList);
	}
      else
	{
	  NSBeep();
	}
    }
  else if ( [[[NSApp keyWindow] delegate] isKindOfClass: [AddressBookController class]] )
    {
#if 0
      // FIXME
      //
      // AddressPboardType
      //
      AddressBookController *addressBookController;
      AddressBook *addressBook;
      NSBrowser *browser;
      Group *aGroup;
      Address *anAddress;

      addressBookController = [AddressBookController singleInstance];
      addressBook = [AddressBook singleInstance];
      browser = [addressBookController browser];
      
      if ( [browser selectedCellInColumn: 1] )
	{
	  aGroup = [addressBook groupAtIndex: [browser selectedRowInColumn: 0]];
	  anAddress = [aGroup addressAtIndex: [browser selectedRowInColumn: 1]];
	  NSDebugLog(@"copy: address \"%@\"", [anAddress name]);

	  [pboard declareTypes: [NSArray arrayWithObjects: NSStringPboardType, AddressPboardType, nil]
		  owner: self];
	  [pboard setData: [NSArchiver archivedDataWithRootObject: anAddress] 
		  forType: AddressPboardType];
	  [pboard setString: [anAddress formattedAddress]
		  forType: NSStringPboardType];
	}
#endif
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) cut: (id) sender
{
  if ( [[[NSApp keyWindow] delegate] isKindOfClass: [MailWindowController class]] && [GNUMail lastMailWindowOnTop] )
    {
      //
      // MessagePboardType
      //
      MailWindowController *aMailWindowController;
      aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];

      // We copy the selected messages to the pasteboard
      [self copy: nil];
      
      if ( !lastCopyOpWasSuccessful )
	{
	  NSDebugLog(@"The previous copy operation failed for some (or all) messages.");
	  return;
	}

      // We mark the selected messages as transferred
      [aMailWindowController setSelectedMessagesAsTransferred];
    }
  else if ( [[[NSApp keyWindow] delegate] isKindOfClass: [AddressBookController class]] )
    {
#if 0 
      // FIXME
      //
      // AddressPboardType
      //
      AddressBookController *addressBookController;
      AddressBook *addressBook;
      NSBrowser *browser;
      Group *group;
      Address *address;

      addressBookController = [AddressBookController singleInstance];
      addressBook = [AddressBook singleInstance];
      browser = [addressBookController browser];
      
      if ( [browser selectedCellInColumn: 1] )
	{
	  group = [addressBook groupAtIndex: [browser selectedRowInColumn: 0]];
	  address = [group addressAtIndex: [browser selectedRowInColumn: 1]];
	  
	  // We copy the selected address to the pasteboard
	  [self copy: nil];

	  // We remove the selected address from its group
	  [group removeAddress: address];
	  
	  [addressBook synchronize];
		      
	  [browser reloadColumn: 1];
	}
      else
	{
	  NSBeep();
	}
#endif
    }
  else
    {
      NSBeep();
    }
}


//
// This method is used to togle the flags of the selected message
// from the last mail window on top to 'undelete' (or delete) if it was
// set to 'deleted' before the call.
//
- (IBAction) deleteOrUndeleteMessage: (id) sender
{  
  if ( [GNUMail lastMailWindowOnTop] ) 
    {
      MailWindowController *aMailWindowController;
      
      aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      // We delete (or undelete) our message
      [aMailWindowController deleteMessage: nil];
      
      // We 'swap' our title and our tag
      if ( [deleteOrUndelete tag] == DELETE_MESSAGE )
	{
	  [deleteOrUndelete setTitle: _(@"Undelete")];
	  [deleteOrUndelete setTag: UNDELETE_MESSAGE];
	}
      else
	{
	  [deleteOrUndelete setTitle: _(@"Delete")];
	  [deleteOrUndelete setTag: DELETE_MESSAGE];
	}
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) enterSelectionInFindPanel: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] ) 
    {
      MailWindowController *aMailWindowController;
      NSTextView *aTextView;

      aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      aTextView = [aMailWindowController textView];
      
      [[[FindWindowController singleInstance] 
	 findField] setStringValue: [[aTextView string] substringWithRange: [aTextView selectedRange]]];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) findNext: (id) sender
{
  [[FindWindowController singleInstance] nextMessage: nil];
}


//
//
//
- (IBAction) findPrevious: (id) sender
{
  [[FindWindowController singleInstance] previousMessage: nil];
}


// 
// This method is used to forward the selected messsage from the last
// mail window that was on top.
//
- (IBAction) forwardMessage: (id) sender
{  
  if ( [GNUMail lastMailWindowOnTop] )
    {
      id aWindowController;
      
      aWindowController = [[GNUMail lastMailWindowOnTop] delegate];
      [aWindowController forwardMessage: nil];
    }
  else
    {
      NSBeep();
    }
}


//
// This method is used to get the new messages if the LocalInboxWindow
// is currently opened and visible.
//
- (IBAction) getNewMessages: (id) sender
{
  id aController;
  
  aController = [GNUMail lastMailWindowOnTop];

  if ( aController )
    {
      aController = [[GNUMail lastMailWindowOnTop] windowController];

      if ( [aController isKindOfClass: [MessageViewWindowController class]] )
	{
	  aController = [(MessageViewWindowController *)aController mailWindowController];
	}
    }
 
  [self checkForNewMail: sender
	controller: aController];
}


//
//
//
- (IBAction) importMailboxes: (id) sender
{
  NSString *aString;
  NSBundle *aBundle;

#ifdef MACOSX
  aString = [[[NSBundle mainBundle] builtInPlugInsPath] 
	      stringByAppendingPathComponent: @"Import.bundle"];
#else
  aString = [[[NSBundle mainBundle] resourcePath] stringByDeletingLastPathComponent];
  aString = [aString stringByDeletingPathExtension];

  // We verify if we must load the bundles in the GNUstep's Local, Network or System dir.
  // We must also be careful in case poeple are using GNUstep with --enable-flattened
  if ( [aString hasSuffix: @"/Applications/GNUMail"] )
    {
      aString = [NSString stringWithFormat: @"%@/Library/GNUMail/Import.bundle",
			  [[aString stringByDeletingLastPathComponent] 
			    stringByDeletingLastPathComponent]];
    }
  else
    {
      aString = [NSString stringWithFormat: @"%@/Bundles/Import/Import.bundle",
			  [aString stringByDeletingLastPathComponent]];
    }
#endif
	
  aBundle = [NSBundle bundleWithPath: aString];

  if ( aBundle )
    {
      [[[aBundle principalClass] singleInstance] showWindow: self];
      return;
    }

  // We were unable to load the Import bundle.
  NSBeep();
}


//
//
//
- (IBAction) markMessageAsReadOrUnread: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] ) 
    {
      MailWindowController *aMailWindowController;
      NSEnumerator *anEnumerator;
      Message *aMessage;
      
      aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      // We mark our message as read (or not)
      anEnumerator = [[aMailWindowController selectedMessages] objectEnumerator];

      while ( (aMessage = [anEnumerator nextObject]) )
	{ 
	  // If we must mark all messages as read
	  if ( [markAsReadOrUnread tag] == MARK_AS_READ )
	    {
	      if ( ![[aMessage flags] contain: SEEN] )
		{
		  Flags *theFlags;

		  theFlags = [[aMessage flags] copy];	  
		  [theFlags add: SEEN];
		  [aMessage setFlags: theFlags];
		  RELEASE(theFlags);
		} 
	    }
	  // Else, we must mark them all as unread
	  else
	    {
	      if ( [[aMessage flags] contain: SEEN] )
		{
		  Flags *theFlags;

		  theFlags = [[aMessage flags] copy];	  
		  [theFlags remove: SEEN];
		  [aMessage setFlags: theFlags];
		  RELEASE(theFlags);
		} 
	    }
	} // while (...)
      
      // We always refresh our dataView and our status label
      [[aMailWindowController dataView] setNeedsDisplay: YES];
      [aMailWindowController updateStatusLabel];
      
      // We 'swap' our title and our tag
      if ( [markAsReadOrUnread tag] == MARK_AS_READ )
	{
	  [markAsReadOrUnread setTitle: _(@"Mark as Unread")];
	  [markAsReadOrUnread setTag: MARK_AS_UNREAD];
	}
      else
	{
	  [markAsReadOrUnread setTitle: _(@"Mark as Read")];
	  [markAsReadOrUnread setTag: MARK_AS_READ];
	}
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) nextUnreadMessage: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] &&
       [[[GNUMail lastMailWindowOnTop] delegate] isKindOfClass: [MailWindowController class]] )
    {
      [[[GNUMail lastMailWindowOnTop] delegate] nextUnreadMessage: sender];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
#ifdef MACOSX
- (IBAction) newViewerWindow: (id) sender
{
  MailWindowController *aMailWindowController;
  
  aMailWindowController = [[MailWindowController alloc] initWithWindowNibName: @"MailWindow"];
  
  [aMailWindowController setFolder: nil];
  [[aMailWindowController window] orderFrontRegardless];
  [[aMailWindowController window] makeKeyAndOrderFront: nil];
}
#endif


//
//
//
- (IBAction) openInNewWindow: (id) sender
{
#ifdef MACOSX
  if ( [GNUMail lastMailWindowOnTop] &&
       [[[GNUMail lastMailWindowOnTop] delegate] isKindOfClass: [MailWindowController class]] )
    {
      [[MailboxManagerController singleInstance] open: sender];
    }
#else
  if ( [[[MailboxManagerController singleInstance] window] isVisible] )
    {
      [[MailboxManagerController singleInstance] open: sender];
    }
#endif
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) paste: (id) sender
{
  NSArray *types, *propertyList;
  NSPasteboard *pboard;
  NSString *aType;

  int i, j, count;
  

  // We verify every supported pasteboard type  
  pboard = [NSPasteboard generalPasteboard];
  types = [pboard types];

  for (i = 0; i < [types count]; i++)
    {
      aType = (NSString*)[types objectAtIndex: i];
      NSDebugLog(@"paste: type %i = %@", i, aType);
      
      if ( [MessagePboardType isEqualToString: aType] )
	{
	  //
	  // MessagePboardType
	  //
	  propertyList = [pboard propertyListForType: MessagePboardType];

	  if ( propertyList )
	    {
	      count = [propertyList count];
	      
	      NSDebugLog(@"paste: %i message(s)", count);
	      
	      if ( [NSApp keyWindow] == [[MailboxManagerController singleInstance] window] )
		{
		  //
		  // The front window is the Mailbox Manager
		  //
		  MailboxManagerController *aMailboxManagerController;
		  NSString *aFolderName;
		  Folder *aFolder;
		  Store *aStore;
		  id item;

		  int row, level;
		  
		  aMailboxManagerController = [MailboxManagerController singleInstance];
		  row = [[aMailboxManagerController outlineView] selectedRow];
		  
		  if ( row < 0 )
		    {
		      NSBeep();
		      return;
		    }
		  
		  item = [[aMailboxManagerController outlineView] itemAtRow: row];
		  level = [[aMailboxManagerController outlineView] levelForItem: item];
		  
		  //
		  // We must verify that:
		  //
		  // a) we have at least one selected row
		  // b) we haven't selected our root, or a store (local or IMAP)
		  //
		  if ( [[aMailboxManagerController outlineView] numberOfSelectedRows] != 1 ||
		       level < 2 )
		    {
		      NSRunInformationalAlertPanel(_(@"Mailbox error!"),
						   _(@"You must select a valid mailbox to open!"),
						   _(@"OK"),
						   NULL, 
						   NULL,
						   NULL);
		      continue;
		    }
		  
		  aFolderName = [Utilities pathOfFolderFromFolderNode: (FolderNode *)item
					   separator: @"/"];
		  
		  // We get our store
		  if ( [[Utilities completePathForFolderNode: (FolderNode *)item separator: @"/"]
			 hasPrefix: [NSString stringWithFormat: @"/%@", _(@"Local")]] )
		    {
		      aStore = [aMailboxManagerController storeForName: @"GNUMAIL_LOCAL_STORE"
							  username: NSUserName()];
		    }
		  else
		    {
		      NSString *aServerName, *aUsername;
		      
		      [Utilities storeKeyForFolderNode: (FolderNode *)item
				 serverName: &aServerName
				 username: &aUsername];
		      
		      aStore = [aMailboxManagerController storeForName: aServerName
							  username: aUsername];
		    }
		  
		  // We get a reference to our destination folder, w/o parsing it if it's not already open.
		  // or w/o selecting it if it's an IMAP store.
		  if ( [(id<NSObject>)aStore isKindOfClass: [IMAPStore class]] )
		    {
		      aFolder = (Folder *)[(IMAPStore *)aStore folderForName: aFolderName
							select: NO];
		    }
		  else
		    {
		      aFolder = (Folder *)[(LocalStore *)aStore folderForName: aFolderName];
		    }
		  
		  for (j = 0; j < count; j++)
		    {
		      // We retrieve the message from the property list
		      NSDictionary *aDictionary;
		      Flags *flags;
		      NSData *rawSource;
		      
		      aDictionary = (NSDictionary*)[propertyList objectAtIndex: j];
		      
		      rawSource = [aDictionary objectForKey: @"message"];
		      flags = (Flags*)[NSUnarchiver unarchiveObjectWithData: (NSData*)[aDictionary objectForKey: 
												     @"flags"]];
		      
		      if (rawSource && flags)
			{
			  // We add the message to the folder
			  if ( ![[MailboxManagerController singleInstance] transferMessageFromRawSource: rawSource
									   flags: flags
									   toFolderWithName: aFolderName
									   orToFolder: aFolder
									   store: aStore] )
			    {
			      // The transfer failed; we exit the loop
			      break;
			    }
			}
		    }
		  
		  // If the folder wasn't open, we just close it.
		  if ( aFolder )
		    {
		      [aFolder close];
		    } 
		}
	      else if ( [[[NSApp keyWindow] delegate] isKindOfClass: [MailWindowController class]] &&
			[GNUMail lastMailWindowOnTop] )
		{
		  //
		  // The front window is a Mail Window
		  //
		  MailWindowController *aMailWindowController;
		  
		  aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
		  
		  for (j = count - 1; j >= 0; j--)
		    {
		      // We retrieve the message from the property list
		      NSDictionary *aDictionary;
		      Flags *flags;
		      NSData *rawSource;
		      
		      aDictionary = (NSDictionary*)[propertyList objectAtIndex: j];
		      
		      rawSource = [aDictionary objectForKey: @"message"];
		      
		      flags = (Flags*)[NSUnarchiver unarchiveObjectWithData: (NSData*)[aDictionary objectForKey:
												     @"flags"]];
		      if (rawSource && flags)
			{
			  // We add the message to the folder
			  if ( ![[MailboxManagerController singleInstance] transferMessageFromRawSource: rawSource
									   flags: flags
									   toFolderWithName: [[aMailWindowController folder] 
											       name]
									   orToFolder: [aMailWindowController folder]
									   store: [[aMailWindowController folder] store]] )
			    {
			      // The transfer failed; we exit the loop
			      break;
			    }
			}
		    }
		  
		  if (count > 0)
		    {
		      // We refresh the table view
		      [aMailWindowController dataViewShouldReloadData];
		      
		      // We update the status label
		      [aMailWindowController updateStatusLabel];
		    }
		}
	      else
		{
		  NSBeep();
		  NSDebugLog(@"-paste Message to unknown destination");
		}
	    }
	}

      else if ( [AddressPboardType isEqualToString: aType] )
	{
#if 0
	  // FIXME
	  //
	  // AddressPboardType
	  //
	  NSData *data;

	  data = [pboard dataForType: AddressPboardType];
	  if ( [data length] )
	    {
	      Address *anAddress;
	      
	      anAddress = (Address*)[NSUnarchiver unarchiveObjectWithData: data];
	      NSDebugLog(@"paste: %@", [anAddress name]);
	      
	      if ( [NSApp keyWindow] == [[AddressBookController singleInstance] window] )
		{
		  //
		  // The front window is the Address Book Manager
		  //
		  AddressBookController *addressBookController;
		  AddressBook *addressBook;
		  NSBrowser *browser;
	  
		  addressBookController = [AddressBookController singleInstance];
		  addressBook = [AddressBook singleInstance];
		  browser = [addressBookController browser];
		  
		  if ( [browser selectedCellInColumn: 0] )
		    {
		      Group *aGroup;

		      aGroup = [addressBook groupAtIndex: [browser selectedRowInColumn: 0]];
		      
		      [aGroup addAddress: anAddress];
		      
		      [addressBook synchronize];
		      
		      [browser reloadColumn: 1];
		    }
		}
	      else if ( [NSApp keyWindow] )
		{
		  [[NSApp keyWindow] insertText: [anAddress name]];
		}
	      else
		{
		  NSBeep();
		  NSDebugLog(@"-paste Address to unknown destination");
		}
	    }
#endif
	}
    }
}


//
//
//
- (IBAction) previousUnreadMessage: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] &&
       [[[GNUMail lastMailWindowOnTop] delegate] isKindOfClass: [MailWindowController class]] )
    {
      [[[GNUMail lastMailWindowOnTop] delegate] previousUnreadMessage: sender];
    }
  else
    {
      NSBeep();
    }
}


//
// This method prints the current selected message - ie., the one
// shown in the MailWindow's (or MessageViewWindow) text view.
//
- (IBAction) printMessage: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] )
    {
      NSPrintInfo *aPrintInfo;
      id aWindowController;

      aWindowController = [[GNUMail lastMailWindowOnTop] delegate];
      aPrintInfo = [NSPrintInfo sharedPrintInfo];
      [aPrintInfo setHorizontalPagination: NSFitPagination];
      
      [[NSPrintOperation printOperationWithView: [aWindowController textView]  printInfo: aPrintInfo] runOperation];
    }
  else
    {
      NSBeep();
    }
}


//
// This method is used to reply to the selected message
// to all recipients.
//
- (IBAction) replyAllMessage: (id) sender
{  
  if ( [GNUMail lastMailWindowOnTop] )
    {
      id aWindowController;

      aWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      [aWindowController replyMessage: self];
    }
  else
    {
      NSBeep();
    }
}


//
// This method is used to reply to the selected message.
//
- (IBAction) replyMessage: (id) sender
{  
  if ( [GNUMail lastMailWindowOnTop] )
    {
      id aWindowController;

      aWindowController = [[GNUMail lastMailWindowOnTop] delegate];
      [aWindowController replyMessage: nil];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) restoreDraft: (id) sender
{
  Message *aMessage;
    
  aMessage = [[MailboxManagerController singleInstance] messageFromDraftsFolder];

  // We create our controller and we show the window if we have a valid message
  if ( aMessage )
    {
      EditWindowController *editWindowController;

      // Initialize the the message if it has not been already
      if ( ![aMessage isInitialized] )
	{
	  [aMessage setInitialized: YES];
	}
      
      editWindowController = [[EditWindowController alloc] initWithWindowNibName: @"EditWindow"];
      
      if ( editWindowController )
	{
	  InternetAddress *aInternetAddress;
	  NSEnumerator *enumerator;
	  NSArray *recipients;

	  [editWindowController setAccountName: [Utilities _guessAccountNameFromMessage: aMessage]];
	  [editWindowController setMessageFromDraftsFolder: aMessage];
	  [editWindowController setShowCc: NO];
	  if ([[aMessage subject] length] != 0)
	    {
	      [[editWindowController window] setTitle: [aMessage subject]];
	    }
	  else
	    {
	      [[editWindowController window] setTitle: _(@"New message...")];
	    }

	  // We verify if we need to show the Cc field
	  recipients = [aMessage recipients];
	  enumerator = [recipients objectEnumerator];

          while ( (aInternetAddress = [enumerator nextObject]) )
	    {
              if ( [aInternetAddress type] == CC )
		{
		  if ( ![editWindowController showCc] )
		    {
		      [editWindowController setShowCc: YES];
		    }
		}
	      else if ( [aInternetAddress type] == BCC )
		{
		  if ( ![editWindowController showBcc] )
		    { 
		      [editWindowController setShowBcc: YES];
		    }
		}
	    }

	  // Tell the controller that it is editing a draft message
	  [editWindowController setEditingDraftMessage: YES];

	  [editWindowController showWindow: self];
	}
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) saveInDrafts: (id) sender
{
  if ( [GNUMail lastAddressTakerWindowOnTop] )
    {
      [[MailboxManagerController singleInstance] saveMessageInDraftsFolderForController: [GNUMail lastAddressTakerWindowOnTop]];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) saveAttachment: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] )
    {
      NSTextAttachment *aTextAttachment;

      aTextAttachment = [sender textAttachment];
      
      [Utilities clickedOnCell: [aTextAttachment attachmentCell]
		 inRect: NSZeroRect
		 atIndex: 0
		 sender: self];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) saveAllAttachments: (id) sender
{
  NSTextAttachment *aTextAttachment;
  NSFileWrapper *aFilewrapper;
  NSSavePanel *aSavePanel;
  
  BOOL useSameDir, ask;
  int i;

  aSavePanel = [NSSavePanel savePanel];
  [aSavePanel setAccessoryView: nil];
  [aSavePanel setRequiredFileType: @""];

  useSameDir = NO;
  ask = YES;

  for ( i = 1; i < [[(GNUMail *)[NSApp delegate] saveAttachmentMenu] numberOfItems]; i++ )
    {
      aTextAttachment = [(ExtendedMenuItem *)[[(GNUMail *)[NSApp delegate] saveAttachmentMenu] itemAtIndex: i] 
					     textAttachment];
      aFilewrapper = [aTextAttachment fileWrapper];
      
      if ( !useSameDir )
        {
          int aChoice;

          aChoice = [aSavePanel runModalForDirectory: [GNUMail currentWorkingPath]
                                file: [aFilewrapper preferredFilename]];
	  
          // if successful, save file under designated name
          if ( aChoice == NSOKButton )
            {
              if ( ![aFilewrapper writeToFile: [aSavePanel filename]
				  atomically: YES
				  updateFilenames: YES] )
                {
                  NSBeep();
                }
	      
              [GNUMail setCurrentWorkingPath: [[aSavePanel filename] stringByDeletingLastPathComponent]];
            }
        }
      else
        {
          // We save the file in the same directory
          if ( ![aFilewrapper writeToFile: [[GNUMail currentWorkingPath] stringByAppendingPathComponent:
									   [aFilewrapper preferredFilename]]
			      atomically: YES
			      updateFilenames: YES] )
            {
              NSBeep();
            }
        }

      if ( ask )
        {
          int use = NSRunAlertPanel(_(@"Information"),
                                    _(@"Use the same directory (%@) to save all other attachments? (override the files with the same name)."),
                                    _(@"Yes"),
                                    _(@"No"),
                                    NULL,
                                    [GNUMail currentWorkingPath]);
          
	  if ( use == NSAlertDefaultReturn )
            {
              useSameDir = YES;
            }
          else if ( use == NSAlertAlternateReturn )
            {
              useSameDir = NO;
            }
	  
          ask = NO;
        }
    }
}



//
// This method is used to save only the textual content of a message
// to the local file system. It skips attachments.
//
- (IBAction) saveTextFromMessage: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] )
    {
      id aWindowController;
      NSSavePanel *aSavePanel;
      NSData *aData;
      int aChoice;
#ifdef MACOSX
      NSWindow *aWindow;
#endif
      
      // We get a reference to our mail window controller
      aWindowController = [[GNUMail lastMailWindowOnTop] delegate];
      
      if ( [aWindowController selectedMessage] )
	{
	  // We get our content of a message (just the text displayed in mail window)
	  aData = [[[aWindowController textView] string] 
		    dataUsingEncoding: NSASCIIStringEncoding
		    allowLossyConversion: YES];
	  
	  aSavePanel = [NSSavePanel savePanel];
	  [aSavePanel setAccessoryView: nil];
	  [aSavePanel setRequiredFileType: @""];
#ifdef MACOSX
	  RETAIN(aData);
	  if ( [sender respondsToSelector:@selector(window)] )
	    {
	      aWindow = [sender window];
	    }
	  else
	    {
	      aWindow = [GNUMail lastMailWindowOnTop];
	    }
	  [aSavePanel beginSheetForDirectory: [GNUMail currentWorkingPath]
		      file: [[aWindowController selectedMessage] subject]
		      modalForWindow: aWindow
		      modalDelegate: self
		      didEndSelector: @selector(_savePanelDidEnd: returnCode: contextInfo:)
		      contextInfo: aData];
#else
	  aChoice = [aSavePanel runModalForDirectory: [GNUMail currentWorkingPath] file: @"Unknown"];
	  
	  // if successful, save file under designated name
	  if (aChoice == NSOKButton)
	    {
	      if ( ![aData writeToFile: [aSavePanel filename]
			   atomically: YES] )
		{
		  NSBeep();
		}

              [GNUMail setCurrentWorkingPath: [[aSavePanel filename] stringByDeletingLastPathComponent]];
        }
#endif
	}
      else
	{
	  NSBeep();
	}
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) showAboutPanel: (id) sender
{
  AboutPanelController *aController;
  
  aController = [AboutPanelController singleInstance];
  [aController showWindow: self];
}


//
//
//
- (IBAction) sortByNumber: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] &&
       [[[GNUMail lastMailWindowOnTop] delegate] isKindOfClass: [MailWindowController class]] ) 
    {
      MailWindowController *aMailWindowController;
      
      aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      [aMailWindowController
	tableView: [aMailWindowController dataView]
	didClickTableColumn: [[aMailWindowController dataView]
			       tableColumnWithIdentifier: @"#"]];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) sortByDate: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] &&
       [[[GNUMail lastMailWindowOnTop] delegate] isKindOfClass: [MailWindowController class]] ) 
    {
      MailWindowController *aMailWindowController;
      
      aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      [aMailWindowController
	tableView: [aMailWindowController dataView]
	didClickTableColumn: [[aMailWindowController dataView]
			       tableColumnWithIdentifier: @"Date"]];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) sortByName: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] &&
       [[[GNUMail lastMailWindowOnTop] delegate] isKindOfClass: [MailWindowController class]] ) 
    {
      MailWindowController *aMailWindowController;
      
      aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      [aMailWindowController
	tableView: [aMailWindowController dataView]
	didClickTableColumn: [[aMailWindowController dataView]
			       tableColumnWithIdentifier: @"From"]];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) sortBySubject: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] &&
       [[[GNUMail lastMailWindowOnTop] delegate] isKindOfClass: [MailWindowController class]] ) 
    {
      MailWindowController *aMailWindowController;
      
      aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      [aMailWindowController
	tableView: [aMailWindowController dataView]
	didClickTableColumn: [[aMailWindowController dataView]
			       tableColumnWithIdentifier: @"Subject"]];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) sortBySize: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] &&
       [[[GNUMail lastMailWindowOnTop] delegate] isKindOfClass: [MailWindowController class]] ) 
    {
      MailWindowController *aMailWindowController;
      
      aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      [aMailWindowController
	tableView: [aMailWindowController dataView]
	didClickTableColumn: [[aMailWindowController dataView]
			       tableColumnWithIdentifier: @"Size"]];
    }
  else
    {
      NSBeep();
    }
}



//
// This method is used to show the address book on the screen.
// We use a singleton.
//
- (IBAction) showAddressBook: (id) sender
{
  [[AddressBookController singleInstance] showWindow: self];
}


//
// This method is used to view or hide all headers in the 
// last mail window on top.
//
- (IBAction) showAllHeaders: (id) sender
{ 
  if ( [GNUMail lastMailWindowOnTop] )
    {
      id aWindowController;
      Message *aMessage;
      
      BOOL aBOOL;
      int row;
      
      aWindowController = [[GNUMail lastMailWindowOnTop] delegate];
      row = -1;

      if ( [aWindowController isKindOfClass: [MailWindowController class]] &&
	   (row = [[aWindowController dataView] selectedRow]) &&
	   (row == -1 || [[aWindowController dataView] numberOfSelectedRows] > 1) )
	{
	  NSBeep();
	  return;
	}
      
      if ( [sender tag] == SHOW_ALL_HEADERS )
	{
	  aBOOL = YES;
	  [aWindowController setShowAllHeaders: aBOOL];
          
          if ( [sender isKindOfClass: [NSButton class]] ||
               [sender isKindOfClass: [NSMenuItem class]] )
            {
	      [sender setTitle: _(@"Filtered Headers")];
            }
          else
            {
              [sender setLabel: _(@"Filtered Headers")];
            }
        
	  [sender setTag: HIDE_ALL_HEADERS];
	}
      else
	{
	  aBOOL = NO;
	  [aWindowController setShowAllHeaders: aBOOL];	
          
          if ( [sender isKindOfClass: [NSButton class]] ||
               [sender isKindOfClass: [NSMenuItem class]] )
            {
	      [sender setTitle: _(@"All Headers")];
            }
          else
            {
              [sender setLabel: _(@"All Headers")];
            }

	  [sender setTag: SHOW_ALL_HEADERS];
	}
      
      [menu sizeToFit];

      if ( [aWindowController isKindOfClass: [MailWindowController class]] )
	{
	  aMessage = [aWindowController selectedMessage];
	}
      else
	{
	  aMessage = [aWindowController message];
	}
      
      [Utilities showMessage: aMessage
		 target: [aWindowController textView]
		 showAllHeaders: aBOOL];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) showConsoleWindow: (id) sender
{
  [[ConsoleWindowController singleInstance] showWindow: self];
}


//
// This method is used to show the find window on the screen.
// We use a singleton.
//
- (IBAction) showFindWindow: (id) sender
{
  [[FindWindowController singleInstance] showWindow: self];
}


//
// This method is used to show the mailbox manager window on the screen.
// We use a singleton.
//
- (IBAction) showMailboxManager: (id) sender
{
#ifdef MACOSX
  if ( [GNUMail lastMailWindowOnTop] &&
       [[[GNUMail lastMailWindowOnTop] delegate] isKindOfClass: [MailWindowController class]] )
    {
      [[[GNUMail lastMailWindowOnTop] delegate] openOrCloseDrawer: self];
    }
#else
  if ( [[[MailboxManagerController singleInstance] window] isVisible] )
    {
      [[[MailboxManagerController singleInstance] window] orderOut: self];
    }
  else
    {
      [[MailboxManagerController singleInstance] showWindow: self];
    }
#endif
}


//
// 
//
- (IBAction) showOrHideDeletedMessages: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] )
    {
      MailWindowController *aMailWindowController;
      
      aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];

      if ( [showOrHideDeleted tag] == SHOW_DELETED_MESSAGES )
	{
	  [self updateShowOrHideDeletedMenuItem: YES];
	  [[aMailWindowController folder] setShowDeleted: YES];
	}
      else
	{
	  [self updateShowOrHideDeletedMenuItem: NO];
	  [[aMailWindowController folder] setShowDeleted: NO];
	}
      
      [aMailWindowController dataViewShouldReloadData];
      [aMailWindowController updateStatusLabel];
    }
  else
    {
      NSBeep();
    }
}


//
// 
//
- (IBAction) showOrHideReadMessages: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] )
    {
      MailWindowController *aMailWindowController;
      
      aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];

      if ( [showOrHideRead tag] == SHOW_READ_MESSAGES )
	{
	  [self updateShowOrHideReadMenuItem: YES];
	  [[aMailWindowController folder] setShowRead: YES];
	}
      else
	{
	  [self updateShowOrHideReadMenuItem: NO];
	  [[aMailWindowController folder] setShowRead: NO];
	}
      
      [aMailWindowController dataViewShouldReloadData];
      [aMailWindowController updateStatusLabel];
    }
  else
    {
      NSBeep();
    }
}


//
// This method is used to show the preferences window on the screen.
// The window is modal.
//
- (IBAction) showPreferencesWindow: (id) sender
{
  [[PreferencesWindowController singleInstance] showWindow: self];
}


//
// This method is used to show the raw source of the selected message
// from the last mail window on top.
//
- (IBAction) showRawSource: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] )
    {
      id aWindowController;
      Message *aMessage;
      
      aWindowController = [[GNUMail lastMailWindowOnTop] delegate];
      
      if ( !aWindowController )
	{
	  NSBeep();
	  return;
	}
      
      if ( [aWindowController isKindOfClass: [MailWindowController class]] )
	{
	  aMessage = [aWindowController selectedMessage];
	}
      else
	{
	  aMessage = [aWindowController message];
	}
      
      if ( aMessage )
	{
	  if ( ![aWindowController showRawSource] )
	    {
	      [aWindowController setShowRawSource: YES];
	      [showRawSource setTitle: _(@"Normal Display")];

	      if ( [sender isKindOfClass: [NSButton class]] ||
		   [sender isKindOfClass: [NSMenuItem class]] )
		{
		  [sender setTitle: _(@"Normal Display")];
		}
	      else
		{
		  [sender setLabel: _(@"Normal Display")];
		}
	      
	      [Utilities showMessageRawSource: aMessage 
			 target: [aWindowController textView]];
	    }
	  else
	    {
	      [aWindowController setShowRawSource: NO];
	      [showRawSource setTitle: _(@"Raw Source")];

	      if ( [sender isKindOfClass: [NSButton class]] ||
		   [sender isKindOfClass: [NSMenuItem class]] )
		{
		  [sender setTitle: _(@"Raw Source")];
		}
	      else
		{
		  [sender setLabel: _(@"Raw Source")];
		}

	      [Utilities showMessage: aMessage
			 target: [aWindowController textView]
			 showAllHeaders: [aWindowController showAllHeaders]];
	    }
	}
      else
	{
	  NSBeep();
	}
      
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) threadOrUnthreadMessages: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] &&
       [[[GNUMail lastMailWindowOnTop] delegate] isKindOfClass: [MailWindowController class]] )
    {
      MailWindowController *aMailWindowController;

      aMailWindowController = [[GNUMail lastMailWindowOnTop] delegate];

      if ( [aMailWindowController dataViewType] == TYPE_OUTLINEVIEW)
	{
	  [self updateThreadOrUnthreadMenuItem: YES];
	  [aMailWindowController setDataViewType: TYPE_TABLEVIEW];
	  [aMailWindowController dataViewShouldReloadData];
	}
      else
	{
	  [self updateThreadOrUnthreadMenuItem: NO];
	  [[aMailWindowController folder] thread];
	  [aMailWindowController setDataViewType: TYPE_OUTLINEVIEW];
	  [aMailWindowController dataViewShouldReloadData];
	}
    }
  else
    {
      NSBeep();
    }
}


//
// delegate methods
//
- (BOOL) applicationShouldTerminate: (NSApplication *) theSender
{
  NSMutableArray *foldersToOpen;
  NSEnumerator *theEnumerator;
  id aStore, aWindow;

  int choice, i;
  
  if ( [tasks count] > 0 )
    {
      choice = NSRunAlertPanel(_(@"Error!"),
			       _(@"Not all tasks finished their execution. You must wait before quitting the application.\nOpen the Console (Windows -> Console) if you want to manage those tasks."),
			       _(@"OK"),          // default
			       _(@"Quit Anyway"), // alternate
			       NULL);
      
      if ( choice == NSAlertDefaultReturn)
	{
	  return NO;
	}
    }

  
  // If the user has left any "edited" EditWindow:s open, we warn him/her about this
  if ( [allEditWindows count] > 0 )
    {
      for (i = 0; i < [allEditWindows count]; i++)
	{
	  if ( [[allEditWindows objectAtIndex: i] isDocumentEdited] )
	    {
	      choice = NSRunAlertPanel(_(@"Quit"),
				       _(@"There are unsent Compose window."),
				       _(@"Review Unsent"), // default
				       _(@"Quit Anyway"),   // alternate
				       _(@"Cancel"),        // other return
				       NULL);
	      
	      // We want to review unsent Compose windows
	      if ( choice == NSAlertDefaultReturn )
		{
		  [[allEditWindows objectAtIndex: i] makeKeyAndOrderFront: self];
		  return NO;
		}
	      // We want to quit
	      else if ( choice == NSAlertAlternateReturn )
		{
		  break;
		}
	      // We want to cancel the quit operation
	      else
		{
		  return NO;
		}
	    }
	}
    }
  
  // We first remove all observers
  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: SelectionOfMessageHasChanged
    object: nil];
  
  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: FiltersHaveChanged
    object: nil];
  
  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: AccountsHaveChanged
    object: nil];
    
  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: NSTextViewDidChangeSelectionNotification
    object: nil];
  
  // We release our array containing all our EditWindow:s
  DESTROY(allEditWindows);
  
  // We stop our timer
  [self stopTimer];

  // We closes all open MailWindow:s. Before doing so, we
  // save them in the FOLDERS_TO_OPEN default variable in order
  // to reopen them upon the application's startup.
  foldersToOpen = [[NSMutableArray alloc] init];

  for (i = ([allMailWindows count]-1); i >= 0; i--)
    {
      aWindow = [allMailWindows objectAtIndex: i];

      if ( ![[aWindow delegate] folder] )
	{
	  [aWindow close];
	  continue;
	}

      if ( [[[aWindow delegate] folder] isKindOfClass: [LocalFolder class]] )
	{
	  [foldersToOpen addObject: [NSString stringWithFormat: @"local://%@/%@",
					      [[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"],
					      [[[aWindow delegate] folder] name]]];
	}
      else
	{
	  [foldersToOpen addObject: [NSString stringWithFormat: @"imap://%@@%@/%@",
					      [(IMAPStore *)[[[aWindow delegate] folder] store] username],
					      [(IMAPStore *)[[[aWindow delegate] folder] store] name],
					      [[[aWindow delegate] folder] name]]];
											     
	}

      [aWindow close];
    }


  // We save our FOLDERS_TO_OPEN value in the user's defaults
  [[NSUserDefaults standardUserDefaults] setObject: foldersToOpen
					 forKey: @"FOLDERS_TO_OPEN"];
  [[NSUserDefaults standardUserDefaults] synchronize];


  // We wait until all our windows are closed
  while ( [allMailWindows count] > 0 )
    {
      [[NSRunLoop currentRunLoop] runMode: NSDefaultRunLoopMode
				  beforeDate: [NSDate distantFuture]];
    }

  // We release our array containing all our MailWindow:s
  DESTROY(allMailWindows);

  // We close all remaining open Store:s
  theEnumerator = [[[MailboxManagerController singleInstance] allStores] objectEnumerator];

  while ( (aStore = [theEnumerator nextObject]) )
    {
      NS_DURING
	[aStore close];
      NS_HANDLER
	// Do nothing
      NS_ENDHANDLER
    }

  // We synchronize our MailboxManagerCache and we close all Stores
  [[[MailboxManagerController singleInstance] mailboxManagerCache] synchronize];

  // Under GNUstep, we also close the window before releasing the singleton
#ifndef MACOSX
  [[[MailboxManagerController singleInstance] window] close];
#endif

  RELEASE([MailboxManagerController singleInstance]);

  // We close our Console
  [[[ConsoleWindowController singleInstance] window] close];
  RELEASE([ConsoleWindowController singleInstance]);

  // We release our current working path
  TEST_RELEASE(currentWorkingPath);

  // We release our array containing all our bundles
  RELEASE(allBundles);

  // We release our array containing our password cache
  RELEASE([Utilities passwordCache]);

  // We release our MimeType, our Filters and our AddressBook
  RELEASE([MimeTypeManager singleInstance]);
  RELEASE([FilterManager singleInstance]);
  RELEASE([AddressBookController singleInstance]);

  // We stop our task manager
  [self stopTaskManager];

  // We finally remove all temporary files and then, we are ready to really close the application.
  [self removeTemporaryFiles];

  return YES;
}


//
//
//
- (void) applicationWillFinishLaunching: (NSNotification *) theNotification
{
#ifdef MACOSX
  unichar backspaceKey;

  backspaceKey = NSBackspaceCharacter;
  [deleteOrUndelete setKeyEquivalent: [NSString stringWithCharacters: &backspaceKey  length: 1]];
  
#else
  SEL action = NULL;
  unichar del, up, down;

  del = NSDeleteCharacter;       // FIXME: We might want to display, instead:
  up = NSUpArrowFunctionKey;     // 8593 (0x2191) U+2191 arrowup UPWARDS ARROW
  down = NSDownArrowFunctionKey; // 8595 (0x2193) U+2193 arrowup DOWNWARDS ARROW

  // We begin by setting our NSApp's logo
  [NSApp setApplicationIconImage: [NSImage imageNamed: @"GNUMail.tiff"]];

  // We continue by creating our NSMenu
  menu = [[NSMenu alloc] init];
  
  [menu addItemWithTitle: _(@"Info") 
	action: action 
	keyEquivalent: @""];
  [menu addItemWithTitle: _(@"Message") 
	action: action 
	keyEquivalent: @""];
  [menu addItemWithTitle: _(@"Mailbox") 
	action: action 
	keyEquivalent: @""];
  [menu addItemWithTitle: _(@"Edit")
	action: action
	keyEquivalent: @""];
  [menu addItemWithTitle: _(@"Find")
	action: action
	keyEquivalent: @""];
  [menu addItemWithTitle: _(@"View")
	action: action
	keyEquivalent: @""];
  [menu addItemWithTitle: _(@"Windows")
	action: action
	keyEquivalent: @""];

  // We verify if scrippting is supported by the application.
  if ( [NSApp isScriptingSupported] )
    {
      [menu addItemWithTitle: _(@"Scripting")
	    action: action
	    keyEquivalent: @""];
    }
  
  [menu addItemWithTitle: _(@"Services")
	action: action
	keyEquivalent: @""];
  [menu addItemWithTitle: _(@"Hide")
	action: @selector (hide:)
	keyEquivalent: @"h"];
  [menu addItemWithTitle: _(@"Quit")
	action: @selector(terminate:)
	keyEquivalent: @"q"];
  
  //
  // Info menu / submenus
  //
  info = [[NSMenu alloc] init];
  [menu setSubmenu: info  forItem: [menu itemWithTitle: _(@"Info")]];
  [info addItemWithTitle: _(@"About GNUMail.app...")
	action: @selector(showAboutPanel:)   
	keyEquivalent: @""];
  [info addItemWithTitle: _(@"Check for Updates...")
	action: @selector(checkForUpdates:)   
	keyEquivalent: @""];
  [info addItemWithTitle: _(@"Preferences...")
	action: @selector(showPreferencesWindow:)
	keyEquivalent: @","];
  [info addItemWithTitle: _(@"Help...")
	action: action
	keyEquivalent: @"?"];
  RELEASE(info);
  

  //
  // Our Message menu / submenus
  //
  message = [[NSMenu alloc] init];
  [menu setSubmenu: message  forItem: [menu itemWithTitle: _(@"Message")]];

  [message addItemWithTitle: _(@"Compose")
	   action: @selector(composeMessage:) 
	   keyEquivalent: @"n"];
  [message addItemWithTitle: _(@"Reply")
	   action: @selector(replyMessage:)  
	   keyEquivalent: @"R"];
  [message addItemWithTitle: _(@"Reply All")
	   action: @selector(replyAllMessage:)   
	   keyEquivalent: @"E"];
  [message addItemWithTitle: _(@"Forward")
	   action: @selector(forwardMessage:)  
	   keyEquivalent: @"W"];
  [message addItemWithTitle: _(@"Bounce")
	   action: @selector(bounceMessage:)  
	   keyEquivalent: @""];
  
  deliver = [[NSMenuItem alloc] init];
  [deliver setTitle: _(@"Deliver")];
  [deliver setAction: NULL];
  [deliver setKeyEquivalent: @"D"];
  [message addItem: deliver];
  RELEASE(deliver);

  markAsReadOrUnread = [[NSMenuItem alloc] init];
  [markAsReadOrUnread setTitle: _(@"Mark as Read")];
  [markAsReadOrUnread setAction: @selector(markMessageAsReadOrUnread:)];
  [markAsReadOrUnread setKeyEquivalent: @"U"];
  [message addItem: markAsReadOrUnread];
  RELEASE(markAsReadOrUnread);

  deleteOrUndelete = [[NSMenuItem alloc] init];
  [deleteOrUndelete setTitle: _(@"Undelete")];
  [deleteOrUndelete setAction: @selector(deleteOrUndeleteMessage:)];
  [deleteOrUndelete setKeyEquivalent: [NSString stringWithCharacters: &del  length: 1]];
  [message addItem: deleteOrUndelete];
  RELEASE(deleteOrUndelete);

  [message addItemWithTitle: _(@"Save Attachment")
	   action: action
	   keyEquivalent: @""];
  
  saveAttachment = [[NSMenu alloc] init];
  [message setSubmenu: saveAttachment  forItem: [message itemWithTitle: _(@"Save Attachment")]];
  RELEASE(saveAttachment);
  
  [message addItemWithTitle: _(@"Save Text from Message")
	   action: @selector(saveTextFromMessage:)  
	   keyEquivalent: @""];

  [message addItemWithTitle: _(@"Drafts")
	   action: action
	   keyEquivalent: @""];
  
  drafts = [[NSMenu alloc] init];
  
  saveInDrafts = [[NSMenuItem alloc] initWithTitle: _(@"Save in Drafts")
				     action: NULL
				     keyEquivalent: @"s"];
  [drafts addItem: saveInDrafts];
  RELEASE(saveInDrafts);
  
  [drafts addItemWithTitle: _(@"Restore Draft")
	  action: @selector(restoreDraft:)
	  keyEquivalent: @""];
  [[message itemWithTitle: _(@"Drafts")] setSubmenu: drafts];
  RELEASE(drafts);

  [message addItemWithTitle: _(@"Text Encodings")
	   action: action
	   keyEquivalent: @""];

  textEncodings = [[NSMenu alloc] init];
  [message setSubmenu: textEncodings  forItem: [message itemWithTitle: _(@"Text Encodings")]];
  RELEASE(textEncodings);

  [message addItemWithTitle: _(@"Apply Manual Filters")
	   action: action
	   keyEquivalent: @""];
  filters = [[NSMenu alloc] init];
  [[message itemWithTitle: _(@"Apply Manual Filters")] setSubmenu: filters];
  RELEASE(filters);

  [message addItemWithTitle: _(@"Print...")
	   action: @selector(printMessage:)
	   keyEquivalent: @"p"];

  // FIXME
  //[message addItemWithTitle: _(@"Send to Address Book")
  //	   action: action
  //	   keyEquivalent: @""];
  //
  // FIXME
  sendToAddressBook = [[NSMenu alloc] init];
  //[message setSubmenu: sendToAddressBook  forItem: [message itemWithTitle: _(@"Send to Address Book")]];
  //RELEASE(sendToAddressBook);
  RELEASE(message);


  //
  // Mailbox menu / submenus
  //
  mailbox = [[NSMenu alloc] init];
  [menu setSubmenu: mailbox  forItem: [menu itemWithTitle: _(@"Mailbox")]];
  [mailbox addItemWithTitle: _(@"Mailboxes...")
	   action: @selector(showMailboxManager:) 
	   keyEquivalent: @"M"];
  
  incomingMailServers = [[NSMenu alloc] init];
  [mailbox addItemWithTitle: _(@"Get New Mail")
	   action: action
	   keyEquivalent: @""];
  [[mailbox itemAtIndex: 1] setSubmenu: incomingMailServers];
  RELEASE(incomingMailServers);

  openInNewWindow = [[NSMenuItem alloc] init];
  [openInNewWindow setTitle: _(@"Open in New Window")];
  [openInNewWindow setAction: @selector(openInNewWindow:)];
  [openInNewWindow setKeyEquivalent: @""];
  [mailbox addItem: openInNewWindow];
  RELEASE(openInNewWindow);

  [mailbox addItemWithTitle: _(@"Import Mailboxes...")
	   action: @selector(importMailboxes:)
	   keyEquivalent: @""];

  [mailbox addItemWithTitle: _(@"Compact...")
	   action: @selector(compactMailbox:)
	   keyEquivalent: @"K"];
  RELEASE(mailbox);
  
  
  //
  // Edit menu / submenus
  //
  edit = [[NSMenu alloc] init];
  [menu setSubmenu: edit  forItem: [menu itemWithTitle: _(@"Edit")]];

  [edit addItemWithTitle: _(@"Cut")
	action: @selector(cut:)
	keyEquivalent: @"x"];
  [edit addItemWithTitle: _(@"Copy")
	action: @selector(copy:)
	keyEquivalent: @"c"];
  [edit addItemWithTitle: _(@"Paste")
	action: @selector(paste:)
	keyEquivalent: @"v"];
  [edit addItemWithTitle: _(@"Paste As Quoted Text")
	action: @selector(pasteAsQuoted:)
	keyEquivalent: @"V"];
  [edit addItemWithTitle: _(@"Undo")
	action: action
	keyEquivalent: @"z"];

  [edit addItemWithTitle: _(@"Spelling...")
	action: @selector(checkSpelling:)
	keyEquivalent: @":"];

  [edit addItemWithTitle: _(@"Check Spelling")
	action: @selector(showGuessPanel:)
	keyEquivalent: @";"];

  [edit addItemWithTitle: _(@"Select All")
	action: @selector(selectAll:)
	keyEquivalent: @"a"];
  RELEASE(edit);


  //
  // Find menu / submenus
  //
  find = [[NSMenu alloc] init];
  [menu setSubmenu: find  forItem: [menu itemWithTitle: _(@"Find")]];

  [find addItemWithTitle: _(@"Find Messages...")
	action: @selector(showFindWindow:)
	keyEquivalent: @"F"];
  [find addItemWithTitle: _(@"Find Text...")
	action: action
	keyEquivalent: @"f"];
  [find addItemWithTitle: _(@"Find Next")
	action: @selector(findNext:)
	keyEquivalent: @"g"];
  [find addItemWithTitle: _(@"Find Previous")
	action: @selector(findPrevious:)
	keyEquivalent: @"d"];

  enterSelection = [[NSMenuItem alloc] initWithTitle: _(@"Enter Selection")
				       action: action
				       keyEquivalent: @"e"];
  [find addItem: enterSelection];
  RELEASE(enterSelection);
  RELEASE(find);

  
  //
  // Our View menu / submenus
  //
  view =  [[NSMenu alloc] init];
  [menu setSubmenu: view  forItem: [menu itemWithTitle: _(@"View")]];
    
  threadOrUnthreadMessages = [[NSMenuItem alloc] init];
  [threadOrUnthreadMessages setTitle: _(@"Thread Messages")];
  [threadOrUnthreadMessages setAction: @selector(threadOrUnthreadMessages:)];
  [threadOrUnthreadMessages setKeyEquivalent: @""];
  [threadOrUnthreadMessages setTag: THREAD_MESSAGES];
  [view addItem: threadOrUnthreadMessages];
  RELEASE(threadOrUnthreadMessages);

  [view addItemWithTitle: _(@"Sorting")
	action: action 
	keyEquivalent: @""];

  sorting = [[NSMenu alloc] init];
  [[view itemAtIndex: 1] setSubmenu: sorting];

  [sorting addItemWithTitle: _(@"Sort by Date")
	   action: @selector(sortByDate:) 
	   keyEquivalent: @""];
  [sorting addItemWithTitle: _(@"Sort by Name")
	   action: @selector(sortByName:)
	   keyEquivalent: @"S"];
  [sorting addItemWithTitle: _(@"Sort by Number")
	   action: @selector(sortByNumber:)
	   keyEquivalent: @""];
  [sorting addItemWithTitle: _(@"Sort by Size")
	   action: @selector(sortBySize:)
	   keyEquivalent: @""];
  [sorting addItemWithTitle: _(@"Sort by Subject")
	   action: @selector(sortBySubject:)
	   keyEquivalent: @""];
  RELEASE(sorting);

  showOrHideDeleted = [[NSMenuItem alloc] init];
  [showOrHideDeleted setTitle: _(@"Hide Deleted")];
  [showOrHideDeleted setAction: @selector(showOrHideDeletedMessages:)];
  [showOrHideDeleted setKeyEquivalent: @""];
  [showOrHideDeleted setTag: HIDE_DELETED_MESSAGES];
  [view addItem: showOrHideDeleted];
  RELEASE(showOrHideDeleted);
	  
  showOrHideRead = [[NSMenuItem alloc] init];
  [showOrHideRead setTitle: _(@"Hide Read")];
  [showOrHideRead setAction: @selector(showOrHideReadMessages:)];
  [showOrHideRead setKeyEquivalent: @""];
  [showOrHideRead setTag: HIDE_READ_MESSAGES];
  [view addItem: showOrHideRead];
  RELEASE(showOrHideRead);
  
  showAllHeaders = [[NSMenuItem alloc] init];
  [showAllHeaders setTitle: _(@"All Headers")];
  [showAllHeaders setAction: @selector(showAllHeaders:)];
  [showAllHeaders setKeyEquivalent: @""];
  [showAllHeaders setTag: SHOW_ALL_HEADERS];
  [view addItem: showAllHeaders];
  RELEASE(showAllHeaders);

  showRawSource = [[NSMenuItem alloc] init];
  [showRawSource setTitle: _(@"Raw Source")];
  [showRawSource setAction: @selector(showRawSource:)];
  [showRawSource setKeyEquivalent: @""];
  [view addItem: showRawSource];
  RELEASE(showRawSource);

  [view addItemWithTitle: _(@"Previous Unread")
	action: @selector(previousUnreadMessage:)
	keyEquivalent: [NSString stringWithCharacters: &up  length: 1]];
  [[view itemWithTitle: _(@"Previous Unread")] setKeyEquivalentModifierMask: NSControlKeyMask];

  [view addItemWithTitle: _(@"Next Unread")
	action: @selector(nextUnreadMessage:)
	keyEquivalent: [NSString stringWithCharacters: &down  length: 1]];
  [[view itemWithTitle: _(@"Next Unread")] setKeyEquivalentModifierMask: NSControlKeyMask];
  RELEASE(view);
  

  //
  // Windows menu
  //
  windows = [[NSMenu alloc] init];
  [menu setSubmenu: windows  forItem: [menu itemWithTitle: _(@"Windows")]];
  [windows addItemWithTitle: _(@"Address Book")
	   action: @selector(showAddressBook:)
	   keyEquivalent: @"A"];
  [windows addItemWithTitle: _(@"Console")
	   action: @selector(showConsoleWindow:)
	   keyEquivalent: @"C"];
  [windows addItemWithTitle: _(@"Arrange")
	   action: @selector(arrangeInFront:)
	   keyEquivalent: @""];
  [windows addItemWithTitle: _(@"Miniaturize")
	   action: @selector(performMiniaturize:)
	       keyEquivalent: @"m"];
  [windows addItemWithTitle: _(@"Close")
	   action: @selector(close:)
	   keyEquivalent: @"w"];
  RELEASE(windows);
  
  if ( [NSApp isScriptingSupported] )
    {
      [menu setSubmenu: [NSApp scriptingMenu]
	    forItem: [menu itemWithTitle: _(@"Scripting")]];
    }
  
  // Our Services menu
  services = [[NSMenu alloc] init];
  [menu setSubmenu: services  forItem: [menu itemWithTitle: _(@"Services")]];
  [NSApp setServicesMenu: services];
  RELEASE(services);
  
  [NSApp setMainMenu: menu];
  [NSApp setWindowsMenu: windows];

  [menu setTitle: @"GNUMail.app"];
  RELEASE(menu);
#endif
}


//
//
//
- (void) applicationDidFinishLaunching: (NSNotification *) theNotification
{
  NSString *pathToLocalMailDir;
  NSFileManager *aFileManager;
  LocalStore *aLocalStore;  
  
  BOOL isDir, mustShowPreferencesWindow;

  aFileManager = [NSFileManager defaultManager];
  mustShowPreferencesWindow = NO;

  // We now verify if the User's Library directory does exist (to store 
  // the AddressBook, the MimeTypes, etc) and if not, we create it.
  if ( [aFileManager fileExistsAtPath: (NSString *)GNUMailUserLibraryPath()
		     isDirectory: &isDir] )
    {
      if ( !isDir )
	{ 
	  NSRunCriticalAlertPanel(_(@"Fatal error!"),
				  _(@"%@ exists but it is a file not a directory.\nThe application will now terminate."),
				  @"OK",
				  NULL,
				  NULL,
				  GNUMailUserLibraryPath());
	  exit(1);
	}
    }
  else 
    {
      if (![aFileManager createDirectoryAtPath: (NSString *)GNUMailUserLibraryPath()
			 attributes: nil] )
	{
	  // Directory creation failed. We warn the user, then quit.
	  NSRunCriticalAlertPanel(_(@"Fatal error!"),
				  _(@"Could not create directory: %@\nThe application will now terminate."),
				  @"OK",
				  NULL,
				  NULL,
				  GNUMailUserLibraryPath());
	  exit(1);
	}
    }
  
  // We register for our notifications
  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(selectionOfMessageHasChanged:)
    name: SelectionOfMessageHasChanged
    object: nil];

  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(updateGetNewMailMenuItems:)
    name: AccountsHaveChanged
    object: nil];

  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(updateFilterMenuItems:)
    name: FiltersHaveChanged
    object: nil];

  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(selectionInTextViewHasChanged:)
    name: NSTextViewDidChangeSelectionNotification
    object: nil];
  
  // We initialize our mutable array containing all open EditWindow:s
  allEditWindows = [[NSMutableArray alloc] init];
  
  // We initialize our mutable array containing all open MailWindow:s
  allMailWindows = [[NSMutableArray alloc] init];
  
  // We initialize our mutable array containing all our bundles
  allBundles = [[NSMutableArray alloc] init];

  // We set the current working path of GNUMail.app to the user's home directory
  [GNUMail setCurrentWorkingPath: NSHomeDirectory()];
  
  
  // Setup our mailbox locations, if LOCALMAILDIR isn't defined, we define
  // it to ~/gnumail under GNUstep and to ~/Library/GNUMail/Mailboxes under Mac OS X
  if ( ![[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"] )
    {
#ifdef MACOSX
      pathToLocalMailDir = [NSHomeDirectory() stringByAppendingPathComponent: @"Library/GNUMail/Mailboxes"];
#else
      pathToLocalMailDir = [NSHomeDirectory() stringByAppendingPathComponent: @"Mailboxes"];
#endif
      
      [[NSUserDefaults standardUserDefaults] setObject: pathToLocalMailDir  forKey: @"LOCALMAILDIR"];
      [[NSUserDefaults standardUserDefaults] setObject: pathToLocalMailDir  forKey: @"LOCALMAILDIR_PREVIOUS"];
    }
  else
    {
      NSString *aString;

      pathToLocalMailDir = [[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"];

      // We remove the trailing slash, if any.
      if ( [pathToLocalMailDir length] > 1 && [pathToLocalMailDir hasSuffix: @"/"] )
	{
	  pathToLocalMailDir = [pathToLocalMailDir substringToIndex: ([pathToLocalMailDir length] - 1)];
	  [[NSUserDefaults standardUserDefaults] setObject: pathToLocalMailDir forKey: @"LOCALMAILDIR"];
	}

      // If LOCALMAILDIR_PREVIOUS isn't set, we set it to LOCALMAILDIR
      if ( ![[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR_PREVIOUS"] )
	{
	  [[NSUserDefaults standardUserDefaults] setObject: pathToLocalMailDir  forKey: @"LOCALMAILDIR_PREVIOUS"];
	}

      aString = [[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR_PREVIOUS"];

      if ( ![pathToLocalMailDir isEqualToString: aString] )
	{
	  [self moveLocalMailDirectoryFromPath: aString
		toPath: pathToLocalMailDir];
	}
    }


  //
  // We migrate our preferences (and other things) from previous GNUMail.app versions.
  // Currently, we can migrate from:  1.1.0  to 1.1.1.
  //                                  1.1.1  to 1.1.2.
  //
  [self update_110_to_111];
  [self update_111_to_112];


  //
  // We verify if GNUMail.app has been configured, if not, we suggest the user to do so.
  //
  if ( ![[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] )
    {
      NSString *pathToDrafts, *pathToInbox, *pathToSent, *pathToTrash;
      NSMutableDictionary *allValues, *theAccount;
      int choice;
            
      choice = NSRunAlertPanel(_(@"Warning!"),
			       _(@"No configuration found for GNUMail.app"),
			       _(@"Configure"), // default
			       _(@"Cancel"),    // alternate
			       NULL);
      
      // If the user doesn't want to configure GNUMail.app, we simply quit 
      if (choice == NSAlertAlternateReturn) 
	{
	  [NSApp terminate: self];
	}
      
      // We create the mailboxes directory for GNUMail.app, if we need to!
      if ( ![[NSFileManager defaultManager]
	      fileExistsAtPath: pathToLocalMailDir
	      isDirectory: &isDir] )
	{
	  [[NSFileManager defaultManager] createDirectoryAtPath: pathToLocalMailDir
					  attributes: nil]; 
	}
      else
	{
	  // If it's a file
	  if ( !isDir )
	    {
	      NSDebugLog(@"%@ is a file and not a directory. Please remove that file before restarting GNUMail.app.",
		    pathToLocalMailDir);
	      [NSApp terminate: self];
	    }
	}
      
      //
      // We create the following Mailboxes automatically, only if we need to:
      //
      // - Inbox  : for the newly received messages
      // - Sent   : for the messages that have been sent
      // - Trash  : for the messages we want to delete and transfer locally in IMAP
      // - Drafts : for un-sent messages
      //
      
      //
      // Inbox
      //     
      pathToInbox = [pathToLocalMailDir stringByAppendingPathComponent: @"Inbox"];
      
      if ( ![[NSFileManager defaultManager] fileExistsAtPath: pathToInbox] )
	{
	  // We first verify if we can find our "Welcome" mailbox. If so, the Inbox will be created
	  // with the content of that file.
	  NSString *pathToFile;
	  
	  pathToFile = [NSString stringWithFormat: @"%@/Welcome", [[NSBundle mainBundle] resourcePath]];
	  
	  if ( [[NSFileManager defaultManager] fileExistsAtPath: pathToFile] )
	    {
	      [[NSFileManager defaultManager] copyPath: pathToFile
					      toPath: pathToInbox
					      handler: nil];
	    }
	  else
	    {
	      [[NSFileManager defaultManager] createFileAtPath: pathToInbox
					      contents: nil
					      attributes: nil];
	    }
	}
      
      //
      // Sent
      //
      pathToSent = [pathToLocalMailDir stringByAppendingPathComponent: @"Sent"];   
      
      if ( ![[NSFileManager defaultManager] fileExistsAtPath: pathToSent] )
	{
	  [[NSFileManager defaultManager] createFileAtPath: pathToSent
					  contents: nil
					  attributes: nil];
	}
      
      //
      // Trash
      //
      pathToTrash = [pathToLocalMailDir stringByAppendingPathComponent: @"Trash"];   
      
      if ( ![[NSFileManager defaultManager] fileExistsAtPath: pathToTrash] )
	{
	  [[NSFileManager defaultManager] createFileAtPath: pathToTrash
					  contents: nil
					  attributes: nil];
	}

      //
      // Drafts
      //
      pathToDrafts = [pathToLocalMailDir stringByAppendingPathComponent: @"Drafts"];   
      
      if ( ![[NSFileManager defaultManager] fileExistsAtPath: pathToDrafts] )
	{
	  [[NSFileManager defaultManager] createFileAtPath: pathToDrafts
					  contents: nil
					  attributes: nil];
	}
       
      //
      // We create a basic account with default preferences values just to get the user started.
      //
      theAccount = [[NSMutableDictionary alloc] init];
      

      //
      // We set the default PERSONAL values.
      //
      allValues = [[NSMutableDictionary alloc] init];
      [allValues setObject: _(@"Your name")           forKey: @"NAME"];
      [allValues setObject: _(@"Your E-Mail address") forKey: @"EMAILADDR"];
      [theAccount setObject: allValues  forKey: @"PERSONAL"];      
      RELEASE(allValues);
      
      //
      // MAILBOXES, we set the INBOX, SENT, .. for this account.
      //
      allValues = [[NSMutableDictionary alloc] init];
      [allValues setObject: [NSString stringWithFormat: @"local://%@", pathToInbox]   forKey: @"INBOXFOLDERNAME"];
      [allValues setObject: [NSString stringWithFormat: @"local://%@", pathToSent]    forKey: @"SENTFOLDERNAME"];
      [allValues setObject: [NSString stringWithFormat: @"local://%@", pathToDrafts]  forKey: @"DRAFTSFOLDERNAME"];
      [allValues setObject: [NSString stringWithFormat: @"local://%@", pathToTrash]   forKey: @"TRASHFOLDERNAME"];
      [theAccount setObject: allValues  forKey: @"MAILBOXES"];      
      RELEASE(allValues);
      
      //
      // SEND, we set the transport method to SMTP
      //
      allValues = [[NSMutableDictionary alloc] init];
      [allValues setObject: [NSNumber numberWithInt: 2]  forKey: @"TRANSPORT_METHOD"];
      [theAccount setObject: allValues  forKey: @"SEND"];      
      RELEASE(allValues);
      
      //
      // We finally save the account's information
      //
      [[NSUserDefaults standardUserDefaults] setObject: [NSDictionary dictionaryWithObject: theAccount
								      forKey: @"General"]
					     forKey: @"ACCOUNTS"];
      RELEASE(theAccount);
      
      //
      // We create the basic set of shown headers and other preference values.
      //
      [[NSUserDefaults standardUserDefaults] setObject: [NSArray arrayWithObjects: @"Date", @"From", @"To", @"Cc", @"Subject", nil]
					     forKey: @"SHOWNHEADERS"];
      [[NSUserDefaults standardUserDefaults] setBool: YES  forKey: @"HIDE_DELETED_MESSAGES"];
      [[NSUserDefaults standardUserDefaults] setBool: YES  forKey: @"HIGHLIGHT_URL"];

      //
      // We add our "Inbox" folder to the list of folders to open.
      //
      [[NSUserDefaults standardUserDefaults] setObject: [NSArray arrayWithObject: [NSString stringWithFormat: @"local://%@", 
											    pathToInbox]]
					     forKey: @"FOLDERS_TO_OPEN"];
      
      mustShowPreferencesWindow = YES;
    } // if ( ![[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] )
  

  //
  // We initialize our store
  //
  aLocalStore = [[LocalStore alloc] initWithPathToDirectory: pathToLocalMailDir];
  
  if ( !aLocalStore )
    {
      NSString *aString;

#ifdef MACOSX
      aString = _(@"Could not open the Local Store.\nThat means GNUMail.app is not properly configured.\nTo fix this problem, do the following commands:\n\n%% mkdir ~/Library/GNUMail/Mailboxes\n%% touch ~/Library/GNUMail/Mailboxes/Inbox\n%% touch ~/Library/GNUMail/Mailboxes/Sent\n%% touch ~/Library/GNUMail/Mailboxes/Trash\n%% touch ~/Library/GNUMail/Mailboxes/Drafts\n\nand restart GNUMail.app after that! You can also remove the ~/Library/Preferences/com.collaboration-world.GNUMail.plist file instead of doing the commands mentionned above.");
#else
      aString = _(@"Could not open the Local Store.\nThat means GNUMail.app is not properly configured.\nTo fix this problem, do the following commands:\n\n%% mkdir ~/Mailboxes\n%% touch ~/Mailboxes/Inbox\n%% touch ~/Mailboxes/Sent\n%% touch ~/Mailboxes/Trash\n%% touch ~/Mailboxes/Drafts\n\nand restart GNUMail.app after that!");
#endif

      NSRunCriticalAlertPanel(_(@"Fatal error!"),
			      aString,
			      @"OK",
			      NULL,
			      NULL);
      
      [NSApp terminate: self];
    }

  // We got a valid Local store, let's add it to our list of open Store:s in the MailboxManagerController
  [[MailboxManagerController singleInstance] setStore: aLocalStore
					     name: @"GNUMAIL_LOCAL_STORE"
					     username: NSUserName()];
  
  [[MailboxManagerController singleInstance] reloadAllFolders];
  [[[MailboxManagerController singleInstance] outlineView] reloadData];
  
  // Sync with the user's defaults
  [[NSUserDefaults standardUserDefaults] synchronize];

  // We create our console
  [ConsoleWindowController singleInstance];

  // We load all our bundles
  [self loadBundles];

  // We start our task manager and our global timer
  [self startTaskManager];
  [self startTimer];

  // We show of MailboxManager window, if we need to.
  // Under GNUstep, we MUST do this _before_ showing any MailWindow:s
  // since the MailboxManager will be the key window otherwise.
#ifndef MACOSX
  if ( [[NSUserDefaults standardUserDefaults] boolForKey: @"OPEN_MAILBOXMANAGER_ON_STARTUP"] )
    {
      [self showMailboxManager: nil];
    }
#endif
    
  // We show the Console window, if we need to.
  if ( [[NSUserDefaults standardUserDefaults] boolForKey: @"OPEN_CONSOLE_ON_STARTUP"] )
    {
      [self showConsoleWindow: nil];
    }  

  // We finally show our Inbox (or any other folders)
  [self showFoldersToOpen];
  
  // We show of MailboxManager window, if we need to.
  // Under OS X, we MUST do this _after_ showing any MailWindow:s
  // since we are using a drawer attached to the window.
#ifdef MACOSX
  if ( [[NSUserDefaults standardUserDefaults] boolForKey: @"OPEN_MAILBOXMANAGER_ON_STARTUP"] )
    {
      [self showMailboxManager: nil];
    }
#endif
  
  // If we must show the Preferences window, we show it right now. That happens if for example,
  // we started GNUMail.app for the first time and the user has chosen to configure it.
  if ( mustShowPreferencesWindow )
    {
      [self showPreferencesWindow: nil];
    }

  // We register our service
  [NSApp setServicesProvider: self];

  // We set up our initial list of incoming mail servers
  [self updateGetNewMailMenuItems: nil];

  // We set up our initial list of filters for the menu
  [self updateFilterMenuItems: nil];

  // We load all the items for the supported encodings (in Pantomime) in the encoding menu
  [self updateTextEncodingsMenu: self];

  // We finally check for new mails on startup, if we need to.
  [self checkForNewMail: self  controller: nil];

  // We are done initing
  doneInit = YES;

  // If a window has requested to be on top, do it now.
  if (requestLastMailWindowOnTop != nil)
    {
      [requestLastMailWindowOnTop makeKeyAndOrderFront: self];
      [GNUMail setLastMailWindowOnTop: requestLastMailWindowOnTop];
      requestLastMailWindowOnTop = nil;
    }

}


//
// methods invoked by notifications
//
- (void) selectionInTextViewHasChanged: (id) sender
{
  NSRange aRange;
  
  aRange = [[sender object] selectedRange];

  if ( aRange.length )
    {
      [enterSelection setAction: @selector(enterSelectionInFindPanel:)];
    }
  else
    {
      [enterSelection setAction: NULL];
    }
}


//
//
//
- (void) selectionOfMessageHasChanged: (id) sender
{
  MessageViewWindowController *messageViewWindowController;
  MailWindowController *aMailWindowController;
  id aWindowDelegate;

  messageViewWindowController = nil;
  aMailWindowController = nil;
  aWindowDelegate = nil;

  if ( [GNUMail lastMailWindowOnTop] )
    { 
      aWindowDelegate = [[GNUMail lastMailWindowOnTop] delegate];
      
      if ( aWindowDelegate )
    	{
	  if ( [aWindowDelegate isKindOfClass: [MailWindowController class]] )
	    {
	      aMailWindowController = (MailWindowController *)aWindowDelegate;
	    }
	  else
	    {
	      messageViewWindowController = (MessageViewWindowController *)aWindowDelegate;
	    }
	  
	  // No matter how the selection has changed, we always remove all items from the
	  // "Send to Address Book" menu.
	  [self removeAllItemsFromMenu: sendToAddressBook];
	  
	  // We we are in a situation with multiple rows selected, we must 
	  // take the first message in our enumerator for flags comparaison.
	  if ( aMailWindowController && [[aMailWindowController dataView] numberOfSelectedRows] > 1 )
	    {
	      Message *aMessage;
	      
	      aMessage = [[aMailWindowController selectedMessages] objectAtIndex: 0];
	      
	      [self updateMenuItemsForMessage: aMessage];
	    }
	  else
	    {
	      Message *aMessage;
	      int i;
	      
	      aMessage = [aWindowDelegate selectedMessage];
	      
	      [self updateMenuItemsForMessage: aMessage];

	      // We finally build our 'Send to Address Book' menu
	      if ( [[aMessage from] unicodeStringValue] )
		{
		  [sendToAddressBook addItemWithTitle: [[aMessage from] unicodeStringValue]
				     action: @selector(addToAddressBook:)
				     keyEquivalent: @""];
		}
	      
	      for (i = 0; i < [aMessage recipientsCount]; i++)
		{
		  if ( [[[aMessage recipients] objectAtIndex: i] unicodeStringValue] )
		    {
		      [sendToAddressBook addItemWithTitle: [[[aMessage recipients] objectAtIndex: i] unicodeStringValue]
					 action: @selector(addToAddressBook:)
					 keyEquivalent: @""];
		    }
		}
	    }
	} // if ( aMailWindowController ) ...
    }
  
  // We now have to decide what the "All Headers" menu item should read.
  // If the top window is a MailWindowController, no matter what, we always 
  // reset our menu item for showing all headers when this action method is called 
  // since we are switching to another message anyway.
  // If the top window is a MessageViewWindowController, we have to find out what the state 
  // the message is in.
  if ( aWindowDelegate &&
       [aWindowDelegate isKindOfClass: [MailWindowController class]] && 
       ![aMailWindowController showAllHeaders] )
    {      
      [showAllHeaders setTitle: _(@"All Headers")];
      [showAllHeaders setTag: SHOW_ALL_HEADERS];
      [showRawSource setTitle: _(@"Raw Source")];
      [aWindowDelegate setShowRawSource: NO];
    }
  else if ( aWindowDelegate &&
	    [aWindowDelegate isKindOfClass: [MessageViewWindowController class]] )
    {
      if ( [aWindowDelegate showAllHeaders] )
        {
	  [showAllHeaders setTitle: _(@"Filtered Headers")];
	  [showAllHeaders setTag: HIDE_ALL_HEADERS];
        }
      else
        {
	  [showAllHeaders setTitle: _(@"All Headers")];
	  [showAllHeaders setTag: SHOW_ALL_HEADERS];
        }
      // Set the menu text for showing raw source
      if ( [aWindowDelegate showRawSource] )
	{
	  [showRawSource setTitle: _(@"Normal Display")];
	}
      else
	{
	  [showRawSource setTitle: _(@"Raw Source")];
	}
    }
}


//
// access / mutation methods
//
+ (NSArray *) allBundles
{
  return allBundles;
}


+ (NSArray *) allMailWindows
{
  return allMailWindows;
}


+ (NSString *) currentWorkingPath
{
  return currentWorkingPath;
}


+ (void) setCurrentWorkingPath: (NSString *) thePath
{
  RETAIN(thePath);
  RELEASE(currentWorkingPath);
  currentWorkingPath = thePath;
}


+ (id) lastAddressTakerWindowOnTop
{
  return lastAddressTakerWindowOnTop;
}


+ (void) setLastAddressTakerWindowOnTop: (id) aWindow
{
  lastAddressTakerWindowOnTop = aWindow;
}


+ (id) lastMailWindowOnTop
{ 
  return lastMailWindowOnTop;
}


//
//
//
+ (void) setLastMailWindowOnTop: (id) aWindow
{
  lastMailWindowOnTop = aWindow;

  if ( [NSApp delegate] && lastMailWindowOnTop )
    {
      [[NSApp delegate] updateShowOrHideDeletedMenuItem:
			  [[[lastMailWindowOnTop windowController] folder] showDeleted]];
      [[NSApp delegate] updateShowOrHideReadMenuItem:
			  [[[lastMailWindowOnTop windowController] folder] showRead]];
    }
}


//
//
//
- (NSMenu *) saveAttachmentMenu
{
  return saveAttachment;
}


//
//
//
- (NSMenuItem *) deliverMenuItem
{
  return deliver;
}

//
//
//
- (void) setEnableSaveInDraftsMenuItem: (BOOL) aBOOL
{
  if ( aBOOL )
    {
      [saveInDrafts setAction: @selector(saveInDrafts:)];
    }
  else
    {
      [saveInDrafts setAction: NULL];
    }
}


//
//
//
- (void) setShowRawSourceMenuItem: (BOOL) aBOOL
{
  if ( aBOOL )
    {
      [showRawSource setTitle: _(@"Raw Source")];
    }
  else
    {
      [showRawSource setTitle: _(@"Normal Display")];
    }
}


//
//
//
- (NSMenuItem *) showAllHeadersMenuItem
{
  return showAllHeaders;
}


//
// other methods
//
- (void) addItemToMenuFromTextAttachment: (NSTextAttachment *) theTextAttachment
{
  NSFileWrapper *aFileWrapper;
  ExtendedMenuItem *menuItem;

  aFileWrapper = [theTextAttachment fileWrapper];
 
  menuItem = [[ExtendedMenuItem alloc] initWithTitle: [aFileWrapper preferredFilename]
				       action: @selector(saveAttachment:)
				       keyEquivalent: @""];
  [menuItem setTextAttachment: theTextAttachment];

  [[self saveAttachmentMenu] addItem: menuItem];

  RELEASE(menuItem);
}


//
//
//
+ (void) addEditWindow: (id) theEditWindow
{
  if (allEditWindows && theEditWindow )
    {
      [allEditWindows addObject: theEditWindow];
    }
}


//
//
//
+ (void) addMailWindow: (id) theMailWindow
{
  if (allMailWindows && theMailWindow )
    {
      [allMailWindows addObject: theMailWindow];
    }
}


//
// FIXME - what to do with this?
//
- (void) addToAddressBook: (id) sender
{
#if 0
  InternetAddress *anInternetAddress;
  Address *anAddress;

  if ( ![sender isKindOfClass: [NSMenuItem class]] )
    {
      return;
    }
  
  anInternetAddress = [[InternetAddress alloc] initWithString: [sender title]];
  
  if ( !anInternetAddress )
    {
      NSDebugLog(@"Could not add %@ to the Address Book.", [sender title]);
      return;
    }
  
  anAddress = [[Address alloc] init];  
  [anAddress setName: [anInternetAddress personal]];
  [anAddress setEmailAddress: [anInternetAddress address]];
  [[AddressBookController singleInstance] addToAddressBook: anAddress];

  RELEASE(anInternetAddress);
  RELEASE(anAddress);
#endif
}


//
// Method used to load all bundles in all domains.
// FIXME: Offer a way to the user to specify a list of bundles
//        that he/she DOES NOT want to load automatically.
//
- (void) loadBundles
{
  NSFileManager *aFileManager;
  NSArray *allFiles, *allPaths;
  NSString *aPath;
  int i, j;


  aFileManager = [NSFileManager defaultManager];
  
  allPaths =  NSSearchPathForDirectoriesInDomains(NSLibraryDirectory,
						  NSLocalDomainMask|NSNetworkDomainMask|NSSystemDomainMask|NSUserDomainMask,
						  YES);
  for (i = 0; i < [allPaths count]; i++)
    {
      aPath = [NSString stringWithFormat: @"%@/GNUMail", [allPaths objectAtIndex: i]];

      NSDebugLog(@"Trying to load bundles from |%@|", aPath);
      
      allFiles = [aFileManager directoryContentsAtPath: aPath];
      
      for (j = 0; j < [allFiles count]; j++)
	{
	  NSString *aString;
	  
	  aString = [allFiles objectAtIndex: j];
   
	  // If we found a bundle, let's load it!
	  if ( [[aString pathExtension] isEqualToString: @"bundle"] )
	    {
	      id<GNUMailBundle> aModule;
	      NSBundle *aBundle;
	      NSString *path;
	      
	      path = [NSString stringWithFormat: @"%@/%@",
			       aPath,
			       aString];
	      
	      aBundle = [NSBundle bundleWithPath: path];
	      
	      if ( aBundle )
		{
		  Class aClass;

		  aClass = [aBundle principalClass];

		  // We ensure our bundle is really a GNUMail.app bundle.
		  if ( ![aClass conformsToProtocol: @protocol(GNUMailBundle)] )
		    {
		      NSDebugLog(@"%@ does not conform to the GNUMailBundle protocol. Skipping.", aString);
		      continue;
		    }

		  aModule = [aClass singleInstance];
		  
		  if ( aModule )
		    {
		      [aModule setOwner: self];
		      [allBundles addObject: aModule];
		      [[ConsoleWindowController singleInstance] addConsoleMessage: [NSString stringWithFormat: 
											       _(@"Loaded bundle at path %@"),
											     path]];
		    }
		  else
		    {
		      [[ConsoleWindowController singleInstance] addConsoleMessage: [NSString stringWithFormat: 
											       _(@"Failed to initialize bundle at path %@"), 
											     path]];
		    }
		}
	      else
		{
		  [[ConsoleWindowController singleInstance] addConsoleMessage: [NSString stringWithFormat: 
											   _(@"Error loading bundle at path %@"), 
											 path]];
		}
	    }
	}
    }
}


//
//
//
- (void) removeAllItemsFromMenu: (NSMenu *) theMenu
{
  int i;
  
  for (i = ([theMenu numberOfItems] - 1); i >= 0; i--)
    {
      [theMenu removeItemAtIndex: i];
    }
}


//
//
//
+ (void) removeEditWindow: (id) theEditWindow
{
  if (allEditWindows && theEditWindow )
    {
      [allEditWindows removeObject: theEditWindow];
    }
}


//
//
//
+ (void) removeMailWindow: (id) theMailWindow
{
  if (allMailWindows && theMailWindow )
    {
      [allMailWindows removeObject: theMailWindow];
    }
}


//
// This method is used to open a set of folders on startup. 
//
- (void) showFoldersToOpen
{ 
  NSMutableArray *foldersToOpen;
  int i;
  
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"FOLDERS_TO_OPEN"] )
    {
      foldersToOpen = [[NSUserDefaults standardUserDefaults] objectForKey: @"FOLDERS_TO_OPEN"];
    }
  else
    {
      foldersToOpen = [NSMutableArray array];
    }
  
  
  for (i = 0; i < [foldersToOpen count]; i++)
    {
      URLName *theURLName;

      theURLName = [[URLName alloc] initWithString: [foldersToOpen objectAtIndex: i]
				    path: [[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"]];
      
      [[MailboxManagerController singleInstance] openFolderWithURLName: theURLName
						 sender: self];
      
      RELEASE(theURLName);
    }
}


//
//
//
- (void) updateGetNewMailMenuItems: (id) sender
{
  NSDictionary *allValues;
  NSMenuItem *aMenuItem;
  NSArray *allKeys;
  NSString *aKey;
  
  int i;
  
  // We first remove all our items in the current menu
  [self removeAllItemsFromMenu: incomingMailServers];
  
  // Our "All" menu item
  aMenuItem = [[NSMenuItem alloc] initWithTitle: _(@"All")
				  action: @selector(getNewMessages:)
				  keyEquivalent: @"N"];
  [aMenuItem setTarget: self];
  [aMenuItem setTag: -1];
  [incomingMailServers addItem: aMenuItem];
  RELEASE(aMenuItem);

  // We sort the array to be sure to keep the order.
  allKeys = [[[Utilities allEnabledAccounts] allKeys]
	      sortedArrayUsingSelector: @selector(compare:)];

  for (i = 0; i < [allKeys count]; i++)
    {
      aKey = [allKeys objectAtIndex: i];
      allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] objectForKey: aKey]
		    objectForKey: @"RECEIVE"];
      
      // We only consider our POP3 and UNIX receiving accounts - both
      // being NOT set to never check mails
      if ( (![allValues objectForKey: @"SERVERTYPE"] ||
	    [[allValues objectForKey: @"SERVERTYPE"] intValue] == POP3 || 
	    [[allValues objectForKey: @"SERVERTYPE"] intValue] == UNIX) &&
	   [[allValues objectForKey: @"RETRIEVEMETHOD"] intValue] != NEVER )
	{
	  aMenuItem = [[NSMenuItem alloc] initWithTitle: aKey
					  action: @selector(getNewMessages:)
					  keyEquivalent: @""];
	  [aMenuItem setTarget: self];
	  [aMenuItem setTag: i];
	  [incomingMailServers addItem: aMenuItem];
	  RELEASE(aMenuItem);
	}
    }
}


//
//
//
#ifdef MACOSX
- (void) updateDockMenu
{
  if ( [GNUMail lastMailWindowOnTop] )
    {
      NSMenuItem *item;
      Folder *aFolder;
      
      aFolder = [[[GNUMail lastMailWindowOnTop] windowController] folder];
      
      [self removeAllItemsFromMenu: dock];
      item = [[NSMenuItem alloc] initWithTitle: [NSString stringWithFormat: _(@"%d unread messages"), [aFolder numberOfUnreadMessages]]
				 action: nil  
				 keyEquivalent: @""];
      [item setEnabled: NO];
      [dock addItem: item];
      RELEASE(item);
      
      item = [[NSMenuItem alloc] initWithTitle: [NSString stringWithFormat: _(@"%d deleted messages"), [aFolder numberOfDeletedMessages]]
				 action: nil  
				 keyEquivalent: @""];
      [item setEnabled: NO];
      [dock addItem: item];
      RELEASE(item);
      
      // We now add our "New Message" dock item
      [dock addItem: [NSMenuItem separatorItem]];
      item = [[NSMenuItem alloc] initWithTitle: _(@"New Message")
      				 action: @selector(composeMessage:)
				 keyEquivalent: @""];
      [dock addItem: item];
      RELEASE(item);
    }
  else
    {
      [self removeAllItemsFromMenu: dock];
    }
}
#endif


//
//
//
- (void) updateFilterMenuItems: (id) sender
{
  BOOL isDir;
  
  if ( [[NSFileManager defaultManager] fileExistsAtPath: PathToFilters()
				       isDirectory: &isDir] &&
       ! isDir )
    {
      FilterManager *aFilterManager;
      NSMenuItem *aMenuItem;
      int i;
  
      // We first remove all our items in the current menu
      [self removeAllItemsFromMenu: filters];
    
      aFilterManager = [FilterManager singleInstance];
      
      // Our "All" menu item
      aMenuItem = [[NSMenuItem alloc] initWithTitle: _(@"All")
				      action: @selector(applyManualFilter:)
				      keyEquivalent: @""];
      [aMenuItem setTag: -1];
      [filters addItem: aMenuItem];
      RELEASE(aMenuItem);

      for (i = 0; i < [[aFilterManager filters] count]; i++)
	{
	  Filter *aFilter;
	  
	  aFilter = [[aFilterManager filters] objectAtIndex: i];
	  
	  aMenuItem = [[NSMenuItem alloc] initWithTitle: [aFilter description]
					  action: @selector(applyManualFilter:)
					  keyEquivalent: @""];
	  [aMenuItem setTag: i];
	  [filters addItem: aMenuItem];
	  RELEASE(aMenuItem);
	}
    }
}


//
//
//
- (void) updateMenuItemsForMessage: (Message *) theMessage
{
  // Initial verification, we never know!
  if ( !theMessage )
    {
      return;
    }
  
  // We verify if we need to delete or undelete our message and 
  // we set the tag to the right value
  if ( [[theMessage flags] contain: DELETED] )
    {
      [deleteOrUndelete setTitle: _(@"Undelete")];
      [deleteOrUndelete setTag: UNDELETE_MESSAGE];
    }
  else
    {
      [deleteOrUndelete setTitle: _(@"Delete")];
      [deleteOrUndelete setTag: DELETE_MESSAGE];
    }
  
  // We verify if we need to mark the message as read or unread
  if ( [[theMessage flags] contain: SEEN] )
    {
      [markAsReadOrUnread setTitle: _(@"Mark as Unread")];
      [markAsReadOrUnread setTag: MARK_AS_UNREAD];
    }
  else
    {
      [markAsReadOrUnread setTitle: _(@"Mark as Read")];
      [markAsReadOrUnread setTag: MARK_AS_READ];
    }      
}


//
//
//
- (void) updateShowOrHideDeletedMenuItem: (BOOL) aBOOL
{
  if ( aBOOL )
    {
      [showOrHideDeleted setTitle: _(@"Hide Deleted")];
      [showOrHideDeleted setTag: HIDE_DELETED_MESSAGES];
    }
  else
    {
      [showOrHideDeleted setTitle: _(@"Show Deleted")];
      [showOrHideDeleted setTag: SHOW_DELETED_MESSAGES];
    }
}


//
//
//
- (void) updateShowOrHideReadMenuItem: (BOOL) aBOOL
{
  if ( aBOOL )
    {
      [showOrHideRead setTitle: _(@"Hide Read")];
      [showOrHideRead setTag: HIDE_READ_MESSAGES];
    }
  else
    {
      [showOrHideRead setTitle: _(@"Show Read")];
      [showOrHideRead setTag: SHOW_READ_MESSAGES];
    }
}


//
//
//
- (void) updateTextEncodingsMenu: (id) sender
{
  NSMutableArray *aMutableArray;
  NSMenuItem *item;
  int i;

  [self removeAllItemsFromMenu: textEncodings];
  
  item = [[NSMenuItem alloc] initWithTitle: _(@"Default")
			     action: @selector(changeTextEncoding:)
			     keyEquivalent: @""];
  [item setTag: -1];
  [textEncodings addItem: item];
  RELEASE(item);

  aMutableArray = [[NSMutableArray alloc] init];
  [aMutableArray addObjectsFromArray: [[MimeUtility allCharsets] allValues]];
  [aMutableArray sortUsingSelector: @selector(compare:)];

  for (i = 0; i < [aMutableArray count]; i++)
    {
      item = [[NSMenuItem alloc] initWithTitle: [aMutableArray objectAtIndex: i]
				 action: @selector(changeTextEncoding:)
				 keyEquivalent: @""];
      [item setTag: i];
      [textEncodings addItem: item];
      RELEASE(item);
    }

  RELEASE(aMutableArray);
}


//
//
//
- (void) updateThreadOrUnthreadMenuItem: (BOOL) aBOOL
{
  if ( aBOOL )
    {
      [threadOrUnthreadMessages setTitle: _(@"Thread Messages")];
      [threadOrUnthreadMessages setTag: THREAD_MESSAGES];
    }
  else
    {
      [threadOrUnthreadMessages setTitle: _(@"Unthread Messages")];
      [threadOrUnthreadMessages setTag: UNTHREAD_MESSAGES];
    }
}

//
// services methods
//
- (void) newMessageWithContent: (NSPasteboard *) pboard
	              userData: (NSString *) userData
                         error: (NSString **) error
{
  EditWindowController *editWindowController;
  NSString *aString;
  NSArray *allTypes;
  Message *aMessage;

  allTypes = [pboard types];

  if ( ![allTypes containsObject: NSStringPboardType] )
    {
      *error = @"No string type supplied on pasteboard";
      return;
    }
  
  aString = [pboard stringForType: NSStringPboardType];
  
  if (aString == nil)
    {
      *error = @"No string value supplied on pasteboard";
      return;
    }
  
  // We create a new message with our pasteboard content
  aMessage = [[Message alloc] init];
  [aMessage setContent: aString];

  // We create our controller and we show the window
  editWindowController = [[EditWindowController alloc] initWithWindowNibName: @"EditWindow"];

  if ( editWindowController )
    {
      [[editWindowController window] setTitle: _(@"New message...")];
      [editWindowController setMessage: aMessage];
      [editWindowController setShowCc: NO];
    
      [[editWindowController window] orderFrontRegardless];
    }
  
  RELEASE(aMessage);
}


//
//
//
- (void) newMessageWithRecipient: (NSPasteboard *) pboard
	                userData: (NSString *) userData
                           error: (NSString **) error
{
  NSString *aString;
  NSArray *allTypes;

  allTypes = [pboard types];

  if ( ![allTypes containsObject: NSStringPboardType] )
    {
      *error = @"No string type supplied on pasteboard";
      return;
    }
  
  aString = [pboard stringForType: NSStringPboardType];
  
  if (aString == nil)
    {
      *error = @"No string value supplied on pasteboard";
      return;
    }

  [self newMessageWithRecipient: aString];
  
}

//
//
//
- (void) newMessageWithAttachments: (NSPasteboard *) pboard
			  userData: (NSString *) userData
                             error: (NSString **) error
{
  EditWindowController *editWindowController;
  NSAutoreleasePool *pool;

  MimeMultipart *aMimeMultipart;
  Part *aPart;
  MimeType *aMimeType;
  Message *aMessage;

  NSEnumerator *enumerator;
  NSFileManager *manager;

  NSString *aFile;
  NSArray *allFiles, *allTypes;
  BOOL isDir;

  pool = [[NSAutoreleasePool alloc] init];
  manager = [NSFileManager defaultManager];
  allTypes = [pboard types];
  aMimeMultipart = nil;
  aMessage = nil;
  
  if ( ![allTypes containsObject: NSFilenamesPboardType] )
    {
      *error = @"No filenames supplied on pasteboard";
      RELEASE(pool);
      return;
    }
  
  allFiles = [pboard propertyListForType: NSFilenamesPboardType];
  
  NSDebugLog(@"Attach %@", allFiles);
  
  if (allFiles == nil)
    {
      *error = @"No files supplied on pasteboard";
      RELEASE(pool);
      return;
    }
  
  // We create a new message with our pasteboard content
  aMessage = [[Message alloc] init];

  aMimeMultipart = [[MimeMultipart alloc] init];

  enumerator = [allFiles objectEnumerator];
  
  while ( (aFile = [enumerator nextObject]) )
    {
      if ( ![manager fileExistsAtPath: aFile  isDirectory: &isDir] )
        {
          NSDebugLog(@"File '%@' does not exists (not adding as attachment)", aFile);
          continue;
        }
      if ( isDir )
        {
          NSDebugLog(@"'%@' is a directory (not adding as attachment)", aFile);
          continue;
        }

      NSDebugLog(@"Adding '%@' as attachement", aFile);
      
      // We first decode our text/plain body
      aPart = [[Part alloc] init];
      
      aMimeType = [[MimeTypeManager singleInstance] bestMimeTypeForFileExtension:
						      [[aFile lastPathComponent] pathExtension]];
      
      if ( aMimeType )
	{
	  [aPart setContentType: [aMimeType mimeType]];
	}
      else
	{
	  [aPart setContentType: @"application/octet-stream"];
	}
      
      [aPart setContentTransferEncoding: BASE64];
      [aPart setContentDisposition: @"attachment"];
      [aPart setFilename: [aFile lastPathComponent]];
      
      [aPart setContent: [NSData dataWithContentsOfFile: aFile]];
      [aMimeMultipart addBodyPart: aPart];
      RELEASE(aPart);
    }
  
  [aMessage setContentTransferEncoding: NONE];
  [aMessage setContentType: @"multipart/mixed"];     
  [aMessage setContent: aMimeMultipart];
  
  // We generate a new boundary for our message
  [aMessage setBoundary: [MimeUtility generateBoundary]];
  
  RELEASE(aMimeMultipart);
  
  // We create our controller and we show the window
  editWindowController = [[EditWindowController alloc] initWithWindowNibName: @"EditWindow"];

  if ( editWindowController )
    {
      [[editWindowController window] setTitle: _(@"New message...")];
      [editWindowController setMessage: aMessage];
      [editWindowController setShowCc: NO];
      
      [[editWindowController window] orderFrontRegardless];
    }
  
  RELEASE(aMessage);
  RELEASE(pool);
}


//
//
//
- (void) newMessageWithRecipient: (NSString *) aString
{
  EditWindowController *editWindowController;
  InternetAddress *anInternetAddress;
  Message *aMessage;
  
  // We create a new message and we set the recipient
  aMessage = [[Message alloc] init];
  anInternetAddress = [[InternetAddress alloc] initWithString: aString];
  [anInternetAddress setType: TO];
  [aMessage addToRecipients: anInternetAddress];
  RELEASE(anInternetAddress);
  
  // We create our controller and we show the window
  editWindowController = [[EditWindowController alloc] initWithWindowNibName: @"EditWindow"];
  
  if ( editWindowController )
    {
      [[editWindowController window] setTitle: _(@"New message...")];
      [editWindowController setMessage: aMessage];
      [editWindowController setShowCc: NO];
      
      // If we just got launched, as to to be put on top after we are done initing
      if ( !doneInit )
	{
	  requestLastMailWindowOnTop = [editWindowController window];
	}
      else
	{
	  [[editWindowController window] makeKeyAndOrderFront: self];
	}
    }
  
  RELEASE(aMessage);
}

@end

//
// Scripting support; still experimental code.
//
#ifdef MACOSX
@implementation GNUMail (KeyValueCoding)

- (BOOL)application:(NSApplication *)sender delegateHandlesKey:(NSString *)key
{
  // NSLog(@"key = %@", key);
  return [key isEqualToString:@"messageCompositions"];
}

// accessors for to-many relationships:
-(NSMutableArray*)messageCompositions
{
  // NSLog(@"messageCompositions...");
  return _messageCompositions;
}

-(void)setMessageCompositions: (NSMutableArray*)messageCompositions
{
  // NSLog(@"setMessageCompositions..."); 
  [_messageCompositions setArray: messageCompositions];
}


// accessors for to-many relationships:
// (See NSScriptKeyValueCoding.h)

- (void) addInMessageCompositions: (MessageComposition *) object
{
  // NSLog(@"GNUMail: addInMessageCompositions 0x%x", object);
  [self insertInMessageCompositions: object atIndex: [_messageCompositions count]];
}

- (void) insertInMessageCompositions: (MessageComposition *) object
{
  // NSLog(@"GNUMail: insertInMessageCompositions 0x%x", object);
  [self insertInMessageCompositions: object atIndex: [_messageCompositions count]];
}

- (void) insertInMessageCompositions: (MessageComposition *) object atIndex: (unsigned) index
{
  // NSLog(@"GNUMail: insertInMessageCompositions 0x%x atIndex %d", object, index);
  [_messageCompositions insertObject: object atIndex: index];
  
}

- (void) replaceInMessageCompositions: (MessageComposition *) object atIndex: (unsigned) index
{
  // NSLog(@"GNUMail: replaceInMessageCompositions 0x%x atIndex %d", object, index);
  [_messageCompositions replaceObjectAtIndex: index withObject: object];
}

- (void) removeFromMessageCompositionsAtIndex: (unsigned) index
{
  // NSLog(@"GNUMail: removeFromMessageCompositionsAtIndex %d", index);
  [_messageCompositions removeObjectAtIndex: index];
}

- (id) valueInMessageCompositionsAtIndex: (unsigned) index
{
  // NSLog(@"valueInMessageCompositionsAtIndex: %d", index);
  return ([_messageCompositions objectAtIndex: index]);
}

@end

#endif // MACOSX


//
// Private methods
//
@implementation GNUMail (Private)

//
// Tries to load the property page. Returns NO in case of a failure
// or if there's no update available.
//
- (BOOL) _checkForUpdate
{
  BOOL checked;

  checked = NO;
  
  NS_DURING
    {
      NSString *aString;
      NSData *aData;
      NSURL *aURL;

      aURL = [NSURL URLWithString: PROPERTY_URL];
      
      // Fetch the property list from PROPERTY_URL
      aData = [aURL resourceDataUsingCache: NO];
      
      // Decode the property list
      aString = [[NSString alloc] initWithData: aData
				   encoding: NSUTF8StringEncoding];
      
      // Check the content
      checked = [self _checkDictionary: [aString propertyList]];
      RELEASE(aString);
    }
  NS_HANDLER
    {
      // Something went wrong, eg. page unavailable
      [localException raise];
      checked = NO;
    }
  NS_ENDHANDLER
  
  return checked;
}


//
// Retrieves the current (running application) and the latest
// (from the web) version numbers
//
- (BOOL) _checkDictionary: (NSDictionary *) theDictionary
{
  NSString *latestVersion;
  int result;
  
  //
  // If dictionary is empty, raise an exception and return
  //
  if ( !theDictionary )
    {
      [NSException raise: @"UpdateException"
		   format: _(@"Unable to retrieve software information.")];
      return NO;
    }
  
  //
  // Get the latest version number as posted on "the net"
  //
  latestVersion = [theDictionary objectForKey: [[NSProcessInfo processInfo] processName]];
  
  //
  // Now compare them 
  //
  result = CompareVersion(GNUMailVersion(), latestVersion);
  
  if ( result == NSOrderedDescending || result == NSOrderedSame )
    {
      return NO;
    }

  //
  // Cool, a new version...
  //
  [self _newVersionAvailable: latestVersion];
  
  return YES;
}


//
// Opens the 'new version available' dialog and offers to
// download if possible (read: on Mac OS X)
//
- (void) _newVersionAvailable: (NSString *) theVersion
{
  NSString *aURL;         // The URL where to get it from
  NSString *firstButton;  // Default (OK) button
  NSString *secondButton; // Alternate (Cancel) button
  NSString *aMessage;     // The message we display to the user.
  
  int result;
  
  aMessage = [NSString stringWithFormat: _(@"The latest version of GNUMail.app is %@.\n"),
		       theVersion];
  
  //
  // GNUstep's NSWorkspae doesn't react to -openURL:
  // but MACOSX does...
  //
#ifdef MACOSX
  aMessage = [aMessage stringByAppendingString: _(@"Would you like to download the new version now?")];
  aURL = DOWNLOAD_URL;
  firstButton = _(@"Update now");
  secondButton = _(@"Update later");
#else
  aMessage = [aMessage stringByAppendingFormat: _(@"Check < %@ > for more details."), APPLICATION_URL];
  aURL = nil;
  firstButton = _(@"OK");
  secondButton = NULL;
#endif
  
  result = NSRunInformationalAlertPanel(_(@"A new version of GNUMail.app is now available."),
					aMessage,
					firstButton,
					secondButton,
					NULL);
  
  if ( result == NSAlertDefaultReturn )
    {
      if ( aURL )
	{
	  [[NSWorkspace sharedWorkspace] openURL: [NSURL URLWithString: aURL]];
	}
    }
}


//
//
//
- (void) _savePanelDidEnd: (NSSavePanel *) theSheet
	       returnCode: (int) returnCode
	      contextInfo: (void  *) contextInfo
{
  
  NSString *aFilename;
  NSData *aData;

  aFilename = [theSheet filename];
  aData = (NSData *)contextInfo;

  // if successful, save file under designated name
  if (returnCode == NSAlertDefaultReturn)
    {
      if ( ![aData writeToFile: [theSheet filename]
		   atomically: YES] )
        {
	  NSBeep();
        }
      
      [GNUMail setCurrentWorkingPath: [aFilename stringByDeletingLastPathComponent]];
    }
  
  RELEASE(aData);
}

@end
 
