/*
**  LocalInboxWindowController.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#import "LocalInboxWindowController.h"

#import "ExtendedCell.h"
#import "GNUMail.h"
#import "GNUMailConstants.h"

#ifndef MACOSX
#import "LocalInboxWindow.h"
#endif

#import "Filter.h"
#import "FilterManager.h"
#import "LocalMailDelivery.h"
#import "PasswordPanelController.h"
#import "Utilities.h"

#import <Pantomime/Folder.h>
#import <Pantomime/LocalFolder.h>
#import <Pantomime/LocalStore.h>
#import <Pantomime/Message.h>
#import <Pantomime/POP3Folder.h>
#import <Pantomime/POP3Store.h>
#import <Pantomime/NSDataExtensions.h>
#import <Pantomime/POP3CacheManager.h>

static LocalInboxWindowController *localInboxWindowController = nil;

@implementation LocalInboxWindowController


//
//
//
- (id) initWithWindowNibName: (NSString *) windowNibName
{
#ifdef MACOSX
  
  self = [super initWithWindowNibName: windowNibName];
#else
  LocalInboxWindow *localInboxWindow;
  
  localInboxWindow = [[LocalInboxWindow alloc] initWithContentRect: NSMakeRect(150,100,750,595)
					       styleMask: NSClosableWindowMask|NSTitledWindowMask|
					       NSMiniaturizableWindowMask|NSResizableWindowMask
					       backing: NSBackingStoreRetained
					       defer: NO];
  
  self = [super initWithWindow: localInboxWindow];

  [localInboxWindow layoutWindow];
  [localInboxWindow setDelegate: self];
  [localInboxWindow setFrame:[Utilities rectForMailWindow] display: NO];
  
  // We link our outlets
  tableScrollView = [localInboxWindow tableScrollView];
  textScrollView = [localInboxWindow textScrollView];

  splitView = [localInboxWindow splitView];
  tableView = [localInboxWindow tableView];
  textView = [localInboxWindow textView];

  icon = [localInboxWindow icon];
  next = [localInboxWindow next];
  previous = [localInboxWindow previous];
  delete = [localInboxWindow delete];
  mailboxes = [localInboxWindow mailboxes];
  compose = [localInboxWindow compose];
  forward = [localInboxWindow forward];
  reply = [localInboxWindow reply];
  addresses = [localInboxWindow reply];
  find = [localInboxWindow find];
  get = [localInboxWindow get];

  label = (NSTextField *)[localInboxWindow label];
  
  statusColumn = [localInboxWindow statusColumn];
  idColumn = [localInboxWindow idColumn];
  dateColumn = [localInboxWindow dateColumn];
  fromColumn = [localInboxWindow fromColumn];
  subjectColumn = [localInboxWindow subjectColumn];
  sizeColumn = [localInboxWindow sizeColumn];

  RELEASE(localInboxWindow);
#endif

  // We set our window title
  [[self window] setTitle: @""];

  // We must retain our columns under OS X after the outlets are linked (we release them in -dealloc)
#ifdef MACOSX
  RETAIN(statusColumn);
  RETAIN(idColumn);
  RETAIN(dateColumn);
  RETAIN(fromColumn);
  RETAIN(subjectColumn);
  RETAIN(sizeColumn);
#endif

  // We set our custom cell
  [statusColumn setDataCell: AUTORELEASE([[ExtendedCell alloc] init])];

  // We load our accessory views
  [self _loadAccessoryViews];

  // We restore our sorting order
  [self _restoreSortingOrder];

  // We restore our split view knob position
  [self _restoreSplitViewSize];

  // We load the right set of columns
  [self reloadTableColumns];

  return self;
}


//
//
//
- (void) dealloc
{
  NSLog(@"LocalInboxWindowController: -dealloc");
 
  // We remove our timers
  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: ShouldStartLocalInboxTimer
    object: nil];
  
  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: ShouldStopLocalInboxTimer
    object: nil];
  
  // We now release our connection and our array of ports
  RELEASE(connection);
  RELEASE(ports);

  // We now release...
  RELEASE(localMailDelivery);
  
  RELEASE(lock);
 
  RELEASE(passwordCache);
  RELEASE(filteredMessageBoxes);

  TEST_RELEASE(timer);

  [super dealloc];
}


//
// action methods
//

- (IBAction) updateAnimatedIcon: (id) sender
{
  if (animation_index == 9)
    {
      animation_index = 1;
    }
  
  [icon setImage: [NSImage imageNamed: [NSString stringWithFormat: @"anim-logo-%d.tiff", animation_index]]];
  [[self window] update];
  
  animation_index += 1;
}

//
//
//
- (IBAction) getNewMessages: (id) sender
{
  NSArray *allPOP3Keys;
  int i;

  // We verify 'who' initiated the call to this method, the timer or the 'Get' button.
  if ( sender == timer )
    {
      initiatedFromTimer = YES;
    }
  else
    {
      initiatedFromTimer = NO;
    }
  
  //
  // We attempt to aquire the lock, the lock is release just after the thread has stopped running...
  //
  while (! [lock tryLock] )
    {
      [[NSRunLoop currentRunLoop] runMode: NSDefaultRunLoopMode
                                  beforeDate: [NSDate distantFuture]];
    }
  
  //
  // We first prompt for the passwords of all our POP3 servers.
  //
  allPOP3Keys = [[[NSUserDefaults standardUserDefaults] objectForKey:@"RECEIVING"] allKeys];
  
  for (i = 0; i < [allPOP3Keys count]; i++)
    {
      NSDictionary *allPOP3Values;
      NSString *servername;

      servername = [allPOP3Keys objectAtIndex: i];
      allPOP3Values = [[[NSUserDefaults standardUserDefaults] objectForKey:@"RECEIVING"] objectForKey: servername];

      if ( ! [allPOP3Values objectForKey: @"SERVERTYPE"] ||
	   [[allPOP3Values objectForKey: @"SERVERTYPE"]  intValue] == POP3) 
	{
	  // We get the password for the POP3 server and we cache it.
	  [self _passwordForServerName: servername
		prompt: YES];
	}
    }

  // Before detaching our thread, we verify if we need to show our "wait panel"
  if ( !initiatedFromTimer )
    {
      if ( [[[NSUserDefaults standardUserDefaults] 
	      objectForKey: @"SHOWWAITPANELS"] intValue] == NSOnState )
	{
	  // We show our wait panel...
	  //panel = NSGetInformationalAlertPanel(_(@"Please wait..."), 
	  //				       _(@"Fetching messages from the server. Please wait."),
	  //				       NULL,
	  //				       NULL,
	  //				       NULL);
	  // 
	  //modalSession = [NSApp beginModalSessionForWindow: panel];
	  //[NSApp runModalSession: modalSession];
	}
    }

  // We disable our automatic fetching timer, to avoid a deadlock
  [self stopTimer: nil];

  // We disable our Get button
  [get setEnabled: NO];

  // We start our animation timer
  animation_index = 1;
  animation =  [NSTimer scheduledTimerWithTimeInterval: 0.1
			target: self
			selector: @selector(updateAnimatedIcon:)
			userInfo: nil
			repeats: YES];
  RETAIN(animation);

  //
  // We detach our thread that will fetch the messages from the POP3 servers and from
  // the local mail spool file (if the user wants to!)
  //
  [NSThread detachNewThreadSelector: @selector(_transferMessagesUsingPorts:)
	    toTarget: self
	    withObject: ports];
}


//
// Methods called trough our NSConnection object
// by our thread that transfer the messages
//

- (void) communicationErrorOccuredOnServer: (NSString *) theServer
{
  NSRunInformationalAlertPanel(_(@"Error!"),
			       _(@"Unable to communicate with the POP3 server (%@)."),
			       _(@"OK"),
			       NULL,
			       NULL,
			       theServer);
}

- (void) invalidPasswordOnServer: (NSString *) theServer
			username: (NSString *) theUsername
{
  NSRunInformationalAlertPanel(_(@"Error!"),
			       _(@"Invalid password %@ @ %@."),
			       _(@"OK"),
			       NULL,
			       NULL,
			       theUsername,
			       theServer);
}

- (oneway void) messageWasReceived: (NSData *) theMessage
			folderName: (NSString *) theFolderName
{
  // We append the message to the folder
  [self _transferMessageFromRawSource: theMessage
	toFolderWithName: theFolderName];
	
  // We update our table view and our label
  [self tableViewShouldReloadData];
  [self updateStatusLabel];
  
  // We set the application icon to GNUMail_Full.tiff
  [NSApp setApplicationIconImage: [NSImage imageNamed: @"GNUMail_Full.tiff"]];

  // We finally force our window to upadte
  [[self window] update];
}


- (void) messagesWereReceived
{
  if ( initiatedFromTimer )
    {
      NSBeep();
    }
  else
    {
      // We verify if we need to inform the user that received mails have been
      // transferred to folders by some filters.
      if (filteredMessageCount > 0 &&
	  [[NSUserDefaults standardUserDefaults] integerForKey: @"SHOW_FILTER_PANEL"] == NSOnState )
	{
	  NSRunInformationalAlertPanel(_(@"Filtered messages..."),
				       _(@"%d messages have been transfered to the following folders by filters:\n%@"),
				       _(@"OK"),
				       NULL,
				       NULL,
				       filteredMessageCount,
				       [filteredMessageBoxes componentsJoinedByString: @"\n"]);
	}
    }
}


- (void) noNewMessagesOnServer: (NSString *) theServer
{
  NSRunInformationalAlertPanel(_(@"No New Messages..."),
			       _(@"There are no new messages on %@."),
			       _(@"OK"),
			       NULL,
			       NULL,
			       theServer);
}

- (void) threadWillExit
{
  // We first stop our animation timer
  [animation invalidate];
  DESTROY(animation);
  
  // FIXME - This causes a segfault...
  // We need to hide our 'wait panel', if it was shown!
  //if ( [[[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWWAITPANELS"] intValue] == NSOnState )
  //	{
  //	  [NSApp endModalSession: modalSession];
  //	  [panel close];
  //	  NSReleaseAlertPanel(panel);
  //          modalSession = 0;
  //	  panel = nil;
  //	}

  // We finally re-enable our Get button
  [get setEnabled: YES];

  // We release our lock
  NSLog(@"Unlocking the lock...");
  [lock unlock];

  // We restart our automatic fetching timer
  if ([[NSUserDefaults standardUserDefaults] integerForKey:@"RETRIEVEMETHOD"] == AUTOMATICALLY)
    {  
      [self startTimer: nil];
    }
}

//
//
//
- (void) startTimer: (id) sender
{
  int nbOfSecondsToWait;
  
  if ([[NSUserDefaults standardUserDefaults] integerForKey:@"RETRIEVEMINUTES"] <= 0)
    {
      nbOfSecondsToWait = 60;
    }
  else
    {
      nbOfSecondsToWait = [[NSUserDefaults standardUserDefaults] integerForKey:@"RETRIEVEMINUTES"] *  60;
    }
  
  
  timer =  [NSTimer scheduledTimerWithTimeInterval: nbOfSecondsToWait
		    target: self
		    selector: @selector(getNewMessages:)
		    userInfo: nil
		    repeats: YES];
  RETAIN(timer);
}


//
//
//
- (void) stopTimer: (id) sender
{
  if ( timer )
    {
      [timer invalidate];
    }
  
  DESTROY(timer);
}

//
// delegate methods
//

- (BOOL) windowShouldClose: (id) sender
{
  // We first verify if the animation timer is running, if it is,
  // we are currently sending a mail so we inform the user about this.
  if ( animation && [animation isValid] )
    {
      NSRunInformationalAlertPanel(_(@"Closing..."),
				   _(@"GNUMail.app is currently receiving E-Mails. Please wait."),
				   _(@"OK"), 
				   NULL, 
				   NULL);
      return NO;
    }

  return YES;
}

- (void) windowWillClose: (NSNotification *) not
{ 
  // We set our static reference to nil
  localInboxWindowController = nil;

  [self stopTimer: nil];

  while (! [lock tryLock] )
    {
      [[NSRunLoop currentRunLoop] runMode: NSDefaultRunLoopMode
                                  beforeDate: [NSDate distantFuture]];
    }
  [lock unlock];

  [super windowWillClose: not];
}


//
//
//
- (void) windowDidLoad
{
  NSPort *port1, *port2;

  [super windowDidLoad];

  // Now, let's create our real lock.
  lock = [[NSLock alloc] init];

  passwordCache = [[NSMutableDictionary alloc] init];
  filteredMessageBoxes = [[NSMutableArray alloc] init];

  modalSession = NULL;
  panel = nil;

  // We initialize our NSConnection object used to communicate between our thread used to
  // transfer the messages and the main one
  port1 = [[NSPort alloc] init];
  port2 = [[NSPort alloc] init];
  
  connection = [[NSConnection alloc] initWithReceivePort: port1
				     sendPort: port2];
  [connection setRootObject: self];

  ports = [NSArray arrayWithObjects: port2, port1, nil];
  
  RETAIN(ports);
  RELEASE(port1);
  RELEASE(port2);

  if ([[NSUserDefaults standardUserDefaults] integerForKey:@"RETRIEVEMETHOD"] == AUTOMATICALLY)
    {  
      [self startTimer: nil];
    }

  // We add our notifications for knowing if we must start or stop the timer
  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(startTimer:)
    name: ShouldStartLocalInboxTimer
    object: nil];

  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(stopTimer:)
    name: ShouldStopLocalInboxTimer
    object: nil];
  
  // We update our static reference to this object
  localInboxWindowController = self;
}

//
// access/mutation methods
//

- (LocalMailDelivery *) localMailDelivery
{
  return localMailDelivery;
}

- (void) setLocalMailDelivery: (LocalMailDelivery *) theObject
{
  RETAIN(theObject);
  RELEASE(localMailDelivery);
  localMailDelivery = theObject;
}


//
// class methods
//

+ (id) localInboxWindowController
{
  return localInboxWindowController;
}

@end



//
// private methods
//

@implementation LocalInboxWindowController (Private)

//
//
//
- (NSString *) _passwordForServerName: (NSString *) theName
			       prompt: (BOOL) aBOOL;
{
  NSDictionary *allPOP3Values;
  NSString *password;

  allPOP3Values = [[[NSUserDefaults standardUserDefaults] objectForKey:@"RECEIVING"] objectForKey: theName];

  // We verify in the user defaults
  password = [Utilities decryptPassword: [allPOP3Values objectForKey:@"PASSWORD"] withKey: theName];
  
  // We verify in our cache
  if (! password )
    {
      password = [passwordCache objectForKey: theName];
    }

  // If we must prompt for the password
  if (! password && aBOOL )
    {
      PasswordPanelController *theController; 
      int result;
      
      theController = [[PasswordPanelController alloc] initWithWindowNibName: @"PasswordPanel"];
      [[theController window] setTitle: theName];
      
      result = [NSApp runModalForWindow: [theController window]];
      
      // If the user has entered a password...
      if (result == NSRunStoppedResponse)
	{
	  password = [theController password];
	  
	  // Let's cache this password...
	  [passwordCache setObject: password
			 forKey: theName];
	}
      else
	{
	  password = nil;
	}
      
      RELEASE(theController);
    }
  
  return password;
}


//
//
//
- (void) _transferMessagesUsingPorts: (NSArray *) thePorts
{
  NSDictionary *allPOP3Values;
  NSArray *allPOP3Keys;
  POP3Store *aStore;
  POP3Folder *aFolder;

  NSNumber *serverTypeValue, *portValue, *retainPeriodValue;
  NSString *servername, *username, *password;
  int i;
  
  NSConnection *serverConnection;

  BOOL leaveOnServer;

  NSAutoreleasePool *threadpool;

  threadpool = [[NSAutoreleasePool alloc] init];
  
  // We first get our connection to our main thread
  serverConnection = [[NSConnection alloc] initWithReceivePort: [thePorts objectAtIndex: 0]
					   sendPort: [thePorts objectAtIndex: 1]];

  // We re-initialize our ivars
  transferredMessageCount = filteredMessageCount = 0;
  
  // We first remove all elements from our array of mailbox names
  [filteredMessageBoxes removeAllObjects];

  allPOP3Keys = [[[NSUserDefaults standardUserDefaults] objectForKey:@"RECEIVING"] allKeys];
  
  for (i = 0; i < [allPOP3Keys count]; i++)
    {
      // We get our servername
      servername = [allPOP3Keys objectAtIndex: i];
      
      allPOP3Values = [[[NSUserDefaults standardUserDefaults] objectForKey: @"RECEIVING"] objectForKey: servername];
      
      serverTypeValue = [allPOP3Values objectForKey: @"SERVERTYPE"];
      portValue = [allPOP3Values objectForKey: @"PORT"];
      retainPeriodValue = [allPOP3Values objectForKey: @"RETAINPERIOD"];
      
      // We use the default POP3 port if it's not defined.
      if ( !portValue )
	{
	  portValue = [NSNumber numberWithInt: 110];
	}

      // We use the default retain period (365 days, if it's not defined)
      if ( !retainPeriodValue)
	{
	  retainPeriodValue = [NSNumber numberWithInt: 365];
	}

      if (!serverTypeValue || [serverTypeValue intValue] == POP3)
	{
	  // We get our username
	  username = [allPOP3Values objectForKey: @"USERNAME"];

	  // We get our password. We prompt the user if we need to.
	  password = [self _passwordForServerName: servername
			   prompt: NO];

	  // 
	  if ( password )
	    {
	      leaveOnServer = ([[allPOP3Values objectForKey: @"LEAVEONSERVER"] intValue] == NSOnState ? YES : NO);
	      
	      aStore = (POP3Store *) [[POP3Store alloc] initWithName: servername
							port: [portValue intValue]];
	      
	      // We verify that we got a valid connection with the POP3 server
	      if (! aStore )
		{
		  [(id)[serverConnection rootProxy] communicationErrorOccuredOnServer: servername];
		}
	      else
		{
		  // We verify if we should use APOP mode
		  if ( [allPOP3Values objectForKey: @"USEAPOP"] )
		    {
		      [aStore setUseAPOP: ([[allPOP3Values objectForKey: @"USEAPOP"] intValue] == NSOnState ? YES : NO) ];
		    }

		  if ( [aStore authenticateWithUsername: username
			       password: password] )
		    {
		      POP3CacheManager *aPOP3CacheManager;		      
		      
		      // We get our POP3 folder
		      aFolder = (POP3Folder *) [aStore folderForName: @"Inbox" prefetch: NO];
		      [aFolder setLeaveOnServer: leaveOnServer];
		      [aFolder setRetainPeriod: [retainPeriodValue intValue]];

		      // We get our POP3 cache
		      aPOP3CacheManager = [POP3CacheManager pop3CacheObjectsFromDiskWithPath:
							      [NSString stringWithFormat: @"%@/POP3Cache_%@",
									GNUMailUserLibraryPath(),
									servername]]; 
		      [aFolder setPOP3CacheManager: aPOP3CacheManager];
		      
		      transferredMessageCount += [self _transferMessagesFromPOP3Folder: aFolder
						       connection: serverConnection];
		      		      
		      if (transferredMessageCount == 0 &&
			  !initiatedFromTimer)
			{
			  [(id)[serverConnection rootProxy] noNewMessagesOnServer: servername];
			}
		      
		      // We now close our folder
		      [aFolder close];
		      
		    } // if ( [aStore authenticateWithUsername... ] )
		  else
		    {
		      // We remove the password from the cache
		      [passwordCache removeObjectForKey: servername];
		      
		      // We inform our main thread the the password isn't valid for this server
		      [(id)[serverConnection rootProxy] invalidPasswordOnServer: servername
			   username: username];
		    }
		  
		  // We now close our store
		  [aStore close];
		  RELEASE(aStore);

		} // else of if ( !aStore ) ...

	    } // if ( password ) ...
	  
	} // if (!serverTypeValue || ... )
      
    } // for (i = 0; ... )


  //
  // We now transfer the emails from the local mail spool file, if the user wants too.
  //
  if ( [[NSUserDefaults standardUserDefaults] integerForKey: @"TRANSFERFROMMAILSPOOLFILE"] == NSOnState )
    {
      NSAutoreleasePool *pool, *inner_pool;
      NSArray *anArray;

      if ( ! [self localMailDelivery] )
	{
	  LocalMailDelivery *lmd;
	  
	  lmd = [[LocalMailDelivery alloc] init];
	  
	  [self setLocalMailDelivery: lmd];
	  
	  RELEASE(lmd);
	}
      
      pool = [[NSAutoreleasePool alloc] init];
      inner_pool = nil;

      anArray = [[self localMailDelivery] messagesFromMailspoolFile];
      transferredMessageCount += [anArray count];
      
      for (i = 0; i < [anArray count]; i++)
	{
	  if ( (i % 3) == 0 )
	    {
	      TEST_RELEASE(inner_pool);
	      inner_pool = [[NSAutoreleasePool alloc] init];
	    }
	  
	  [self _matchFilterRuleFromRawSource: [anArray objectAtIndex: i]
		connection: serverConnection];
	}
      
      TEST_RELEASE(inner_pool);
      RELEASE(pool);
    }
  
  
  // If we have received some messages, and we are done fetching them, let's
  // inform our main thread from this situation
  if ( transferredMessageCount > 0 )
    {
      [(id)[serverConnection rootProxy] messagesWereReceived];
    }
    
  // We inform the main thread that this thread is about to exit
  [(id)[serverConnection rootProxy] threadWillExit];

  // We release our server connection object
  RELEASE(serverConnection);
  
  // We release our autorelease pool used in that thread and we exit that thread!
  RELEASE(threadpool);
  
  NSLog(@"Exiting the thread...");
  [NSThread exit];
}


//
// This method is used to transfer message from a POP3 folder
// to a local folder.
//
// It verifies for various filters to be sure to transfer to the
// right folder.
//
- (int) _transferMessagesFromPOP3Folder: (POP3Folder *) theFolder
			     connection: (NSConnection *) theConnection
{
  NSData *aData;
  int i, count;

  if (! theFolder )
    {
      return 0;
    }
  
  transferredMessageCount = 0;
  count = [theFolder count];

  // We get all messages..
  for (i = 1; i <= count; i++)
    {
      aData = [theFolder prefetchMessageAtIndex: i];
      // If it's a new message (the string isn't nil when it's a new message)
      if ( aData )
	{
	  [self _matchFilterRuleFromRawSource: aData
		connection: theConnection];
	  transferredMessageCount++;
	}
    } // for 

  // We mark the messages as deleted if we need to
  if (! [theFolder leaveOnServer] )
    {
      for (i = 1; i <= count; i++)
	{
	  [theFolder deleteMessageAtIndex: i];
	}
    }

  return transferredMessageCount;
}


//
//
//
- (void) _matchFilterRuleFromRawSource: (NSData *) theRawSource
			    connection: (NSConnection *) theConnection
{
  NSString *aFolderName;
  NSRange aRange;
  
  aRange = [theRawSource rangeOfCString: "\n\n"];
  aFolderName = [[self folder] name];

  if ( aRange.length )
    {
      Message *aMessage;
      Filter *aFilter;
      
      aMessage = [[Message alloc] initWithHeadersFromData: [theRawSource subdataToIndex: aRange.location + 1] ];
      
      aFilter = [filterManager matchedFilterForMessage: aMessage];
      
      // If we have found a filter that matches our message
      if ( aFilter )
	{
	  // We verify if the operation is TRANSFER_TO_FOLDER and if folder DIFFERENT from Inbox
	  if ( [aFilter action] == TRANSFER_TO_FOLDER &&
	       ! [[aFilter actionFolderName] isEqualToString:
					       [[NSUserDefaults standardUserDefaults] 
						 objectForKey: @"INBOXFOLDERNAME"]] )
	    {
	      // We verify that our folder hasn't been deleted!
	      // If it was deleted, we transfer it to the Inbox Folder
	      if (! [self _folderNameExists: [aFilter actionFolderName]
			  forStore: [(GNUMail *)[NSApp delegate] localStore]] )
		{
		  aFolderName = [[self folder] name];
		}
	      // No it hasn't so let's transfer the message to that specific folder instead of Inbox.
	      else
		{
		  aFolderName = [aFilter actionFolderName];
		  
		  filteredMessageCount += 1;

		  if ( ![filteredMessageBoxes containsObject: [aFilter actionFolderName]] )
		    {
		      [filteredMessageBoxes addObject: [aFilter actionFolderName]];
		    }
		}
	    }
	  // We verify if the operation is DELETE. If so, we transfer the message to the trash folder.
	  else if ( [aFilter action] == DELETE )
	    {      
	      aFolderName = [[NSUserDefaults standardUserDefaults] objectForKey: @"TRASHFOLDERNAME"];
	    }
	  // We have an UKKNOWN action. We simply add the message to our Inbox folder.
	  else
	    {
	      aFolderName = [[self folder] name];
	    }
	}
      // We haven't found a filter, let's add it to our Inbox Folder
      else
	{
	  aFolderName = [[self folder] name];
	}
      

      [(id)[theConnection rootProxy] messageWasReceived: theRawSource
	   folderName: aFolderName];
      
      RELEASE(aMessage);
      
    } // if ( aRange.length )
}

//
// This method is used to transfer a message (from raw source) to
// the specified folder.
//
// If the folder isn't opened, it is opened w/o parsing.
// If it was opened, we refresh the tableView that shows
// the messages to the user.
//
- (void) _transferMessageFromRawSource: (NSData *) theRawSource
		      toFolderWithName: (NSString *) theFolderName
{
  NSMutableData *aMutableData;
  LocalFolder *localFolder;
  LocalStore *localStore;
  id aWindow = nil;
		      
  // We get our local store
  localStore = [(GNUMail *)[NSApp delegate] localStore];
  
  // We get a reference to our folder, w/o parsing it if it's not already open.
  localFolder = (LocalFolder *)[localStore folderForName: theFolderName];
  
  // Our folder is opened, let's get it from the list of opened windows.
  if (! localFolder )
    {
      aWindow = [Utilities windowForFolderName: theFolderName]; 
      localFolder = (LocalFolder *)[[aWindow windowController] folder];
    } 
  
  // We transfer the new message
  aMutableData = [[NSMutableData alloc] init];
  [aMutableData appendBytes: "From -\n"
		length: 7];
  [aMutableData appendData: theRawSource];
  [localFolder appendMessageFromRawSource: aMutableData];
  RELEASE(aMutableData);
  
  // If the folder was open, we refresh its view. Otherwise, we just close it.
  if ( aWindow )
    {
      [[aWindow delegate] tableViewShouldReloadData];
    }
  else
    {
      [localFolder close];
    }
}


//
// This method is used to verify if a folder with name "theFolderName"
// exists in the specified local store.
//
- (BOOL) _folderNameExists: (NSString *) theFolderName
                  forStore: (LocalStore *) theLocalStore
{
  NSEnumerator *anEnumerator;
  NSString *aString;

  anEnumerator = [theLocalStore folderEnumerator];

  while ( (aString = [anEnumerator nextObject]) )
    {
      if ( [aString isEqualToString: theFolderName] )
	{
	  return YES;
	}
    }
  
  return NO;
}

@end
