/*
**  FilterManager.m
**
**  Copyright (c) 2001
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import <AppKit/AppKit.h>

#import "GNUMailConstants.h"
#import "FilterManager.h"
#import "Filter.h"
#import "NSRegEx.h"
#import "NSRegExRange.h"
#import "NSStringExtensions.h"
#import "Utilities.h"

#import <Pantomime/InternetAddress.h>
#import <Pantomime/Message.h>

NSString *PathToFilters()
{
  return [NSString stringWithFormat: @"%@/%@",
		   GNUMailUserLibraryPath(), @"Filters"];
}

@implementation FilterManager

- (id) init
{
  self = [super init];

  [self setFilters: [NSMutableArray array]];

  return self;
}

- (void) dealloc
{
  RELEASE(filters);

  [super dealloc];
}


- (BOOL) synchronize
{
  return [NSArchiver archiveRootObject: self 
		     toFile: PathToFilters()];
}

//
// NSCoding protocol
//

- (void) encodeWithCoder: (NSCoder *) theCoder
{
  [theCoder encodeObject: [self filters]];
}

- (id) initWithCoder: (NSCoder *) theCoder
{
  self = [super init];

  [self setFilters: [theCoder decodeObject]];

  return self;
}


//
// access/mutation methods
//

- (Filter *) filterAtIndex: (int) theIndex
{
  return [filters objectAtIndex: theIndex];
}

- (void) addFilter: (Filter *) theFilter
{
  [filters addObject: theFilter];
}

- (void) addFilter: (Filter *) theFilter
	   atIndex: (int) theIndex
{
  [filters insertObject: theFilter
	   atIndex: theIndex];
}

- (void) removeFilter: (Filter *) theFilter;
{
  [filters removeObject: theFilter];
}

- (NSArray *) filters
{
  return filters;
}

- (void) setFilters: (NSArray *) theFilters
{
  if ( theFilters )
    {
      NSMutableArray *newFilters;

      newFilters = [[NSMutableArray alloc] initWithArray: theFilters];
      RELEASE(filters);
      filters = newFilters;
    }
  else
    {
      RELEASE(filters);
      filters = nil;
    }
}


//
// This method returns the *first* filter that matches
// for the specified message. If no filter matches, nil
// is returned.
//
- (Filter *) matchedFilterForMessage: (Message *) theMessage
{
  int i;

  if (! theMessage )
    {
      return nil;
    }
  
  
  for (i = 0; i < [filters count]; i++)
    {
      Filter *aFilter;

      aFilter = [filters objectAtIndex: i];

      if ( [aFilter isActive] )
	{
	  NSArray *anArray;
	  int j;
	  
	  anArray = [self _stringsFromMessage: theMessage
			  filter: aFilter];

	  for (j = 0; j < [anArray count]; j++)
	    {
	      NSString *aString;
	      
	      aString = [anArray objectAtIndex: j];

	      if ( [self _matchString: aString
			 operation: [aFilter criteriaFindOperation]
			 criteria: [aFilter criteriaString]] )
		{
		  return aFilter;
		}
	    }
	}
    }
  

  return nil;
}

- (NSColor *) colorForMessage: (Message *) theMessage
{
  int i;

  if (! theMessage )
    {
      return nil;
    }
  
  
  for (i = 0; i < [filters count]; i++)
    {
      Filter *aFilter;
      
      aFilter = [filters objectAtIndex: i];
      
      if ( [aFilter action] == SET_COLOR  && [aFilter isActive])
	{
	  NSArray *anArray;
	  int j;
	  
	  anArray = [self _stringsFromMessage: theMessage
			  filter: aFilter];

	  for (j = 0; j < [anArray count]; j++)
	    {
	      NSString *aString;

	      aString = [anArray objectAtIndex: j];

	      if ( [self _matchString: aString
			 operation: [aFilter criteriaFindOperation]
			 criteria: [aFilter criteriaString]] )
		{
		  return [aFilter actionColor];
		}
	    }
	}
    }
  

  return nil;
}

//
// class methods
//

+ (id) filtersFromDisk
{
  return [NSUnarchiver unarchiveObjectWithFile: PathToFilters()];
}

@end

//
// private methods
//

@implementation FilterManager (Private)

- (BOOL) _matchString: (NSString *) theString
            operation: (int) theOperation
             criteria: (NSString *) theCriteria
{
  // Variables used in this method
  NSArray *anArray;
  NSRange aRange;

  // We must be sure to have a valid criteria.
  if ( !theCriteria || [theCriteria length] == 0 )
    {
      return NO;
    }
  
  switch ( theOperation )
    {
    case CONTAINS:
      aRange = [theString rangeOfString: theCriteria
			  options: NSCaseInsensitiveSearch];
      
      if ( aRange.length )
	{
	  return YES;
	}
      break;

    case IS_EQUAL:
      if ( [theString caseInsensitiveCompare: theCriteria] == NSOrderedSame )
	{
	  return YES;
	}
      break;

    case HAS_PREFIX:
      if ( [[theString lowercaseString] hasPrefix: [theCriteria lowercaseString]] )
	{
	  return YES;
	}
      break;

    case HAS_SUFFIX:
      if ( [[theString lowercaseString] hasSuffix: [theCriteria lowercaseString]] )
	{
	  return YES;
	}
      break;
      
    case MATCH_REGEXP:
      anArray = [NSRegEx matchString: theString
			 withPattern : theCriteria
			 isCaseSensitive: YES];

      if ( [anArray count] > 0 )
	{
	  return YES;
	}
      break;

    default:
      break;
    }

  return NO;
}

- (NSArray *) _stringsFromMessage: (Message *) theMessage
                           filter: (Filter *) theFilter
{
  NSMutableArray *aMutableArray;
  NSArray *allRecipients;
  NSString *aString;
  int i, theSource;

  aMutableArray = [[NSMutableArray alloc] init];

  theSource = [theFilter criteriaSource];

  switch ( theSource )
    {
    case TO:
      allRecipients = [theMessage recipients];
      
      for (i = 0; i < [allRecipients count]; i++)
	{
	  InternetAddress *anInternetAddress;

	  anInternetAddress = [allRecipients objectAtIndex: i];

	  if ( [anInternetAddress type] == TO )
	    {
	      aString = [anInternetAddress unicodeStringValue];
	      
	      if ( aString )
		{
		  [aMutableArray addObject: aString];
		}
	    }

	}
      break;

    case CC:
      allRecipients = [theMessage recipients];
      
      for (i = 0; i < [allRecipients count]; i++)
	{
	  InternetAddress *anInternetAddress;

	  anInternetAddress = [allRecipients objectAtIndex: i];

	  if ( [anInternetAddress type] == CC )
	    {
	      aString = [anInternetAddress unicodeStringValue];
	      
	      if ( aString )
		{
		  [aMutableArray addObject: aString];
		}
	    }

	}
      break;

    case TO_OR_CC:
      allRecipients = [theMessage recipients];
      
      for (i = 0; i < [allRecipients count]; i++)
	{
	  InternetAddress *anInternetAddress;

	  anInternetAddress = [allRecipients objectAtIndex: i];

	  if ( [anInternetAddress type] == TO ||
	       [anInternetAddress type] == CC )
	    {
	      aString = [anInternetAddress unicodeStringValue];
	      
	      if ( aString )
		{
		  [aMutableArray addObject: aString];
		}
	    }

	}
      break;
      
    case SUBJECT:
      aString = [theMessage subject];

      if ( aString )
	{
	  [aMutableArray addObject: aString];
	}
      break;
      
    case FROM:
      aString = [[theMessage from] unicodeStringValue];

      if ( aString )
	{
	  [aMutableArray addObject: aString];
	}
      break;
      
    case EXPERT:
      if ( [theFilter criteriaHeaders] && [[theFilter criteriaHeaders] count] > 0)
	{
	  for (i = 0; i < [[theFilter criteriaHeaders] count]; i++)
	    {
	      aString = [theMessage headerValueForName: [[theFilter criteriaHeaders] objectAtIndex: i]];
	      
	      if ( aString )
		{
		  [aMutableArray addObject: aString];
		}
	    }
	}
      else
	{
	  // We do nothing.. so we won't have any matches.
	}
      
      break;

    default:
      break;
    }

  return AUTORELEASE(aMutableArray);
}

@end
