/*
**  MIMEViewController.m
**
**  Copyright (c) 2001
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import "MIMEViewController.h"

#import "GNUMailConstants.h"
#import "MailcapWindowController.h"
#import "MimeTypeEditorWindowController.h"
#import "MimeType.h"
#import "MimeTypeManager.h"

#ifndef MACOSX
#import "MIMEView.h"
#endif

static MIMEViewController *singleInstance = nil;

@implementation MIMEViewController

- (id) initWithNibName: (NSString *) theName
{
  BOOL isDir;

  self = [super init];

  if ( [[NSFileManager defaultManager] fileExistsAtPath: PathToMimeTypes()
				       isDirectory: &isDir] &&
       ! isDir )
    {
      mimeTypeManager = [MimeTypeManager mimeTypesFromDisk];
      RETAIN(mimeTypeManager);
    }
  else
    {
      // create a new address book and save it to disk
      mimeTypeManager = [[MimeTypeManager alloc] init];
      [NSArchiver archiveRootObject: mimeTypeManager toFile: PathToMimeTypes()];
    }

#ifdef MACOSX
  
  if (! [NSBundle loadNibNamed: theName
		  owner: self] )
    {
      NSLog(@"Fatal error occured while loading the MIMEView nib file");
      AUTORELEASE(self);
      return nil;
    }
  
  RETAIN(view);
  
  mimeTypesColumn = [tableView tableColumnWithIdentifier: @"mime-type"];
  fileExtensionsColumn = [tableView tableColumnWithIdentifier: @"file-extensions"];

  [tableView setTarget: self];
  [tableView setDoubleAction: @selector(edit:)];


#else
  // We link our view
  view = [[MIMEView alloc] initWithParent: self];
  [view layoutView];

  // We link our outlets
  tableView = [view tableView];
  mimeTypesColumn = [view mimeTypesColumn];
  fileExtensionsColumn = [view fileExtensionsColumn];
  add = [view add];
  delete = [view delete];
  edit = [view edit];
  importFromMailcap = [view importFromMailcap];
#endif
    
  // We get our defaults for this panel
  [self initializeFromDefaults];

  return self;
}


- (void) dealloc
{
  singleInstance = nil;

  RELEASE(mimeTypeManager);
  RELEASE(view);

  [super dealloc];
}

//
// Data Source methods
//

- (int) numberOfRowsInTableView: (NSTableView *)aTableView
{
  return [[mimeTypeManager mimeTypes] count];
}

- (id) tableView: (NSTableView *)aTableView
objectValueForTableColumn: (NSTableColumn *)aTableColumn 
             row: (int)rowIndex
{
  MimeType *aMimeType = [[mimeTypeManager mimeTypes] objectAtIndex: rowIndex];
 
  if (aTableColumn == mimeTypesColumn)
    {
      return [aMimeType mimeType];
    }
  else
    {
      return [aMimeType stringValueOfFileExtensions];
    }
}


//
// action methods
//

- (IBAction) add : (id) sender
{
  MimeTypeEditorWindowController *mimeTypeEditorWindowController;
  int result;

  mimeTypeEditorWindowController = [[MimeTypeEditorWindowController alloc] 
				     initWithWindowNibName: @"MimeTypeEditorWindow"];
  [mimeTypeEditorWindowController setMimeTypeManager: mimeTypeManager];
  [mimeTypeEditorWindowController setMimeType: nil];

  result = [NSApp runModalForWindow: [mimeTypeEditorWindowController window]];
  
  if (result == NSRunStoppedResponse)
    {
      [tableView reloadData];
    }
}


//
//
//
- (IBAction) delete: (id) sender
{
  int choice;

  if ([tableView numberOfSelectedRows] == 0 || [tableView numberOfSelectedRows] > 1)
    {
      NSBeep();
      return;
    }
  
  choice = NSRunAlertPanel(_(@"Delete..."),
			   _(@"Are you sure you want to delete this Mime-Type entry?"),
			   _(@"Cancel"), // default
			   _(@"Yes"),    // alternate
			   nil);

  // If we delete it...
  if (choice == NSAlertAlternateReturn)
    {
      [mimeTypeManager removeMimeType: [mimeTypeManager mimeTypeAtIndex:[tableView selectedRow]]];
      [tableView reloadData];
    }
}


//
//
//
- (IBAction) edit: (id) sender

{
  MimeTypeEditorWindowController *mimeTypeEditorWindowController;
  MimeType *aMimeType;
  int result;

  if ([tableView numberOfSelectedRows] == 0 || [tableView numberOfSelectedRows] > 1)
    {
      NSBeep();
      return;
    }

  aMimeType = [mimeTypeManager mimeTypeAtIndex: [tableView selectedRow]];
			      
  mimeTypeEditorWindowController = [[MimeTypeEditorWindowController alloc] 
				     initWithWindowNibName: @"MimeTypeEditorWindow"];
  [mimeTypeEditorWindowController setMimeTypeManager: mimeTypeManager];
  [mimeTypeEditorWindowController setMimeType: aMimeType];


  result = [NSApp runModalForWindow: [mimeTypeEditorWindowController window]];

  if (result == NSRunStoppedResponse)
    {
      NSLog(@"We update...");
      [tableView reloadData];
    }
}


//
//
//
- (IBAction) importFromMailcap: (id) sender
{
  NSArray *filesToOpen;
  NSOpenPanel *anOpenPanel;
  int result, i;
  
  anOpenPanel = [NSOpenPanel openPanel];
  [anOpenPanel setAllowsMultipleSelection:NO];
  result = [anOpenPanel runModalForDirectory:NSHomeDirectory() file:nil types:nil];
  
  if (result == NSOKButton)
    {
      NSArray *chosenMimeTypes;
      
      filesToOpen = [anOpenPanel filenames];
      
      if ( [filesToOpen count] > 0 )
        {
          MailcapWindowController *mailcapWindowController;

          mailcapWindowController = [[MailcapWindowController alloc] initWithWindowNibName: @"MailcapWindow"];
	  [mailcapWindowController setFilename: [filesToOpen objectAtIndex: 0]];
	  
	  result = [NSApp runModalForWindow: [mailcapWindowController window]];
	  
	  if (result == NSRunStoppedResponse)
	    {
	      chosenMimeTypes = [mailcapWindowController getChoices];
	      
	      for (i = 0; i < [chosenMimeTypes count]; i++ )
		{
		  MimeType *aMimeType;
		  aMimeType = [chosenMimeTypes objectAtIndex: i];
		  [aMimeType setIcon: [NSImage imageNamed: @"common_Unknown.tiff"]];
		  [mimeTypeManager addMimeType: aMimeType];
		}
	      
	      [tableView reloadData];
	    }
	  
	  RELEASE(mailcapWindowController);
        }
    }


}

//
// access methods
//

- (NSImage *) image
{
  NSBundle *aBundle;
  
  aBundle = [NSBundle bundleForClass: [self class]];
  
  return AUTORELEASE([[NSImage alloc] initWithContentsOfFile:
					[aBundle pathForResource: @"mime" ofType: @"tiff"]]);
}

- (NSString *) name
{
  return _(@"MIME");
}

- (NSView *) view
{
  return view;
}


//
//
//

- (BOOL) hasChangesPending
{
  return YES;
}


- (void) initializeFromDefaults
{

}


- (void) saveChanges
{
  [mimeTypeManager synchronize];

  [[NSNotificationCenter defaultCenter]
    postNotificationName: MimeTypesHaveChanged
    object: nil
    userInfo: nil];
}

//
// class methods
//
+ (id) singleInstance
{
  if (! singleInstance)
    {
      singleInstance = [[MIMEViewController alloc] initWithNibName: @"MIMEView"];
    }

  return singleInstance;
}

@end
