#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it 
# and/or modify it under the terms of the GNU General Public 
# License as published by the Free Software Foundation; either 
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be 
# useful, but WITHOUT ANY WARRANTY; without even the implied 
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000, 2001 Free Software Foundation
#
#
# FILE:
# GObjects.py
#
# DESCRIPTION:
# Base class for GNUe objects which can be represented as XML
#
# NOTES:
#
import sys

try:
  from xml.sax import saxutils
except ImportError:
  print """
   This GNUe tool requires PyXML to be installed.
   Please download and install PyXML from:

      http://pyxml.sourceforge.net/

"""
  sys.exit()

import GDebug
import string
import types
from GParserHelpers import GContent
import GTypecast

#
# Class GObj
#
# Base class for GNUe objects which can be represented as XML
#
class GObj:
  def __init__(self, parent=None, type='GObj'):
    self._type = type
    self._parent = parent
    self._children = []
    self._attributes = {}
    self._inits = []
    if parent :
      parent.addChild(self)

  #
  # initialize
  #
  # This will be the base of the new initialization system for gnue
  # each childs initialization function must call GObj.initialize()
  #
  #def initialize(self):
  #  GDebug.printMesg(6,"Initializing %s" % self.getObjectType())
  #  if self._children:
  #    for child in self._children:
  #      if isinstance(child, GObj):
  #        child.initialize()

  #
  # phaseInit
  #
  def phaseInit(self, iterations=5):
    for phase in range(iterations):
      self._phaseInit(phase)
#      phase = phase + 1

  def _phaseInit(self,phase):
    if (len(self._inits) > phase) and (self._inits[phase] != None):
      GDebug.printMesg(6,"%s: Init Phase %s" % (self.getObjectType(), phase+1))
      self._inits[phase]()

    if self._children:
      for child in self._children:
        if isinstance(child, GObj):
          child._phaseInit(phase)


  #
  # getObjectType
  #
  # Return the _type of this object
  #
  def getObjectType(self):
    return self._type

  # This function is called after the parsers have completely
  # constructed. All children should be in place and
  # attributes and content should be set at this point.
  # Return the number of phaseInit iterations your object will
  # need.
  #
  # NOTE: Do not initialize datasources, etc at this point.
  #       This is only so content can be set, etc, after
  #       loading from XML.
  #
  def _buildObject(self):
    return len(self._inits)

  def getChildrenAsContent(self):
    content = ""
    for child in self._children:
      if isinstance(child, GContent):
        content = content + child._content
    return content

  def showTree(self, indent=0): 
    print ' '*indent + `self._type` + `self`
    for child in self._children: 
       child.showTree(indent + 2)

  #
  # addChild
  #
  # Adds an object to an instances list of children
  #
  def addChild(self, child):
    self._children.append(child)
  
  def toXML(self):
    xml_outer = string.lower(self._type[2:])
    r = "<" + xml_outer + ">\n  <options>\n"
    for k in self.__dict__.keys():
      # skip keys beginning with _
      if k[0] == "_":
        continue
      val = self.__dict__[k]
      if isinstance(val, _types.StringType):
        str = val
      elif isinstance(val, _types.IntType) or isinstance(val, _types.FloatType):
        str = repr(val)
      else:
        continue
      r = r + "    <" + k + ">" + str + "</" + k + ">\n"
    r = r + "  </options>\n</" + xml_outer + ">\n"
    return r

  #
  # dumpXML
  #
  # Dumps an XML representation of the object
  #
  def dumpXML(self, lookupDict, treeDump=None, gap="  "):
    xmlEntity = string.lower(string.replace(self._type[2:],'_','-'))
    xmlString = "%s<%s" % (gap[:-2],xmlEntity)

    indent = len(xmlString)
    pos = indent
    for attribute in self.__dict__.keys():
      # skip keys beginning with _
      if attribute[0] == "_":
        continue
      val = self.__dict__[attribute]
      if lookupDict[xmlEntity].has_key('Attributes') and \
         lookupDict[xmlEntity]['Attributes'].has_key(attribute):
        if val != None and \
           (not lookupDict[xmlEntity]['Attributes'][attribute].has_key('Default') or \
            (lookupDict[xmlEntity]['Attributes'][attribute]['Default']) != (val)):
          typecast = lookupDict[xmlEntity]['Attributes'][attribute]['Typecast']
          if typecast == GTypecast.boolean \
             and val == 1:
            addl = ' %s=""' % (attribute)
          elif typecast == GTypecast.names:
            addl = ' %s="%s"' % \
                (attribute, string.join(val,','))
          else:
            addl = ' %s="%s"' % (attribute, saxutils.escape('%s' % val))
          if len(addl) + pos > 78:
            xmlString = xmlString + "\n" + " " * indent + addl
            pos = indent
          else:
            xmlString = xmlString + addl
            pos = pos + len(addl)

    if len(self._children):
      hasContent = 0
      for child in self._children:
        hasContent = hasContent or isinstance(child,GContent)
      if hasContent:
        xmlString = xmlString + ">"
      else:
        xmlString = xmlString + ">\n"

      if treeDump:
        for child in self._children:
          xmlString = xmlString + child.dumpXML(lookupDict, 1,gap+"  ")

      if hasContent:
        xmlString = xmlString + "</%s>\n" % (xmlEntity)
      else:
        xmlString = xmlString + "%s</%s>\n" % (gap[:-2], xmlEntity)
    else:
      xmlString = xmlString + "/>\n"
    return xmlString


  def walk(self, function):
    function(self)
    if self._children:
      for child in self._children:
        if isinstance(child, GObj):
          child.walk(function)

  #
  # findParentOfType
  #
  # Moves upward though the parents of an object till
  # it finds the parent of the specified type
  #
  def findParentOfType(self,type):
    parentObject = self
    while (parentObject._parent != None and parentObject.getObjectType() != type):
      parentObject = parentObject._parent

    if parentObject.getObjectType() == type: 
      return parentObject
    else: 
      return None

  #
  # getDescription
  #
  # Return a useful description of this object
  # Used by designer clients
  #
  def getDescription(self): 
    if hasattr(self,'_description'): 
      return self._description
    elif hasattr(self,'name'): 
      return self.name + " (%s)" % self.getObjectType()[2:]
    else: 
      return self._type[2:] + " (%s)" % self.getObjectType()[2:]

