#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000, 2001 Free Software Foundation
#
# FILE:
# GDataSource.py
#
# DESCRIPTION:
# Class that implements a provider-independent DataSource object
#
# NOTES:
#
# HISTORY:
#

import GDebug
import GDataObjects
import GObjects
import sys, string
import GConnections
import GTypecast
import GConditions


########################################################################
#
# Class that handles DataSources.  This is a subclass of GObj, which
# means this class can be created from XML markup and stored in an
# Object tree (e.g., a Forms tree).
#
########################################################################

class GDataSource(GObjects.GObj):

  def __init__(self, parent=None, type="GDataSource"):
    GObjects.GObj.__init__(self, parent, type)
    self.type = ""
    self.database = None
    self._connectionManager = None
    self._dataObject = None
    self._connectionComment = ""
    self._fieldReferences = {}
    self._unboundFieldReferences = {}
    self._defaultValues = {}

  #
  # This method should be called after the object is created
  # but before any other methods are called
  #
  def setConnectionManager(self, connectionManager):
    self._connectionManager = connectionManager

  def initialize(self):

    if not self.database:
      # We are a connectionless datasource (virtual?)
      # We have to bind to something, so bind to empty driver
      from gnue.common.dbdrivers._empty import DBdriver
      GDebug.printMesg (7, 'Using empty data driver')
      self._dataObject = DBdriver.supportedDataObjects['object']()

    elif self._connectionManager:
      self.database = string.lower(self.database)
      # This will throw a GConnections.NotFoundError if an unknown
      # connection name is specified.  The calling method should
      # catch this exception and handle it properly (exit w/message)
      self._dataObject = \
         self._connectionManager.getDataObject(self.database, self.type)
      GDebug.printMesg (7, "GDataSource.py bound to %s " % self._dataObject)

    self._dataObject._fieldReferences = self._fieldReferences
    self._dataObject._unboundFieldReferences = self._unboundFieldReferences
    self._dataObject._defaultValues = self._defaultValues
    for child in self._children:
      if isinstance(child, GConditions.GCondition):
        self._dataObject._staticCondition = child
        break


    # Copy all attributes from XML to the dataObject
    tagAttributes = getXMLelements()['datasource']['Attributes']
    for attribute in tagAttributes.keys():
      if hasattr(self, attribute):
        self._dataObject.__dict__[attribute] = self.__dict__[attribute]


  def connect(self):
    if self.database != None:
      self._connectionManager.\
           requestConnection(self._dataObject, self.database)

  def getDataObject(self):
    return self._dataObject

  def referenceField(self, field, defaultValue=None):
    GDebug.printMesg(7,'Field %s implicitly referenced' % field)
    self._fieldReferences[field] = ""

    if defaultValue != None:
      self._defaultValues[field] = defaultValue

  def referenceUnboundField(self, field, defaultValue=None):
    GDebug.printMesg(7,'Field %s implicitly referenced' % field)
    self._unboundFieldReferences[field] = 1

    if defaultValue != None:
      self._defaultValues[field] = defaultValue


  #
  # The following are a simple wrapper arround the datasource's dataobject
  # to hide the dataobject from the app programmer
  #
  def hasMaster(self):
    return self._dataObject != None and self._dataObject.hasMaster()

  def createResultSet(self, conditions={}, readOnly=0):
      return self._dataObject.createResultSet(conditions,readOnly)

  def addDetailDataObject(self, dataObject, handler=None):
      self._dataObject.addDetailDataObject(dataObject, handler)

  def createEmptyResultSet(self, readOnly=0):
      return self._dataObject.createEmptyResultSet(readOnly)

######
#
#
#
######
class GSql(GObjects.GObj):
  def __init__(self, parent=None):
     GObjects.GObj(self, parent, type="GDSql")

######
#
# Used by client GParsers to automatically pull supported xml tags
#
######

#
# Return any XML elements associated with
# GDataSources.  Bases is a dictionary of tags
# whose values are update dictionaries.
# For example: bases={'datasource': {'BaseClass':myDataSource}}
# sets xmlElements['datasource']['BaseClass'] = myDataSource
#
def getXMLelements(updates={}):

  xmlElements = {
      'datasource': {
         'BaseClass': GDataSource,
         'Attributes': {
            'name':        {
               'Required': 1,
               'Unique':   1,
               'Typecast': GTypecast.name },
            'type':        {
               'Typecast': GTypecast.name,
               'Default':  "object" },
            'database':    {
               'Typecast': GTypecast.name },
            'table':       {
               'Typecast': GTypecast.name },
            'cache':       {
               'Typecast': GTypecast.whole,
               'Default':  5 },
            'prequery':    {
               'Typecast': GTypecast.boolean,
               'Default':  0 },
            'order_by':    {
               'Typecast': GTypecast.text },
            'master':      {
               'Typecast': GTypecast.name },
            'masterlink':  {
               'Typecast': GTypecast.text },
            'detaillink':  {
               'Typecast': GTypecast.text } },
         'ParentTags': None },
  }

  # Add conditional elements
  xmlElements.update(
      GConditions.getXMLelements(
          {'conditions':{'ParentTags':('datasource',) } } ))

  for alteration in updates.keys():
    xmlElements[alteration].update(updates[alteration])

  return xmlElements



