#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000, 2001 Free Software Foundation
#
# FILE:
# GDataObjects.py
#
# DESCRIPTION:
# Class
#
# NOTES:
#
# HISTORY:
#

import GDebug
import GConfig
import GConditions
import string

class Error(StandardError):
  # Base exception
  pass

class LoginError(Error):
  # Raised when invalid login user/pass was provided
  # Client should attempt to get better information and
  # try again
  pass

class ConnectError(Error):
  # Raised when connection data is invalid (e.g., host not found, etc).
  # Client should probably not attempt to relogin.  Exit gracefully
  # with a reason.
  pass

class ProviderNotSupportedError(Error):
  # Raised when a datasource type is requested that the dbdriver
  # does not support (e.g., not all dbdrivers support raw sql mode.)
  pass

class ObjectTypeNotAvailableError(Error):
  # Raised when a datasource type is requested that the dbdriver
  # does not support (e.g., not all dbdrivers support raw sql mode.)
  pass

class ReadOnlyError(Error):
  # Raised when an attempt is made to update a read-only data object.
  pass

class MasterDetailFieldMismatch(Error):
  # Raised when a the number of master fields doesn't match the
  # number of detail fields. (e.g., masterlink="id,subid"
  # and detaillink="id" would be a problem; must be 1:1)
  pass

class ConnectionError(Error):
  # Generic error reading from the database connection
  pass



###########################################################
#
#
#
###########################################################
class DataObject:

  def __init__(self):
    # Set by
    self.masterlink = ""
    self.detaillink = ""

    self._masterfields = []
    self._detailfields = []
    self._staticCondition = None

    self._masterObject = None
    self._detailObjects = []
    self._dataConnection = None
    self._resultSetClass = ResultSet
    self._fieldReferences = {}  # Set by GDataSource; lists all fields
                                # a client explicits references

    self._unboundFieldReferences = {}  # Contains names of all unbound
                                       # field references

    self._defaultValues = {}
    self.triggerExtensions = None


  # Do we have a master datasource?
  def hasMaster(self):
    return self._masterObject != None

  # Do not over-ride by vendor code
  def createResultSet(self, conditions={}, readOnly=0, masterRecordSet=None):
    return self._createResultSet(
       GConditions.combineConditions(conditions, self._staticCondition),
       readOnly=readOnly, masterRecordSet=masterRecordSet)

  # Designed to be replaced by vendor-specific code
  def _createResultSet(self, conditions={}, readOnly=0, masterRecordSet=None):
    pass

  # Do not over-ride by vendor code
  def createEmptyResultSet(self, readOnly=0, masterRecordSet=None):
    return self._createEmptyResultSet(readOnly, masterRecordSet)

  # Designed to be replaced by vendor-specific code
  def _createEmptyResultSet(self, readOnly=0, masterRecordSet=None):
    cond = GConditions.GCondition()
    ceq = GConditions.GCeq(cond)
    GConditions.GCConst(ceq,1,"number")
    GConditions.GCConst(ceq,0,"number")
    return self.createResultSet(conditions=cond, readOnly=readOnly,
                                masterRecordSet=masterRecordSet)


  # Add a detail data object.  This dataobject will create a new resultset
  # everytime this dataobject changes (new record, etc).  The optional
  # handler will be called after the detail dataobject is notified.  The
  # client application may wish to add a handler to know when the detail
  # has been requeried.  handler is a method that takes two arguments:
  # the master ResultSet and the detail ResultSet
  def addDetailDataObject(self, dataObject, handler=None, **params):

    for param in params.keys():
      dataObject.__dict__[param] = params[param]

    GDebug.printMesg (1,"Adding a master/detail relationship to DataObject")
    dataObject._masterObject = self
    dataObject._masterfields = string.split(hasattr(dataObject,'masterlink') and \
                                 string.lower(dataObject.masterlink) or "", ',')
    dataObject._detailfields = string.split(hasattr(dataObject,'detaillink') and \
                                 string.lower(dataObject.detaillink) or "", ',')

    if len(dataObject._masterfields) != len(dataObject._detailfields):
      raise MasterDetailFieldMismatch, "master=%s; detail=%s" % \
          (dataObject._masterfields, dataObject._detailfields)

    # Make sure "master" fields will be in our future query
    for field in dataObject._masterfields:
      self._fieldReferences[string.strip(field)] = ""

    for field in dataObject._detailfields:
      dataObject._fieldReferences[string.strip(field)] = ""

    self._detailObjects.append ([dataObject, handler])


  # Return a list of necessary login fields (e.g., user/pass).
  # Each list item is another list of ["field label", isPassword?]
  def getLoginFields(self):
    return []


  #
  # Connect to database. Design to be replaced by vendor-specific code.
  #
  # NOTE: This will only be called for the FIRST dataobject
  #   using a particular connection.  Any subsequent dataobjects
  #   will only get setDataConnection() called. Therefore, any
  #   routines that must be run for EACH dataobject after a connection
  #   has been establisted should be placed in _postConnect(), not connect().
  #
  #   As such, if the vendor-specific drivers replace this method, the last
  #   line of the new method should be self._postConnect()!
  #
  def connect(self, connectData={}):
    self._postConnect()


  #
  # Post connection routines. Design to be replaced by vendor-specific code.
  #
  # NOTE: See note for connect()
  #
  def _postConnect(self):
    pass


  #
  # Set the associated data connection. Used by GConnections.
  # THERE IS NO NEED FOR VENDOR-CLASSES TO REPLACE THIS METHOD!
  #
  # NOTE: See note for connect()
  #
  def setDataConnection(self, connection):
    self._dataConnection = connection
    self._postConnect()


  def getDataConnection(self):
    return self._dataConnection


  def commit(self):
    pass

  def rollback(self):
    pass


  #
  # Schema (metadata) functions
  #

  # Return a list of the types of Schema objects this driver provides
  # Contains tuples of (key, description, dataSource??)
  # dataSource?? is true if this schema type can be a datasource
  def getSchemaTypes(self):
    return []

  # Return a list of Schema objects
  def getSchemaList(self, type=None):
    return []

  # Find a schema object with specified name
  def getSchemaByName(self, name, type=None):
    return None


  # Called when new record master in master/detail is queried
  def _masterRecordChanged(self, master):
    GDebug.printMesg (5, 'Master Record Changed')
    criteria = {}

    # If a detail result set has already been created for a particular
    # master record set, then just return/reuse this old set (after all,
    # it may contain uncommitted changes)
    if (not master.current._cachedDetailResultSets.has_key(self)) or \
        ( not int(GConfig.get('CacheDetailRecords','1')) and \
	  not master.current._cachedDetailResultSets[self].isPending() ):
      for i in range(0, len(self._masterfields)):
        GDebug.printMesg(10,"Adding criteria")
        criteria[string.strip(self._detailfields[i])] = \
            master.current.getField(string.strip(self._masterfields[i]))
        GDebug.printMesg(10,master.current.getField(self._masterfields[i]))
      master.current._cachedDetailResultSets[self] = \
          self.createResultSet(conditions=criteria, masterRecordSet=master.current)
    return master.current._cachedDetailResultSets[self]


###########################################################
#
#
#
###########################################################
class ResultSet:

  def __init__(self, dataObject, cursor=None,defaultValues={},masterRecordSet=None):
     self._dataObject = dataObject
     self._recordSetClass = RecordSet
     self._cursor = cursor
     self._cachedRecords = []
     self._currentRecord = -1
     self._masterRecordSet = masterRecordSet
     self._readonly = 0
     self._recordCount = 0

     self._defaultValues = {}
     for key in defaultValues.keys():
       self._defaultValues[key] = defaultValues[key]

     self.current = None

     if masterRecordSet:
       masterRecordSet.addDetailResultSet(self)


  # Returns whether this result set is read only or not
  def isReadOnly(self):
    return self._readonly


  # Returns 1=At first record, 0=Not first record
  def isFirstRecord(self):
    return (self._currentRecord == 1)


  # returns -1=No records in memory, #=Current record #
  def getRecordNumber(self):
    return self._currentRecord


  # returns # of records currently loaded
  def getCacheCount(self):
    return len(self._cachedRecords)

  # returns # of records the
  def getRecordCount(self):
    return self._recordCount  > 0 and self._recordCount or self.getCacheCount()

  # Get a specific record (0=based)
  def getRecord(self, record):
    while (record + 1 > len(self._cachedRecords)) and self._loadNextRecord():
      pass

    if record + 1 > len(self._cachedRecords):
      return None
    else:
      return self._cachedRecords[record]


  # move to record #, returns 1=New record loaded, 0=invalid #
  def setRecord(self, record):

    while record - 1 > len(self._cachedRecords) and self._loadNextRecord():
      pass

    if record + 1 >= len(self._cachedRecords):
      return 0
    else:
      self._currentRecord = record
      self.current = self._cachedRecords[self._currentRecord]
      self.notifyDetailObjects()
      return 1
      
  # returns 1=New record loaded, 0=No more records
  def nextRecord(self):
    if self._currentRecord + 1 == len(self._cachedRecords):
      if not self._loadNextRecord():
        return 0

    self._currentRecord += 1
    self.current = self._cachedRecords[self._currentRecord]
    self.notifyDetailObjects()
    return 1


  # returns 1=New record loaded, 0=At first record
  def prevRecord(self):
    if self._currentRecord < 1:
      return 0
    else:
      self._currentRecord = self._currentRecord - 1
      self.current = self._cachedRecords[self._currentRecord]
      self.notifyDetailObjects()
      return 1


  # returns 1=at first record, 0=No records loaded
  def firstRecord(self):
    if self._currentRecord < 0:
      if not self._loadNextRecord():
        return 0

    self._currentRecord = 0
    self.current = self._cachedRecords[0]
    self.notifyDetailObjects()
    return 1



  # returns 1=at last record, 0=No records loaded
  def lastRecord(self):
    if self._currentRecord == -1:
      return 0
    else:
      while self._loadNextRecord():
        pass
      self._currentRecord = len(self._cachedRecords) - 1
      self.current = self._cachedRecords[self._currentRecord]
      self.notifyDetailObjects()
      return 1



  # Insert a blank record after the current record
  def insertRecord(self):
    if self.isReadOnly():
      # Provide better feedback??
      raise ReadOnlyError, "Attempted to insert into a read only datasource"
    else:
      GDebug.printMesg(7,'Inserting a blank record')
      self._currentRecord += 1
      self._cachedRecords.insert(self._currentRecord, self._createEmptyRecord())
      self._recordCount += 1
      self.current = self._cachedRecords[self._currentRecord]

      # Set any dataobject-wide default values
      for field in self._dataObject._defaultValues.keys():
        self.current.setField(field, self._dataObject._defaultValues[field],0)

      # Set any resultset specific values
      for field in self._defaultValues.keys():
        self.current.setField(field, self._defaultValues[field],0)

      # Pull any primary keys from a master record set
      if self._masterRecordSet != None and hasattr(self._dataObject, '_masterfields'):
        i = 0
        for field in self._dataObject._masterfields:
          self.current.setField(self._dataObject._detailfields[i],self._masterRecordSet.getField(field),0)
          i += 1

      self.notifyDetailObjects()
      return 1


  # Returns 1=DataObject has uncommitted changes
  def isPending(self):
    isPending = 0
    for rec in (self._cachedRecords):
      isPending = isPending or rec.isPending()
      if isPending:
        break
    return isPending



  # Returns 1=DataObject has uncommitted changes
  def isRecordPending(self):
    return self.current.isPending()



  # Post changes to the database
  def post(self, foreign_keys={}):
    # post our changes
    self._update_cursor = self._dataObject._dataConnection.cursor()

    i = 0
    for record in self._cachedRecords:
      # Flip the flag for 'default' values to true so that hidden
      # default fields are included in insert statements
      for field in self._dataObject._defaultValues.keys():
        if record.isPending():
          print "Setting %s to %s" % (field, record.getField(field))
          record.setField(field, record.getField(field))

      for field in foreign_keys.keys():
        record._fields[field] = foreign_keys[field]
        # Some DBs will throw an exception if you update a Primary Key
        # (even if you are updating to the same value)
        if record._insertFlag:
          record._modifiedFlags[field] = 1

      delete = record._emptyFlag or record._deleteFlag
      record.post()

      # Adjust the current record if a preceding record is deleted
      if delete and i <= self._currentRecord:
        self._currentRecord -= 1

      if delete:
        self._cachedRecords.pop(i)
        self._recordCount -= 1
      else:
        i += 1

    # Move to record 0 if all preceding records were deleted
    # (or set to -1 if all records were deleted)
    if self._currentRecord < 0:
      if len(self._cachedRecords):
        self._currentRecord = 0
      else:
        self._currentRecord = -1
    if self._currentRecord >= self._recordCount:
      self._currentRecord = self._recordCount - 1



  def notifyDetailObjects(self):
    GDebug.printMesg(5,'Master record changed; Notifying Detail Objects')
    for detail in self._dataObject._detailObjects:
       rs = detail[0]._masterRecordChanged(self)
       if detail[1]:
         detail[1].masterResultSetChanged(self, rs)


  # Returns 1=Field is bound to a database field
  def isFieldBound(self, fieldName):
    return self._dataObject._fieldReferences.has_key(string.lower(fieldName))


  ###
  ### Methods below should be overridden by Vendor Specific functions
  ### (_createEmptyRecord may not need to be overridden in all cases)
  ###

  # Load cacheCount number of new records
  def _loadNextRecord(self):
    return 0


  # Create an empty recordset
  def _createEmptyRecord(self):
    return self._recordSetClass(self)


###########################################################
#
#
#
###########################################################
class RecordSet:

  def __init__(self, parent, initialData={}, dbIdentifier=None, defaultData={}):
    self._detailObjects = []
    self._dbIdentifier = dbIdentifier
    self._deleteFlag = 0
    self._updateFlag = 0
    self._parent = parent
    self._modifiedFlags = {}      # If field name is present as a key,
                                  # then field has been modified

    self._cachedDetailResultSets = {}

    self._initialData = initialData

    if self._initialData and len(self._initialData):
      self._insertFlag = 0
      self._emptyFlag = 0
      self._fields = {}
      for key in initialData.keys():
       self._fields[key] = initialData[key]
    else:
      self._insertFlag = 1
      self._emptyFlag = 1
      self._fields = {}
      for key in defaultData.keys():
       self._fields[key] = defaultData[key]


  # Returns 1=Record has uncommitted changes
  def isPending(self):

    # The _insertFlag and _deleteFlag takes care of records that
    # were inserted, but then deleted before a save (i.e., nothing to do)
    if self._emptyFlag or self._insertFlag and self._deleteFlag:
      return 0
    else:
      return self._insertFlag or self._deleteFlag or self._updateFlag


  # Returns 1=Record is pending a deletion
  def isDeleted(self):
    if self._emptyFlag:
      return 0
    else:
      return self._deleteFlag


  # Returns 1=Record is pending an update
  def isModified(self):
    if self._emptyFlag or self._insertFlag:
      return 0
    else:
      return self._updateFlag


  # Returns 1=Record is pending an insertion
  def isInserted(self):
    if self._emptyFlag:
      return 0
    else:
      return self._insertFlag


  # Returns 1=Record is empty (inserted, but no data set)
  def isEmpty(self):
    return self._emptyFlag


  # Returns current value of "field"
  def getField(self, field):
    if self._fields.has_key(string.lower(field)):
      return self._fields[string.lower(field)]
    else:
      return None


  # Sets current value of "field"
  # If trackMod is set to 0 then the modification flag isn't raised
  def setField(self, field, value, trackMod = 1):
    # If this field is bound to a datasource and the datasource is read only,
    # generate an error.
    if self._parent.isFieldBound(field) and self._parent.isReadOnly():
      # Provide better feedback??
      raise ReadOnlyError, "Attempted to modify read only field '%s'" % field
    else:
      fn = string.lower(field)
      self._fields[fn] = "%s" % value
      if trackMod == 1:
        self._emptyFlag = 0
        if self._parent.isFieldBound(field):
          self._updateFlag = 1
          if self._modifiedFlags.has_key(fn):
            flag = self._modifiedFlags[fn] + 1
          else:
            flag = 1
          self._modifiedFlags[fn] = flag


  # Returns 1=Field has been modified
  def isFieldModified(self, fieldName):
    return self._modifiedFlags.has_key (string.lower(fieldName))


  # Mark the current record as deleted
  def delete(self):
    if self._parent.isReadOnly():
      # Provide better feedback??
      raise ReadOnlyError, "Attempted to delete from a read only datasource"
    else:
      self._deleteFlag = 1


  # Posts changes to database
  def post(self):
    # Should a post() to a read only datasource cause a ReadOnlyError?
    # It does no harm to attempt to post since nothing will be posted,
    # But does this allow sloppy programming?

    GDebug.printMesg(5,'Preparing to post datasource %s' %  self._parent._dataObject.name)

    if self.isPending():
      GDebug.printMesg(5,'Posting datasource %s' % self._parent._dataObject.name)
      self._postChanges()

    # Post all detail records
    for child in (self._cachedDetailResultSets.keys()):
      c = self._cachedDetailResultSets[child]._dataObject
      # Set the primary key for any new child records
      fk = {}
      for i in range(len(c._masterfields)):
        fk[c._detailfields[i]] = self.getField(c._masterfields[i])

      self._cachedDetailResultSets[child].post(foreign_keys=fk)


  # Sets the ResultSet associated with this master record
  def addDetailResultSet(self, ds):
    self._detailObjects.append(ds)


  ###
  ### Methods below should be over-written by Vendor Specific functions
  ###

  # Post any changes to database
  def _postChanges(self):
    return 1


# Used to store schema data
class Schema:
  def __init__(self, attrs={}, getChildSchema=None):
    self.__dict__.update(attrs)
    if getChildSchema != None:
      self.getChildSchema = getChildSchema

  def getChildSchema(self, parent):
    return ()




