#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it 
# and/or modify it under the terms of the GNU General Public 
# License as published by the Free Software Foundation; either 
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be 
# useful, but WITHOUT ANY WARRANTY; without even the implied 
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public 
# License along with program; see the file COPYING. If not, 
# write to the Free Software Foundation, Inc., 59 Temple Place 
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000, 2001 Free Software Foundation
#
# FILE:
# GBaseApp.py
#
# DESCRIPTION:
# Class that provides a basis for GNUe applications.
#
# NOTES:
# Typically, this class will not be called; rather, a tool will
# be a GClientApp or GServerApp.
#
# HISTORY:
#

import GDebug
import getopt
import sys
import string
import os
import GConnections
import GConfig

class GBaseApp:

  # Attributes to be overwritten by subclasses
  VERSION = "0.0.0"
  NAME = "GNUe Application"
  COMMAND_OPTIONS = []  # Should be in same format as _base_options below
  SUMMARY = "A brief summary of the program goes here."
  COMMAND = "app"
  USAGE = "%s [options] file"


  #  Run the program
  #  Should be overwritten by subclasses
  def run(self):
    pass


  # Attributes that will be set by GClientApp after __init__ has run
  OPTIONS = {}        # Will contain a hash containing command line options
  ARGUMENTS = []      # Will contain an array of command line arguments
  connections = None  # Will contain a GConnection object




  def __init__(self, connections=None, application=None):
    # format of COMMAND_OPTIONS and _base_options:
    #  [option, option] where option is:
    #       [dictionary key name, short option, long option,
    #        argument follows option?, argument name, help message]
    self._base_options = [
         [ 'version',None,'version',0,0, None,
           'Displays the version information for this program.' ],
         [ 'debug_level','d','debug-level',1,0, "level",
           'Enables debugging messages.  Argument specifies the ' + \
           'level of messages to display (e.g., "-d5" displays ' + \
           'all debugging messages at level 5 or below.)' ],
         [ 'debug_file',None,'debug-file',1,None, "file",
           'Sends all debugging messages to a specified file ' + \
           '(e.g., "--debug-file trace.log" sends all output to "trace.log")' ],
         [ 'connections', None, 'connections', 1, None, "loc",
           'Specifies the location of the connection definition file. ' + \
           '<loc> may specify a file name ' + \
           '(/usr/local/gnue/etc/connections.conf), ' + \
           'or a URL location ' + \
           '(http://localhost/connections.conf). '+ \
           'If this option is not specified, the environent variable ' + \
           'GNUE_CONNECTIONS is checked.' ],
         [ 'help', None, 'help', 0, None, None,
           'Displays this help screen.' ],
         [ 'profile', None, 'profile', 0, None, None,
           "Run Python's built-in profiler and display the resulting " + \
           "run statistics." ],
    ]

    #
    # Get all command line options and arguments
    #
    shortoptions = ""
    longoptions = []
    lookup = {}
    for optionset in [self._base_options, self.COMMAND_OPTIONS]: 
      for option in optionset:  
        self.OPTIONS[option[0]] = option[4] 
        if option[1] != None: 
          shortoptions = shortoptions + option[1]
          lookup["-" + option[1]] = option[0]
        lookup["--" + option[2]] = option[0]
        lo = option[2]
        if option[3]:   
          lo = lo + '='
          shortoptions = shortoptions + ':'
        longoptions.append(lo)
 
    try:
      opt, self.ARGUMENTS = getopt.getopt(sys.argv[1:], shortoptions, longoptions)
    except getopt.error, msg: 
      self.handleStartupError(msg)

    for o in opt: 
      if len(o[1]):
        self.OPTIONS[lookup[o[0]]] = o[1]
      else:
        self.OPTIONS[lookup[o[0]]] = 1

    if self.OPTIONS['help']:
      self.printHelp()
      sys.exit()
    elif self.OPTIONS['version']:
      self.printVersion()
      sys.exit()

    # Should we profile?
    self._run = self.run
    if self.OPTIONS['profile']:
      self.run = self._profile

    # Setup debugging
    try:
      GDebug.setDebug(int("%s" % self.OPTIONS['debug_level']),
          self.OPTIONS['debug_file'])
    except ValueError:
      self.handleStartupError('The debug_level option ("-d") expects a numerical value.')

    GDebug.printMesg(1,"Run Options: %s" % opt)
    GDebug.printMesg(1,"Run Arguments: %s" % self.ARGUMENTS)

    # Read the config files
    if application:
      GConfig.GConfig(application)

    # Get the connection definitions
    if connections != None:
      GDebug.printMesg(1,"Reusing connections instance")
      self.connections = connections
    else:
      if self.OPTIONS['connections']:
        self.connections_file = self.OPTIONS['connections']
      elif os.environ.has_key('GNUE_CONNECTIONS'):
        self.connections_file = os.environ['GNUE_CONNECTIONS']
      else:
        self.connections_file = "" # temp
#        self.handleStartupError(
#            'Unable to load the connections definition file.\n' \
#            + '\n   Please set the environmental variable GNUE_CONNECTIONS or ' \
#            + '\n   use the "-f" command option.')

      GDebug.printMesg(1, 'Connection Definition: "%s"' % self.connections_file)

      try:
        self.connections = GConnections.GConnections(self.connections_file)
      except GConnections.InvalidFormatError, msg:
        self.handleStartupError(
            'Unable to load the connections definition file.\n' \
            + '\n   The connections file is in an invalid format. ' \
            + '\n   %s' \
               % msg)
      except IOError:
        self.handleStartupError(
             'Unable to load the connections definition file.\n' \
             + '\n   The connections file specified either does ' \
             + '\n   not exist or is not readable by your account.\n' \
             + '\n   Location: "%s"'
                % self.connections_file)

  #
  #  Display version information for this application
  #
  def printVersion(self): 
    print "\n%s\nVersion %s\n" % (self.NAME, self.VERSION)

  #
  #  Display version information for this program
  #
  def printHelp(self): 
    allOptions = {}
    descriptors = {}
    maxLength = 0

    for optionset in [self._base_options, self.COMMAND_OPTIONS]: 
      for option in optionset:  
        allOptions[string.upper(option[2])] = option

        if option[5]: 
          descr = '--%s <%s>' % (option[2], option[5])
        else: 
          descr = '--%s' % (option[2])
        if option[1]: 
          descr += ', -%s' % option[1]

        descriptors[string.upper(option[2])] = descr

        if len(descr) > maxLength: 
          maxLength = len(descr)

    if maxLength > 20: 
      maxLength = 20

    sorted = allOptions.keys()
    sorted.sort()

    dispOptions = ""

    for optionKey in sorted: 

      margin = maxLength + 4
      width = 78 - margin
      pos = 0

      if len(descriptors[optionKey]) > maxLength: 
        dispOptions += "\n  %s\n%s" % (descriptors[optionKey], " " * margin)
      else: 
        dispOptions += "\n  %s  %s" % (descriptors[optionKey],  
               " " * (maxLength - len(descriptors[optionKey])))

      for word in string.split(allOptions[optionKey][6]): 
        if (len(word) + pos) > width: 
          pos = 0
          dispOptions = dispOptions + "\n" + " " * margin

        pos = pos + len(word) + 1

        dispOptions = dispOptions + word + " "

      dispOptions = dispOptions + "\n"

    self.printVersion()
    print "Usage:  %s\n\n%s\n\nAvailable command line options:\n%s" % \
       (self.USAGE, self.SUMMARY, dispOptions)
    print "Report bugs to bugs@gnue.org\n"


  #
  #  Display a startup error and exit gracefully with a message on
  #  how to get help
  #
  def handleStartupError(self, msg):
      self.printVersion()

      # if msg is multiline, then surround with dashes to set it apart
      if string.find("%s" % msg, "\n") + 1:
        print '-' * 60
      print "Error: %s" % msg
      if string.find("%s" % msg, "\n") + 1:
        print '-' * 60
      print "\nFor help, type:\n   %s --help\n" % (self.COMMAND)
      sys.exit()

  #
  #  Used when profiling
  #
  def _profile(self):

    import profile
    prof = profile.Profile()
    prof.runctx( 'self._run()', globals(), locals() )

    import pstats
    p = pstats.Stats(prof)
    p.sort_stats('time').print_stats(50)
    p.sort_stats('cumulative').print_stats(50)



