"======================================================================
|
|   Numerical methods - Test Suite
|
|
 ======================================================================"

"======================================================================
|
| Written by Didier Besset.
|
| This file is distributed together with GNU Smalltalk.
|
 ======================================================================"



TestCase subclass: DhbTestCase [
    
    <comment: nil>
    <category: 'DHB Numerical SUnits'>

    defaultLogPolicyClass [
	<category: 'logging'>
	^TestVerboseLog
    ]
]



DhbTestCase subclass: DhbTestGamma [
    
    <comment: nil>
    <category: 'DHB Numerical SUnits'>

    DhbTestGamma class >> epsilon [
	<category: 'utility'>
	^0.00005
    ]

    gammaDiff: aNumber expected: result [
	<category: 'helpers'>
	| a |
	a := DhbLanczosFormula new gamma: aNumber.
	^(a - result) abs
    ]

    testGamma [
	<category: 'tests'>
	| diff |
	diff := self gammaDiff: 4 expected: 6.
	self assert: diff < self class epsilon.
	diff := self gammaDiff: 1 / 3 expected: 2.6789.
	self assert: diff < self class epsilon.
	diff := self gammaDiff: 4 / 5 expected: 1.1642.
	self assert: diff < self class epsilon
    ]
]



DhbTestCase subclass: DhbTestIterators [
    
    <comment: nil>
    <category: 'DHB Numerical SUnits'>

    integratorTest: anIntegratorClass [
	<category: 'integration'>
	| a |
	a := anIntegratorClass 
		    function: [:x | x sin]
		    from: 0
		    to: 0.
	self assert: (a evaluate equalsTo: 0).
	a := anIntegratorClass 
		    function: [:x | x sin]
		    from: 0
		    to: FloatD pi.
	self assert: a evaluate - 2 < 0.00001.
	a := anIntegratorClass 
		    function: [:x | x sin]
		    from: FloatD pi negated
		    to: 0.
	self assert: (a evaluate + 2) abs < 0.00001.
	a := anIntegratorClass 
		    function: [:x | x sin]
		    from: FloatD pi negated
		    to: FloatD pi.
	self assert: (a evaluate equalsTo: 0)
    ]

    integratorTestInversedBounds: anIntegratorClass [
	<category: 'integration'>
	"Integrator cannot deal with nversed bounds"

	| a |
	a := anIntegratorClass 
		    function: [:x | x sin]
		    from: FloatD pi
		    to: 0.
	self deny: (a evaluate equalsTo: -2)
    ]

    testRomberg [
	<category: 'integration'>
	self integratorTest: DhbRombergIntegrator
    ]

    testRombergInversedBounds [
	"Integrator cannot deal with nversed bounds"

	<category: 'integration'>
	self integratorTestInversedBounds: DhbRombergIntegrator
    ]

    testSimpson [
	<category: 'integration'>
	self integratorTest: DhbSimpsonIntegrator
    ]

    testSimpsonInversedBounds [
	"Integrator cannot deal with nversed bounds"

	<category: 'integration'>
	self integratorTestInversedBounds: DhbSimpsonIntegrator
    ]

    testTrapeze [
	<category: 'integration'>
	self integratorTest: DhbTrapezeIntegrator
    ]

    testTrapezeInversedBounds [
	"Integrator cannot deal with nversed bounds"

	<category: 'integration'>
	self integratorTestInversedBounds: DhbTrapezeIntegrator
    ]

    testRootFind [
	<category: 'polynomial'>
	| a roots |
	a := DhbPolynomial coefficients: #(0 1 1).
	roots := a roots asSortedCollection.
	self
	    assert: roots size = 2;
	    assert: (roots first equalsTo: -1);
	    assert: (roots last equalsTo: 0)
    ]

    testRootFindComplex [
	"Does not support complex roots"

	<category: 'polynomial'>
	| a roots |
	a := DhbPolynomial coefficients: #(1 0 1).
	roots := a roots.
	self assert: roots size = 0
    ]

    testBisectionZeroFinder [
	<category: 'zero finders'>
	| finder |
	finder := DhbBisectionZeroFinder function: [:x | x + 1].
	finder
	    setNegativeX: -70;
	    setPositiveX: 10.
	finder evaluate.
	self
	    assert: finder hasConverged;
	    assert: (finder result equalsTo: -1)
    ]

    testBisectionZeroFinderNoZero [
	<category: 'zero finders'>
	| finder |
	finder := DhbBisectionZeroFinder function: [:x | x * x + 1].
	self should: [finder findNegativeXFrom: -30 range: 20] raise: Error
    ]

    testBisectionZeroFinderSquared [
	<category: 'zero finders'>
	| finder |
	finder := DhbBisectionZeroFinder function: [:x | x * x - 1].
	finder
	    setNegativeX: -0.9000000000000001;
	    setPositiveX: 10.
	finder evaluate.
	self
	    assert: finder hasConverged;
	    assert: (finder result equalsTo: 1)
    ]

    testNewtonZeroFinder [
	<category: 'zero finders'>
	| finder |
	finder := DhbNewtonZeroFinder function: [:x | x + 1] derivative: [:x | 1].
	finder evaluate.
	self
	    assert: finder hasConverged;
	    assert: (finder result equalsTo: -1)
    ]

    testNewtonZeroFinderSquared [
	<category: 'zero finders'>
	| finder |
	finder := DhbNewtonZeroFinder function: [:x | x * x - 1]
		    derivative: [:x | 2 * x].
	finder evaluate.
	self
	    assert: finder hasConverged;
	    assert: (finder result abs equalsTo: 1)
    ]
]



DhbTestCase subclass: DhbNumericalMethodsTestCase [
    
    <comment: nil>
    <category: 'DHB Numerical SUnits'>

    testClusterCovariance [
	<category: 'data mining'>
	| dataServer clusters finder |
	dataServer := DhbMemoryBasedDataServer new.
	dataServer data: (self generatedPoints: 1000).
	finder := DhbClusterFinder 
		    new: 5
		    server: dataServer
		    type: DhbCovarianceCluster.
	finder minimumRelativeClusterSize: 0.1.
	clusters := finder evaluate.
	self should: [clusters size = 3]
    ]

    testClusterEuclidean [
	<category: 'data mining'>
	| dataServer clusters finder |
	dataServer := DhbMemoryBasedDataServer new.
	dataServer data: (self generatedPoints: 1000).
	finder := DhbClusterFinder 
		    new: 5
		    server: dataServer
		    type: DhbEuclideanCluster.
	finder minimumRelativeClusterSize: 0.15.
	clusters := finder evaluate.
	self should: [clusters size = 3]
    ]

    testCovarianceAccumulation [
	"Code example 12.2"

	<category: 'data mining'>
	| accumulator average covarianceMatrix |
	accumulator := DhbCovarianceAccumulator new: 3.
	#(#(1 2 3) #(2 3 4) #(1 3 2) #(4 3 1) #(1 3 1) #(1 4 2) #(3 1 2) #(3 4 2)) 
	    do: [:x | accumulator accumulate: x asVector].
	average := accumulator average.
	self should: [(average at: 1) equalsTo: 2.0].
	self should: [(average at: 2) equalsTo: 2.875].
	self should: [(average at: 3) equalsTo: 2.125].
	covarianceMatrix := accumulator covarianceMatrix.
	self should: [((covarianceMatrix rowAt: 1) at: 1) equalsTo: 1.25].
	self should: [((covarianceMatrix rowAt: 1) at: 2) equalsTo: -0.125].
	self should: [((covarianceMatrix rowAt: 2) at: 1) equalsTo: -0.125].
	self should: [((covarianceMatrix rowAt: 1) at: 3) equalsTo: -0.25].
	self should: [((covarianceMatrix rowAt: 3) at: 1) equalsTo: -0.25].
	self should: [((covarianceMatrix rowAt: 2) at: 2) equalsTo: 0.859375].
	self should: [((covarianceMatrix rowAt: 2) at: 3) equalsTo: -0.109375].
	self should: [((covarianceMatrix rowAt: 3) at: 2) equalsTo: -0.109375].
	self should: [((covarianceMatrix rowAt: 3) at: 3) equalsTo: 0.859375]
    ]

    testMahalanobisCenter [
	"Code example 12.5"

	<category: 'data mining'>
	| center distance |
	center := DhbMahalanobisCenter new: 3.
	#(#(1 2 3) #(2 3 4) #(1 3 2) #(4 3 1) #(1 3 1) #(1 4 2) #(3 1 2) #(3 4 2)) 
	    do: [:x | center accumulate: x asVector].
	center computeParameters.
	distance := center distanceTo: #(1 2 3) asVector.
	self should: [distance equalsTo: 2.26602282704126]
    ]

    testFTest [
	<category: 'estimation'>
	| accC accMM confidenceLevel |
	accC := DhbStatisticalMoments new.
	#(5.560000000000001 5.89 4.660000000000001 5.690000000000001 5.34 4.79 4.8 7.860000000000001 3.64 5.700000000000001) 
	    do: [:x | accC accumulate: x].
	accMM := DhbStatisticalMoments new.
	#(7.480000000000001 6.75 3.77 5.71 7.25 4.730000000000001 6.230000000000001 5.600000000000001 5.940000000000001 4.58) 
	    do: [:x | accMM accumulate: x].
	confidenceLevel := accC fConfidenceLevel: accMM.
	self should: [(accC average - 5.393) abs < 0.000000001].
	self should: [(accC standardDeviation - 1.0990809292) abs < 0.000000001].
	self should: [(accMM average - 5.804000000000001) abs < 0.000000001].
	self should: [(accMM standardDeviation - 1.19415428) abs < 0.000000001].
	self should: [(confidenceLevel - 79.8147614536) abs < 0.000000001]
    ]

    testInterpolationNewton [
	<category: 'estimation'>
	| interpolator |
	interpolator := DhbNewtonInterpolator new.
	1 to: 45
	    by: 2
	    do: [:x | interpolator add: x @ x degreesToRadians sin].
	self should: 
		[((interpolator value: 8) - 8 degreesToRadians sin) abs < 0.00000000000001]
    ]

    testLeastSquare [
	"Code example 10.9"

	"Note: the seemingly large error on the fit results is due to the binning of the histogram."

	<category: 'estimation'>
	| count shape scale genDistr hist fit fittedDistr parameters |
	count := 10000.
	shape := 0.
	scale := 1.
	hist := DhbHistogram new.
	hist freeExtent: true.
	genDistr := DhbFisherTippettDistribution shape: shape scale: scale.
	count timesRepeat: [hist accumulate: genDistr random].
	fit := DhbLeastSquareFit histogram: hist
		    distributionClass: DhbFisherTippettDistribution.
	fittedDistr := fit evaluate.
	parameters := fittedDistr parameters.
	self should: [((parameters at: 1) - shape) abs < 0.1].
	self should: [((parameters at: 2) - scale) abs < 0.1].
	self should: [((parameters at: 3) - count) abs < 100]
    ]

    testLeastSquarePolynomial [
	"Code example 10.5"

	<category: 'estimation'>
	| fit estimation |
	fit := DhbPolynomialLeastSquareFit new: 3.
	fit
	    add: (DhbWeightedPoint point: 1 @ 2.0);
	    add: (DhbWeightedPoint point: 2 @ 21.0);
	    add: (DhbWeightedPoint point: 3 @ 72.0);
	    add: (DhbWeightedPoint point: 4 @ 173.0);
	    add: (DhbWeightedPoint point: 5 @ 342.0);
	    add: (DhbWeightedPoint point: 6 @ 597.0);
	    add: (DhbWeightedPoint point: 7 @ 956.0);
	    add: (DhbWeightedPoint point: 8 @ 1437.0);
	    add: (DhbWeightedPoint point: 9 @ 2058.0);
	    add: (DhbWeightedPoint point: 10 @ 2837.0).
	estimation := fit evaluate.
	self should: [((estimation value: 4.5) - 247.875) abs < 0.000000001].
	self should: [((estimation error: 4.5) - 0.5215298) abs < 0.00001].
	self should: 
		[((estimation value: 7.150000000000001) - 1019.932625) abs 
		    < (estimation error: 7.150000000000001)]
    ]

    testLinearRegression [
	"Code example 10.5"

	<category: 'estimation'>
	| linReg estimation |
	linReg := DhbLinearRegression new.
	linReg
	    add: 1 @ 0.72;
	    add: 2 @ 3.25;
	    add: 3 @ 5.75;
	    add: 4 @ 8.210000000000001;
	    add: 5 @ 10.71;
	    add: 6 @ 13.38;
	    add: 7 @ 15.82;
	    add: 8 @ 18.39;
	    add: 9 @ 20.72;
	    add: 10 @ 23.38.
	self should: [(linReg slope - 2.514727272727) abs < 0.000000000001].
	self should: [(linReg intercept + 1.798) abs < 0.000000000001].
	self should: 
		[(linReg correlationCoefficient - 0.999966922113) abs < 0.000000000001].
	estimation := linReg asEstimatedPolynomial.
	self 
	    should: [((estimation value: 4.5) - 9.518272727272701) abs < 0.000000000001].
	self should: 
		[((estimation value: 7.150000000000001) - 16.1823) abs < 0.000000000001]
    ]

    testMaximumLikelihood [
	"Code example 10.11"

	"Note: the seemingly large error on the fit results is due to the binning of the histogram."

	<category: 'estimation'>
	| count shape scale genDistr hist fit fittedDistr parameters |
	count := 10000.
	shape := 0.
	scale := 1.
	hist := DhbHistogram new.
	hist freeExtent: true.
	genDistr := DhbFisherTippettDistribution shape: shape scale: scale.
	count timesRepeat: [hist accumulate: genDistr random].
	fit := DhbMaximumLikekihoodHistogramFit histogram: hist
		    distributionClass: DhbFisherTippettDistribution.
	fittedDistr := fit evaluate.
	parameters := fittedDistr parameters.
	self should: [((parameters at: 1) - shape) abs < 0.1].
	self should: [((parameters at: 2) - scale) abs < 0.1].
	self should: [((parameters at: 3) - count) abs < 100]
    ]

    testTTest [
	<category: 'estimation'>
	| accC accMM confidenceLevel |
	accC := DhbStatisticalMoments new.
	#(5.560000000000001 5.89 4.660000000000001 5.690000000000001 5.34 4.79 4.8 7.860000000000001 3.64 5.700000000000001) 
	    do: [:x | accC accumulate: x].
	accMM := DhbStatisticalMoments new.
	#(7.480000000000001 6.75 3.77 5.71 7.25 4.730000000000001 6.230000000000001 5.600000000000001 5.940000000000001 4.58) 
	    do: [:x | accMM accumulate: x].
	confidenceLevel := accC tConfidenceLevel: accMM.
	self should: [(accC average - 5.393) abs < 0.000000001].
	self should: [(accC standardDeviation - 1.0990809292) abs < 0.000000001].
	self should: [(accMM average - 5.804000000000001) abs < 0.000000001].
	self should: [(accMM standardDeviation - 1.19415428) abs < 0.000000001].
	self should: [(confidenceLevel - 56.63207399890001) abs < 0.000000001]
    ]

    testBeta [
	"Code example 2.14"

	<category: 'function evaluation'>
	| value |
	value := 2.5 gamma * 5.5 gamma / 8 gamma.
	self should: [((2.5 beta: 5.5) - value) abs < 0.00000000000001]
    ]

    testBetaLog [
	"Code example 2.15"

	<category: 'function evaluation'>
	| value |
	value := 2.5 logGamma + 5.5 logGamma - 8 logGamma.
	self should: [((2.5 logBeta: 5.5) - value) abs < 0.0000000000001]
    ]

    testErrorFunctionCentile [
	"Code example 2.5"

	<category: 'function evaluation'>
	| weight average stDev centile |
	weight := 2.85.
	average := 3.39.
	stDev := 0.44.
	centile := ((weight - average) / stDev) errorFunction * 100.
	self should: [(centile - 10.986012) abs < 0.000001]
    ]

    testGamma [
	"Code example 2.10"

	<category: 'function evaluation'>
	| value |
	value := FloatD pi sqrt * 3 / 4.
	self should: [(2.5 gamma - value) abs < 0.00000000000001]
    ]

    testGammaLog [
	"Code example 2.11"

	<category: 'function evaluation'>
	| value |
	value := 2.5 gamma ln.
	self should: [(2.5 logGamma - value) abs < 0.0000000000001]
    ]

    testGammaLow [
	<category: 'function evaluation'>
	| value |
	value := FloatD pi sqrt / 2.
	self should: [((3 / 2) gamma - value) abs < 0.00000000000001]
    ]

    testGammaNegative [
	<category: 'function evaluation'>
	| value |
	value := FloatD pi / (1.5 gamma * (FloatD pi / -2) sin).
	self should: [((-1 / 2) gamma - value) abs < 0.00000000000001]
    ]

    testInterpolationBulirschStoer [
	<category: 'function evaluation'>
	| interpolator |
	interpolator := DhbBulirschStoerInterpolator new.
	1 to: 45
	    by: 2
	    do: [:x | interpolator add: x @ x degreesToRadians sin].
	self should: 
		[((interpolator value: 8) - 8 degreesToRadians sin) abs < 0.00000000000001]
    ]

    testInterpolationLagrange [
	"Code example 3.2"

	<category: 'function evaluation'>
	| interpolator |
	interpolator := DhbLagrangeInterpolator new.
	1 to: 45
	    by: 2
	    do: [:x | interpolator add: x @ x degreesToRadians sin].
	self should: 
		[((interpolator value: 8) - 8 degreesToRadians sin) abs < 0.00000000000001]
    ]

    testInterpolationLagrangeLinear [
	"Code example 3.1"

	<category: 'function evaluation'>
	| interpolator |
	interpolator := DhbLagrangeInterpolator 
		    points: (Array with: 1 @ 2 with: 3 @ 1).
	self should: [((interpolator value: 2.2) - 1.4) abs < 0.00000000000001]
    ]

    testInterpolationNeville [
	<category: 'function evaluation'>
	| interpolator |
	interpolator := DhbNevilleInterpolator new.
	1 to: 45
	    by: 2
	    do: [:x | interpolator add: x @ x degreesToRadians sin].
	self should: 
		[((interpolator value: 8) - 8 degreesToRadians sin) abs < 0.00000000000001]
    ]

    testInterpolationNevilleLinear [
	"Code example 3.1"

	<category: 'function evaluation'>
	| interpolator |
	interpolator := DhbNevilleInterpolator 
		    points: (Array with: 1 @ 2 with: 3 @ 1).
	self should: [((interpolator value: 2.2) - 1.4) abs < 0.00000000000001]
    ]

    testInterpolationNewtonLinear [
	"Code example 3.1"

	<category: 'function evaluation'>
	| interpolator |
	interpolator := DhbNewtonInterpolator 
		    points: (Array with: 1 @ 2 with: 3 @ 1).
	self should: [((interpolator value: 2.2) - 1.4) abs < 0.00000000000001]
    ]

    testInterpolationSpline [
	<category: 'function evaluation'>
	| interpolator |
	interpolator := DhbSplineInterpolator new.
	1 to: 45
	    by: 2
	    do: [:x | interpolator add: x @ x degreesToRadians sin].
	self 
	    should: [((interpolator value: 8) - 8 degreesToRadians sin) abs < 0.0000001]
    ]

    testInterpolationSplineLinear [
	"Code example 3.1"

	<category: 'function evaluation'>
	| interpolator |
	interpolator := DhbSplineInterpolator 
		    points: (Array with: 1 @ 2 with: 3 @ 1).
	self should: [((interpolator value: 2.2) - 1.4) abs < 0.00000000000001]
    ]

    testPolynomialAddition [
	<category: 'function evaluation'>
	| polynomial |
	polynomial := (DhbPolynomial coefficients: #(2 -3 1)) 
		    + (DhbPolynomial coefficients: #(-3 7 2 1)).
	self should: [(polynomial at: 0) = -1].
	self should: [(polynomial at: 1) = 4].
	self should: [(polynomial at: 2) = 3].
	self should: [(polynomial at: 3) = 1].
	self should: [(polynomial at: 4) = 0]
    ]

    testPolynomialDerivative [
	"Code example 2.3"

	<category: 'function evaluation'>
	| polynomial |
	polynomial := (DhbPolynomial coefficients: #(-3 7 2 1)) derivative.
	self should: [(polynomial at: 0) = 7].
	self should: [(polynomial at: 1) = 4].
	self should: [(polynomial at: 2) = 3].
	self should: [(polynomial at: 3) = 0].
	self should: [(polynomial at: 4) = 0]
    ]

    testPolynomialDivision [
	<category: 'function evaluation'>
	| pol1 pol2 polynomial |
	pol1 := DhbPolynomial coefficients: #(2 -3 1).
	pol2 := DhbPolynomial coefficients: #(-6 23 -20 3 -1 1).
	polynomial := pol2 / pol1.
	self should: [(polynomial at: 0) = -3].
	self should: [(polynomial at: 1) = 7].
	self should: [(polynomial at: 2) = 2].
	self should: [(polynomial at: 3) = 1].
	self should: [(polynomial at: 4) = 0].
	self should: [(polynomial at: 5) = 0].
	self should: [(polynomial at: 6) = 0]
    ]

    testPolynomialEvaluation [
	"Code example 2.2"

	<category: 'function evaluation'>
	| polynomial |
	polynomial := DhbPolynomial coefficients: #(2 -3 1).
	self should: [0 = (polynomial value: 1)]
    ]

    testPolynomialIntegral [
	"Code example 2.3"

	<category: 'function evaluation'>
	| polynomial |
	polynomial := (DhbPolynomial coefficients: #(-3 7 2 1)) integral.
	self should: [(polynomial at: 0) = 0].
	self should: [(polynomial at: 1) = -3].
	self should: [(polynomial at: 2) = (7 / 2)].
	self should: [(polynomial at: 3) = (2 / 3)].
	self should: [(polynomial at: 4) = (1 / 4)].
	self should: [(polynomial at: 5) = 0]
    ]

    testPolynomialIntegralWithConstant [
	"Code example 2.3"

	<category: 'function evaluation'>
	| polynomial |
	polynomial := (DhbPolynomial coefficients: #(-3 7 2 1)) integral: 5.
	self should: [(polynomial at: 0) = 5].
	self should: [(polynomial at: 1) = -3].
	self should: [(polynomial at: 2) = (7 / 2)].
	self should: [(polynomial at: 3) = (2 / 3)].
	self should: [(polynomial at: 4) = (1 / 4)].
	self should: [(polynomial at: 5) = 0]
    ]

    testPolynomialMultiplication [
	"Code example 2.3"

	<category: 'function evaluation'>
	| pol1 pol2 polynomial |
	pol1 := DhbPolynomial coefficients: #(2 -3 1).
	pol2 := DhbPolynomial coefficients: #(-3 7 2 1).
	polynomial := pol1 * pol2.
	self should: [(polynomial at: 0) = -6].
	self should: [(polynomial at: 1) = 23].
	self should: [(polynomial at: 2) = -20].
	self should: [(polynomial at: 3) = 3].
	self should: [(polynomial at: 4) = -1].
	self should: [(polynomial at: 5) = 1].
	self should: [(polynomial at: 6) = 0]
    ]

    testPolynomialNumberAddition [
	<category: 'function evaluation'>
	| polynomial |
	polynomial := 2 + (DhbPolynomial coefficients: #(2 -3 1)).
	self should: [(polynomial at: 0) = 4].
	self should: [(polynomial at: 1) = -3].
	self should: [(polynomial at: 2) = 1].
	self should: [(polynomial at: 3) = 0]
    ]

    testPolynomialNumberAdditionInverse [
	<category: 'function evaluation'>
	| polynomial |
	polynomial := (DhbPolynomial coefficients: #(2 -3 1)) + 2.
	self should: [(polynomial at: 0) = 4].
	self should: [(polynomial at: 1) = -3].
	self should: [(polynomial at: 2) = 1].
	self should: [(polynomial at: 3) = 0]
    ]

    testPolynomialNumberDivision [
	<category: 'function evaluation'>
	| polynomial |
	polynomial := (DhbPolynomial coefficients: #(2 -3 1)) / 2.
	self should: [(polynomial at: 0) = 1].
	self should: [(polynomial at: 1) = (-3 / 2)].
	self should: [(polynomial at: 2) = (1 / 2)].
	self should: [(polynomial at: 3) = 0]
    ]

    testPolynomialNumberMultiplication [
	<category: 'function evaluation'>
	| polynomial |
	polynomial := 2 * (DhbPolynomial coefficients: #(2 -3 1)).
	self should: [(polynomial at: 0) = 4].
	self should: [(polynomial at: 1) = -6].
	self should: [(polynomial at: 2) = 2].
	self should: [(polynomial at: 3) = 0]
    ]

    testPolynomialNumberMultiplicationInverse [
	<category: 'function evaluation'>
	| polynomial |
	polynomial := (DhbPolynomial coefficients: #(2 -3 1)) * 2.
	self should: [(polynomial at: 0) = 4].
	self should: [(polynomial at: 1) = -6].
	self should: [(polynomial at: 2) = 2].
	self should: [(polynomial at: 3) = 0]
    ]

    testPolynomialNumberSubtraction [
	<category: 'function evaluation'>
	| polynomial |
	polynomial := 2 - (DhbPolynomial coefficients: #(2 -3 1)).
	self should: [(polynomial at: 0) = 0].
	self should: [(polynomial at: 1) = 3].
	self should: [(polynomial at: 2) = -1].
	self should: [(polynomial at: 3) = 0]
    ]

    testPolynomialNumberSubtractionInverse [
	<category: 'function evaluation'>
	| polynomial |
	polynomial := (DhbPolynomial coefficients: #(2 -3 1)) - 2.
	self should: [(polynomial at: 0) = 0].
	self should: [(polynomial at: 1) = -3].
	self should: [(polynomial at: 2) = 1].
	self should: [(polynomial at: 3) = 0]
    ]

    testPolynomialSubtraction [
	<category: 'function evaluation'>
	| polynomial |
	polynomial := (DhbPolynomial coefficients: #(2 -3 1)) 
		    - (DhbPolynomial coefficients: #(-3 7 2 1)).
	self should: [(polynomial at: 0) = 5].
	self should: [(polynomial at: 1) = -10].
	self should: [(polynomial at: 2) = -1].
	self should: [(polynomial at: 3) = -1].
	self should: [(polynomial at: 4) = 0]
    ]

    testBissection [
	"Code Example 5.1"

	<category: 'iterative algorithms'>
	| zeroFinder result |
	zeroFinder := DhbBisectionZeroFinder 
		    function: [:x | x errorFunction - 0.9000000000000001].
	zeroFinder
	    setPositiveX: 10.0;
	    setNegativeX: 0.0.
	result := zeroFinder evaluate.
	self should: [zeroFinder hasConverged].
	self should: [(result - 1.28155193291605) abs < 0.00000000000001]
    ]

    testIncompleteBetaFunction [
	<category: 'iterative algorithms'>
	| function |
	function := DhbIncompleteBetaFunction shape: 2 shape: 5.
	self should: 
		[((function value: 0.8000000000000001) - 0.9984000000000001) abs < 0.00001]
    ]

    testIncompleteGammaFunction [
	<category: 'iterative algorithms'>
	| function |
	function := DhbIncompleteGammaFunction shape: 2.
	self should: [((function value: 2) - 0.59399414981) abs < 0.00000000001]
    ]

    testIntegrationRomberg [
	<category: 'iterative algorithms'>
	| integrator ln2 ln3 |
	integrator := DhbRombergIntegrator 
		    function: [:x | 1.0 / x]
		    from: 1
		    to: 2.
	ln2 := integrator evaluate.
	integrator from: 1 to: 3.
	ln3 := integrator evaluate.
	self should: [(2.0 ln - ln2) abs < (2 * integrator precision)].
	self should: [(3.0 ln - ln3) abs < (2 * integrator precision)]
    ]

    testIntegrationSimpson [
	<category: 'iterative algorithms'>
	| integrator ln2 ln3 |
	integrator := DhbSimpsonIntegrator 
		    function: [:x | 1.0 / x]
		    from: 1
		    to: 2.
	ln2 := integrator evaluate.
	integrator from: 1 to: 3.
	ln3 := integrator evaluate.
	self should: [(2.0 ln - ln2) abs < integrator precision].
	self should: [(3.0 ln - ln3) abs < integrator precision]
    ]

    testIntegrationTrapeze [
	"Code Example 6.1"

	<category: 'iterative algorithms'>
	| integrator ln2 ln3 |
	integrator := DhbTrapezeIntegrator 
		    function: [:x | 1.0 / x]
		    from: 1
		    to: 2.
	ln2 := integrator evaluate.
	integrator from: 1 to: 3.
	ln3 := integrator evaluate.
	self should: [(2.0 ln - ln2) abs < integrator precision].
	self should: [(3.0 ln - ln3) abs < integrator precision]
    ]

    testNewtonZeroFinder [
	"Code Example 5.3"

	<category: 'iterative algorithms'>
	| zeroFinder result |
	zeroFinder := DhbNewtonZeroFinder 
		    function: [:x | x errorFunction - 0.9000000000000001].
	zeroFinder initialValue: 1.0.
	result := zeroFinder evaluate.
	self should: [zeroFinder hasConverged].
	self should: [(result - 1.28155193867885) abs < zeroFinder precision]
    ]

    testPolynomialRoots [
	"Code Example 5.5"

	<category: 'iterative algorithms'>
	| polynomial roots |
	polynomial := DhbPolynomial coefficients: #(-10 -13 -2 1).
	roots := polynomial roots asSortedCollection asArray.
	self should: [roots size = 3].
	self should: 
		[((roots at: 1) + 2) abs 
		    < DhbFloatingPointMachine new defaultNumericalPrecision].
	self should: 
		[((roots at: 2) + 1) abs 
		    < DhbFloatingPointMachine new defaultNumericalPrecision].
	self should: 
		[((roots at: 3) - 5) abs 
		    < DhbFloatingPointMachine new defaultNumericalPrecision]
    ]

    testDeterminant [
	<category: 'linear algebra'>
	| m |
	m := DhbMatrix rows: #(#(3 2 4) #(2 -5 -1) #(1 -2 2)).
	self should: [m determinant = -42]
    ]

    testEigenvalues [
	"Code Example 8.15"

	<category: 'linear algebra'>
	| m charPol roots eigenvalues finder |
	m := DhbMatrix rows: #(#(3 -2 0) #(-2 7 1) #(0 1 5)).
	charPol := DhbPolynomial coefficients: #(82 -66 15 -1).
	roots := charPol roots asSortedCollection asArray reverse.
	finder := DhbJacobiTransformation matrix: m.
	finder desiredPrecision: 0.000000001.
	eigenvalues := finder evaluate.
	self should: [eigenvalues size = 3].
	self should: [((roots at: 1) - (eigenvalues at: 1)) abs < 0.000000001].
	self should: [((roots at: 2) - (eigenvalues at: 2)) abs < 0.000000001].
	self should: [((roots at: 3) - (eigenvalues at: 3)) abs < 0.000000001]
    ]

    testEigenvaluesLargest [
	"Code Example 8.13"

	<category: 'linear algebra'>
	| m charPol roots eigenvalue finder |
	m := DhbMatrix rows: #(#(3 -2 0) #(-2 7 1) #(0 1 5)).
	charPol := DhbPolynomial coefficients: #(82 -66 15 -1).
	roots := charPol roots asSortedCollection asArray reverse.
	finder := DhbLargestEigenValueFinder matrix: m.
	finder desiredPrecision: 0.00000001.
	eigenvalue := finder evaluate.
	self should: [((roots at: 1) - eigenvalue) abs < 0.00000001].
	finder := finder nextLargestEigenValueFinder.
	eigenvalue := finder evaluate.
	self should: [((roots at: 2) - eigenvalue) abs < 0.00000001]
    ]

    testLUPDecomposition [
	"Code Example 8.10"

	<category: 'linear algebra'>
	| s sol1 sol2 |
	s := DhbLUPDecomposition equations: #(#(3 2 4) #(2 -5 -1) #(1 -2 2)).
	sol1 := s solve: #(16 6 10).
	sol2 := s solve: #(7 10 9).
	self should: [sol1 size = 3].
	self should: [(sol1 at: 1) = 2].
	self should: [(sol1 at: 2) = -1].
	self should: [(sol1 at: 3) = 3].
	self should: [sol2 size = 3].
	self should: [(sol2 at: 1) = 1].
	self should: [(sol2 at: 2) = -2].
	self should: [(sol2 at: 3) = 2]
    ]

    testLinearEquations [
	"Code Example 8.6"

	<category: 'linear algebra'>
	| s sol1 sol2 |
	s := DhbLinearEquationSystem equations: #(#(3 2 4) #(2 -5 -1) #(1 -2 2))
		    constants: #(#(16 6 10) #(7 10 9)).
	sol1 := s solutionAt: 1.
	sol2 := s solutionAt: 2.
	self should: [sol1 size = 3].
	self should: [(sol1 at: 1) = 2].
	self should: [(sol1 at: 2) = -1].
	self should: [(sol1 at: 3) = 3].
	self should: [sol2 size = 3].
	self should: [(sol2 at: 1) = 1].
	self should: [(sol2 at: 2) = -2].
	self should: [(sol2 at: 3) = 2]
    ]

    testLinearEquationsSingle [
	<category: 'linear algebra'>
	| s sol |
	s := DhbLinearEquationSystem equations: #(#(1 2 0) #(3 5 4) #(5 6 3))
		    constant: #(0.1 12.5 10.3).
	sol := s solution.
	self should: [sol size = 3].
	self should: [(sol at: 1) equalsTo: 0.5].
	self should: [(sol at: 2) equalsTo: -0.2].
	self should: [(sol at: 3) equalsTo: 3.0]
    ]

    testLinearEquationsSingular [
	<category: 'linear algebra'>
	| s sol |
	s := DhbLinearEquationSystem equations: #(#(1 2 0) #(10 12 6) #(5 6 3))
		    constant: #(0.1 12.5 10.3).
	sol := s solution.
	self should: [sol isNil]
    ]

    testMatrixInversionSmall [
	<category: 'linear algebra'>
	| m c |
	m := DhbMatrix rows: #(#(3 2 4) #(2 -5 -1) #(1 -2 2)).
	c := m inverse * m.
	self should: [c numberOfRows = 3].
	self should: [c numberOfColumns = 3].
	self should: [((c rowAt: 1) at: 1) = 1].
	self should: [((c rowAt: 1) at: 2) = 0].
	self should: [((c rowAt: 1) at: 3) = 0].
	self should: [((c rowAt: 2) at: 1) = 0].
	self should: [((c rowAt: 2) at: 2) = 1].
	self should: [((c rowAt: 2) at: 3) = 0].
	self should: [((c rowAt: 3) at: 1) = 0].
	self should: [((c rowAt: 3) at: 2) = 0].
	self should: [((c rowAt: 3) at: 3) = 1]
    ]

    testMatrixOperation [
	"Code Example 8.1"

	<category: 'linear algebra'>
	| a b c |
	a := DhbMatrix rows: #(#(1 0 1) #(-1 -2 3)).
	b := DhbMatrix rows: #(#(1 2 3) #(-2 1 7) #(5 6 7)).
	c := a * b.
	self should: [c numberOfRows = 2].
	self should: [c numberOfColumns = 3].
	self should: [((c rowAt: 1) at: 1) = 6].
	self should: [((c rowAt: 1) at: 2) = 8].
	self should: [((c rowAt: 1) at: 3) = 10].
	self should: [((c rowAt: 2) at: 1) = 18].
	self should: [((c rowAt: 2) at: 2) = 14].
	self should: [((c rowAt: 2) at: 3) = 4]
    ]

    testVectorMatrixOperation [
	"Code Example 8.1"

	<category: 'linear algebra'>
	| a u v |
	a := DhbMatrix rows: #(#(1 0 1) #(-1 -2 3)).
	u := #(1 2 3) asVector.
	v := a * u.
	self should: [v size = 2].
	self should: [(v at: 1) = 4].
	self should: [(v at: 2) = 4]
    ]

    testVectorOperation [
	"Code Example 8.1"

	<category: 'linear algebra'>
	| u v w |
	u := #(1 2 3) asVector.
	v := #(3 4 5) asVector.
	w := 4 * u + (3 * v).
	self should: [w size = 3].
	self should: [(w at: 1) = 13].
	self should: [(w at: 2) = 20].
	self should: [(w at: 3) = 27]
    ]

    testVectorOperationInverse [
	<category: 'linear algebra'>
	| u v w |
	u := #(1 2 3) asVector.
	v := #(3 4 5) asVector.
	w := v * 4 - (3 * u).
	self should: [w size = 3].
	self should: [(w at: 1) = 9].
	self should: [(w at: 2) = 10].
	self should: [(w at: 3) = 11]
    ]

    testVectorProduct [
	"Code Example 8.1"

	<category: 'linear algebra'>
	| u v |
	u := #(1 2 3) asVector.
	v := #(3 4 5) asVector.
	self should: [u * v = 26]
    ]

    testVectorTransposeMatrixOperation [
	"Code Example 8.1"

	<category: 'linear algebra'>
	| c v w |
	c := DhbMatrix rows: #(#(6 8 10) #(18 14 4)).
	v := #(4 4) asVector.
	w := c transpose * v.
	self should: [w size = 3].
	self should: [(w at: 1) = 96].
	self should: [(w at: 2) = 88].
	self should: [(w at: 3) = 56]
    ]

    testOptimize [
	"General optimizer to test genetic algorithm"

	<category: 'optimization'>
	| fBlock finder result |
	fBlock := 
		[:x | 
		| r |
		r := x * x.
		r = 0 ifTrue: [1] ifFalse: [r sqrt sin / r]].
	finder := DhbMultiVariableGeneralOptimizer maximizingFunction: fBlock.
	finder desiredPrecision: 0.000001.
	finder
	    origin: #(0.5 1.0 0.5) asVector;
	    range: #(2 2 2) asVector.
	result := finder evaluate.
	self should: [finder precision < 0.000001].
	self should: [(result at: 1) abs < 0.000001].
	self should: [(result at: 2) abs < 0.000001].
	self should: [(result at: 3) abs < 0.000001]
    ]

    testOptimizeOneDimension [
	"Code example 11.1"

	<category: 'optimization'>
	| distr finder maximum |
	distr := DhbGammaDistribution shape: 2 scale: 5.
	finder := DhbOneVariableFunctionOptimizer maximizingFunction: distr.
	finder desiredPrecision: 0.000001.
	maximum := finder evaluate.
	self should: [(maximum - 5) abs < 0.000001].
	self should: [finder precision < 0.000001]
    ]

    testOptimizePowell [
	"Code example 11.3"

	<category: 'optimization'>
	| fBlock hillClimber educatedGuess result |
	fBlock := [:x | (x * x) negated exp].
	educatedGuess := #(0.5 1.0 0.5) asVector.
	hillClimber := DhbHillClimbingOptimizer maximizingFunction: fBlock.
	hillClimber initialValue: educatedGuess.
	hillClimber desiredPrecision: 0.000001.
	result := hillClimber evaluate.
	self should: [hillClimber precision < 0.000001].
	self should: [(result at: 1) abs < 0.000001].
	self should: [(result at: 2) abs < 0.000001].
	self should: [(result at: 3) abs < 0.000001]
    ]

    testOptimizeSimplex [
	"Code example 11.5"

	<category: 'optimization'>
	| fBlock simplex educatedGuess result |
	fBlock := [:x | (x * x) negated exp].
	educatedGuess := #(0.5 1.0 0.5) asVector.
	simplex := DhbSimplexOptimizer maximizingFunction: fBlock.
	simplex initialValue: educatedGuess.
	simplex desiredPrecision: 0.000001.
	result := simplex evaluate.
	self should: [simplex precision < 0.000001].
	self should: [(result at: 1) abs < 0.000001].
	self should: [(result at: 2) abs < 0.000001].
	self should: [(result at: 3) abs < 0.000001]
    ]

    accumulateAround: aVector size: aNumber into: aCollection [
	"Private - Generate a random point around the given center and insert it into the collection.
	 aNumber is the sigma for the distance to the center"

	<category: 'privateMethods'>
	| r phi psi localVector |
	r := (DhbNormalDistribution new: 0 sigma: aNumber) random.
	phi := FloatD pi random.
	psi := FloatD pi random.
	localVector := DhbVector new: 3.
	localVector
	    at: 1 put: phi sin * psi sin * r;
	    at: 2 put: phi cos * psi sin * r;
	    at: 3 put: psi cos * r.
	aCollection add: localVector + aVector
    ]

    generatedPoints: anInteger [
	"Private - Generate random points into aCollection. 3 clusters are used"

	<category: 'privateMethods'>
	| centers results |
	centers := Array new: 3.
	centers
	    at: 1 put: #(200 200 200) asVector;
	    at: 2 put: #(-200 200 200) asVector;
	    at: 3 put: #(200 200 -200) asVector.
	results := OrderedCollection new.
	anInteger timesRepeat: 
		[self 
		    accumulateAround: (centers at: 3 random + 1)
		    size: 1
		    into: results].
	^results
    ]

    setUp [
	"Reset the seed of the random numbers (to get consistent results)"

	<category: 'privateMethods'>
	DhbMitchellMooreGenerator reset: 0
    ]

    testGammaDistribution [
	<category: 'statistics'>
	| dist |
	dist := DhbGammaDistribution shape: 3.4 scale: 1.7.
	self should: [(dist average - (3.4 * 1.7)) abs < 0.000000001].
	self 
	    should: [(dist standardDeviation - (3.4 sqrt * 1.7)) abs < 0.000000001].
	self should: [((dist value: 4.5) - 0.1446067652) abs < 0.000000001].
	self 
	    should: [((dist distributionValue: 4.5) - 0.3982869736) abs < 0.000000001]
    ]

    testHistogram [
	<category: 'statistics'>
	| histogram |
	histogram := DhbHistogram new.
	histogram 
	    setRangeFrom: 0.0
	    to: 48.0
	    bins: 8.
	#(36 13 27 16 33 24 4 20 15 23 37 23 31 15 47 22 6 15 41 22 14 14 31 42 3 42 22 8 37 41) 
	    do: [:x | histogram accumulate: x].
	histogram
	    accumulate: -1;
	    accumulate: 55;
	    accumulate: 56.
	self should: [histogram count = 30].
	self should: [histogram underflow = 1].
	self should: [histogram overflow = 2].
	self should: [(histogram countAt: 1) = 3].
	self should: [(histogram countAt: 8.5) = 4].
	self should: [(histogram countAt: 16) = 8].
	self should: [(histogram countAt: 23.5) = 4].
	self should: [(histogram countAt: 31) = 6].
	self should: [(histogram countAt: 38.5) = 4].
	self should: [(histogram countAt: 46) = 1].
	self should: [(histogram average - 24.1333333333) abs < 0.000000001].
	self 
	    should: [(histogram standardDeviation - 12.461619237603) abs < 0.000000001].
	self should: [(histogram skewness - 0.116659884676) abs < 0.000000001].
	self should: [(histogram kurtosis + 1.004665562311) abs < 0.000000001]
    ]

    testNormalDistribution [
	<category: 'statistics'>
	| dist |
	dist := DhbNormalDistribution new: 3.4 sigma: 1.7.
	self should: [(dist average - 3.4) abs < 0.000000001].
	self should: [(dist standardDeviation - 1.7) abs < 0.000000001].
	self should: [((dist value: 4.5) - 0.1903464693) abs < 0.000000001].
	self should: 
		[((dist distributionValue: 4.5) - 0.7412031298000001) abs < 0.000000001]
    ]

    testStatisticalMoments [
	"comment"

	<category: 'statistics'>
	| accumulator |
	accumulator := DhbStatisticalMoments new.
	#(36 13 27 16 33 24 4 20 15 23 37 23 31 15 47 22 6 15 41 22 14 14 31 42 3 42 22 8 37 41) 
	    do: [:x | accumulator accumulate: x].
	self should: [(accumulator average - 24.1333333333) abs < 0.000000001].
	self 
	    should: [(accumulator standardDeviation - 12.461619237603) abs < 0.000000001].
	self should: [(accumulator skewness - 0.116659884676) abs < 0.000000001].
	self should: [(accumulator kurtosis + 1.004665562311) abs < 0.000000001]
    ]

    testStatisticalMomentsFast [
	<category: 'statistics'>
	| accumulator |
	accumulator := DhbFastStatisticalMoments new.
	#(36 13 27 16 33 24 4 20 15 23 37 23 31 15 47 22 6 15 41 22 14 14 31 42 3 42 22 8 37 41) 
	    do: [:x | accumulator accumulate: x].
	self should: [(accumulator average - 24.1333333333) abs < 0.000000001].
	self 
	    should: [(accumulator standardDeviation - 12.461619237603) abs < 0.000000001].
	self should: [(accumulator skewness - 0.116659884676) abs < 0.000000001].
	self should: [(accumulator kurtosis + 1.004665562311) abs < 0.000000001]
    ]

    testStatisticalMomentsFixed [
	<category: 'statistics'>
	| accumulator |
	accumulator := DhbFixedStatisticalMoments new.
	#(36 13 27 16 33 24 4 20 15 23 37 23 31 15 47 22 6 15 41 22 14 14 31 42 3 42 22 8 37 41) 
	    do: [:x | accumulator accumulate: x].
	self should: [(accumulator average - 24.1333333333) abs < 0.000000001].
	self 
	    should: [(accumulator standardDeviation - 12.461619237603) abs < 0.000000001].
	self should: [(accumulator skewness - 0.116659884676) abs < 0.000000001].
	self should: [(accumulator kurtosis + 1.004665562311) abs < 0.000000001]
    ]
]



DhbTestCase subclass: DhbTestNumericPrecision [
    
    <comment: nil>
    <category: 'DHB Numerical SUnits'>

    testDecimalAdd [
	<category: 'decimal floating number'>
	| a b |
	a := DhbDecimalFloatingNumber new: 1.
	b := DhbDecimalFloatingNumber new: 2.
	self assert: (a + b) value = 3.
	a := DhbDecimalFloatingNumber new: 1.56.
	b := DhbDecimalFloatingNumber new: 2.2.
	self assert: (a + b) value = 3.76
    ]

    testDecimalMultiple [
	<category: 'decimal floating number'>
	| a b result |
	a := DhbDecimalFloatingNumber new: 2.
	b := DhbDecimalFloatingNumber new: 2.
	self assert: (a * b) value = 4.
	a := DhbDecimalFloatingNumber new: 1.5.
	b := DhbDecimalFloatingNumber new: 2.5.
	result := (a * b) value asFloat - (1.5 * 2.5).
	self assert: result abs < 0.00001
    ]

    testComputeLargestNumber [
	<category: 'floating point machine'>
	| machine |
	machine := DhbFloatingPointMachine new.
	machine computeLargestNumber.
	self assert: machine largestNumber > 1.0000001e25
    ]

    testComputeSmallestNumber [
	<category: 'floating point machine'>
	| machine |
	machine := DhbFloatingPointMachine new.
	machine computeSmallestNumber.
	self assert: machine smallestNumber < 1.0e-25
    ]

    testMachinePrecision [
	<category: 'floating point machine'>
	| machine |
	machine := DhbFloatingPointMachine new.
	machine computeMachinePrecision.
	self assert: machine machinePrecision < 0.0000099999998e
    ]
]



DhbTestCase subclass: DhbTestFunctions [
    
    <comment: nil>
    <category: 'DHB Numerical SUnits'>

    testAdd [
	<category: 'polynomial'>
	| a b c |
	a := DhbPolynomial coefficients: #(2 1 1).
	c := a + 3.
	self assert: c coefficients = #(5 1 1).
	b := DhbPolynomial coefficients: #(1 5).
	c := a + b.
	self assert: c coefficients = #(3 6 1)
    ]

    testDerivative [
	<category: 'polynomial'>
	| a answer |
	a := DhbPolynomial coefficients: #(5 2 3).
	answer := DhbPolynomial coefficients: #(2 6).
	self assert: a derivative = answer
    ]

    testDivide [
	<category: 'polynomial'>
	| a b c answer |
	a := DhbPolynomial coefficients: #(9 6 3).
	c := a / 3.
	self assert: c coefficients = #(3 2 1).
	answer := DhbPolynomial coefficients: #(3 3).
	b := DhbPolynomial coefficients: #(1 1).
	a := answer * b.
	c := a / b.
	self assert: c = answer
    ]

    testEqual [
	<category: 'polynomial'>
	| a b |
	a := DhbPolynomial coefficients: #(5 2 3).
	b := DhbPolynomial coefficients: #(5 2 3).
	self assert: a = b
    ]

    testIntegral [
	<category: 'polynomial'>
	| a b |
	a := DhbPolynomial coefficients: #(0 2 3).
	b := a derivative.
	self assert: b integral = a
    ]

    testMultiply [
	<category: 'polynomial'>
	| a b c |
	a := DhbPolynomial coefficients: #(3 2 1).
	c := a * 3.
	self assert: c coefficients = #(9 6 3).
	b := DhbPolynomial coefficients: #(1 1).
	c := a * b.
	self assert: c coefficients = #(3 5 3 1)
    ]

    testValue [
	<category: 'polynomial'>
	| square |
	square := DhbPolynomial coefficients: #(2 1 1).
	self
	    assert: (square value: 1) = 4;
	    assert: (square value: 3) = (2 + (1 * 3) + (1 * 3 * 3)).
	square := DhbPolynomial coefficients: #(2).
	self
	    assert: (square value: 1) = 2;
	    assert: (square value: 0) = 2
    ]
]



DhbTestCase subclass: DhbTestBeta [
    
    <comment: nil>
    <category: 'DHB Numerical SUnits'>

    testBeta [
	<category: 'tests'>
	self assert: (2 beta: 4) isNumber
    ]
]

