//  Gnomoradio - roboradio/song-list-search.cc
//  Copyright (C) 2003  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version)
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "roboradio/song-list-search.h"
#include "roboradio/sort-songs.h"
#include "roboradio/init.h"

#include <libxml++/libxml++.h>
#include <algorithm>

using namespace Roboradio;
using namespace Rainbow;
using namespace xmlpp;
using namespace Glib;
using namespace std;

namespace Roboradio {

	static bool cmp_less (int a, int b) { return a < b; }
	static bool cmp_greater (int a, int b) { return a > b; }
	static bool cmp_less_equal (int a, int b) { return a <= b; }
	static bool cmp_greater_equal (int a, int b) { return a >= b; }
	static bool cmp_equal (int a, int b) { return a == b; }
	static bool cmp_not_equal (int a, int b) { return a != b; }

	class Criteria {
	public:		
 		static Criteria *create (Node *c);

		Criteria () {}
		
		virtual bool test (const SongRef &song) {
			return false;
		}

	};

	class CriteriaInt : public Criteria
	{
	public:
		CriteriaInt (Element *node) {
			TextNode *n = node->get_child_text();
			num = n ? atoi(n->get_content().c_str()) : 1;
			
			// determine comparison operator
			compare = cmp_less; // by default
			Attribute *a = node->get_attribute("compare");
			if (a) {
				ustring cmp = a->get_value();
				if (cmp == "less") {
					compare = cmp_less;
				} else if (cmp == "greater") {
					compare = cmp_greater;
				} else if (cmp == "less_equal") {
					compare = cmp_less_equal;
				} else if (cmp == "greater_equal") {
					compare = cmp_greater_equal;
				} else if (cmp == "equal") {
					compare = cmp_equal;
				} else if (cmp == "not_equal") {
					compare = cmp_not_equal;
				}
			}
		}
		
		virtual ~CriteriaInt () {}

	protected:
		int num;
		bool (*compare)(int,int);
	};

	class CriteriaRating : public CriteriaInt
	{
	public:
		CriteriaRating (Element *n) : CriteriaInt(n) {}
		virtual ~CriteriaRating () {}

		bool test (const SongRef &song) {
			return compare(song->get_rating(), num);
		}
	};

	class CriteriaLength : public CriteriaInt
	{
	public:
		CriteriaLength (Element *n) : CriteriaInt(n) {}
		virtual ~CriteriaLength () {}

		bool test (const SongRef &song) {
			return compare(song->get_length(), num);
		}
	};

	class CriteriaLast : public CriteriaInt
	{
	public:
		CriteriaLast (Element *node) : CriteriaInt(node) {
			Attribute *a = node->get_attribute("unit");
			if (!a)
				return;
			ustring unit = a->get_value();
			
			if (unit == "minutes")
				num *= 60;
			else if (unit == "hours")
				num *= 60 * 60;
			else if (unit == "days")
				num *= 60 * 60 * 24;
			else if (unit == "weeks")
				num *= 60 * 60 * 24 * 7;
		}

		virtual ~CriteriaLast () {}

		bool test (const SongRef &song) {
			return compare(time(0) - song->last_played(), num);
		}
	};

	class CriteriaInfo : public Criteria {
	protected:
		vector<ustring> keys, values;

	public:
		CriteriaInfo (Element *n) {
			TextNode *tn = n->get_child_text();
			if (tn) {
				values.push_back(tn->get_content()); // FIXME: split
			}
			Attribute *k = n->get_attribute("key");
			if (k)
				keys.push_back(k->get_value());

			// lowercase all
			for (vector<ustring>::iterator i = values.begin();
			     i != values.end(); ++i)
				(*i) = i->lowercase();
			for (vector<ustring>::iterator j = keys.begin();
			     j != keys.end(); ++j)
				(*j) = j->lowercase();
		}

		bool test (const SongRef &song) {
			if (keys.empty()) {
				vector<ustring> k, v;
				song->get_info(k, v);
				for (vector<ustring>::iterator i = v.begin();
				     i != v.end(); ++i) {
					for (vector<ustring>::iterator j = values.begin();
					     j != values.end(); ++j) {
						ustring v_lower = i->lowercase();
						if (v_lower.find(*j) != ustring::npos)
							return true;
					}
				}
			} else {
				for (vector<ustring>::iterator i = keys.begin(); i != keys.end(); ++i) {
					for (vector<ustring>::iterator j = values.begin();
					     j != values.end(); ++j) {
						ustring v_lower = song->get_info(*i).lowercase();
						if (v_lower.find(*j) != ustring::npos)
							return true;
					}
				}
			}

			return false;
		}

		virtual ~CriteriaInfo () {}
	};

	class CriteriaBoolean : public Criteria
	{
	public:
		CriteriaBoolean (Element *node) {
			is_and = true; // by default
			Attribute *a = node->get_attribute("type");
			if (a && a->get_value() == "or")
				is_and = false;

			Node::NodeList subnodes = node->get_children();
			for (Node::NodeList::iterator i = subnodes.begin();
			     i != subnodes.end(); ++i) {
				Element *el = dynamic_cast<Element*>(*i);
				if (el)
					sub_criteria.push_back(Criteria::create(el));
			}
		}

		~CriteriaBoolean () {
			for (vector<Criteria*>::iterator i = sub_criteria.begin();
			     i != sub_criteria.end(); ++i)
				delete *i;
		}

		bool test (const SongRef &song) {
			if (sub_criteria.size() == 0)
				return false;

			for (vector<Criteria*>::iterator i = sub_criteria.begin();
			     i != sub_criteria.end(); ++i) {
				if ((*i)->test(song) != is_and)
					return !is_and;
			}

			return is_and;
		}

	protected:
		vector<Criteria*> sub_criteria;
		bool is_and; // 1 for and, 0 for or
	};

	Criteria *Criteria::create (Node *c)
	{
		Element *crit;
		if (!(c && (crit = dynamic_cast<Element*>(c))))
			return new Criteria;
		else if (crit->get_name() == "rating")
			return new CriteriaRating(crit);
		else if (crit->get_name() == "info")
			return new CriteriaInfo(crit);
		else if (crit->get_name() == "length")
			return new CriteriaLength(crit);
		else if (crit->get_name() == "last")
			return new CriteriaLast(crit);
		else if (crit->get_name() == "boolean")
			return new CriteriaBoolean(crit);
		else
			return new Criteria;
	}
		

}

Roboradio::SongListSearch::SongListSearch (const ustring &c,
					   const ustring &name)
	: SongList(name),
	  criteria(0),
	  online(0),
	  http(0)
{
	do_upcoming_ref = false;

	set_criteria(c);

	Song::signal_global_new_song.connect(mem_fun(*this, &SongListSearch::on_song_changed));
	Song::signal_global_song_info_changed.connect(mem_fun(*this, &SongListSearch::on_song_changed));
	Song::signal_global_song_rating_changed.connect(mem_fun(*this, &SongListSearch::on_song_changed));
	Song::signal_global_song_length_changed.connect(mem_fun(*this, &SongListSearch::on_song_changed));
	Song::signal_global_song_done.connect(mem_fun(*this, &SongListSearch::on_song_changed));
}

Roboradio::SongListSearch::~SongListSearch ()
{
}

void Roboradio::SongListSearch::set_criteria (const ustring &c)
{
	clear_except_playing();

	if (criteria) {
		delete criteria;
		criteria = 0;
	}

	DomParser tree;
	try {
		tree.parse_memory(c);
	} catch (...) {
		return;
	}
	criteria = Criteria::create(tree.get_document()->get_root_node());

	SortSongs sort(SortSongs::Artist);
	for (SortSongs::iterator i = sort.begin(); i != sort.end(); ++i) {
		if (criteria->test(*i)) {
			push_back(*i);
			if (&*current_song && &**i == &**current_song) {
				current_song = rbegin();
				signal_current_song_changed();
			}
		}
	}
	if (&*current_song)
		pop_front();

	crit = c;

	online = false;
}

ustring Roboradio::SongListSearch::get_criteria () const
{
	return crit;
}

void Roboradio::SongListSearch::search_online (const ustring &search_string)
{
	if (online || http || !&Roboradio::Init::get_rainbow())
		return;
	online = true;

	http = new HttpClient("search.gnomoradio.org");
	http->signal_request_done.connect(mem_fun(*this, &SongListSearch::on_online_search_done));
	http->get("/search.php?q=" + Rainbow::HttpClient::url_encode(search_string.c_str()));
}

void Roboradio::SongListSearch::on_online_search_done (bool success)
{
	if (!success || !online) {
		// !online means that the list has changed since the search started
		delete http;
		http = 0;
		return;
	}

	DomParser tree;
	try {
		tree.parse_memory(http->get_buffer());
	} catch (...) {
		delete http;
		http = 0;
		return;
	}
	delete http;
	http = 0;

	Node *root = tree.get_document()->get_root_node();

	Node::NodeList results = root->get_children();
	for (Node::NodeList::iterator result = results.begin(); result != results.end(); ++result) {
		xmlpp::Element *song = dynamic_cast<xmlpp::Element*>(*result);
		if (!song)
			continue;

		Attribute *url = song->get_attribute("url");
		if (!url)
			continue;

		SongRef s(url->get_value(), false, true);
		Node::NodeList tags = song->get_children();
		for (Node::NodeList::iterator tag = tags.begin(); tag != tags.end(); ++tag) {
			xmlpp::Element *el = dynamic_cast<xmlpp::Element*>(*tag);
			if (!el)
				continue;

			if (s->get_info(el->get_name()) == "") {
				TextNode *val = el->get_child_text();
				if (val)
					s->set_info(el->get_name(), val->get_content());
			}
		}

		SongList::iterator pos = begin();
		while (pos != end() && &*s != &**pos)
			++pos;
		if (pos == end())
			push_back(s);
	}
}

void Roboradio::SongListSearch::on_song_changed (SongRef song)
{
	iterator pos = begin();
	while (pos != end() && &*song != &**pos)
		++pos;

	if (criteria->test(song)) {
		if (pos == end())
			push_back(song);
	} else {
		if (pos != end())
			remove(pos);
	}
}
