/********************************************
  Allows for a list of common away messages
  By David A. Hepkin
  for Jeremy Wise's GnomeICU
*********************************************/

#include "common.h"
#include "awaymsglist.h"
#include "gnomeicu.h"
#include "icons.h"

gchar *away_msgs[AWAY_MSGS_NUM];	/* The Away Message List */
static void cb_menu_position (GtkMenu * menu, gint * x, gint * y,
			      gpointer user_data);

/* popup_away_msgs():
   Invoked by: changeaway.c::change_away_window()
   Description: Creates the popup list of away messages when the user
     clicks on the "Away Messages" button in the away messages window.
     The popup menu contains AWAY_MSGS_NUM number of away messages.  To
     the left of each away message is the icon corresponding to the
     status to which the user is changing (i.e., N/A icon if switching
     to N/A mode).
   Pre-Conditions: main_win is a signal handler that should be called
     by the "Away Button" widget in changeaway.c.  Thus, main_win should
     be the "Away Button" widget and stuff should be NULL.  Also, in
     main_win's data, there must be a field "text" that points to the
     text window that contains the actual away message text.
*/
int popup_away_msgs (GtkWidget * main_win, gpointer stuff)
{
	GtkWidget *msg_menu;	/* Actual popup msg menu */
	GtkWidget *msg;
	GtkWidget *msg_box, *msg_name;
	GtkWidget *msg_pix;
	GdkBitmap *status_bitmap;
	GdkPixmap *status_pixmap;
	gint pos;		/* Loop counter */

	msg_menu = gtk_menu_new ();
	/* Makes menu cuter */
	status_bitmap = get_bitmap_for_status (Current_Status);
	status_pixmap = get_pixmap_for_status (Current_Status);

	/* This creates the menu entry for each away message, along with the
	   status icon to its left */
	for (pos = 0; pos < AWAY_MSGS_NUM; pos++)
	{
		if( away_msgs[pos] != NULL && strlen( away_msgs[pos] ) )
		{
			msg_box = gtk_hbox_new (FALSE, 0);
			/* This is for the status icon */
			msg_pix = gtk_pixmap_new (status_pixmap, status_bitmap);
			gtk_box_pack_start (GTK_BOX (msg_box), msg_pix, FALSE,
			                    FALSE, 5);
			gtk_widget_show (msg_pix);

			/* This creates the actual menu item with the away message */
			msg = gtk_menu_item_new ();
			msg_name = gtk_label_new (away_msgs[pos]);
			gtk_label_set_line_wrap( GTK_LABEL( msg_name ), TRUE );
			gtk_box_pack_start (GTK_BOX (msg_box), msg_name, FALSE,
			                    FALSE, 0);
			gtk_widget_show (msg_name);

			gtk_container_add (GTK_CONTAINER (msg), msg_box);
			gtk_widget_show (msg_box);
			gtk_menu_append (GTK_MENU (msg_menu), msg);

			/* The generalized signal handler to update the away message
			window must know: 1) the away message number and 2) the widget
			that is the text window to be updated */
			gtk_object_set_data (GTK_OBJECT (msg), "Num",
			                     GINT_TO_POINTER (pos));
			gtk_object_set_data (GTK_OBJECT (msg), "text",
			                     gtk_object_get_data (GTK_OBJECT
			                     (main_win),
			                     "text"));

			gtk_signal_connect (GTK_OBJECT (msg), "activate",
			                    GTK_SIGNAL_FUNC (update_away_text),
			                    NULL);

			gtk_widget_show (msg);
		}
	}

	gtk_widget_show (msg_menu);	/* Shows the popup menu */

	gtk_menu_popup (GTK_MENU (msg_menu), NULL, NULL, cb_menu_position,
	                main_win, 0, 0);

	return (TRUE);
}

/* This is just a copy of gtkfunc.c's cb_menu_position */
void cb_menu_position (GtkMenu * menu, gint * x, gint * y, gpointer user_data)
{
	GtkWidget *widget, *button;
	gint screen_width, screen_height;

	GtkRequisition requisition;

	widget = GTK_WIDGET (menu);
	button = GTK_WIDGET (user_data);

	gtk_widget_size_request (widget, &requisition);

	gdk_window_get_origin (button->window, x, y);
	*y -= requisition.height;

	screen_width = gdk_screen_width ();
	screen_height = gdk_screen_height ();

	if (*x + requisition.width > screen_width)
	  {
		  *x -= *x + requisition.width - screen_width;
	  }
}

/* get_config_file_away_msgs():
   Invoked By: gnomecfg.c::Read_RC_File()
   Description: Reads the away message list from the GnomeICU
     configuration file.
   Pre-Conditions: The name of the config file must have already been
     set.
*/
void get_config_file_away_msgs (void)
{
	const gulong buf_size = 16;
	char away_field[buf_size];
	int pos;
	for (pos = 0; pos < AWAY_MSGS_NUM; pos++)
	  {
		  g_free (away_msgs[pos]);
		  g_snprintf (away_field, buf_size, "Away/Msg%d", pos);
		  away_msgs[pos] = gnome_config_get_string (away_field);
	  }
}

/* set_config_file_away_msgs():
   Invoked By: gnomecfg.c::Save_RC()
   Description: Writes the away message list to the GnomeICU
     configuration file under the section "Away"
   Pre-Conditions: The name of the config file must have already been
     set.
   Post-Conditions: Each away message in away_msgs has been written to
     the configuration file under the section "Away."*/
void set_config_file_away_msgs (void)
{
	const gulong buf_size = 16;
	char away_field[buf_size];
	int pos;
	for (pos = 0; pos < AWAY_MSGS_NUM; pos++)
	  {
		  g_snprintf (away_field, buf_size, "Away/Msg%d", pos);
		  gnome_config_set_string (away_field, away_msgs[pos]);
	  }
}

/* update_away_text():
   Invoked By: changeaway.c::change_away_window()
   Description: Updates the away message text box in the the change
   away message window when one of the away messages is choosen from
   the away message list (away_msgs).
*/
int update_away_text (GtkWidget * away_button, gpointer text_win)
{
	int *data;
	GtkWidget *wid;
	data = gtk_object_get_data (GTK_OBJECT (away_button), "Num");
	wid = gtk_object_get_data (GTK_OBJECT (away_button), "text");
	gtk_editable_delete_text (GTK_EDITABLE (wid), 0, -1);
	gtk_text_insert (GTK_TEXT (wid), NULL, NULL, NULL,
			 away_msgs[GPOINTER_TO_INT (data)], -1);
	return (TRUE);
}

/* apply_away_msgs_change:
   Invoked By: gtkconf.c::save_changes()
   Description: Updates the away message list with any new or changed
   away messages from the options menu.
*/
void apply_away_msgs_change (gchar ** temp_away_msgs, const int num)
{
	gint pos;		/* Loop counter */

	if (num < 0 || num > AWAY_MSGS_NUM)
		return;
	for (pos = 0; pos < num; pos++)
	  {
		  g_free (away_msgs[pos]);
		  away_msgs[pos] = g_strdup (temp_away_msgs[pos]);
	  }
}

/* get_away_msg_from_list:
   Invoked By: Anyone
   Description: Because the away message list is "private" to the
   scope of this file, this allows outside functions to get a message
   from the away message list.
*/
const gchar *get_away_msg_from_list (const int num)
{
	if (num < 0 || num >= AWAY_MSGS_NUM)
		return NULL;
	return (away_msgs[num]);
}
