/* gnome-sudo.c -- GUI frontend to sudo.
 *
 * $Progeny: gnome-sudo/src/gnome-sudo.c,v 1.16 2001/11/01 18:06:26 epg Exp $
 *
 * Copyright (C) 2001  Progeny Linux Systems, Inc.
 * AUTHORS: Eric Gillespie, Jr. <epg@progeny.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2,  as
 * published by the Free Software Foundation.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/wait.h>

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <gnome.h>
#include <gdk/gdkx.h>
#include <X11/Xlib.h>

static char *pw = NULL;

static void
bomb(const char *format, ...)
{
	va_list ap;
	GtkWidget *dialog;
	char *msg;

	va_start(ap, format);
	msg = g_strdup_vprintf(format, ap);
	va_end(ap);

	dialog = gnome_error_dialog(msg);
	gtk_widget_show(dialog);
	gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
	exit(EXIT_FAILURE);
}

/* Write all of buf, even if write(2) is interrupted. */
static ssize_t
full_write(int d, const char *buf, size_t nbytes)
{
	ssize_t r, w = 0;

        /* Loop until nbytes of buf have been written. */
	while (w < nbytes) {
                /* Keep trying until write succeeds without interruption. */
                do {
                        r = write(d, buf + w, nbytes - w);
                } while (r < 0 && errno == EINTR);

		if (r < 0) {
			return -1;
                }

		w += r;
	}

	return w;
}

static void
copy(const char *fn, const char *dir)
{
	int in, out;
	int r;
	char *newfn;
	char buf[BUFSIZ];

	newfn = g_strdup_printf("%s/%s", dir, g_basename(fn));

	out = open(newfn, O_WRONLY | O_CREAT | O_EXCL);
	if (out == -1) {
		if (errno == EEXIST)
			bomb(_("The X authority file i am trying to create for"
			       " root already exists! This is highly"
			       " suspicous!"));
		else
			bomb(_("Error copying '%s' to '%s': %s"),
			     fn, dir, strerror(errno));
	}

	in = open(fn, O_RDONLY);
	if (in == -1)
		bomb(_("Error copying '%s' to '%s': %s"),
		     fn, dir, strerror(errno));

	while ((r = read(in, buf, BUFSIZ)) > 0) {
		if (full_write(out, buf, r) == -1)
			bomb(_("Error copying '%s' to '%s': %s"),
			     fn, dir, strerror(errno));
	}

	if (r == -1)
		bomb(_("Error copying '%s' to '%s': %s"),
		     fn, dir, strerror(errno));
}

/******************************************************************************
 * These next two functions come almost un-modified from gnome-ssh-askpass:

 * Copyright (c) 2000 Damien Miller.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

static void
report_failed_grab (void)
{
	GtkWidget *err;

	err = gnome_message_box_new("Could not grab keyboard or mouse.\n"
		"A malicious client may be eavesdropping on your session.",
				    GNOME_MESSAGE_BOX_ERROR, "EXIT", NULL);
	gtk_window_set_position(GTK_WINDOW(err), GTK_WIN_POS_CENTER);
	gtk_object_set(GTK_OBJECT(err), "type", GTK_WINDOW_POPUP, NULL);

	gnome_dialog_run_and_close(GNOME_DIALOG(err));
}

static void
passphrase_dialog(char *message)
{
	char *passphrase;
	char **messages;
	int result, i;
	
	GtkWidget *dialog, *entry, *label;

	dialog = gnome_dialog_new("OpenSSH", GNOME_STOCK_BUTTON_OK,
	    GNOME_STOCK_BUTTON_CANCEL, NULL);

	messages = g_strsplit(message, "\\n", 0);
	if (messages)
		for(i = 0; messages[i]; i++) {
			label = gtk_label_new(messages[i]);
			gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox),
			    label, FALSE, FALSE, 0);
		}

	entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), entry, FALSE, 
	    FALSE, 0);
	gtk_entry_set_visibility(GTK_ENTRY(entry), FALSE);
	gtk_widget_grab_focus(entry);

	/* Center window and prepare for grab */
	gtk_object_set(GTK_OBJECT(dialog), "type", GTK_WINDOW_POPUP, NULL);
	gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);
	gtk_window_set_position (GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
	gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, TRUE);
	gnome_dialog_close_hides(GNOME_DIALOG(dialog), TRUE);
	gtk_container_set_border_width(GTK_CONTAINER(GNOME_DIALOG(dialog)->vbox),
	    GNOME_PAD);
	gtk_widget_show_all(dialog);

	/* Grab focus */
	XGrabServer(GDK_DISPLAY());
	if (gdk_pointer_grab(dialog->window, TRUE, 0, NULL, NULL, 
	    GDK_CURRENT_TIME))
		goto nograb;
	if (gdk_keyboard_grab(dialog->window, FALSE, GDK_CURRENT_TIME))
		goto nograbkb;

	/* Make <enter> close dialog */
	gnome_dialog_editable_enters(GNOME_DIALOG(dialog), GTK_EDITABLE(entry));

	/* Run dialog */
	result = gnome_dialog_run(GNOME_DIALOG(dialog));
        gtk_widget_hide(GTK_WIDGET(dialog));

	/* Ungrab */
	XUngrabServer(GDK_DISPLAY());
	gdk_pointer_ungrab(GDK_CURRENT_TIME);
	gdk_keyboard_ungrab(GDK_CURRENT_TIME);
	gdk_flush();

	/* Report passphrase if user selected OK */
	passphrase = gtk_entry_get_text(GTK_ENTRY(entry));
	if (result == 0)
		pw = passphrase;
		
	/* Kill the entry's copy of the passphrase. */
	gtk_entry_set_text(GTK_ENTRY(entry), "");
			
	gnome_dialog_close(GNOME_DIALOG(dialog));
	return;

	/* At least one grab failed - ungrab what we got, and report
	   the failure to the user.  Note that XGrabServer() cannot
	   fail.  */
 nograbkb:
	gdk_pointer_ungrab(GDK_CURRENT_TIME);
 nograb:
	XUngrabServer(GDK_DISPLAY());
	gnome_dialog_close(GNOME_DIALOG(dialog));
	
	report_failed_grab();
}

static char **
argv_insert(int argc, char *argv[], char *dirname)
{
	char **new_argv;
	int i;

	// 8 for the 8 items inserted below, 1 for the final NULL
	new_argv = g_new(char *, argc + 8 + 1);

	new_argv[0] = g_strdup(SUDO); /* sudo binary */
	new_argv[1] = g_strdup("-H"); /* Make sudo set $HOME */
	new_argv[2] = g_strdup("-S"); /* Make sudo read from stdin */
	new_argv[3] = g_strdup("-p"); /* Make sudo use next arg as prompt */
	new_argv[4] = g_strdup("GNOME_SUDO_PASS ");	/* prompt */
	new_argv[5] = g_strdup("--"); /* Make sudo stop processing options */
	new_argv[6] = g_strdup(LIBEXEC "/gnome-sudo-helper");
	new_argv[7] = g_strdup(dirname); /* Where .Xauthority is stored */
	
	/* This looks at each argument passed to gnome-sudo and
	   copies it into new_argv, starting at element 8, since
	   the first 8 were created above. Then it sets the last
	   element to NULL. */
	for (i = 0; argv[i] != NULL; i++)
		new_argv[i + 8] = g_strdup(argv[i]);
	new_argv[i + 8] = NULL;

	return new_argv;
}

int
main(int argc, char *argv[])
{
	char template[] = "/tmp/" PACKAGE "-XXXXXX";
	char *dir, *xauth;
	pid_t pid;
        int status;
	int i;
	FILE *infile, *outfile;
	int parent_pipe[2];	/* For talking to the parent */
	int child_pipe[2];	/* For talking to the child */
	char buf[16];
	size_t r;

	poptContext pctx;
	char **args;
	int args_len;
	char *target_user = NULL;
	struct poptOption options[] = {
		{ "user", 'u', POPT_ARG_STRING, &target_user, 0,
		  N_("Run the command as USER instead of root. To specify"
		     " a UID instead of a username, use the form \"#UID\"."),
		  N_("USER") },

		{ NULL, '\0', 0, NULL, 0, NULL, NULL }
	};

	bindtextdomain(PACKAGE, GNOMELOCALEDIR);  
	textdomain(PACKAGE);

	gnome_init_with_popt_table(PACKAGE, VERSION, argc, argv,
				   options, 0, &pctx);

	args = (char **) poptGetArgs(pctx);
	if (!args)
		bomb(_("No command specified!"));

	// Determine how many arguments there are
	for (i = 0; args[i] != NULL; i++)
		;
	args_len = i;

	dir = mkdtemp(template);
	if (!dir)
		bomb(strerror(errno));

	xauth = g_strdup_printf("%s/.Xauthority", g_get_home_dir());
	copy(xauth, dir);

	if ((pipe(parent_pipe)) == -1)
		bomb(strerror(errno));

	if ((pipe(child_pipe)) == -1)
		bomb(strerror(errno));

	pid = fork();
	if (pid == -1)
		bomb(strerror(errno));
	else if (pid == 0) {
		// Child

		close(child_pipe[1]);
		dup2(child_pipe[0], STDIN_FILENO);
		dup2(parent_pipe[1], STDERR_FILENO);

		args = argv_insert(args_len, args, dir);
		execv(args[0], args);

		bomb(strerror(errno));
	} else {
		// Parent

		close(parent_pipe[1]);

                r = read(parent_pipe[0], buf, 16);
                if (r < 16) {
                        perror("read");
			exit(EXIT_FAILURE);
		}

		infile = fdopen(parent_pipe[0], "r");
		if (!infile)
			exit(EXIT_FAILURE);

		outfile = fdopen(child_pipe[1], "w");
		if (!outfile)
			exit(EXIT_FAILURE);

		if (strncmp(buf, "GNOME_SUDO_PASS ", 16) == 0) {
                        passphrase_dialog(_("Enter the root password:"));
			if (!pw)
				exit(EXIT_SUCCESS);

			fprintf(outfile, "%s\n", pw);
			fclose(outfile);

			/* Zero passphrase in memory */
			memset(pw, '\0', strlen(pw));

		} else if (strncmp(buf, "GNOME_SUDO_DONE ", 16) != 0) {
			/* This is the only other input we should
			   be getting so, if it's *not* what we
			   get, something is wrong. */
			exit(EXIT_FAILURE);
		}

		while (waitpid(pid, &status, 0) == -1)
			;

		fclose(infile);
	}

	if (WIFEXITED(status)) {
		return WEXITSTATUS(status);
	}

	return EXIT_FAILURE;
}
