/*
 *  Hearts - hearts.c
 *  Copyright 2006 Sander Marechal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "hearts.h"

GladeXML 	*xml;
GdkPixmap 	*backbuffer; 
GdkGC 		*backbuffer_gc;
GameBackground background;

GameRules	game_rules;
Trick 		game_trick;
gint 		game_state;
gint 		game_pass;
gint 		game_trick_winner;
gboolean 	game_hearts_broken;
CardsDeck 	*game_deck;
GList	 	*game_score_labels;
GList		*game_ai_scripts;
GHashTable	*game_card_styles;
GString		*game_card_style;
GString		*game_background;
gboolean	game_background_tiled;
lua_State	*game_hint_lua_state;

CardsImage 	*cards_image;

Player *player[4]; 	/* the players of the game */
gint user;			/* which playes is the user */

/* Load a pixmap */
GdkPixmap* get_pixmap(const char *filename)
{
	GdkPixmap *ret;
	GdkPixbuf *im;

	if (filename == NULL)
		return NULL; 

	im = gdk_pixbuf_new_from_file (filename, NULL);
	if (im != NULL)
	{
		gdk_pixbuf_render_pixmap_and_mask_for_colormap (im, gdk_colormap_get_system(), &ret, NULL, 127);
		gdk_pixbuf_unref (im);
	}
	else
	{
		ret = NULL;
	} 

	return ret;
}

/* start a new game */
void game_new(void)
{
	gint i;
	GList *l;
	GtkTable *score_table;
	GtkWindow *score_window;
	GtkLabel *label;
	GtkWidget *widget;
	
	/* reset the score_table. To do this, we need to remove the child widgets and then resize */
	score_table = (GtkTable*)glade_xml_get_widget(xml, "score_table");
	for (l = game_score_labels; l; l = l->next)
	{
		GtkLabel *label;
		label = l->data;
		gtk_widget_destroy((GtkWidget*)label);
	}
	g_list_free(game_score_labels);
	game_score_labels = NULL;
	gtk_table_resize(score_table, 2, 4);

	/* set the score window title */
	score_window = (GtkWindow*)glade_xml_get_widget(xml, "score");
	gtk_window_set_title(score_window, _("Scores"));
	
	/* refresh the AI's */
	for (i = 0; i < 4; i++)
		player_free(player[i]);
	
	widget = glade_xml_get_widget(xml, "playingarea");
	player[0] = player_new(FACE_DOWN, NORTH, game_rules.north_ai, widget);
	player[1] = player_new(FACE_DOWN, EAST,  game_rules.east_ai,  widget);
	player[2] = player_new(FACE_UP,   SOUTH, NULL,                widget);
	player[3] = player_new(FACE_DOWN, WEST,  game_rules.west_ai,  widget);
	
	/* set the user's name */
	label = (GtkLabel*)glade_xml_get_widget(xml, "south_name");
	gtk_label_set_text(label, g_get_real_name());
	
	gchar *name = g_strconcat("<span foreground=\"white\"><b>", g_get_real_name(), "</b></span>", NULL);
	pango_layout_set_markup(player[user]->layout, name, -1);
	g_free(name);
	
	game_pass = 0;
	game_new_round();
}

/* start a new round in this game */
void game_new_round(void)
{
	gint i;
	
	/* reset the game */
	cards_deck_reset(game_deck);
	trick_reset(&game_trick);
	game_pass = (game_pass + 1) % 4;
	game_hearts_broken = FALSE;
	
	/* new hands for everyone */
	for (i = 0; i < 4; i++)
	{
		cards_hand_free_cards(player[i]->hand);
		cards_hand_draw(player[i]->hand, game_deck, 13);
		player[i]->point_cards = 0;
		
		/* have the AI's select cards to pass on if we're passing cards this round */
		if (i != user && game_pass > 0)
			player_select_cards(player[i]);
	}
	
	/* set the game state */
	if (game_pass > 0)
	{
		game_set_status(_("Select three cards that you wish to pass on."));
		game_state = GAME_SELECT_CARDS;
	}
	else
	{
		game_set_status(_("Click on the card you wish to play."));
		game_state = GAME_PLAY;
		game_open_round();
	}
}

/* restart the current round */
void game_restart_round(void)
{
	gint i;
	
	/* reset the game */
	trick_reset(&game_trick);
	game_hearts_broken = FALSE;
	
	/* reset the hands for everyone */
	for (i = 0; i < 4; i++)
	{
		cards_hand_reset(player[i]->hand);
		player[i]->point_cards = 0;
		
		/* have the AI's select cards to pass on if we're passing cards this round */
		if (i != user && game_pass > 0)
			player_select_cards(player[i]);
	}
	
	/* set the game state */
	if (game_pass > 0)
	{
		game_set_status(_("Select three cards that you wish to pass on."));
		game_state = GAME_SELECT_CARDS;
	}
	else
	{
		game_set_status(_("Click on the card you wish to play."));
		game_state = GAME_PLAY;
		game_open_round();
	}
}

/* start the round and play untill the player is due */
void game_open_round (void)
{
	gint i, starter = user;
	Card *two_of_clubs;
	
	/* figure out which playes has to start */
	if (game_rules.clubs_lead)
	{
		two_of_clubs = &g_array_index(game_deck, Card, CARD_ID(CARDS_CLUBS, CARD_TWO));
		for (i = 0; i < 4; i++)
		{
			if (g_list_find(player[i]->hand->list, two_of_clubs))
				starter = i;
		}
	}
	else
	{
		starter = (game_pass + 1) % 4;
	}
	
	/* play */
	i = starter;
	while (i != user)
	{
		player_play(player[i], &game_trick);
		i = (i + 1) % 4;
	}
}

/* test for the end of the round and show this game's score */
gboolean game_end_test(void)
{
	gint rows, i, score_value = 0;
	GtkWidget *label, *score;
	GtkTable *score_table;
	gchar *label_text;
	GList *l;
	
	/* end of round test */
	if (g_list_length(player[user]->hand->list) == 0)
	{
		/* Signal the players that the round is over */
		player_round_end(player[NORTH]);
		player_round_end(player[EAST]);
		player_round_end(player[WEST]);
		
		/* strike out the previous row */
		i = 0;
		l = game_score_labels;
		while (l != NULL && i < 4)
		{
			GtkLabel *label = l->data;
			gchar *text = g_strdup_printf("<s>%s</s>", gtk_label_get_label(label));
			gtk_label_set_markup(label, text);
			g_free(text);
			
			l = l->next;
			i++;
		}

		/* resize the score_table widget */
		score_table = (GtkTable*)glade_xml_get_widget(xml, "score_table");
		g_object_get(score_table, "n-rows", &rows, NULL);
		gtk_table_resize(score_table, rows + 1, 4);

		/* fill the new row */		
		for (i = 0; i < 4; i++)
		{
			/* update the scores */
			if (player[i]->point_cards == 14) /* shot the moon? */
				player[i]->score_total -= player[i]->score_round;
			else
				player[i]->score_total += player[i]->score_round;

			player[i]->score_round = 0;
			
			label_text = g_strdup_printf("%d", player[i]->score_total);
			label = gtk_label_new(label_text);
			game_score_labels = g_list_prepend(game_score_labels, label);
			g_free(label_text);
				
			gtk_label_set_justify((GtkLabel*)label, GTK_JUSTIFY_CENTER);
			gtk_table_attach(score_table, label, i, i + 1, rows, rows + 1,
					 GTK_EXPAND, GTK_SHRINK, 2, 0);
		}

		/* grab the score widget */		
		score = glade_xml_get_widget(xml, "score");

		/* end of game test */
		for (i = 0; i < 4; i++)
			score_value = MAX(score_value, player[i]->score_total);
		
		if (score_value >= (game_rules.ruleset == RULESET_SPOT_HEARTS ? 500 : 100))
		{
			game_state = GAME_END;
			
			gint winner = NORTH;
			for (i = 0; i < 4; i++)
			{
				if (player[i]->score_total < score_value)
				{
					score_value = player[i]->score_total;
					winner = i;
				}
			}
			
			if (winner == user)
				gtk_window_set_title((GtkWindow*)score, _("Game over - You have won!"));
			else
				gtk_window_set_title((GtkWindow*)score, _("Game over - You have lost"));
			
			game_set_status(_("Click somewhere to start a new game."));
		}
		else
		{
			game_set_status(_("Click somewhere to continue the game."));
			game_state = GAME_ROUND_END;
		}
		
		/* show the score */
		gtk_widget_show_all(score);

		return TRUE;
	}
	
	return FALSE;
}

/* pass cards to the next player */
gboolean game_pass_cards(GtkWidget *widget)
{
	gint x, y, width, height, mouse_x, mouse_y, p;
	GList *temp[4];
	
	/* get the area we're supposed to click on */
	gtk_widget_get_pointer(widget, &mouse_x, &mouse_y);
	cards_hand_get_area(player[(user + game_pass) % 4]->hand, widget, cards_image, &x, &y, &width, &height, TRUE);
	
	if (mouse_x >= x && mouse_x < x + width && mouse_y >= y && mouse_y < y + height)
	{
		/* pass the cards and make a copy in the history for trick reset support */
		for (p = 0; p < 4; p++)
			temp[p] = cards_hand_draw_selected(player[p]->hand);
		
		for (p = 0; p < 4; p++)
			cards_hand_add(player[(p + game_pass) % 4]->hand, temp[p]);
		
		/* redraw */
		draw_playingarea();
		return TRUE;
	}
	
	return FALSE;
}

/* (de)select cards in the passing round */
gint game_select_cards(GtkWidget *widget)
{
	GList *i;
	Card *card, *active;
	gint x, y, width, height, num_selected = 0;
	
	i = player[user]->hand->list;
	active = NULL;
	
	/* loop through the hand to find the active cards and the number of selected cards */			
	do
	{
		card = (Card*)i->data;

		if (card->active)
			active = card;

		if (card->selected)
			num_selected++;
							
		i = g_list_next(i);
	} while (i != NULL);

	/* You cannot select the active card if there are already 3 selected cards */				
	if (active != NULL && (num_selected < 3 || active->selected == TRUE))
	{
		active->selected = !active->selected;
		
		/* adjust the number of selected cards */
		if (active->selected)
			num_selected++;
		else
			num_selected--;
	}

	/* redraw the screen */
	cards_hand_get_area(player[user]->hand, widget, cards_image, &x, &y, &width, &height, TRUE);
	background_render(backbuffer, x, y, width, height);
	cards_hand_render(player[user]->hand, cards_image, backbuffer, backbuffer_gc, x, y);
	gtk_widget_queue_draw_area(widget, x, y, width, height);
	
	return num_selected;
}

/* Publish the game's score to Lua */
void game_score_publish(lua_State *state)
{
	gint i;
	
	lua_newtable(state);							/* table */
	
	for (i = 0; i < 4; i++)
	{
		/* push the player ID */
		lua_pushnumber(state, i + 1);				/* table, player */
		
		/* start a new {this_hand, total} */
		lua_newtable(state);						/* table, player, table */
		
		/* push the this_hand */
		lua_pushnumber(state, 1);					/* table, player, table, key */
		if (game_state == GAME_ROUND_END || game_state == GAME_END)
			lua_pushnil(state);						/* table, player, table, key, nil */
		else
			lua_pushnumber(state, player[i]->score_round);	/* table, player, table, key, this_hand */
		lua_settable(state, -3);							/* table, player, table */
		
		/* push the total */
		lua_pushnumber(state, 2);							/* table, player, table, key */
		lua_pushnumber(state, player[i]->score_total);		/* table, player, table, key, total */
		lua_settable(state, -3);							/* table, player, table */
		
		/* finish this player */
		lua_settable(state, -3);					/* table */
	}

	/* close the score list */
	lua_setglobal(state, "score");					/* <empty> */
}

/* returns wether a card is valid to play from this hand on the stack */
gboolean game_is_valid_card(Card *card, CardsHand *hand, Trick *trick)
{
	/* special case: opening with a two of clubs (variant) */
	if (game_rules.clubs_lead && g_list_length(hand->list) == 13 && trick->num_played == 0)
	{
		if (card->suit == CARDS_CLUBS && card->rank == CARD_TWO)
			return TRUE;
		else
			return FALSE;
	}
	
	/* special case, no blood in the first round, unless we have to */
	if (game_rules.no_blood
		&& (card->suit == CARDS_HEARTS || (card->suit == CARDS_SPADES && card->rank == CARD_QUEEN))
		&& g_list_length(hand->list) == 13)
	{
		GList *l;
		gint num_pointcards = 0;
		for (l = hand->list; l; l = l->next)
		{
			Card *lcard = l->data;
			if (lcard->suit == CARDS_HEARTS || (lcard->suit == CARDS_SPADES && lcard->rank == CARD_QUEEN))
				num_pointcards++;
		}
		
		if (num_pointcards < 13)
			return FALSE;
	}
	
	/* special case: hearts must have been "broken" to be played as trump (variant).
	If a player has all hearts, then one must be played anyway */
	if (game_rules.hearts_break 
		&& trick->num_played == 0
		&& game_hearts_broken == FALSE
		&& card->suit == CARDS_HEARTS
		&& cards_hand_num_suit(hand, CARDS_HEARTS) != cards_hand_length(hand)
	)
		return FALSE;
	
	/* follow the trump if possible */
	if (trick->trump > -1 && card->suit != trick->trump && cards_hand_num_suit(hand, trick->trump) > 0)
		return FALSE;
	
	/* no more reason to disallow a card */
	return TRUE;
}

/* get a hint string */
gchar* game_get_hint(void)
{
	/* publish the hand */
	cards_hand_publish(player[user]->hand, game_hint_lua_state);
	
	if (game_state == GAME_PLAY)
	{
		if (game_trick.num_played == 4)
			return g_strdup(_("Click somewhere to continue the game."));
		
		/* publish the trick */
		trick_publish(&game_trick, game_hint_lua_state);
	
		/* call the hint play_card() function */
		lua_getglobal(game_hint_lua_state, "play_card");		/* "play_card" */
		if (lua_pcall(game_hint_lua_state, 0, 1, 0) != 0)		/* table */
		{
			printf(_("Error: %s"), lua_tostring(game_hint_lua_state, -1));
			g_assert_not_reached();
		}
	
		/* extract the card */
		Card *card = lua_pop_card(game_hint_lua_state);
		g_assert(g_list_find(player[user]->hand->list, card) != NULL);
		
		g_assert(game_is_valid_card(card, player[user]->hand, &game_trick));
		
		/* Hint: %s is the name of the card the player should play */
		return g_strdup_printf(_("Play %s."), card_get_name(card));
	}
	else if (game_state == GAME_PASS_CARDS)
	{
		gchar *name = NULL;
		switch ((user + game_pass) % 4)
		{
			case NORTH: name = game_rules.north_ai->str; break;
			case EAST:  name = game_rules.east_ai->str;  break;
			case WEST:  name = game_rules.west_ai->str;  break;
		}
			
		return g_strdup_printf(_("Click on the hand of %s to pass the cards."), name);
	}
	else if (game_state == GAME_SELECT_CARDS)
	{
		/* call the hint select_cards() function */
		lua_getglobal(game_hint_lua_state, "select_cards");	/* select_cards */
		if (lua_pcall(game_hint_lua_state, 0, 1, 0) != 0)		/* table */
		{
			printf(_("Error: %s"), lua_tostring(game_hint_lua_state, -1));
			g_assert_not_reached();
		}
	
		/* start reading from the table */					
		lua_pushnil(game_hint_lua_state);						/* table, nil */

		/* read three cards show the first one that the player hasn't selected yet */
		gint i;
		for (i = 0; i < 3; i++)
		{
			/* read next card */
			lua_next(game_hint_lua_state, -2);			/* table, key, table */

			Card *card = lua_pop_card(game_hint_lua_state);
			g_assert(g_list_find(player[user]->hand->list, card) != NULL);
			
			if (!card->selected)
				/* Hint: %s is the name of the card the player should pass on */
				return g_strdup_printf(_("Pass %s."), card_get_name(card));
		}
	}
	
	return g_strdup(_("Sorry, I can't help you."));
}

/* Set the text of the status bar */
void game_set_status(gchar *message)
{
	GtkStatusbar *statusbar = (GtkStatusbar*)glade_xml_get_widget(xml, "statusbar");
	gint context_id = gtk_statusbar_get_context_id(statusbar, "hearts user feedback");
	gtk_statusbar_pop(statusbar, context_id);
	gtk_statusbar_push(statusbar, context_id, message);
}

/* draw the playingarea */
void draw_playingarea()
{
	gint x, y, width, height, n_x, n_y, n_width, n_height, i;
	GtkWidget *playingarea_widget;

	playingarea_widget = glade_xml_get_widget (xml, "playingarea");
	
	/* Fill the buffer with the background image */
	background_render(backbuffer, 0, 0, playingarea_widget->allocation.width, playingarea_widget->allocation.height);
	
	/* render the hands */
	for (i = 0; i < 4; i++)
	{
		cards_hand_get_area(player[i]->hand, playingarea_widget, cards_image, &x, &y, &width, &height, TRUE);
		cards_hand_render(player[i]->hand, cards_image, backbuffer, backbuffer_gc, x, y);

		pango_layout_get_pixel_size(player[i]->layout, &n_width, &n_height);
		
		n_x = 0;
		n_y = 0;
		switch(i)
		{
			case NORTH:
				n_x = (playingarea_widget->allocation.width - n_width) / 2;
				n_y = y + height + 5;
				break;

			case EAST:
				n_x = x - n_width - 15;
				n_y = (playingarea_widget->allocation.height - n_height) / 2;
				break;

			case SOUTH:
				n_x = (playingarea_widget->allocation.width - n_width) / 2;
				n_y = y - n_height - 5;
				break;

			case WEST:
				n_x = x + width + 15;
				n_y = (playingarea_widget->allocation.height - n_height) / 2;
				break;
		}
		gdk_draw_layout(backbuffer, backbuffer_gc, n_x, n_y, player[i]->layout);
	}
	
	/* render the trick */
	trick_render(&game_trick, playingarea_widget, cards_image, backbuffer, backbuffer_gc);
	
	/* update the screen */
	gtk_widget_queue_draw_area(playingarea_widget, 0, 0, playingarea_widget->allocation.width, playingarea_widget->allocation.height);
}

/* callback to add card styles to the hashmap */
void card_style_add(gpointer key, gpointer value, gpointer *first_style)
{
	static int i = 0;
	GtkComboBox *card_style_box = (GtkComboBox*)glade_xml_get_widget(xml, "card_style");

	gtk_combo_box_append_text(card_style_box, (gchar*)key);
		
	if (strcmp(game_card_style->str, (gchar*)key) == 0)
	{
		gtk_combo_box_set_active(card_style_box, i);
		cards_image = cards_image_from_file(value);
	}
	
	if (*first_style == NULL)
		*first_style = g_string_new(value);
	
	i++;
}

/* create a new ~/.gnome-hearts.cfg file */
void new_cfg_file()
{
	gchar *new_cfg;
	gchar *cfg_path;
	gsize size;

	cfg_path = g_strconcat(g_get_home_dir(), "/.gnome-hearts.cfg", NULL);
	g_file_get_contents(PACKAGE_DATA_DIR"/gnome-hearts/gnome-hearts.cfg", &new_cfg, &size, NULL);
	g_file_set_contents(cfg_path, new_cfg, size, NULL);

	g_free(cfg_path);
	g_free(new_cfg);
}

/* set sensible configuration default. Will be overridden by ~/.gnome-hearts.cfg */
void set_cfg_default()
{
	game_rules.ruleset = RULESET_STANDARD;
	game_rules.clubs_lead   = TRUE;
	game_rules.hearts_break = TRUE;
	game_rules.no_blood     = TRUE;
	game_rules.north_ai = g_string_new("John");
	game_rules.east_ai  = g_string_new("John");
	game_rules.west_ai  = g_string_new("John");
	game_card_style = g_string_new("dondorf.svg");
	game_background = g_string_new(PACKAGE_DATA_DIR"/pixmaps/gnome-hearts/baize.png");
	game_background_tiled = TRUE;
}

/* load ~/.gnome-hearts.cfg and parse all the settings */
void load_cfg_file()
{
	GKeyFile *cfg;
	gchar *cfg_path;
	
	cfg = g_key_file_new();
	cfg_path = g_strconcat(g_get_home_dir(), "/.gnome-hearts.cfg", NULL);
	if (!g_file_test(cfg_path, G_FILE_TEST_EXISTS))
		new_cfg_file();
	
	g_assert(g_key_file_load_from_file(cfg, cfg_path, G_KEY_FILE_KEEP_COMMENTS, NULL));
	
	/* Check the cfg version number */
	GError *error = NULL;
	gint version = g_key_file_get_integer(cfg, "hearts", "version", &error);

	if (version < 2 || error != NULL)
	{
		g_clear_error(&error);
		g_key_file_free(cfg);
		cfg = g_key_file_new();
		new_cfg_file();		
		g_assert(g_key_file_load_from_file(cfg, cfg_path, G_KEY_FILE_KEEP_COMMENTS, NULL));
	}
	
	/* load the game rules */
	gchar *ruleset = g_key_file_get_string(cfg, "hearts", "ruleset", NULL);
	if (g_ascii_strcasecmp(ruleset, "standard") == 0)	 game_rules.ruleset = RULESET_STANDARD;
	if (g_ascii_strcasecmp(ruleset, "omnibus") == 0)	 game_rules.ruleset = RULESET_OMNIBUS;
	if (g_ascii_strcasecmp(ruleset, "omnibus_alt") == 0) game_rules.ruleset = RULESET_OMNIBUS_ALT;
	if (g_ascii_strcasecmp(ruleset, "spot_hearts") == 0) game_rules.ruleset = RULESET_SPOT_HEARTS;
	g_free(ruleset);
	
	game_rules.clubs_lead 	= g_key_file_get_boolean(cfg, "hearts", "clubs_lead", NULL);
	game_rules.hearts_break = g_key_file_get_boolean(cfg, "hearts", "hearts_break", NULL);
	game_rules.no_blood 	= g_key_file_get_boolean(cfg, "hearts", "no_blood", NULL);
	
	/* set the game rules in the preferences dialog */
	GtkWidget* prefs = glade_xml_get_widget(xml, "ruleset");
	gtk_combo_box_set_active((GtkComboBox*)prefs, game_rules.ruleset);

	prefs = glade_xml_get_widget(xml, "clubs_lead");
	gtk_toggle_button_set_active((GtkToggleButton*)prefs, game_rules.clubs_lead);
	
	prefs = glade_xml_get_widget(xml, "hearts_break");
	gtk_toggle_button_set_active((GtkToggleButton*)prefs, game_rules.hearts_break);
	
	prefs = glade_xml_get_widget(xml, "no_blood");
	gtk_toggle_button_set_active((GtkToggleButton*)prefs, game_rules.no_blood);

	/* load the game AI strings */
	gchar *ai;
	ai = g_key_file_get_string(cfg, "hearts", "north_ai", NULL);
	g_string_assign(game_rules.north_ai, ai);
	g_free(ai);
	
	ai = g_key_file_get_string(cfg, "hearts", "east_ai", NULL);
	g_string_assign(game_rules.east_ai,  ai);
	g_free(ai);

	ai = g_key_file_get_string(cfg, "hearts", "west_ai", NULL);
	g_string_assign(game_rules.west_ai,  ai);
	g_free(ai);
	
	/* load the card style */
	gchar *style = g_key_file_get_string(cfg, "hearts", "card_style", NULL);
	g_string_assign(game_card_style, style);
	g_free(style);
	
	/* load the background */
	style = g_key_file_get_string(cfg, "hearts", "background_image", NULL);
	g_string_assign(game_background, style);
	gtk_file_chooser_set_filename((GtkFileChooser*)glade_xml_get_widget(xml, "background"), style);
	g_free(style);
	
	game_background_tiled = g_key_file_get_boolean(cfg, "hearts", "background_tiled", NULL);
	
	if (game_background_tiled)
		prefs = glade_xml_get_widget(xml, "background_tile");
	else
		prefs = glade_xml_get_widget(xml, "background_stretch");
	gtk_toggle_button_set_active((GtkToggleButton*)prefs, TRUE);
	
	/* All done! */
	g_key_file_free(cfg);
	g_free(cfg_path);
}

/* free the configuration */
void free_cfg()
{
	g_string_free(game_rules.north_ai, TRUE);
	g_string_free(game_rules.east_ai, TRUE);
	g_string_free(game_rules.west_ai, TRUE);
	g_string_free(game_card_style, TRUE);
	g_string_free(game_background, TRUE);
}

/* load all of the AI scripts */
void load_ai_scripts()
{
	DIR *directory = opendir(PACKAGE_DATA_DIR"/gnome-hearts/scripts/players/");
	struct dirent *file;
	GString *name;

	g_assert(directory != NULL);
	while ((file = readdir(directory)))
	{
		if (g_str_has_suffix(file->d_name, ".lua"))
		{
			name = g_string_new(file->d_name);
			name = g_string_truncate(name, strlen(file->d_name) - 4);
			game_ai_scripts = g_list_append(game_ai_scripts, name);
		}
	}
	closedir(directory);
	g_assert(game_ai_scripts);
	
	/* load the list in the combo box */
	GtkComboBox *north_ai = (GtkComboBox*)glade_xml_get_widget(xml, "north_ai");
	GtkComboBox *east_ai  = (GtkComboBox*)glade_xml_get_widget(xml, "east_ai");
	GtkComboBox *west_ai  = (GtkComboBox*)glade_xml_get_widget(xml, "west_ai");
	
	gint i = 0;
	GList *l = NULL;
	
	for (l = game_ai_scripts; l; l = l->next)
	{
		name = l->data;
		gtk_combo_box_append_text(north_ai, name->str);
		gtk_combo_box_append_text(east_ai,  name->str);
		gtk_combo_box_append_text(west_ai,  name->str);
		
		if (strcmp(game_rules.north_ai->str, name->str) == 0)
			gtk_combo_box_set_active(north_ai, i);
		if (strcmp(game_rules.east_ai->str, name->str) == 0)
			gtk_combo_box_set_active(east_ai, i);
		if (strcmp(game_rules.west_ai->str, name->str) == 0)
			gtk_combo_box_set_active(west_ai, i);
		
		i++;
	}

	/* make sure that a valid AI is listed in the rules */
	name = game_ai_scripts->data;
	if (gtk_combo_box_get_active(north_ai) == -1)
	{
		gtk_combo_box_set_active(north_ai, 1);
		g_string_assign(game_rules.north_ai, name->str);
	}
	if (gtk_combo_box_get_active(east_ai) == -1)
	{
		gtk_combo_box_set_active(east_ai, 1);
		g_string_assign(game_rules.east_ai, name->str);
	}
	if (gtk_combo_box_get_active(west_ai) == -1)
	{
		gtk_combo_box_set_active(west_ai, 1);
		g_string_assign(game_rules.west_ai, name->str);
	}

	GtkLabel *label;
	label = (GtkLabel*)glade_xml_get_widget(xml, "north_name");
	gtk_label_set_text(label, game_rules.north_ai->str);
	label = (GtkLabel*)glade_xml_get_widget(xml, "east_name");
	gtk_label_set_text(label, game_rules.east_ai->str);
	label = (GtkLabel*)glade_xml_get_widget(xml, "west_name");
	gtk_label_set_text(label, game_rules.west_ai->str);
}

/* free the AI script lists */
void free_ai_scripts()
{
	GList *l;
	
	for (l = game_ai_scripts; l; l = l->next)
		g_string_free((GString*)l->data, TRUE);
	g_list_free(game_ai_scripts);
}

/* build the lists of card styles */
void load_card_styles()
{
	DIR *directory = opendir(PACKAGE_DATA_DIR"/pixmaps/gnome-hearts/cards/");
	struct dirent *file;
	GString *first_style = NULL;

	game_card_styles = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);

	/* read cards from the hearts card dir */
	g_assert(directory != NULL);
	while ((file = readdir(directory)))
	{
		if (g_str_has_suffix(file->d_name, ".png") || g_str_has_suffix(file->d_name, ".svg"))
			g_hash_table_insert(game_card_styles, g_strdup(file->d_name), g_strconcat(PACKAGE_DATA_DIR"/pixmaps/gnome-hearts/cards/", file->d_name, NULL));
	}
	closedir(directory);
	
	/* read cards from the gnome-games card dir */
	directory = opendir(PACKAGE_DATA_DIR"/pixmaps/gnome-games-common/cards/");
	if (directory != NULL)
	{
		while ((file = readdir(directory)))
		{
			if (g_str_has_suffix(file->d_name, ".png") || g_str_has_suffix(file->d_name, ".svg"))
				g_hash_table_insert(game_card_styles, g_strdup(file->d_name), g_strconcat(PACKAGE_DATA_DIR"/pixmaps/gnome-games-common/cards/", file->d_name, NULL));
		}
		closedir(directory);
	}
	
	/* loop over he hash table, add all card styles to the combo box and set the active one */
	g_hash_table_foreach(game_card_styles, (GHFunc)card_style_add, &first_style);
	
	/* Make sure a card style was loaded. If the configured style wasn't found then no style has been loaded
	   at this point. See Debian bugs 396043 and 395551. */
	if (cards_image == NULL)
	{
		g_assert(first_style != NULL);
		cards_image = cards_image_from_file(first_style->str);
		g_string_free(first_style, TRUE);
	}
}

/* initialize the background */
void background_new()
{
	background.tiled = FALSE;
	background.image = NULL;
	background.tile = NULL;
	background.buffer = NULL;
	background.gc = NULL;
}

/* Load the background specified in the configuration */
void background_load(gchar* path)
{
	background_free(TRUE);
	background.image = preimage_new_from_file(path);
	
	/* If the defined background cannot be loaded, fall back to baize.png */
	if (!background.image)
	{
		background.image = preimage_new_from_file(PACKAGE_DATA_DIR"/pixmaps/gnome-hearts/baize.png");
		background.tiled = TRUE;
		game_background_tiled = TRUE;
		GtkWidget *prefs = glade_xml_get_widget(xml, "background_tile");
		gtk_toggle_button_set_active((GtkToggleButton*)prefs, TRUE);
		
		if (!background.image)
		{
			printf("**Error**: The default background could not be loaded.\n");
			g_assert_not_reached();
		}
	}
	
	background_set_tiled(background.tiled);
}

void background_set_size(gint width, gint height)
{
	if (background.tiled)
		return;
	
	if (background.buffer)
		g_object_unref(background.buffer);
	
	background.buffer = preimage_render(background.image, width, height);
}

void background_set_tiled(gboolean tiled)
{
	background_free(FALSE);
	
	GtkWidget *playingarea_widget = glade_xml_get_widget (xml, "playingarea");
	background.gc = gdk_gc_new(playingarea_widget->window);

	if (tiled)
	{
		gdk_pixbuf_render_pixmap_and_mask_for_colormap (background.image->pixbuf, gdk_colormap_get_system(), &background.tile, NULL, 127);
		gdk_gc_set_tile (background.gc, background.tile);
		gdk_gc_set_fill (background.gc, GDK_TILED);
	}
	
	background.tiled = tiled;
}

void background_render(GdkPixmap *target, gint x, gint y, gint width, gint height)
{
	if (background.tiled)
	{
		gdk_draw_rectangle (target,
		    background.gc,
		    TRUE,
		    x, y,
		    width,
		    height);
	}
	else
	{
		gdk_pixbuf_render_to_drawable(background.buffer,
			target,
			background.gc,
			x, y,
			x, y,
			width, height,
			GDK_RGB_DITHER_NONE,
			0, 0);
	}
}

/* free the background */
void background_free(gboolean free_image)
{
	if (free_image && background.image != NULL)
	{
		preimage_free(background.image);
		background.image = NULL;
	}
	
	if (background.tile != NULL)
	{
		gdk_pixmap_unref(background.tile);
		background.tile = NULL;
	}
	
	if (background.buffer != NULL)
	{
		g_object_unref(background.buffer);
		background.buffer = NULL;
	}
	
	if (background.gc != NULL)
	{
		g_object_unref(background.gc);
		background.gc = NULL;
	}
}

int main (int argc, char *argv[])
{
	GtkWidget *playingarea_widget;

	#ifdef ENABLE_NLS
		bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
		textdomain (PACKAGE);
	#endif

	gnome_program_init (PACKAGE, VERSION, LIBGNOMEUI_MODULE, argc, argv, GNOME_PARAM_APP_DATADIR, PACKAGE_DATA_DIR, NULL);
	glade_gnome_init ();

	/* Load the glade UI file */
	xml = glade_xml_new(PACKAGE_DATA_DIR"/gnome-hearts/gnome-hearts.glade", NULL, NULL);

	/* Hide the "leave fullscreen" buttons" */
	gtk_widget_hide(glade_xml_get_widget(xml, "leave_fullscreen"));
	gtk_widget_set_sensitive(glade_xml_get_widget(xml, "leave_fullscreen"), FALSE);
	gtk_widget_hide(glade_xml_get_widget(xml, "leave_fullscreen_button"));
	gtk_widget_set_sensitive(glade_xml_get_widget(xml, "leave_fullscreen_button"), FALSE);

	/* Gray out unused items for 0.1 branch */
	gtk_widget_set_sensitive(glade_xml_get_widget(xml, "undo_move1"), FALSE);
	gtk_widget_set_sensitive(glade_xml_get_widget(xml, "redo_move1"), FALSE);

	/* Turn off double buffering. We do that ourselves */
	playingarea_widget = glade_xml_get_widget (xml, "playingarea");
	gtk_widget_set_double_buffered(playingarea_widget, FALSE);

	/* Add a filer to the background filechooser */
	GtkFileFilter *filter = gtk_file_filter_new();
	gtk_file_filter_set_name(filter, _("Images"));
	gtk_file_filter_add_mime_type(filter, "image/png");
	gtk_file_filter_add_mime_type(filter, "image/svg+xml");
	gtk_file_filter_add_mime_type(filter, "image/jpeg");
	gtk_file_filter_add_mime_type(filter, "image/gif");
	gtk_file_chooser_set_filter((GtkFileChooser*)glade_xml_get_widget(xml, "background"), filter);
	
	/* Get the Graphics Context for the table */
	backbuffer_gc = gdk_gc_new (playingarea_widget->window);

	/* init the game's global variables */
	game_score_labels = NULL;
	game_state = GAME_SELECT_CARDS;
	game_pass = 0;
	game_hearts_broken = FALSE;
	game_deck = cards_deck_new(TRUE, FALSE);
	game_ai_scripts = NULL;
	trick_reset(&game_trick);
	user = SOUTH;
	game_hint_lua_state = NULL;
	
	/* Set the default configuration */
	set_cfg_default();
	
	/* Load the gnome-hearts.cfg file */
	load_cfg_file();
	
	/* build a list of all available AI's */
	load_ai_scripts();
	
	/* Create a list of all card styles */
	load_card_styles();

	/* load the background */
	background_new();
	background.tiled = game_background_tiled;
	background_load(game_background->str);

	/* autoconnect the glade signals */
	glade_xml_signal_autoconnect(xml);

	/* start the game */
	game_new();
	gtk_main ();
	
	/* free the score table */
	GList *l;
	for (l = game_score_labels; l; l = l->next)
	{
		GtkLabel *label;
		
		label = l->data;
		gtk_widget_destroy((GtkWidget*)label);
	}
	g_list_free(game_score_labels);
	
	/* free the players */
	gint i;
	for (i = 0; i < 4; i++)
		player_free(player[i]);
	
	if (game_hint_lua_state != NULL)
		lua_close(game_hint_lua_state);

	/* free various other game data */	
	free_ai_scripts();
	free_cfg();
	background_free(TRUE);

	cards_deck_free(game_deck);
	cards_image_free(cards_image);
	g_hash_table_destroy(game_card_styles);

	g_object_unref(backbuffer_gc);
	g_object_unref(filter);
	g_object_unref(xml);

	return 0;
}
