/* gtkcombo - combo widget for gtk+
 * Copyright 1997 Paolo Molaro
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Modified by the GTK+ Team and others 1997-1999.  See the AUTHORS
 * file for a list of people on the GTK+ Team.  See the ChangeLog
 * files for a list of changes.  These files are distributed with
 * GTK+ at ftp://ftp.gtk.org/pub/gtk/. 
 */

/*
 * Modified by Marcus Bjurman <marbj499@student.liu.se> 2001
 * The orginal comments are left intact above
 */

#include <config.h>
#include <string.h>
#include <gtk/gtkarrow.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkclist.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkeventbox.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkmain.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkwindow.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtkframe.h>
#include "gnome-cmd-combo.h"
#include "gnome-cmd-style.h"

const gchar *gnome_cmd_combo_string_key = "gnome-cmd-combo-string-value";

#define COMBO_LIST_MAX_HEIGHT	(400)
#define	EMPTY_LIST_HEIGHT	(15)

static GtkHBoxClass *parent_class = NULL;



/*******************************
 * Utility functions
 *******************************/

static gint
focus_idle (GnomeCmdCombo *combo)
{
	if (combo)
    {
		GDK_THREADS_ENTER ();
		gtk_widget_grab_focus (combo->entry);
		GDK_THREADS_LEAVE ();
    }
	return FALSE;
}

/*
static void
item_destroy (GtkObject *object)
{
	//FIXME: Why isn't this used ?
	gchar *key;

	key = gtk_object_get_data (object, gnome_cmd_combo_string_key);
	if (key)
		g_free (key);
}
*/

static void
size_allocate (GtkWidget     *widget,
			   GtkAllocation *allocation)
{
	GnomeCmdCombo *combo;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (GNOME_CMD_IS_COMBO (widget));
	g_return_if_fail (allocation != NULL);

	GTK_WIDGET_CLASS (parent_class)->size_allocate (widget, allocation);
  
	combo = GNOME_CMD_COMBO (widget);

	if (combo->entry->allocation.height > combo->entry->requisition.height)
    {
		GtkAllocation button_allocation;

		button_allocation = combo->button->allocation;
		button_allocation.height = combo->entry->requisition.height;
		button_allocation.y = combo->entry->allocation.y + 
			(combo->entry->allocation.height - combo->entry->requisition.height) 
			/ 2;
		gtk_widget_size_allocate (combo->button, &button_allocation);
    }
}


static void
get_pos (GnomeCmdCombo *combo, gint *x, gint *y, gint *height, gint *width)
{
	GtkBin *popwin;
	GtkWidget *widget;
	GtkScrolledWindow *popup;
  
	gint real_height;
	GtkRequisition list_requisition;
	gboolean show_hscroll = FALSE;
	gboolean show_vscroll = FALSE;
	gint avail_height;
	gint min_height;
	gint alloc_width;
	gint work_height;
	gint old_height;
	gint old_width;
  
	widget = GTK_WIDGET(combo);
	popup  = GTK_SCROLLED_WINDOW (combo->popup);
	popwin = GTK_BIN (combo->popwin);
  
	gdk_window_get_origin (combo->entry->window, x, y);
	real_height = MIN (combo->entry->requisition.height, 
					   combo->entry->allocation.height);
	*y += real_height;
	avail_height = gdk_screen_height () - *y;
  
	gtk_widget_size_request (combo->list, &list_requisition);
	min_height = MIN (list_requisition.height, 
					  popup->vscrollbar->requisition.height);
	if (!GTK_CLIST (combo->list)->rows)
		list_requisition.height += EMPTY_LIST_HEIGHT;
  
	alloc_width = (widget->allocation.width -
				   2 * popwin->child->style->klass->xthickness -
				   2 * GTK_CONTAINER (popwin->child)->border_width -
				   2 * GTK_CONTAINER (combo->popup)->border_width -
				   2 * GTK_CONTAINER (GTK_BIN (popup)->child)->border_width - 
				   2 * GTK_BIN (popup)->child->style->klass->xthickness) + 100;
  
	work_height = (2 * popwin->child->style->klass->ythickness +
				   2 * GTK_CONTAINER (popwin->child)->border_width +
				   2 * GTK_CONTAINER (combo->popup)->border_width +
				   2 * GTK_CONTAINER (GTK_BIN (popup)->child)->border_width +
				   2 * GTK_BIN (popup)->child->style->klass->xthickness)+20;
  
	do 
    {
		old_width = alloc_width;
		old_height = work_height;
      
		if (!show_hscroll &&
			alloc_width < list_requisition.width)
		{
			work_height += popup->hscrollbar->requisition.height +
				GTK_SCROLLED_WINDOW_CLASS 
				(GTK_OBJECT (combo->popup)->klass)->scrollbar_spacing;
			show_hscroll = TRUE;
		}
		if (!show_vscroll && 
			work_height + list_requisition.height > avail_height)
		{
			if (work_height + min_height > avail_height && 
				*y - real_height > avail_height)
			{
				*y -= (work_height + list_requisition.height + real_height);
				break;
			}
			alloc_width -= 
				popup->vscrollbar->requisition.width +
				GTK_SCROLLED_WINDOW_CLASS 
				(GTK_OBJECT (combo->popup)->klass)->scrollbar_spacing;
			show_vscroll = TRUE;
		}
    } while (old_width != alloc_width || old_height != work_height);
  
	*width = widget->allocation.width + 100;
	if (show_vscroll)
		*height = avail_height;
	else
		*height = work_height + list_requisition.height;
  
	if (*x < 0)
		*x = 0;
}


void
popup_list (GnomeCmdCombo *combo, GtkWidget *file_list)
{
	gint height, width, x, y;
	gint old_width, old_height;

	combo->file_list = file_list;

	old_width = combo->popwin->allocation.width;
	old_height  = combo->popwin->allocation.height;

	get_pos (combo, &x, &y, &height, &width);

	/* workaround for gtk_scrolled_window_size_allocate bug */
	if (old_width != width || old_height != height)
    {
		gtk_widget_hide (GTK_SCROLLED_WINDOW (combo->popup)->hscrollbar);
		gtk_widget_hide (GTK_SCROLLED_WINDOW (combo->popup)->vscrollbar);
    }

	gtk_widget_set_uposition (combo->popwin, x, y);
	gtk_widget_set_usize (combo->popwin, width, height);
	gtk_widget_realize (combo->popwin);
	gdk_window_resize (combo->popwin->window, width, height);
	gtk_widget_show (combo->popwin);

	gtk_widget_grab_focus (combo->popwin);
}





/*******************************
 * Callbacks
 *******************************/

static int
on_entry_key_press (GtkEntry *entry, GdkEventKey *event, GnomeCmdCombo *combo)
{
	return FALSE;
}


static gint
on_entry_focus_out (GtkEntry *entry, GdkEventFocus *event, GnomeCmdCombo *combo)
{

	if (combo->value_in_list)
    {
		if (combo->ok_if_empty && !strcmp (gtk_entry_get_text (entry), ""))
			return FALSE;

#ifdef TEST
		printf ("INVALID ENTRY: `%s'\n", gtk_entry_get_text (entry));
#endif
		gtk_grab_add (GTK_WIDGET (combo));
		/* this is needed because if we call gtk_widget_grab_focus() 
		   it isn't guaranteed it's the *last* call before the main-loop,
		   so the focus can be lost anyway...
		   the signal_emit_stop doesn't seem to work either...
		*/
		gtk_idle_add ((GtkFunction) focus_idle, combo);
		
		return TRUE;
    }
	return FALSE;
}


static void        
on_entry_activate (GtkWidget        *widget,
				   GnomeCmdCombo    *combo)
{
	popup_list (combo, NULL);

	if (!GTK_WIDGET_HAS_FOCUS (combo->entry))
		gtk_widget_grab_focus (combo->entry);

	gtk_grab_add (combo->popwin);

	gdk_pointer_grab (combo->popwin->window, TRUE,
					  GDK_BUTTON_PRESS_MASK | 
					  GDK_BUTTON_RELEASE_MASK |
					  GDK_POINTER_MOTION_MASK, 
					  NULL, NULL, GDK_CURRENT_TIME);
}


static void
on_entry_changed (GtkEntry *entry, GnomeCmdCombo *combo)
{
	gtk_grab_remove (GTK_WIDGET (combo));
}




static gboolean        
on_popup_button_press (GtkWidget        *button,
					   GdkEventButton   *event,
					   GnomeCmdCombo    *combo)
{
	if (!GTK_WIDGET_HAS_FOCUS (combo->entry))
		gtk_widget_grab_focus (combo->entry);
	if (!combo->current_button && (event->button == 1))
		popup_list (combo, NULL);

	combo->current_button = event->button;
  
	//GTK_LIST (combo->list)->drag_selection = TRUE;
	
	gdk_pointer_grab (combo->list->window, TRUE,
					  GDK_POINTER_MOTION_HINT_MASK |
					  GDK_BUTTON1_MOTION_MASK |
					  GDK_BUTTON_RELEASE_MASK,
					  NULL, NULL, event->time);
		
	gtk_grab_add (combo->list);

	return FALSE;
}


static void         
on_popup_button_leave_notify (GtkWidget        *button,
							  GdkEventCrossing *event,
							  GnomeCmdCombo    *combo)
{
	if (combo->current_button)
		gtk_signal_emit_stop_by_name (GTK_OBJECT (button), "leave_notify_event");
}





static void
on_list_select_row (GtkCList *list, gint row, gint column,
					GdkEventButton *event, GnomeCmdCombo *combo)
{
	gchar *text;

	gtk_clist_get_text (list, row, 0, &text);
	gtk_entry_set_text (GTK_ENTRY (combo->entry), text);
}


static void
on_list_unselect_row (GtkCList *list, gint row, gint column,
					  GdkEventButton *event, GnomeCmdCombo *combo)
{
}


static gboolean
on_list_button_release (GtkWidget *widget, GdkEvent *event, GnomeCmdCombo *combo)
{
	GtkWidget *child;

	if ((combo->current_button != 0) && (event->button.button == 1))
    {
		/* This was the initial button press */
		GdkEventCrossing tmp_event;

		combo->current_button = 0;

		if (widget != combo->button)
			gtk_widget_event (combo->button, event);

		/* Un-pre-hightlight */      
		tmp_event.type = GDK_LEAVE_NOTIFY;
		tmp_event.window = combo->button->window;
		tmp_event.send_event = TRUE;
		tmp_event.subwindow = NULL;
		tmp_event.detail = GDK_NOTIFY_ANCESTOR;
      
		gtk_widget_event (combo->button, (GdkEvent *)&tmp_event);

		/* Check to see if we released inside the button */
		child = gtk_get_event_widget ((GdkEvent*) event);

		while (child && child != (combo->button))
			child = child->parent;

		if (child == combo->button)
		{
			gtk_grab_add (combo->popwin);
			gdk_pointer_grab (combo->popwin->window, TRUE,
							  GDK_BUTTON_PRESS_MASK | 
							  GDK_BUTTON_RELEASE_MASK |
							  GDK_POINTER_MOTION_MASK, 
							  NULL, NULL, GDK_CURRENT_TIME);

			return FALSE;
		}
    }
	else
    {
		/* The user has clicked inside the popwin and released */
		if (GTK_WIDGET_HAS_GRAB (combo->popwin))
		{
			gtk_grab_remove (combo->popwin);
			gdk_pointer_ungrab (event->button.time);
		}
    }
  
	gtk_widget_hide (combo->popwin);

	return TRUE;
}


static int
on_list_key_press (GtkWidget *widget, GdkEventKey *event, GnomeCmdCombo *combo)
{
	if (event->keyval == GDK_Escape)
    {
		if (GTK_WIDGET_HAS_GRAB (combo->popwin))
		{
			gtk_grab_remove (combo->popwin);
			gdk_pointer_ungrab (GDK_CURRENT_TIME);
		}
		
		gtk_widget_hide (combo->popwin);
		
		if (GTK_WIDGET_HAS_GRAB (combo->button))
		{
			combo->current_button = 0;
			GTK_BUTTON (combo->button)->in_button = FALSE;
			gtk_button_released (GTK_BUTTON (combo->button));
			gtk_grab_remove (combo->button);
		}

		if (GTK_WIDGET_HAS_GRAB (combo->list))
		{
			gtk_grab_remove (combo->list);

		}
		
		if (combo->file_list)
		{
			gtk_widget_grab_focus (combo->file_list);
			combo->file_list = NULL;
		}
		
		return TRUE;
    }

	return FALSE;
}


static void
on_list_scroll_vertical                  (GtkCList          *clist,
										  GtkScrollType      scroll_type,
										  gfloat             position,
										  GnomeCmdCombo     *combo)
{
	if (clist->focus_row >= 0)
		gtk_clist_select_row (clist, clist->focus_row, 0);
}


static gboolean
on_popupwin_button_press (GtkWidget *widget, GdkEvent *event, GnomeCmdCombo *combo)
{
	GtkWidget *child;

	child = gtk_get_event_widget (event);

	/* We don't ask for button press events on the grab widget, so
	 *  if an event is reported directly to the grab widget, it must
	 *  be on a window outside the application (and thus we remove
	 *  the popup window). Otherwise, we check if the widget is a child
	 *  of the grab widget, and only remove the popup window if it
	 *  is not.
	 */
	
	if (child != widget)
    {
		while (child)
		{
			if (child == widget)
				return FALSE;
			child = child->parent;
		}
    }
	
	gtk_widget_hide (combo->popwin);
	gtk_grab_remove (combo->popwin);
	gdk_pointer_ungrab (event->button.time);
	gtk_grab_remove (combo->list);
	
	return TRUE;
}


static int
on_popupwin_key_press (GtkWidget *widget, GdkEventKey *event, GnomeCmdCombo *combo)
{
	if (event->keyval == GDK_Return || event->keyval == GDK_KP_Enter)
    {
		if (GTK_WIDGET_VISIBLE (combo->popwin))
		{
			gtk_widget_hide (combo->popwin);
	  
			if (GTK_WIDGET_HAS_GRAB (combo->popwin))
			{
				gtk_grab_remove (combo->popwin);
				gdk_pointer_ungrab (event->time);
			}
		}

		gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "key_press_event");

		if (GTK_WIDGET_HAS_GRAB (combo->list))
		{
			gtk_grab_remove (combo->list);

		}
		
		if (combo->file_list)
		{
			gtk_widget_grab_focus (combo->file_list);
			combo->file_list = NULL;
		}

		return TRUE;
    }

	return FALSE;
}




static gint         
on_button_enter_notify (GtkWidget        *widget,
						GdkEventCrossing *event,
						GnomeCmdCombo    *combo)
{
	GtkWidget *event_widget;

	event_widget = gtk_get_event_widget ((GdkEvent*) event);
  
	if ((event_widget == combo->list) &&
		(combo->current_button != 0) && 
		(!GTK_WIDGET_HAS_GRAB (combo->list)))
    {
		GdkEvent tmp_event;
		gint x, y;
		GdkModifierType mask;

		gtk_grab_remove (combo->popwin);

		/* Transfer the grab over to the list by synthesizing
		 * a button press event
		 */
		gdk_window_get_pointer (combo->list->window, &x, &y, &mask);

		tmp_event.button.type = GDK_BUTTON_PRESS;
		tmp_event.button.window = combo->list->window;
		tmp_event.button.send_event = TRUE;
		tmp_event.button.time = GDK_CURRENT_TIME; /* bad */
		tmp_event.button.x = x;
		tmp_event.button.y = y;
		/* We leave all the XInput fields unfilled here, in the expectation
		 * that GtkList doesn't care.
		 */
		tmp_event.button.button = combo->current_button;
		tmp_event.button.state = mask;

		gtk_widget_event (combo->list, &tmp_event);
    }

	return FALSE;
}



/*******************************
 * Gtk class implementation
 *******************************/

static void
destroy (GtkObject *combo)
{
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (combo);
}


static void
class_init (GnomeCmdComboClass *klass)
{
	GtkObjectClass *oclass;
	GtkWidgetClass *widget_class;

	parent_class = gtk_type_class (gtk_hbox_get_type ());
	oclass = (GtkObjectClass *) klass;
	widget_class = (GtkWidgetClass *) klass;

	oclass->destroy = destroy;
  
	widget_class->size_allocate = size_allocate;
}

static void
init (GnomeCmdCombo *combo)
{
	int i;
	GtkWidget *arrow;
	GtkWidget *frame;
	GtkWidget *event_box;
	GdkCursor *cursor;
	gchar *list_titles[] = {"name", "comment", NULL};

	combo->value_in_list = 0;
	combo->ok_if_empty = 1;
	combo->file_list = NULL;
	
	combo->entry = gtk_entry_new ();
	gtk_widget_ref (combo->entry);
	gtk_object_set_data_full (GTK_OBJECT (combo),
							  "entry", combo->entry,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_set_style (combo->entry, list_style);
	gtk_widget_show (combo->entry);
	
	combo->button = gtk_button_new ();
	gtk_widget_ref (combo->button);
	gtk_object_set_data_full (GTK_OBJECT (combo),
							  "button", combo->button,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (combo->button);
	combo->current_button = 0;
	
	arrow = gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_OUT);
	gtk_widget_ref (arrow);
	gtk_object_set_data_full (GTK_OBJECT (combo),
							  "arrow", arrow,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (arrow);
	
	gtk_container_add (GTK_CONTAINER (combo->button), arrow);
	gtk_box_pack_start (GTK_BOX (combo), combo->entry, TRUE, TRUE, 0);
	gtk_box_pack_end (GTK_BOX (combo), combo->button, FALSE, FALSE, 0);
	GTK_WIDGET_UNSET_FLAGS (combo->button, GTK_CAN_FOCUS);

	/* connect entry signals */
	gtk_signal_connect (
		GTK_OBJECT (combo->entry), "changed",
		(GtkSignalFunc) on_entry_changed, combo);
	gtk_signal_connect (
		GTK_OBJECT (combo->entry), "key-press-event",
		(GtkSignalFunc) on_entry_key_press, combo);
	gtk_signal_connect_after (
		GTK_OBJECT (combo->entry), "focus-out-event",
		(GtkSignalFunc) on_entry_focus_out, combo);
	gtk_signal_connect (
		GTK_OBJECT (combo->entry), "activate",
		(GtkSignalFunc) on_entry_activate, combo);

	/* connect button signals */
	gtk_signal_connect_after (
		GTK_OBJECT (combo->button), "button-press-event",
		(GtkSignalFunc) on_popup_button_press, combo);
	gtk_signal_connect (
		GTK_OBJECT (combo->button), "leave-notify-event",
		(GtkSignalFunc) on_popup_button_leave_notify, combo);

	
	combo->popwin = gtk_window_new (GTK_WINDOW_POPUP);
	gtk_widget_ref (combo->popwin);
	gtk_object_set_data_full (GTK_OBJECT (combo),
							  "popwin", combo->popwin,
							  (GtkDestroyNotify) gtk_widget_unref);	
	gtk_window_set_policy (GTK_WINDOW (combo->popwin), 1, 1, 0);

	/* connect popupwin signals */
	gtk_signal_connect (
		GTK_OBJECT (combo->popwin), "key-press-event",
		GTK_SIGNAL_FUNC (on_popupwin_key_press), combo);
	gtk_signal_connect (
		GTK_OBJECT (combo->popwin), "button-press-event",
		GTK_SIGNAL_FUNC (on_popupwin_button_press), combo);
  
	gtk_widget_set_events (combo->popwin, GDK_KEY_PRESS_MASK);

	event_box = gtk_event_box_new ();
	gtk_widget_ref (event_box);
	gtk_object_set_data_full (GTK_OBJECT (combo),
							  "event_box", event_box,
							  (GtkDestroyNotify) gtk_widget_unref);	
	gtk_container_add (GTK_CONTAINER (combo->popwin), event_box);
	gtk_widget_show (event_box);

	gtk_widget_realize (event_box);
	cursor = gdk_cursor_new (GDK_TOP_LEFT_ARROW);
	gdk_window_set_cursor (event_box->window, cursor);
	gdk_cursor_destroy (cursor);

	frame = gtk_frame_new (NULL);
	gtk_widget_ref (frame);
	gtk_object_set_data_full (GTK_OBJECT (combo),
							  "frame", frame,
							  (GtkDestroyNotify) gtk_widget_unref);	
	gtk_container_add (GTK_CONTAINER (event_box), frame);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_OUT);
	gtk_widget_show (frame);

	combo->popup = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_ref (combo->popup);
	gtk_object_set_data_full (GTK_OBJECT (combo),
							  "combo->popup", combo->popup,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (combo->popup),
									GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	GTK_WIDGET_UNSET_FLAGS (GTK_SCROLLED_WINDOW (combo->popup)->hscrollbar, GTK_CAN_FOCUS);
	GTK_WIDGET_UNSET_FLAGS (GTK_SCROLLED_WINDOW (combo->popup)->vscrollbar, GTK_CAN_FOCUS);
	gtk_container_add (GTK_CONTAINER (frame), combo->popup);
	gtk_widget_show (combo->popup);

	combo->list = gtk_clist_new_with_titles (2,list_titles);
	gtk_widget_ref (combo->list);
	gtk_object_set_data_full (GTK_OBJECT (combo),
							  "combo->list", combo->list,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_clist_set_selection_mode (GTK_CLIST (combo->list), GTK_SELECTION_SINGLE);
	gtk_widget_set_style (combo->list, list_style);
	for ( i=0 ; i<GTK_CLIST (combo->list)->columns ; i++ )
		gtk_widget_set_style (
			GTK_BIN(
				GTK_BIN (
					GTK_CLIST (combo->list)->column[i].button)->child)->child,
				main_style);
	
	/* We'll use enter notify events to figure out when to transfer
	 * the grab to the list
	 */
	gtk_widget_set_events (combo->list, GDK_ENTER_NOTIFY_MASK);

	gtk_scrolled_window_add_with_viewport (
		GTK_SCROLLED_WINDOW (combo->popup), combo->list);
	gtk_container_set_focus_vadjustment (
		GTK_CONTAINER (combo->list),
		gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (combo->popup)));
	gtk_container_set_focus_hadjustment (
		GTK_CONTAINER (combo->list),
		gtk_scrolled_window_get_hadjustment (GTK_SCROLLED_WINDOW (combo->popup)));
	gtk_widget_show (combo->list);

	/* connect list signals */
	gtk_signal_connect (
		GTK_OBJECT (combo->list), "select-row",
		GTK_SIGNAL_FUNC (on_list_select_row), combo);
	gtk_signal_connect (
		GTK_OBJECT (combo->list), "unselect-row",
		GTK_SIGNAL_FUNC (on_list_unselect_row), combo);
	gtk_signal_connect_after (
		GTK_OBJECT (combo->list), "button-release-event",
		GTK_SIGNAL_FUNC (on_list_button_release), combo);
	gtk_signal_connect (
		GTK_OBJECT (combo->list), "key-press-event",
		GTK_SIGNAL_FUNC (on_list_key_press), combo);
	gtk_signal_connect_after (
		GTK_OBJECT (combo->list), "scroll-vertical",
		GTK_SIGNAL_FUNC (on_list_scroll_vertical), combo);

	
	/* We connect here on the button, because we'll have a grab on it
	 * when the event occurs. But we are actually interested in enters
	 * for the combo->list.
	 */
	gtk_signal_connect (
		GTK_OBJECT (combo->button), "enter-notify-event",
		GTK_SIGNAL_FUNC (on_button_enter_notify), combo);
}



/***********************************
 * Public functions
 ***********************************/

guint
gnome_cmd_combo_get_type (void)
{
	static guint combo_type = 0;

	if (!combo_type)
    {
		static const GtkTypeInfo combo_info =
		{
			"GnomeCmdCombo",
			sizeof (GnomeCmdCombo),
			sizeof (GnomeCmdComboClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};
		combo_type = gtk_type_unique (gtk_hbox_get_type (), &combo_info);
    }
	return combo_type;
}

GtkWidget *
gnome_cmd_combo_new (void)
{
	return GTK_WIDGET (gtk_type_new (gnome_cmd_combo_get_type ()));
}


void
gnome_cmd_combo_set_popdown_strings (GnomeCmdCombo *combo, GList *strings)
{
	GList *list = strings;

	g_return_if_fail (combo != NULL);
	g_return_if_fail (GNOME_CMD_IS_COMBO (combo));
	g_return_if_fail (strings != NULL);

	gtk_clist_clear (GTK_CLIST (combo->list));

	while (list)
    {
		gchar *text[3];
		text[0] = (gchar*)list->data;
		text[1] = "description";
		text[2] = NULL;
		gtk_clist_append (GTK_CLIST (combo->list), text);
		list = list->next;
    }
}


void
gnome_cmd_combo_clear (GnomeCmdCombo *combo)
{
	gtk_clist_clear (GTK_CLIST (combo->list));
}


void
gnome_cmd_combo_append (GnomeCmdCombo *combo, gchar **text)
{
	g_return_if_fail (combo != NULL);
	g_return_if_fail (GNOME_CMD_IS_COMBO (combo));
	g_return_if_fail (text != NULL);

	gtk_clist_append (GTK_CLIST (combo->list), text);
}


void
gnome_cmd_combo_set_selection (GnomeCmdCombo *combo, const gchar *text)
{
	g_return_if_fail (GNOME_CMD_IS_COMBO (combo));
	g_return_if_fail (text != NULL);

	gtk_entry_set_text (GTK_ENTRY (combo->entry), text);	
}


void
gnome_cmd_combo_update_style (GnomeCmdCombo *combo)
{
	gint i;
	
	gtk_widget_set_style (combo->entry, list_style);
	gtk_widget_set_style (combo->list, list_style);
	
	for ( i=0 ; i<GTK_CLIST (combo->list)->columns ; i++ )
		gtk_widget_set_style (
			GTK_BIN(
				GTK_BIN (
					GTK_CLIST (combo->list)->column[i].button)->child)->child,
				main_style);	
}
