/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* engine-pgn.c
 *
 * Copyright (C) 2001  JP Rosevear.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: JP Rosevear
 */

#include <config.h>
#include <gtk/gtk.h>
#include "makros.h"
#include "pgn.h"
#include "board-window.h"
#include "engine-pgn.h"

extern BoardWindow *main_board_window;

static EngineClass *parent_class = NULL;

static void class_init (EnginePgnClass *klass);
static void init (EnginePgn *pgn);

static void game_selected_cb (GtkWidget *widget, int row, int column, GdkEventButton *event, gpointer user_data);

static void column_clicked_cb (GtkWidget *widget, int column,  gpointer user_data);

struct _EnginePgnPrivate 
{
	GtkWidget *view;

	GtkWidget *sw;
	GtkWidget *games;

	pgn_info *pgn_info;
	EngineInfo *info;
};

GtkType
engine_pgn_get_type (void)
{
  static GtkType type = 0;

  if (type == 0)
    {
      static const GtkTypeInfo info =
      {
        "EnginePgn",
        sizeof (EnginePgn),
        sizeof (EnginePgnClass),
        (GtkClassInitFunc) class_init,
        (GtkObjectInitFunc) init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      type = gtk_type_unique (engine_get_type (), &info);
    }

  return type;
}

static void
class_init (EnginePgnClass *klass)
{
	GtkObjectClass *object_class;

	object_class = GTK_OBJECT_CLASS (klass);

	parent_class = gtk_type_class (engine_get_type ());

}


static void
init (EnginePgn *engine)
{
	EnginePgnPrivate *priv;
	static char *titles[] ={ N_("White"), N_("Black"), N_("Result")};
	int i;

	priv = g_new0 (EnginePgnPrivate, 1);

	engine->priv = priv;

	priv->info = g_new0 (EngineInfo, 1);
	
	priv->sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (priv->sw),
					GTK_POLICY_AUTOMATIC,     
					GTK_POLICY_AUTOMATIC);
	gtk_widget_show (priv->sw);
	
	for (i = 0; i < 3; i++)
		titles[i] = _(titles[i]);
	priv->games = gtk_clist_new_with_titles (3, titles);

	gtk_signal_connect (GTK_OBJECT (priv->games), "select-row",
			    (GtkSignalFunc) game_selected_cb,
			    engine);
	gtk_signal_connect (GTK_OBJECT (priv->games), "click-column",
			    (GtkSignalFunc) column_clicked_cb,
			    engine);
	gtk_widget_show (priv->games);
	
	gtk_container_add (GTK_CONTAINER (priv->sw), priv->games);

	priv->view = game_view_new_with_engine (ENGINE (engine));
	board_window_add_view (main_board_window, GAME_VIEW (priv->view));
}



GtkObject *
engine_pgn_new (char *filename)
{
	EnginePgn *engine;
	EnginePgnPrivate *priv;
	GList *l;
	
	engine = gtk_type_new (engine_pgn_get_type ());
	priv = engine->priv;
	
	priv->pgn_info = pgn_open (filename);

	for (l = priv->pgn_info->game_tags; l != NULL; l = l->next) {
		Pgn_Tag *tag = l->data;
		char *text[3] = { "", "","" };
		gint row;
		
		row = gtk_clist_append (GTK_CLIST (priv->games), text);
		gtk_clist_set_text (GTK_CLIST (priv->games), row , 0, tag->white);
		gtk_clist_set_text (GTK_CLIST (priv->games), row , 1, tag->black);
		gtk_clist_set_text (GTK_CLIST (priv->games), row , 2, tag->result);
	}
	board_window_add_info (main_board_window, g_basename (filename), priv->sw);

	gtk_clist_columns_autosize(GTK_CLIST(priv->games));

	return GTK_OBJECT (engine);
}

static void
game_selected_cb (GtkWidget *widget, int row, int column, GdkEventButton *event, gpointer user_data)
{
	EnginePgn *engine = user_data;
	EnginePgnPrivate *priv;
	gchar *text;

	priv = engine->priv;

	pgn_select_game (priv->pgn_info, GAME_VIEW (priv->view), row);
	move_list_move_end (game_view_get_movelist (GAME_VIEW (priv->view)));

	if (priv->info->white)
		g_free (priv->info->white);
	gtk_clist_get_text (GTK_CLIST (priv->games), row, 0, &text);
	priv->info->white = g_strdup (text);
	
	if (priv->info->black)
		g_free (priv->info->black);
	gtk_clist_get_text (GTK_CLIST (priv->games), row, 1, &text);
	priv->info->black = g_strdup (text);
	
	gtk_clist_get_text (GTK_CLIST (priv->games), row, 2, &text);
	
	if (!strcmp (text, "1-0"))
		priv->info->result = GAME_WHITE;
	else if (!strcmp (text, "0-1"))
		priv->info->result = GAME_BLACK;
	else if (!strcmp (text, "1/2-1/2"))
		priv->info->result = GAME_DRAW;
		
	gtk_signal_emit_by_name (GTK_OBJECT (engine), "info", priv->info);
}

static void
column_clicked_cb (GtkWidget *widget, int column,  gpointer user_data)
{
	GtkCList *clist = GTK_CLIST(widget);

	if (column == clist->sort_column) {
		if (clist->sort_type == GTK_SORT_ASCENDING)
			clist->sort_type = GTK_SORT_DESCENDING;
		else
			clist->sort_type = GTK_SORT_ASCENDING;
	}
	else
		gtk_clist_set_sort_column (clist, column);
	
	gtk_clist_set_compare_func (clist, NULL);
	gtk_clist_sort (clist);
}


