/*
 * GNoise
 *
 * Copyright (C) 1999-2001 Dwight Engen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: oss.c,v 1.8 2002/01/13 02:51:16 dengen Exp $
 *
 */

#include <errno.h>
#include <fcntl.h>
#include <linux/types.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/soundcard.h>
#include <unistd.h>
#include "oss.h"
#include "gnoise.h"
#include "gtklevels.h"
#include "gtkwaveset.h"
#include "snd_buf.h"


static void     oss_check_clip(char *buf, guint32 buf_size, snd_info_t *si);
static int	oss_open_dsp(int rec, snd_info_t *si);
static guint32	oss_play_sleep(int sd, GtkWaveSet *ws);

guint32 play_sample;
guint32 play_start;
guint32 play_end;

static gint fragsize;



/*
 * FUNC: open dsp device, setting sample size and rate
 *   IN: rec is TRUE if you want to read the dsp, FALSE for write
 *	 sample_bits is the number of bits per sample
 *	 sample_rate is the rate in Hz to sample at
 *  OUT: file descriptor to read or write on or -1 on failure
 */
static int oss_open_dsp(int rec, snd_info_t *in_si)
{
    int			sd;
    audio_buf_info	ab_info;
    snd_info_t		si;

    si = *in_si;
    sd = open("/dev/dsp", rec ? O_RDONLY : O_WRONLY | O_NONBLOCK);
    if (sd < 0)
    {
	log("DAC", "Unable to open /dev/dsp %s\n", strerror(errno));
	goto err1;
    }

    /* reset O_NONBLOCK */
    fcntl(sd, F_SETFL, 0);	

    /* Set number of channels */
    if (ioctl(sd, SNDCTL_DSP_CHANNELS, &si.channels) < 0)
    {
	log("DAC", "ioctl SNDCTL_DSP_CHANNELS failed %s\n", strerror(errno));
	goto err2;
    }

    /* Set sample bit size (8, 16 etc...) */
    if (ioctl(sd, SNDCTL_DSP_SAMPLESIZE, &si.sample_bits) < 0)
    {
	log("DAC", "ioctl SNDCTL_DSP_SAMPLESIZE failed %s\n", strerror(errno));
	goto err2;
    }

    /* Set sample rate */
    if (ioctl(sd, SNDCTL_DSP_SPEED, &si.sample_rate) < 0)
    {
	log("DAC", "ioctl SNDCTL_DSP_SPEED failed %s\n", strerror(errno));
	goto err2;
    }

    /* Get drivers frag size */
    ioctl(sd, SNDCTL_DSP_GETBLKSIZE, &fragsize);
    ioctl(sd, SNDCTL_DSP_GETOSPACE, &ab_info);

    return(sd);

err2:
    close(sd);
err1:
    return(-1);
}



/*
 * FUNC: sleep the thread until we need to redraw
 *	 the playline or write more data
 */
static guint32 oss_play_sleep(int sd, GtkWaveSet *ws)
{
    guint32     dac_left;
    guint32     play_sleep;
    guint32     draw_sleep;
    guint32     u_sleep;
    snd_info_t *si = &ws->sb->info;

    ioctl(sd, SNDCTL_DSP_GETODELAY, &dac_left);

    play_sleep = (dac_left / (si->sample_bits/8) / si->channels)
		 *(1000000/si->sample_rate) / 2;
    draw_sleep = (1<<ws->zoom)*(1000000/si->sample_rate);

    u_sleep = MIN(play_sleep, draw_sleep)/2;
#   if 0
    {
	static guint32 old_u_sleep = 0;

	if (old_u_sleep != u_sleep)
	{
	    log("DAC", "%s sleep:%d dac:%d\n",
		play_sleep < draw_sleep ? "play" : "draw", u_sleep, dac_left);
	    old_u_sleep = u_sleep;
	}
    }
#   endif
    usleep(u_sleep);
    return dac_left;
}



/*
 * FUNC: if there are level meters displayed, see if the audio in buf clips
 *	 to set the clip indicators
 */
static void oss_check_clip(char *buf, guint32 buf_size, snd_info_t *si)
{
    if ((playing && win_levels) || (recording))
    {
	guint8	   smp_max[MAX_CHANNELS];
	gboolean   clipped[MAX_CHANNELS];
	gint8	   channel;
	guint32	   bps;
	guint32	   smp_indx;
	gint16	   smp16;
	gint8	   smp8=0;

	/* set initial values */
	for(channel = 0; channel < si->channels; channel++)
	{
	    smp_max[channel] = 0;
	    clipped[channel] = FALSE;
	}

	bps = (si->sample_bits / 8) * si->channels;	/* bytes per sample */
	for(smp_indx = 0;
	    smp_indx < fragsize / bps &&
	    smp_indx *  bps < buf_size;
	    smp_indx += si->channels)
	{
	    for(channel = 0; channel < si->channels; channel++)
	    {
		switch(si->sample_bits)
		{
		    case 8:
			smp16 = ((guint8 *)buf)[smp_indx + channel];
			if (smp16 >= 255)
			    clipped[channel] = TRUE;
			smp8 = smp16-128;
			break;

		    case 16:
			smp16 = ((gint16 *)buf)[smp_indx + channel];
			if (ABS(smp16) >= 32767)
			    clipped[channel] = TRUE;
			smp8 = (smp16 << 8) / 65535;
			break;

		    /* now: -128 < smp8 < 127 */
		}

		smp_max[channel] = MAX(ABS(smp8), smp_max[channel]);
	    }
	}
	lpeak = rpeak = smp_max[0];
	if (si->channels > 1)
	    rpeak = smp_max[1];

	if (clipped[0])
	    lclip = rclip = TRUE;
	if (si->channels > 1)
	    rclip = clipped[1];
    }
}


/*
 * FUNC: play thread
 */
void *oss_play(void *thread_arg)
{
    int sd;
    GtkWaveSet *ws = GTK_WAVE_SET(thread_arg);
    count_info ci;
    int count;
    int size;
    int buf_indx;
    int wrote;
    int left;


    /* open dsp device to play on */
    sd = oss_open_dsp(FALSE, &ws->sb->info);
    if (sd < 0)
    {
	queue_cmd(CMD_REAP_PLAY_THREAD, NULL);
	return((void *)FALSE);
    }

loop:
    /* calculate total bytes of audio we're going to write */
    size = count = (play_end - play_start) * ws->sb->info.channels
		    * (ws->sb->info.sample_bits/8);
    buf_indx = play_start * (ws->sb->info.sample_bits/8) * ws->sb->info.channels;
    playing = TRUE;
    playing_stop = FALSE;

    while(!playing_stop && count)
    {
	char *buf;

	int c = count;
	audio_buf_info ab_info;

	if (c > fragsize)
	    c = fragsize;

	buf = &((char *)ws->sb->data)[buf_indx];
	if ((wrote = write(sd, buf, c)) != c)
	{
	    log("DAC", "write failed %s\n", strerror(errno));
	    break;
	}
	count -= wrote;
	buf_indx += wrote;

	oss_check_clip(buf, c, &ws->sb->info);
	for(; playing ;)
	{
	    /* update playline */
	    if (ioctl(sd, SNDCTL_DSP_GETOPTR, &ci) >= 0)
		play_sample = play_start + ci.bytes / ((ws->sb->info.sample_bits/8) * ws->sb->info.channels);

	    /* see if we need to write more */
	    ioctl(sd, SNDCTL_DSP_GETOSPACE, &ab_info);
	    if (ab_info.bytes > fragsize)
		break;

	    /* some cards (ie ad1848) apparently need a kick in the ass
	     * to get going if we didn't write a full fragment
	     */
	    if (ci.bytes == 0)
		ioctl(sd, SNDCTL_DSP_POST, NULL);

	    oss_play_sleep(sd, ws);
	}
    }

    /* keep updating playline while last fragment runs out... */
    for(left = 1; !playing_stop && left > 0 && ci.bytes < size;)
    {
	/* update playline */
	if (ioctl(sd, SNDCTL_DSP_GETOPTR, &ci) >= 0)
	    play_sample = MIN(play_start + ci.bytes / 
		((ws->sb->info.sample_bits/8) * ws->sb->info.channels), play_end);

	left = oss_play_sleep(sd, ws);
    }

    ioctl(sd, SNDCTL_DSP_RESET, 0);
    if (playing_looped != LOOP_NONE && !playing_stop)
    {
	if (playing_looped == LOOP_FILE)
	    play_start = 0;
	scrolling = FALSE;
	goto loop;
    }

    close(sd);
    playing = FALSE;
    playing_looped = LOOP_NONE;
    queue_cmd(CMD_REAP_PLAY_THREAD, NULL);
    return((void *)TRUE);
}



/*
 * FUNC: record thread
 */
void *oss_record(void *thread_arg)
{
    record_info_t	*rec = thread_arg;
    int			sd;
    int			fd;
    int			count;
    guint32		bytes_recorded = 0;
    char		*buf;
    gboolean		ret = FALSE;

    /* open file to write to */
    fd = open(rec->file, O_WRONLY | O_CREAT | O_TRUNC, 0644);
    if (fd < 0)
    {
	log("DAC", "Unable to open file %s %s\n", rec->file, strerror(errno));
	goto out1;
    }

    /* seek past where header will go */
    lseek(fd, sizeof(wave_header_t) + sizeof(data_chunk_t), SEEK_SET);

    /* open dsp device to record from */
    sd = oss_open_dsp(TRUE, &rec->si);
    if (sd < 0)
    {
	goto out2;
    }

    buf = malloc(fragsize);
    if (buf == NULL)
    {
	log("DAC", "Unable to allocate recording buffer");
	goto out3;
    }
    memset(buf, 0, fragsize);
    recording = TRUE;
    recording_stop = FALSE;

    while(!recording_stop)
    {
	if ((count = read(sd, buf, fragsize)) != fragsize)
	{
	    log("DAC", "read failed %s\n", strerror(errno));
	    break;
	}

	if (write(fd, buf, fragsize) < 0)
	{
	    log("DAC", "write failed %s\n", strerror(errno));
	    break;
	}

	bytes_recorded += count;
	oss_check_clip(buf, fragsize, &rec->si);
    }

    ioctl(sd, SNDCTL_DSP_RESET, 0);
    rec->si.samples = bytes_recorded / rec->si.channels / (rec->si.sample_bits / 8);
    snd_file_header_save(fd, rec->si);
    ret = TRUE;
    free(buf);
out3:
    close(sd);
out2:
    close(fd);
out1:
    recording = FALSE;
    queue_cmd(CMD_REAP_RECORD_THREAD, rec);
    return((void *)ret);
}
