/*
 * GNoise
 *
 * Copyright (C) 1999-2001 Dwight Engen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: gtkwaveset.c,v 1.6 2002/01/13 02:51:16 dengen Exp $
 *
 */

#include <stdio.h>
#include <string.h>
#include "gtkwaveset.h"

#define SAMPLES_PER_PIXEL                         (1<<ws->zoom)
#define SEL_PIX				          4

static void       gtk_wave_set_class_init         (GtkWaveSetClass *klass);
static void       gtk_wave_set_init               (GtkWaveSet *ws);
static void       gtk_wave_set_size_allocate      (GtkWidget *widget, GtkAllocation *allocation);
static gint       gtk_wave_set_button_press       (GtkWidget *widget, GdkEventButton *event);
static gint       gtk_wave_set_button_release     (GtkWidget *widget, GdkEventButton *event);
static gint       gtk_wave_set_motion_notify      (GtkWidget *widget, GdkEventMotion *event);

static GdkCursor *gtk_wave_set_cursor_get         (GtkWaveSet *WaveSet, guint cursor);
static void       gtk_wave_set_cursor_set         (GtkWaveSet *ws, guint new_cursor);
static void       gtk_wave_set_zoom_mid           (GtkWaveSet *ws, smpl_indx mid);
static void       gtk_wave_set_scrolled           (GtkAdjustment *adj, GtkWaveSet *ws);
static void       gtk_wave_set_scrolladj          (GtkWaveSet *ws);

static void       gtk_wave_set_selection_notify   (GtkWaveSet *ws, gint x, GtkWaveDisplay *wd);
static void       gtk_wave_set_selection_validate (GtkWaveSet *ws, gboolean invalidate);

static gint32    __inline__ smpl_to_xpos          (GtkWaveSet *ws, smpl_indx smp);
static smpl_indx __inline__ xpos_to_smpl          (GtkWaveSet *ws, guint32 xpos);

static GtkContainerClass *parent_class = NULL;


GtkType
gtk_wave_set_get_type (void)
{
    static GtkType wave_set_type = 0;

    if (!wave_set_type)
    {
	static const GtkTypeInfo wave_set_info =
	{
	    "GtkWaveSet",
	    sizeof (GtkWaveSet),
	    sizeof (GtkWaveSetClass),
	    (GtkClassInitFunc) gtk_wave_set_class_init,
	    (GtkObjectInitFunc) gtk_wave_set_init,
	    NULL,
	    NULL,
	    (GtkClassInitFunc) NULL,
	};

	wave_set_type = gtk_type_unique (GTK_TYPE_VBOX, &wave_set_info);
    }

    return wave_set_type;
}


static void
gtk_wave_set_class_init (GtkWaveSetClass *klass)
{
    GtkWidgetClass *widget_class;
    GtkObjectClass *object_class;

    object_class = (GtkObjectClass *) klass;
    widget_class = (GtkWidgetClass *) klass;

    parent_class = gtk_type_class (GTK_TYPE_VBOX);

    klass->cursor_selext = gdk_cursor_new(CURSOR_SELEXT);
    klass->cursor_normal = gdk_cursor_new(CURSOR_NORMAL);
    widget_class->size_allocate = gtk_wave_set_size_allocate;
}


static void
gtk_wave_set_init (GtkWaveSet *ws)
{
    GtkBox *box;

    box = GTK_BOX (ws);
    box->spacing = 0;
    box->homogeneous = FALSE;
    gtk_container_set_border_width(GTK_CONTAINER(box), 0);

    memset(ws->wd, 0, sizeof(ws->wd));
    ws->sb = NULL;
    ws->zoom = -1;
    ws->win_frst = -1;
    ws->win_size = -1;

    /* create event box and vbox in it to hold wave display's */
    ws->eventbox = gtk_event_box_new();
    gtk_widget_set_events(ws->eventbox, GDK_POINTER_MOTION_MASK |
					GDK_BUTTON_PRESS_MASK |
					GDK_BUTTON_RELEASE_MASK);
    gtk_widget_show(ws->eventbox);
    gtk_box_pack_start(GTK_BOX(ws), ws->eventbox, TRUE, TRUE, 0);
    gtk_signal_connect(GTK_OBJECT(ws->eventbox), "button_press_event",
		       GTK_SIGNAL_FUNC(gtk_wave_set_button_press), ws);
    gtk_signal_connect(GTK_OBJECT(ws->eventbox), "button_release_event",
		       GTK_SIGNAL_FUNC(gtk_wave_set_button_release), ws);
    gtk_signal_connect(GTK_OBJECT(ws->eventbox), "motion_notify_event",
		       GTK_SIGNAL_FUNC(gtk_wave_set_motion_notify), ws);

    ws->wdvbox = gtk_vbox_new (TRUE, 0);
    gtk_widget_show(ws->wdvbox);
    gtk_container_add(GTK_CONTAINER(ws->eventbox), ws->wdvbox);

    /* create scrollbar */
    ws->scrollbar = gtk_hscrollbar_new(GTK_ADJUSTMENT(
	gtk_adjustment_new(0, 0, 0, 0, 0, 0)));
    gtk_signal_connect(GTK_OBJECT(GTK_SCROLLBAR(ws->scrollbar)->range.adjustment),
		       "value_changed", GTK_SIGNAL_FUNC(gtk_wave_set_scrolled),
		       ws);
    gtk_widget_show(ws->scrollbar);
    gtk_box_pack_start(GTK_BOX(ws), ws->scrollbar, FALSE, FALSE, 0);
}


GtkWidget*
gtk_wave_set_new (void)
{
    return GTK_WIDGET (gtk_type_new (gtk_wave_set_get_type ()));
}


static void
gtk_wave_set_size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
    GtkWaveSet *ws;

    g_return_if_fail(widget != NULL);
    g_return_if_fail(GTK_IS_WAVE_SET(widget));
    g_return_if_fail(allocation != NULL);

    GTK_WIDGET_CLASS (parent_class)->size_allocate(widget, allocation);

    ws = GTK_WAVE_SET(widget);
    if (GTK_WIDGET_REALIZED(widget))
    {
	smpl_indx old_mid;

	old_mid = ws->win_frst + ws->win_size / 2;
	ws->win_size = allocation->width * (1<<ws->zoom);
	gtk_wave_set_zoom_mid(ws, old_mid);
    }
}



void
gtk_wave_set_snd_attach (GtkWaveSet *ws, snd_buf_t *sb)
{
    chnl_indx chnl;

    g_return_if_fail (ws != NULL);
    g_return_if_fail (sb != NULL);
    g_return_if_fail (GTK_IS_WAVE_SET (ws));

    /* set first sample, size, and initial zoom level */
    ws->sb = sb;
    ws->win_frst = 0;
    gtk_wave_set_zoom_out_full(ws, FALSE);
    ws->win_size = GTK_WIDGET(ws)->allocation.width * (1<<ws->zoom);

    /* create the wave display's */
    for (chnl=0; chnl < sb->info.channels; chnl++)
    {
	GtkWaveDisplay *wd;
	char            wd_name[20];

	ws->wd[chnl] = wd = GTK_WAVE_DISPLAY(gtk_wave_display_new());
	sprintf(wd_name, "WaveDisplay%d", chnl);
	gtk_widget_set_name(GTK_WIDGET(wd), wd_name);
	gtk_widget_show(GTK_WIDGET(wd));
	gtk_box_pack_start(GTK_BOX(ws->wdvbox), GTK_WIDGET(wd), TRUE, TRUE, 0);
	gtk_wave_display_cursor_set(wd,
		gtk_wave_set_cursor_get(ws, CURSOR_NORMAL));
	gtk_wave_display_snd_set(wd, sb, chnl);
	gtk_wave_display_zoom_set(wd, ws->zoom);
	gtk_widget_queue_draw(GTK_WIDGET(wd));
    }

    ws->selecting = SELECTING_NOTHING;
    gtk_wave_set_scrolladj(ws);
}


void
gtk_wave_set_snd_detach (GtkWaveSet *ws)
{
    chnl_indx chnl;

    g_return_if_fail (ws != NULL);
    g_return_if_fail (GTK_IS_WAVE_SET (ws));

    for (chnl=0; chnl < ws->sb->info.channels; chnl++)
    {
	gtk_widget_destroy(GTK_WIDGET(ws->wd[chnl]));
	ws->wd[chnl] = NULL;
    }

    ws->sb = NULL;
    ws->zoom = -1;
    ws->win_frst = -1;
    ws->win_size = -1;
    gtk_wave_set_scrolladj(ws);
}


gboolean
gtk_wave_set_scroll (GtkWaveSet *ws, smpl_indx play_smpl)
{
    chnl_indx		channel;
    GtkWaveDisplay	*wd;
    gint		scroll_pix;
    smpl_indx		desire_smpl;

    /* update wd playsample without redrawing */
    for(channel = 0; channel < ws->sb->info.channels; channel++)
	gtk_wave_display_play_set(ws->wd[channel], play_smpl, FALSE);

    /* determine amount we need to scroll */

    desire_smpl = ws->win_frst + ws->win_size - ws->win_size/3;
    scroll_pix = (play_smpl - desire_smpl) / SAMPLES_PER_PIXEL;
    if (scroll_pix <= 0)
	return FALSE;

    ws->win_frst += scroll_pix * SAMPLES_PER_PIXEL;

    for(channel = 0; channel < ws->sb->info.channels; channel++)
    {
	wd = GTK_WAVE_DISPLAY(ws->wd[channel]);

	gtk_wave_display_scroll(wd, scroll_pix);
	gtk_wave_display_win_set(wd, ws->win_frst, FALSE);
    }
    gtk_wave_set_scrolladj(ws);
    return TRUE;
}


void
gtk_wave_set_win_set(GtkWaveSet *ws, smpl_indx frst, gboolean sbadjust)
{
    chnl_indx chnl;

    g_return_if_fail(ws != NULL);
    g_return_if_fail(GTK_IS_WAVE_SET(ws));

    ws->win_frst = frst;
    for (chnl=0; chnl < ws->sb->info.channels; chnl++)
	gtk_wave_display_win_set(ws->wd[chnl], frst, TRUE);

    if (sbadjust)
	gtk_wave_set_scrolladj(ws);
}

static void
gtk_wave_set_scrolladj (GtkWaveSet *ws)
{
    GtkScrollbar	*scrollbar;
    GtkAdjustment	*scrolladj;

    scrollbar = GTK_SCROLLBAR(ws->scrollbar);
    scrolladj = scrollbar->range.adjustment;
    if (ws->sb)
    {
	/* scrolladj->lower is always 0 */
	scrolladj->upper = ws->sb->info.samples + ws->win_size/2;
	scrolladj->value = ws->win_frst;
	scrolladj->step_increment = ws->win_size/32;
	scrolladj->page_increment = ws->win_size;
	scrolladj->page_size = ws->win_size;
    } else {
	scrolladj->upper = 0;
	scrolladj->value = 0;
	scrolladj->page_size = 0;
    }
    
    gtk_adjustment_changed(scrolladj);
    //log("SCROLLBAR", "ws:%d wl:%d sbv:%f upper:%f sti:%f\n", ws->win_frst, ws->win_size, scrolladj->value, scrolladj->upper, scrolladj->step_increment);
}

static void
gtk_wave_set_scrolled (GtkAdjustment *adj, GtkWaveSet *ws)
{
    g_return_if_fail(adj != NULL && ws != NULL);
    g_return_if_fail(GTK_IS_WAVE_SET(ws));

    if (ws->sb)
	gtk_wave_set_win_set(ws, adj->value, FALSE);
}







/* mouse/selection handling */

static gint
gtk_wave_set_button_press (GtkWidget *widget, GdkEventButton *event)
{
    GtkWaveSet     *ws;
    GtkWaveDisplay *wd;

    g_return_val_if_fail(widget != NULL, FALSE);

    ws = GTK_WAVE_SET(widget->parent);
    if (!ws->sb)
	return FALSE;
    wd = GTK_WAVE_DISPLAY(event->window->user_data);

    if (event->button == 3)
    {
	gtk_wave_set_play_set(ws, xpos_to_smpl(ws, event->x));
    } else {
	ws->just_clicked = TRUE;
	gtk_wave_set_selection_notify(ws, event->x, wd);
    }
    
    return FALSE;
}

static gint
gtk_wave_set_button_release (GtkWidget *widget, GdkEventButton *event)
{
    GtkWaveSet     *ws;

    g_return_val_if_fail(widget != NULL, FALSE);

    ws = GTK_WAVE_SET(widget->parent);
    if (!ws->sb)
	return FALSE;

    ws->just_clicked = FALSE;

    if (ws->selecting)
    {
	ws->selecting = SELECTING_NOTHING;
	gtk_wave_set_cursor_set(ws, CURSOR_NORMAL);
    }
    gtk_wave_set_selection_validate(ws, TRUE);

    return FALSE;
}

static gint
gtk_wave_set_motion_notify (GtkWidget *widget, GdkEventMotion *event)
{
    GtkWaveSet     *ws;
    GtkWaveDisplay *wd;

    g_return_val_if_fail(widget != NULL, FALSE);

    ws = GTK_WAVE_SET(widget->parent);
    if (!ws->sb)
	return FALSE;

    wd = GTK_WAVE_DISPLAY(event->window->user_data);

    gtk_wave_set_selection_notify(ws, event->x, wd);

    if (prefs.show_time)
	set_time(xpos_to_smpl(ws, event->x));

    return FALSE;
}



static void
gtk_wave_set_selection_notify(GtkWaveSet *ws, gint x, GtkWaveDisplay *wd)
{
    smpl_indx ss = wd->selection_start;
    smpl_indx se = wd->selection_end;
    guint32 xp;
    guint32 ssx1, ssx2;
    guint32 sex1, sex2;
    gint new_cursor = -1;
    gint8 channel;

    /* determine x ranges around selection start and selection end */
    ssx1 = ssx2 = smpl_to_xpos(ws, ss);
    if (ssx1 > SEL_PIX)
	ssx1 -= SEL_PIX;
    else
	ssx1 = 0;
    ssx2 += SEL_PIX;

    sex1 = sex2 = smpl_to_xpos(ws, se);
    if (sex1 > SEL_PIX)
	sex1 -= SEL_PIX;
    else
	sex1 = 0;
    sex2 += SEL_PIX;


    /* check for new or existing selection, and update mouse cursor */
    if (ws->just_clicked)
    {
	if ((wd->selection) &&
	    (x >= ssx1) &&
	    (x <= ssx2))
		ws->selecting = SELECTING_START;
	else
	if ((wd->selection) &&
	    (x >= sex1) &&
	    (x <= sex2))
		ws->selecting = SELECTING_END;
	else
	{
	   /* brand new selection */
	   ss = se = xpos_to_smpl(ws, x);
	   ws->selecting = SELECTING_NEW;
	   wd->selection = FALSE;
	   new_cursor = CURSOR_SELEXT;
	}
    }
    else if (wd->selection)
    {
	if ((ss != se) &&
	    (((x >= ssx1) && (x <= ssx2)) ||
	     ((x >= sex1) && (x <= sex2)) ) )
		new_cursor = CURSOR_SELEXT;
	else
		new_cursor = CURSOR_NORMAL;
    }

    /* change cursors if necessary */
    if (new_cursor > 0)
	gtk_wave_set_cursor_set(ws, new_cursor);

    if(ws->selecting == SELECTING_NOTHING)
	return;

    if(x < 0)
	x = 0;
    else if (x >= GTK_WIDGET(ws)->allocation.width)
	x = GTK_WIDGET(ws)->allocation.width;

    xp = xpos_to_smpl(ws, x);
    if (xp != ss)
	wd->selection = TRUE;

    if (xp < 0)
	return;

    if (xp > ws->sb->info.samples)
	xp = ws->sb->info.samples;

    switch(ws->selecting)
    {
	case SELECTING_START:
	    if(xp < se) {
		ss = xp;
	    } else {
		ss = se - 1;
		se = xp;
		ws->selecting = SELECTING_END;
	    }
	    break;

	case SELECTING_END:
	    if(xp > ss) {
		se = xp;
	    } else {
		se = ss;
		ss = xp;
		ws->selecting = SELECTING_START;
	    }
	    break;
	case SELECTING_NEW:
	    ws->selecting = SELECTING_END;
	    break;

	default:
	    log("SHIT", "selecting was %d\n", ws->selecting);
	    g_assert_not_reached();
	    break;
    }

    /* see if selection changed */
    if(wd->selection_start != ss || wd->selection_end != se)
    {
	gint8 starting_channel;
	gint8 ending_channel;

	/* set up channels to redraw */
	if (prefs.selection_bond)
	{
	    starting_channel = 0;
	    ending_channel = ws->sb->info.channels;
	} else {
	    starting_channel = wd->channel;
	    ending_channel = starting_channel+1;
	}

	/* redraw them */
	for(channel = starting_channel; channel < ending_channel; channel++)
	{
	    GtkWaveDisplay *lwd = ws->wd[channel];
	    
	    lwd->selection_start = ss;
	    lwd->selection_end   = se;
	    lwd->selection = TRUE;

	    /* redraw just area that changed */
	    if (lwd->selection_start != lwd->selection_start_old)
	    {
		gtk_wave_set_redraw_area(ws,
		    MIN(lwd->selection_start, lwd->selection_start_old),
		    MAX(lwd->selection_start, lwd->selection_start_old),
		    channel);
		lwd->selection_start_old = lwd->selection_start;
	    }

	    /* redraw just area that changed */
	    if (lwd->selection_end != lwd->selection_end_old)
	    {
		gtk_wave_set_redraw_area(ws,
		    MIN(lwd->selection_end, lwd->selection_end_old),
		    MAX(lwd->selection_end, lwd->selection_end_old),
		    channel);
		lwd->selection_end_old = lwd->selection_end;
	    }
	}

	queue_cmd(CMD_SELECTION_DIALOG_UPDATE, NULL);
    }

    ws->just_clicked = FALSE;
}


static void
gtk_wave_set_selection_validate(GtkWaveSet *ws, gboolean invalidate)
{
    chnl_indx channel;
    
    for(channel = 0; channel < ws->sb->info.channels; channel++)
    {
	if (ws->wd[channel]->selection_start == ws->wd[channel]->selection_end)
	{
	    ws->wd[channel]->selection = FALSE;
	    if (invalidate)
	    {
		ws->wd[channel]->selection_start =
		ws->wd[channel]->selection_end = -1;
		queue_cmd(CMD_SELECTION_DIALOG_UPDATE, NULL);
	    }
	}
    }
}


static GdkCursor *
gtk_wave_set_cursor_get (GtkWaveSet *ws, guint cursor)
{
    GtkWaveSetClass	*klass;

    klass = GTK_WAVE_SET_CLASS(GTK_OBJECT(ws)->klass);
    if (cursor == CURSOR_SELEXT)
	    return klass->cursor_selext;
    return klass->cursor_normal;
}

static void
gtk_wave_set_cursor_set (GtkWaveSet *ws, guint new_cursor)
{
    chnl_indx chnl;

    for(chnl = 0; chnl < ws->sb->info.channels; chnl++)
    {
	gtk_wave_display_cursor_set(ws->wd[chnl],
		gtk_wave_set_cursor_get(ws, new_cursor));
    }
}



void
gtk_wave_set_redraw_area(GtkWaveSet *ws, smpl_indx start,
			 smpl_indx end, chnl_indx channel)
{
    chnl_indx channel_start;
    chnl_indx channel_end;
    GdkRectangle area;

    if (channel == CHANNEL_ALL)
    {
	channel_start = 0;
	channel_end = ws->sb->info.channels;
    } else {
	channel_start = channel;
	channel_end   = channel+1;
    }

    area.x      = smpl_to_xpos(ws, start);
    area.y      = 0;
    area.width  = MAX(1, smpl_to_xpos(ws, end)-smpl_to_xpos(ws, start));

    for(channel = channel_start; channel < channel_end; channel++)
    {
	area.height = GTK_WIDGET(ws->wd[channel])->allocation.height;
	//log("WAVE_SET", "queue redraw area x:%d width:%d y:%d height:%d\n",
	//	area.x, area.width, area.y, area.height);
	gtk_widget_queue_draw_area(GTK_WIDGET(ws->wd[channel]),
		area.x, area.y, area.width, area.height);
    }
}



void
gtk_wave_set_zoom (GtkWaveSet *ws, gboolean in)
{
    /* adjust to new zoom level if still in range */
    if (in)
    {
	if (ws->zoom <= 0)
	    return;
	ws->zoom--;
    } else {
	if (ws->zoom >= ZOOM_LEVELS-1)
	    return;
	ws->zoom++;
    }

    /* make previous middle of window new middle */
    gtk_wave_set_zoom_mid(ws, ws->win_frst + ws->win_size / 2);
}

void
gtk_wave_set_zoom_selection (GtkWaveSet *ws)
{
    smpl_indx ns = ws->sb->info.samples;
    smpl_indx ne = -1;
    smpl_indx nl = 0;
    chnl_indx channel;

    /* calculate the starting and ending sample of the selection(s) */
    for(channel = 0; channel < ws->sb->info.channels; channel++)
    {
	/* skip if there's not a valid selection for this channel */
	if (ws->wd[channel]->selection_start < 0)
	    continue;

	ns = MIN(ns, ws->wd[channel]->selection_start);
	ne = MAX(ne, ws->wd[channel]->selection_end);
	nl = ne-ns;
    }

    /* make sure we found a selection */
    if (nl == 0)
    {
	gtk_wave_set_zoom_out_full(ws, TRUE);
	return;
    }

    /* calculate zoom level that can contain our selection(s) length */
    for (ws->zoom = 0; ws->zoom < ZOOM_LEVELS; ws->zoom++)
    {
	if (ws->wd[0]->widget.allocation.width * (1<<ws->zoom) > nl)
	    break;
    }

    /* center display on middle of selection */
    gtk_wave_set_zoom_mid(ws, ns + ((ne - ns) / 2));
}


void
gtk_wave_set_zoom_out_full(GtkWaveSet *ws, gboolean redraw)
{
    for (ws->zoom = 0; ws->zoom < ZOOM_LEVELS-1; ws->zoom++)
    {
	ws->win_size = GTK_WIDGET(ws)->allocation.width * (1<<ws->zoom);
	if (ws->win_size > ws->sb->info.samples)
	    break;
    }

    if (redraw)
	gtk_wave_set_zoom_mid(ws, ws->win_frst + ws->win_size / 2);
}


static void
gtk_wave_set_zoom_mid (GtkWaveSet *ws, smpl_indx mid)
{
    smpl_indx ns;		/* new window start */
    smpl_indx ne;		/* new window end */
    smpl_indx nl;		/* new window length */
    chnl_indx channel;

    if (!ws->sb)
	return;

    nl = GTK_WIDGET(ws)->allocation.width * (1<<ws->zoom);

    if (nl > ws->sb->info.samples)
    {
	nl = ws->sb->info.samples;
	ns = 0;
    } else
	ns = mid-nl/2;

    if (ns < 0)
	ns = 0;

    ne = ns + nl;

    ws->win_frst = ns;
    ws->win_size = nl;

    queue_cmd(CMD_ADJUST_ZOOM, NULL);
    gtk_wave_set_scrolladj(ws);

    /* adjust all channels' start and end */
    for(channel = 0; channel < ws->sb->info.channels; channel++)
    {
	gtk_wave_display_zoom_set(ws->wd[channel], ws->zoom);
	gtk_wave_display_win_set(ws->wd[channel], ns, TRUE);
    }
    scrolling = FALSE;
}


void
gtk_wave_set_marker_add (GtkWaveSet *ws, smpl_indx smp)
{
    smpl_indx old_sel = -1;

    g_return_if_fail (ws != NULL);
    g_return_if_fail (GTK_IS_WAVE_SET(ws));

    /* remember currently selected marker */
    if (ws->sb->marker != NULL && ws->sb->markers > 0)
	old_sel = ws->sb->marker[ws->sb->marker_sel];

    snd_file_marker_add(ws->sb, smp);

    if (old_sel > 0)
	gtk_wave_set_redraw_area(ws, old_sel, old_sel, CHANNEL_ALL);
    gtk_wave_set_redraw_area(ws, smp, smp, CHANNEL_ALL);
}


void
gtk_wave_set_marker_del (GtkWaveSet *ws, smpl_indx smp)
{
    snd_buf_t	*sb;
    int		 i;

    g_return_if_fail (ws != NULL);
    g_return_if_fail (GTK_IS_WAVE_SET(ws));

    sb = ws->sb;
    for(i = 0; i < sb->markers; i++)
    {
	if (sb->marker[i] == smp)
	{
	    snd_file_marker_del(sb, i);
	    gtk_wave_set_redraw_area(ws, smp, smp, CHANNEL_ALL);

	    /* redraw new currently selected marker */
	    if (sb->markers > 0)
		gtk_wave_set_redraw_area(ws, sb->marker[sb->marker_sel],
					 sb->marker[sb->marker_sel],
					 CHANNEL_ALL);
	    return;
	}
    }
}


void
gtk_wave_set_marker_del_all (GtkWaveSet *ws)
{
    g_return_if_fail (ws != NULL);
    g_return_if_fail (GTK_IS_WAVE_SET(ws));

    while(ws->sb->markers > 0)
	gtk_wave_set_marker_del(ws, ws->sb->marker[ws->sb->markers-1]);
}


void
gtk_wave_set_marker_del_sel (GtkWaveSet *ws)
{
    snd_buf_t	*sb;
    int		 i;

    sb = ws->sb;
    for(i = 0; i < sb->markers; i++)
    {
	/* FIXME: assumption of channel bond... */
	if (sb->marker[i] >= ws->wd[0]->selection_start &&
	    sb->marker[i] <= ws->wd[0]->selection_end)
	{
	    gtk_wave_set_marker_del(ws, sb->marker[i]);
	    i--;
	}
    }
}


void
gtk_wave_set_marker_del_cur (GtkWaveSet *ws)
{
    snd_buf_t	*sb;

    sb = ws->sb;
    gtk_wave_set_marker_del(ws, sb->marker[sb->marker_sel]);
}



void
gtk_wave_set_marker_set_cur (GtkWaveSet *ws, gboolean next)
{
    smpl_indx	 sel;
    snd_buf_t	*sb;

    sb = ws->sb;
    sel = sb->marker[ws->sb->marker_sel];
    if (next)
    {
	if (sb->marker_sel >= sb->markers-1)
	    return;
	gtk_wave_set_redraw_area(ws, sel, sel, CHANNEL_ALL);
	sb->marker_sel++;
    }
    else
    {
	if (sb->marker_sel == 0)
	    return;
	gtk_wave_set_redraw_area(ws, sel, sel, CHANNEL_ALL);
	sb->marker_sel--;
    }
    sel = sb->marker[ws->sb->marker_sel];
    gtk_wave_set_redraw_area(ws, sel, sel, CHANNEL_ALL);
}



void
gtk_wave_set_play_set (GtkWaveSet *ws, smpl_indx smp)
{
    chnl_indx chnl;
    smpl_indx play_pos;

    g_return_if_fail (ws != NULL);
    g_return_if_fail (GTK_IS_WAVE_SET(ws));

    play_pos = MIN(smp, ws->sb->info.samples);
    for (chnl = 0; chnl < ws->sb->info.channels; chnl++)
	gtk_wave_display_play_set(ws->wd[chnl], play_pos, TRUE);
}


void
gtk_wave_set_selection_set (GtkWaveSet *ws, smpl_indx start,
			    smpl_indx end, chnl_indx channel)
{
    chnl_indx first;
    chnl_indx last;

    g_return_if_fail (ws != NULL);
    g_return_if_fail (GTK_IS_WAVE_SET(ws));
    
    if ((!ws->sb) || (channel > ws->sb->info.channels) || start > end)
	return;

    if (end > ws->sb->info.samples)
	end = ws->sb->info.samples;

    if (channel == CHANNEL_ALL)
    {
	first = 0;
	last = ws->sb->info.channels;
    } else {
	first = channel;
	last = first+1;
    }

    for(channel = first; channel < last; channel++)
    {
	/* redraw just area in just channel that changed */
	gtk_wave_set_redraw_area(ws,
				 MIN(ws->wd[channel]->selection_start, start),
				 MAX(ws->wd[channel]->selection_end, end),
				 channel);
	ws->wd[channel]->selection_start     = start;
	ws->wd[channel]->selection_start_old = start;
	ws->wd[channel]->selection_end       = end;  
	ws->wd[channel]->selection_end_old   = end;  
	ws->wd[channel]->selection = TRUE;
	ws->selecting = SELECTING_NOTHING;
    }
}


static smpl_indx __inline__
xpos_to_smpl (GtkWaveSet *ws, guint32 xpos)
{
    return MIN(ws->win_frst + xpos * SAMPLES_PER_PIXEL, ws->sb->info.samples);
}

static gint32 __inline__
smpl_to_xpos (GtkWaveSet *ws, smpl_indx smp)
{
    if (smp < ws->win_frst)
	return -1;
    return (smp - ws->win_frst) / SAMPLES_PER_PIXEL;
}
