/******************************************************************************\
 gnofin/record-type.c   $Revision: 1.3 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

#include "common.h"
#include "record-type.h"
#include "data-if.h"


/******************************************************************************
 * Lookups
 */

RecordType *
get_record_type_by_name (const GList * types, const gchar * name)
{
  trace ("");
  g_return_val_if_fail (name, NULL);

  for (; types; types=types->next)
  {
    RecordType * type = LIST_DEREF (RecordType, types);

    if (strcmp (type->name, name) == 0)
      return type;
  }
  return NULL;
}


/******************************************************************************
 * RecordType info
 */

guint
record_type_info_copy (RecordTypeInfo *dest, const RecordTypeInfo *src, guint mask)
{
  trace ("");
  g_return_val_if_fail (src, 0);
  g_return_val_if_fail (dest, 0);

  if (mask == 0)
    mask = RECORD_TYPE_ALL_WRITABLE_FIELDS;
  
  if (mask & RECORD_TYPE_FIELD_NAME)
    dest->name = g_strdup (src->name);
  
  if (mask & RECORD_TYPE_FIELD_DESCRIPTION)
    dest->description = g_strdup (src->description);

  if (mask & RECORD_TYPE_FIELD_USAGE_COUNT)
    dest->usage_count = src->usage_count;
  
  if (mask & RECORD_TYPE_FIELD_NUMBERED)
    dest->numbered = src->numbered;
  
  if (mask & RECORD_TYPE_FIELD_LINKED)
    dest->linked = src->linked;
  
  if (mask & RECORD_TYPE_FIELD_SIGN)
    dest->sign = src->sign;

  return mask;
}

void
record_type_info_clear (RecordTypeInfo *info, guint mask)
{
  trace ("");
  g_return_if_fail (info);

  if (mask == 0)
    mask = RECORD_TYPE_ALL_WRITABLE_FIELDS;
  
  if (mask & RECORD_TYPE_FIELD_NAME)
    g_free (info->name);

  if (mask & RECORD_TYPE_FIELD_DESCRIPTION)
    g_free (info->description);
}

gboolean
record_type_info_diff (const RecordTypeInfo *a, const RecordTypeInfo *b, guint mask)
{
  trace ("");
  g_return_val_if_fail (a, TRUE);
  g_return_val_if_fail (b, TRUE);
  g_return_val_if_fail (mask, TRUE);

  if (mask & RECORD_TYPE_FIELD_NAME)
    if (strcmp (a->name, b->name)) return TRUE;
  
  if (mask & RECORD_TYPE_FIELD_DESCRIPTION)
    if (strcmp (a->description, b->description)) return TRUE;
  
  if (mask & RECORD_TYPE_FIELD_USAGE_COUNT)
    if (a->usage_count != b->usage_count) return TRUE;
  
  if (mask & RECORD_TYPE_FIELD_NUMBERED)
    if (a->numbered != b->numbered) return TRUE;
 
  if (mask & RECORD_TYPE_FIELD_LINKED)
    if (a->linked != b->linked) return TRUE;
 
  if (mask & RECORD_TYPE_FIELD_SIGN)
    if (a->sign != b->sign) return TRUE;
  
  return FALSE;
}


/******************************************************************************
 * RecordType
 */

RecordType *
record_type_new ()
{
  RecordType * type;

  trace ("");

  type = g_new0 (RecordType, 1);
  record_type_ref (type);

  return type;
}

void
record_type_ref (RecordType * type)
{
  trace ("%p", type);
  g_return_if_fail (type);

  type->ref_count++;
}

void
record_type_unref (RecordType * type)
{
  trace ("%p", type);

  if (type && (--type->ref_count == 0))
    record_type_destroy (type);
}

void
record_type_destroy (RecordType * type)
{
  trace ("%p", type);
  g_return_if_fail (type);

  if (type->parent)
    record_type_detach (type);
  
  g_free (type->name);
  g_free (type->description);
  g_free (type);
}

RecordType *
record_type_copy (const RecordType * type)
{
  RecordType * copy;

  trace ("%p", type);
  g_return_val_if_fail (type, NULL);

  copy = record_type_new ();
  memcpy (copy, type, sizeof (*type));
  copy->parent = NULL;
  copy->ref_count = 1;

  return copy;
}

RecordType *
record_type_copy_and_unref (RecordType * type)
{
  RecordType * copy;

  trace ("%p", type);
  g_return_val_if_fail (type, NULL);

  copy = record_type_copy (type);
  record_type_unref (type);
  return copy;
}

RecordType *
record_type_attach (RecordType * type, Bankbook * parent)
{
  trace ("%p, %p", type, parent);
  g_return_val_if_fail (type, NULL);
  g_return_val_if_fail (parent, NULL);

  /* If already parented, then ok provided parent matches arg */
  if (type->parent == parent)
    return type;
  
  g_return_val_if_fail (type->parent == NULL, NULL);

  if (!parent->record_types)
    parent->record_types = g_list_prepend (NULL, type);
  else
  {
    GList * it;
    gint cmp = 0;

    for (it=parent->record_types; it; it=it->next)
    {
      RecordType * temp;
      
      temp = LIST_DEREF (RecordType, it);
      cmp = strcmp (temp->name, type->name);

      if (cmp == 0)
      {
        if (temp != type)
	{
	  record_type_ref (temp);
	  record_type_unref (type);
	}  
        return temp;
      }
      else if (cmp > 0)
      {
	parent->record_types = g_list_first (g_list_prepend (it, type)); 
	break;
      }
      else if (it->next == NULL)
      {
        g_list_append (it, type);
	break;
      }
    }
  }
  type->parent = parent;

  /* The parent now references this record type */
  record_type_ref (type);
  return type;
}

void
record_type_detach (RecordType * type)
{
  trace ("%p", type);
  g_return_if_fail (type);

  if (type->parent)
  {
    type->parent->record_types =
      g_list_remove (type->parent->record_types, type);
    type->parent = NULL;

    /* The parent no longer references this record type */
    record_type_unref (type);
  }
}

gboolean
record_type_set_info (RecordType * type, guint mask, const RecordTypeInfo * info)
{
  trace ("%p", type);
  g_return_val_if_fail (type, FALSE);
  g_return_val_if_fail (info, FALSE);

  if (mask == 0)
    mask = (guint) -1;
  
  if (mask & RECORD_TYPE_FIELD_NAME)
  {
    g_return_val_if_fail (info->name, FALSE);

    if ((!type->name) || (strcmp (type->name, info->name) != 0))
    {
#ifndef G_DISABLE_CHECKS
      /* Verify that name does not already exist, if attached */
      if (type->parent)
      {
	RecordType * temp;

	temp = get_record_type_by_name (type->parent->record_types, info->name);
	g_return_val_if_fail (temp == NULL, FALSE);
      }
#endif
      g_free (type->name);
      type->name = g_strdup (info->name);
    }
  }

  if (mask & RECORD_TYPE_FIELD_DESCRIPTION)
  {
    g_free (type->description);
    if (info->description)
      type->description = g_strdup (info->description);
    else
      type->description = NULL;
  }

  if (mask & RECORD_TYPE_FIELD_NUMBERED)
    type->numbered = info->numbered;
  
  if (mask & RECORD_TYPE_FIELD_LINKED)
    type->linked = info->linked;

  if (mask & RECORD_TYPE_FIELD_SIGN)
    type->sign = info->sign;

  return TRUE;
}

void
record_type_get_info (const RecordType *type, guint mask, RecordTypeInfo *info)
{
  trace ("%p", type);
  g_return_if_fail (type);
  g_return_if_fail (info);

  if (mask == 0)
    mask = (guint) -1;
  
  if (mask & RECORD_TYPE_FIELD_NAME)
    info->name = type->name;
  
  if (mask & RECORD_TYPE_FIELD_DESCRIPTION)
    info->description = type->description;
  
  if (mask & RECORD_TYPE_FIELD_USAGE_COUNT)
    info->usage_count = type->usage_count;

  if (mask & RECORD_TYPE_FIELD_NUMBERED)
    info->numbered = type->numbered;
  
  if (mask & RECORD_TYPE_FIELD_LINKED)
    info->linked = type->linked;
  
  if (mask & RECORD_TYPE_FIELD_SIGN)
    info->sign = type->sign;
}

gint
record_type_index (const RecordType * type)
{
  trace ("%p", type);
  g_return_val_if_fail (type, -1);
  g_return_val_if_fail (type->parent, -1);

  return g_list_index (type->parent->record_types, (gpointer) type);
}

void
record_type_dump (const RecordType * type)
{
  g_print ("  T - %p[rc=%u,uc=%u] {n=\"%s\",d=\"%s\",n=%d,l=%d,s=%s}\n",
           type,
	   type->ref_count,
	   type->usage_count,
	   type->name,
	   type->description,
	   type->numbered,
	   type->linked,
	   type->sign == 0 ? "ANY" : (type->sign == 1 ? "POS" : "NEG"));
}

// vim: ts=8 sw=2
