-------------------------------------------------------------------------------
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/contrib/gsql/project_file.adb,v $
--  Description     : project file handling                                  --
--  Author          : Michael Erdmann                                        --
--  Created         : 8.8.2001                                               --
--  Last Modified By: $Author: me $
--  Last Modified On: $Date: 2001/09/30 20:14:27 $
--  Status          : $State: Exp $
--
--  Copyright (C) 2000 Michael Erdmann                                       --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  Functional description                                                   --
--  ======================                                                   --
--  The project file contains defintio sections. A defintion section for     --
--  a attribute consists one line with the attribute name and a free number  --
--  of following lines containing the actual data of the attrbiute as for    --
--  example:                                                                 --
--                                                                           --
--  Path                                                                     --
--   2                                                                       --
--   /usr/bin                                                                --
--   /usr/local/bin                                                          --
--                                                                           --
--  Author                                                                   --
--  ======                                                                   --
--                                                                           --
--  Author: Michael Erdmann <michael.erdmann@snafu.de>                       --
--                                                                           --
--  GNADE is implemented to work with GNAT, the GNU Ada compiler.            --
--                                                                           --
-------------------------------------------------------------------------------
with Ada.Text_IO;                 use Ada.Text_IO;
with Ada.Characters.Handling;     use Ada.Characters.Handling;
with Ada.Strings;                 use Ada.Strings;
with Ada.Strings.Unbounded;       use Ada.Strings.Unbounded;
with Ada.Strings.Fixed;           use Ada.Strings.Fixed;
with Unchecked_Deallocation;
with Ada.Characters.Latin_1;      use Ada.Characters.Latin_1;

package body Project_File is

    Version        : constant String := "$Id: project_file.adb,v 1.6 2001/09/30 20:14:27 me Exp $";

    type Attribute_Record( N : Positive ) is record
          Class_Name : Unbounded_String := Null_Unbounded_String;
          Name       : Unbounded_String := Null_Unbounded_String;
          Value      : Text_Buffer( 1..N );
       end record;

    type Attribute is access Attribute_Record;

    procedure Free is
       new Unchecked_Deallocation( Attribute_Record, Attribute);

    Attribute_List : array( 1..Max_Catalog_Length ) of Attribute;
    Next_Free      : Positive := 1;
    Changed        : Boolean  := False;
    Next_Read      : Positive := 1;
    --- ******************************************************************* ---
    --- ***            L O C A L  S U P P O R T    PROCEDURES           *** ---
    --- ******************************************************************* ---

    ----------------
    -- Class_Name --
    ----------------
    function Class_Name(
       S : in String ) return Unbounded_String is
       -- fetch the class name from a line
    begin
       for I in S'Range loop
          if S(I) = ':' then
             return To_Unbounded_String( S(1..I-1) );
          end if;
       end loop;

       return To_Unbounded_String(S);
    end Class_Name;

    ----------------
    -- Entry_Name --
    ----------------
    function Entry_Name(
       S : in String ) return Unbounded_String is
    begin
       for I in S'Range loop
          if S(I) = ':' then
             return To_Unbounded_String( S(I+1..S'Last) );
          end if;
       end loop;
       return To_Unbounded_String(S);
    end Entry_Name;

    -----------
    -- Match --
    -----------
    function Match(
       Id   : in Positive;
       Cls  : in String;
       Name : in String ) return Boolean is
    begin
       return To_String( Attribute_List(Id).Name ) = Name and
              To_String( Attribute_List(Id).Class_Name ) = Cls;
    end;

    --- ******************************************************************* ---
    --- ***          HANDLE P R O J E C T    F I L E S                  *** ---
    --- ******************************************************************* ---

    ----------
    -- Open --
    ----------
    procedure Open(
       Path   : in String  := "";
       Quiet  : in Boolean := False ) is
       -- load a complete project file into the memory
       File   : File_Type;
       Line   : String (1 .. 1024);
       Last   : Natural;

       function Read_Paragraph(
          Name_Line    : in String ) return Attribute is
          Nbr_Of_Lines : Natural;
          Result       : Attribute;
          Name         : Unbounded_String := Entry_Name( Name_Line );
          Cls_Name     : Unbounded_String := Class_Name( Name_Line );
       begin
          Get_Line( File, Line, Last );
          Nbr_Of_Lines := Natural'Value(Line(2..Last));

          Result := new Attribute_Record( Nbr_Of_Lines );
          Result.Name       := Name ;
          Result.Class_Name := Cls_Name;

          for I in 1..Nbr_Of_Lines loop
             Get_Line( File, Line, Last );
             Result.Value(I) := To_Unbounded_String(Line(1..Last));
          end loop;

          return Result;
       end Read_Paragraph;

    begin
       if Path /= "" then
          Open( File, In_File, Path );
       else
          Open( File, In_File, To_String(File_Name) );
       end if;

       while not End_Of_File( File ) loop
          Get_Line( File, Line, Last );

          Attribute_List( Next_Free ) := Read_Paragraph( Trim( Line(1..Last), Right ) );
          Next_Free := Next_Free + 1;
       end loop;

       Close( File );

    exception
       when others =>
          if Quiet then
             return;
          else
             raise;
          end if;
    end Open;

    -----------
    -- Close --
    -----------
    procedure Close is
       -- close the file and release all resources allocated to the file
       procedure Free is
          new Unchecked_Deallocation( Attribute_Record, Attribute);
    begin
       for I in Attribute_List'Range loop
          if Attribute_List(I) /= null then
             Free( Attribute_List(I) );
             Attribute_List(I) := null;
          end if;
       end loop;

    end Close;

    ----------
    -- Save --
    ----------
    procedure Save(
       Path : in String := "" ) is
       -- save the project data into a file
       File : File_Type;
    begin
       if Path /= "" then
          Create( File => File, Mode => Out_File, Name => Path );
       else
          Create( File => File, Mode => Out_File, Name => To_String(File_Name) );
       end if;

       for I in Attribute_List'Range loop
          if Attribute_List(I) /= null then
             Put_Line(
                File,
                To_String( Attribute_List(I).Class_Name ) & ":" &
                To_String( Attribute_List(I).Name ) );
             Put_Line( File, Positive'Image( Attribute_List(I).Value'Length ) );

             for J in Attribute_List(I).Value'Range loop
                Put_Line( File, To_String( Attribute_List(I).Value(J)) );
             end loop;
          end if;

       end loop;

       Close( File );
    end Save;

    --- ******************************************************************* ---
    --- **         A C C E S S    FUNCTIONS FOR   S E C T I O N S       *** ---
    --- ******************************************************************* ---

    ----------
    -- Find --
    ----------
    function Find(
       Cls  : in String;
       Name : in String ) return Attribute is
    begin
       for I in Attribute_List'Range loop
          if Attribute_List(I) /= null and then
             Match( I, Cls, Name )
          then
             return Attribute_List(I);
          end if;
       end loop;

       raise Section_Not_Found;
    end Find;

    ---------
    -- Get --
    ---------
    procedure Get(
       Cls    : in  String;
       Name   : in  String;
       Result : out Text_Buffer;
       Size   : out Positive ) is
       -- get the contents of a certain section
       Comp   : Attribute;
    begin
       Comp := Find( Cls, Name );

       for J in Result'Range loop
          exit when not ( J in Comp.Value'Range);

          Size := J;
          Result(J) := Comp.Value(J);
       end loop;
    end Get;

    ---------
    -- Get --
    ---------
    function Get(
       Cls    : in String;
       Name   : in String ) return String is
       -- get the contents of a certain section
       Result : Unbounded_String := Null_Unbounded_String;
       Comp   : Attribute := Find( Cls, Name );
    begin
       for J in Comp.Value'Range loop
          Result := Result & Comp.Value(J);
       end loop;

       return To_String( Result );
    end Get;

    ---------
    -- Add --
    ---------
    procedure Add(
       Cls  : in String;
       Name : in String;
       Data : in Text_Buffer ) is
       -- add a section to the project file
       Item : Attribute := new Attribute_Record( Data'Length );
    begin
       Item.Name       := To_Unbounded_String( Name );
       Item.Class_Name := To_Unbounded_String( Cls );
       Item.Value      := Data;

       if Next_Free in Attribute_List'Range then
          Attribute_List(Next_Free) := Item;
          Next_Free := Next_Free + 1;
       else
          raise No_More_Components;
       end if;
    end Add;

    ---------
    -- Add --
    ---------
    procedure Add(
       Cls  : in String;
       Name : in String;
       Data : in String ) is
       -- add a section to the project file
       T    : Text_Buffer( 1..Max_Section_Length );
       Last : Positive := 1;
    begin
       To_Text_Buffer( Data, T, Last );
       Add( Cls, Name, T(1..Last) );
    end Add;

    ------------
    -- Update --
    ------------
    procedure Update(
       Cls  : in String;
       Name : in String;
       Data : in String ) is
       -- update a section with a single string. The string might
       -- contain LF.
    begin
       if Section_Exists( Cls, Name ) then
          Remove( Cls, Name );
       end if;
       Add( Cls, Name, Data );
    end Update;

    ------------
    -- Update --
    ------------
    procedure Update(
       Cls  : in String;
       Name : in String;
       Data : in Text_Buffer ) is
       -- update the section by means of a text buffer
    begin
       if Section_Exists( Cls, Name ) then
          Remove( Cls, Name );
       end if;
       Add( Cls, Name, Data );
    end Update;

    --------------------
    -- Section_Exists --
    --------------------
    function Section_Exists(
       Cls    : in String;
       Name   : in String ) return Boolean is
       -- return true if a section exists
       Result : Attribute;
    begin
       Result := Find( Cls, Name );
       return True;

    exception
       when Section_Not_Found =>
          return False;
    end Section_Exists;

    ------------
    -- Remove --
    ------------
    procedure Remove(
       Cls   : in String;
       Name  : in String;
       Quiet : in Boolean := True ) is
    begin
       for I in Attribute_List'Range loop
          if Attribute_List(I) /= null and then
             Match( I, Cls, Name )
          then
             Free(Attribute_List(I));
             Attribute_List(I) := null;
             return;
          end if;
       end loop;
       if not Quiet then
          raise Section_Not_Found;
       end if;
    end Remove;

    -------------
    -- Catalog --
    -------------
    procedure Catalog(
       Name   : in String;
       Cat    : out Text_Buffer ) is
       J      : Positive := 1;
    begin
       for I in Cat'Range loop
          Cat(I) := Null_Unbounded_String;
       end loop;

       for I in Attribute_List'Range loop
          exit when not ( J in Cat'Range);

          if Attribute_List(I) /= null and then
             To_String( Attribute_List(I).Class_Name ) = Name
          then
             Cat(J) := Attribute_List(I).Name;
             J := J + 1;
          end if;
       end loop;
    end Catalog;

end Project_File;
