/*--------------------------------------------------------------------
 *	$Id: grdreformat.c,v 1.27 2006/04/13 06:20:35 pwessel Exp $
 *
 *	Copyright (c) 1991-2006 by P. Wessel and W. H. F. Smith
 *	See COPYING file for copying and redistribution conditions.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; version 2 of the License.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	Contact info: gmt.soest.hawaii.edu
 *--------------------------------------------------------------------*/
/*
 * grdreformat.c reads a grd file in one format and outputs it in another
 *
 * Author:	Paul Wessel
 * Date:	3-JAN-1991-2006
 * Version:	4
 */

#include "gmt.h"

struct GRDREFORMAT_CTRL {
	struct N {	/* -N */
		BOOLEAN active;
	} N;
};

int main (int argc, char **argv)
{
	BOOLEAN error = FALSE, global = FALSE;

	int i, nfiles = 0, nx, ny, no_header, type[2];

	size_t nm;
	
	double w, e, s, n;

	float *z;

	char *file[2], fname[2][BUFSIZ], line[BUFSIZ];

	struct GRD_HEADER grd;
	struct GRDREFORMAT_CTRL *Ctrl;

	FILE *fp;

	void *New_Grdreformat_Ctrl (), Free_Grdreformat_Ctrl (struct GRDREFORMAT_CTRL *C);

	argc = GMT_begin (argc, argv);

	Ctrl = (struct GRDREFORMAT_CTRL *)New_Grdreformat_Ctrl ();	/* Allocate and initialize a new control structure */
	
	file[0] = file[1] = CNULL;
	w = e = s = n = 0.0;

	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
				/* Common parameters */

				case 'R':
				case 'V':
				case '\0':
					error += GMT_parse_common_options (argv[i], &w, &e, &s, &n);
					break;

				case 'N':
					Ctrl->N.active = TRUE;
					break;
				default:
					error = TRUE;
					GMT_default_error (argv[i][1]);
					break;
			}
		}
		else if (nfiles < 2) {
			file[nfiles] = argv[i];
			nfiles++;
		}
		else
			nfiles++;
	}

	if (argc == 1 || GMT_give_synopsis_and_exit) {	/* Display usage */
		fprintf (stderr, "grdreformat %s - Converting between different grdfile formats\n\n", GMT_VERSION);
		fprintf( stderr, "usage: grdreformat ingrdfile[=id[/scale/offset]] outgrdfile[=id[/scale/offset]] [-N]\n\t[%s] [-V]\n", GMT_Rgeo_OPT);

		if (GMT_give_synopsis_and_exit) exit (EXIT_FAILURE);

		fprintf (stderr, "	ingrdfile is the grd file to convert\n");
		fprintf (stderr, "	outgrdfile is the new converted grd file\n");
		fprintf( stderr, "	   scale and offset, if given, will multiply data by scale and add offset.\n");
		fprintf (stderr, "\n\tOPTIONS:\n");
		fprintf (stderr, "\t-N  Do NOT write the header (native grids only - ignored otherwise).\n");
		fprintf (stderr, "\t\t  Useful when creating files to be used by grdraster.\n");
		GMT_explain_option ('r');
		GMT_explain_option ('V');

		fprintf (stderr, "\n	The following grdfile formats are supported:\n\n");
		sprintf (line, "%s%cshare%cgmtformats.d", GMTHOME, DIR_DELIM, DIR_DELIM);
		if ((fp = fopen (line, "r")) == NULL) {
			fprintf (stderr, "%s: GMT ERROR: Cannot read file %s\n", GMT_program, line);
			exit (EXIT_FAILURE);
		}
		while (fgets (line, BUFSIZ, fp)) {
			if (line[0] == '#') continue;
			fprintf (stderr, "\t%s", &line[4]);
		}
		fclose (fp);
		exit (EXIT_FAILURE);
	}

	if (nfiles != 2) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR: Must specify both input and output file names\n", GMT_program);
		error++;
	}

	if (error) exit (EXIT_FAILURE);

	GMT_put_history (argc, argv);	/* Update .gmtcommands4 */

	no_header = (Ctrl->N.active) ? 64 : 0;
	type[0] = GMT_grd_get_format (file[0], &grd);
	strcpy (fname[0], grd.name);
	type[1] = GMT_grd_get_format (file[1], &grd);
	strcpy (fname[1], grd.name);

	if (type[1] == 20) {	/* Golden Surfer format 7 (double) is read-only */
		fprintf (stderr, "%s: Grid format sd (Golden Software Surfer foramt 7 (double) is read-only!\n", GMT_program);
		exit (EXIT_FAILURE);
	}
	
	if (gmtdefs.verbose) {
		if (file[0][0] == '=') strcpy (fname[0], "<stdin>");
		if (file[1][0] == '=') strcpy (fname[1], "<stdout>");
		fprintf (stderr, "%s: Translating file %s (format = %d) to file %s (format = %d)\n", GMT_program, fname[0], type[0], fname[1], type[1]);
		if (no_header && GMT_grdformats[type[1]][0] != 'c' && GMT_grdformats[type[1]][0] != 'n') fprintf (stderr, "%s: No grd header will be written\n", GMT_program);
	}

	if (GMT_read_grd_info (file[0], &grd)) {
		fprintf (stderr, "%s: Error opening file %s\n", GMT_program, fname[0]);
		exit (EXIT_FAILURE);
	}

	if (e > w && n > s) {
		global = (fabs (grd.x_max - grd.x_min) == 360.0);
		if (!global && (w < grd.x_min || e > grd.x_max)) error = TRUE;
		if (s < grd.y_min || n > grd.y_max) error = TRUE;
		if (error) {
			fprintf (stderr, "%s: Subset exceeds data domain!\n", GMT_program);
			exit (EXIT_FAILURE);
		}
		nx = GMT_get_n (w, e, grd.x_inc, grd.node_offset);
		ny = GMT_get_n (s, n, grd.y_inc, grd.node_offset);
		nm = ((size_t)nx) * ((size_t)ny);

		z = (float *) GMT_memory (VNULL, nm, sizeof (float), GMT_program);

		if (GMT_read_grd (file[0], &grd, z, w, e, s, n, GMT_pad, FALSE)) {
			fprintf (stderr, "%s: Error reading file %s\n", GMT_program, fname[0]);
			exit (EXIT_FAILURE);
		}
	}
	else {
		nm = ((size_t)grd.nx) * ((size_t)grd.ny);
		z = (float *) GMT_memory (VNULL, nm, sizeof (float), GMT_program);

		if (GMT_read_grd (file[0], &grd, z, 0.0, 0.0, 0.0, 0.0, GMT_pad, FALSE)) {
			fprintf (stderr, "%s: Error reading file %s\n", GMT_program, fname[0]);
			exit (EXIT_FAILURE);
		}
	}

	grd.type = type[1];

	GMT_grd_init (&grd, argc, argv, TRUE);

	if (GMT_write_grd (file[1], &grd, z, 0.0, 0.0, 0.0, 0.0, GMT_pad, no_header)) {
		fprintf (stderr, "%s: Error writing file %s\n", GMT_program, fname[1]);
		exit (EXIT_FAILURE);
	}

	GMT_free ((void *)z);

	Free_Grdreformat_Ctrl (Ctrl);	/* Deallocate control structure */

	GMT_end (argc, argv);

	exit (EXIT_SUCCESS);
}

void *New_Grdreformat_Ctrl () {	/* Allocate and initialize a new control structure */
	struct GRDREFORMAT_CTRL *C;
	
	C = (struct GRDREFORMAT_CTRL *) GMT_memory (VNULL, 1, sizeof (struct GRDREFORMAT_CTRL), "New_Grdreformat_Ctrl");
	
	/* Initialize values whose defaults are not 0/FALSE/NULL */
	
	return ((void *)C);
}

void Free_Grdreformat_Ctrl (struct GRDREFORMAT_CTRL *C) {	/* Deallocate control structure */
	GMT_free ((void *)C);	
}
