/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <globus_io.h>
#include "globus_rls_client.h"
#include "globus_rls_rpc.h"
#include "misc.h"
#include "db.h"
#include "event.h"
#include "lock.h"
#include "bloom.h"
#include "rli.h"
#include "update.h"
#include <syslog.h>
#include <sys/types.h>
#include <dirent.h>
#include <string.h>
#define __USE_XOPEN	/* Stops compiler warning on linux	*/
#include <time.h>

extern int	errno;
extern int	loglevel;
extern char	*myurl;
extern int	update_buftime;
extern int	update_bf_int;
extern int	update_ll_int;
extern int	update_immediate;
extern int	update_factor;
extern char	*rli_bloomfilter_dir;
extern int	quitting;
extern int	rli_bloomfilter;
extern int	rli_expire_int;
extern int	rli_expire_stale;
extern char	*rli_dbname;
extern char	*db_user;
extern char	*db_pwd;

SENDER		*senderlist;
LOCK		senderlistlock;
RLI		*rli_rlilist;
LOCK		rli_rlilistlock;
static int	senderlistcb(void *a, char *url, char *lastupdate);
static void	*expire(void *a);
static SENDER	*newsender(char *senderurl, char *lrcurl);
static void	freesender(SENDER *lrc);
static int	readsender(char *dbname, char *db_user, char *db_pwd);
static void	readbf();
static void	freesenderlist();
static void	*rli_bfupdates(void *a);
static void	*rli_bfiupdates(void *a);
static void	*rli_llupdates(void *a);
static int	rli_update_ll_cb(void *a, char *lfn, char *lrc);

int
rli_init(char *dbname, char *db_user, char *db_pwd)

{
  int	rc;
  void	*rlih;
  char	errbuf[BUFSIZ];

  if (loglevel > 1)
    logit(LOG_DEBUG, "rli_init: %s %s", dbname, db_user);

  if ((rc = rls_lock_init(&senderlistlock)) != GLOBUS_RLS_SUCCESS)
    return rc;
  senderlist = NULL;
  if (rli_bloomfilter) {
    if (rli_bloomfilter_dir)
      readbf(&senderlist);
  } else
    if ((rc = readsender(dbname, db_user, db_pwd)) != GLOBUS_RLS_SUCCESS)
      return rc;
  if ((rc = rls_lock_init(&rli_rlilistlock)) != GLOBUS_RLS_SUCCESS)
    return rc;
  rli_rlilist = NULL;

  /*
   * Get list of RLIs we update from db.
   */
  if (db_open(dbname, db_user, db_pwd, 0, &rlih, errbuf) != GLOBUS_RLS_SUCCESS)
    logit(LOG_WARNING, "rli_init(%s): %s", dbname, errbuf);
  else {
    update_readrli(rlih, &rli_rlilistlock, &rli_rlilist);
    db_close(rlih);
  }

  /*
   * Queue periodic RLI events (expire, updates).
   */
  if (rli_expire_int > 0)
    event_queue(expire, &rli_expire_int);
  event_queue(rli_llupdates, &update_ll_int);
  event_queue(rli_bfupdates, &update_bf_int);

  if (loglevel > 1)
    logit(LOG_DEBUG, "rli_init succeeded");
  return GLOBUS_RLS_SUCCESS;
}

/*
 * Clean up when exiting or if no longer an RLI server.
 */
void
rli_end()

{
  if (loglevel > 1)
    logit(LOG_INFO, "rli_end:");
  event_cancel(expire);
  if (event_exists(rli_llupdates))
    event_cancel(rli_llupdates);
  if (event_exists(rli_bfupdates))
    event_cancel(rli_bfupdates);
  if (event_exists(rli_bfiupdates))
    event_cancel(rli_bfiupdates);

  if (senderlist) {
    rls_lock_get(&senderlistlock, writelock);
    freesenderlist();
    rls_lock_release(&senderlistlock, writelock);
  }
  rls_lock_destroy(&senderlistlock);

  if (rli_rlilist) {
    rls_lock_get(&rli_rlilistlock, writelock);
    update_freelist(&rli_rlilist);
    rls_lock_release(&rli_rlilistlock, writelock);
  }
  rls_lock_destroy(&rli_rlilistlock);
}

/*
 * Find sender by url in senderlist, insert if doesn't exist.
 */
SENDER *
rli_findsender(char *senderurl, char *lrcurl)

{
  SENDER	*sender;

  rls_lock_get(&senderlistlock, readlock);
  for (sender = senderlist; sender; sender = sender->nxt)
    if (strcasecmp(senderurl, sender->senderurl) == 0 &&
	strcasecmp(lrcurl, sender->lrcurl) == 0)
      break;
  rls_lock_release(&senderlistlock, readlock);
  if (sender)
    return sender;

  if ((sender = newsender(senderurl, lrcurl)) == NULL) {
    logit(LOG_WARNING, "rli_findsender(%s): No memory for %s", senderurl);
    return NULL;
  }
  rls_lock_get(&senderlistlock, writelock);
  sender->nxt = senderlist;
  senderlist = sender;
  rls_lock_release(&senderlistlock, writelock);
  return sender;
}

/*
 * Requeue update events to run immediately.
 */
void
rli_updatenow()

{
  event_now(rli_llupdates);
  event_now(rli_bfupdates);
}

/*
 * Callback to add sender to senderlist.  senderlistlock should already be
 * write locked.
 */
static int
senderlistcb(void *a, char *url, char *lastupdate)

{
  SENDER	*sender;
  struct tm	t;

  if ((sender = newsender(url, url)) == NULL) {
    logit(LOG_WARNING, "senderlistcb: %s: No memory", url);
    return GLOBUS_FALSE;
  }
  strptime(lastupdate, "%Y-%m-%d %H:%M:%S", &t);
  t.tm_isdst = -1;
  sender->lastupdate = mktime(&t);
  sender->nxt = senderlist;
  senderlist = sender;
  return GLOBUS_TRUE;
}

/*
 * Expiring old rli entries depends on whether we're updated with bloom
 * filters or not.  With bloom filters we just look at the time of the
 * last received filter, and discard it if its too old.  Without them
 * we must scan the database and remove old lfn,lrc mappings.
 * In either case we try to use the update interval that the sender is
 * sending updates at (plus 20%) as the stale time.  In the case of
 * bloom filters the update interval is sent with each update.  In
 * the case of lfns the list of lrcs are read from the database,
 * and as they're read the lrc servers are queried to get the
 * current update interval.
 */
static void *
expire(void *a)

{
  EVENTCONTROL		*ec = (EVENTCONTROL *) a;
  void			*rlih = NULL;
  SENDER		*sender;
  char			errbuf[MAXERRMSG];
  int			si;
  time_t		now;
  globus_rls_handle_t	*h;
  globus_rls_rli_info_t	info;
  globus_result_t	r;

  if (loglevel)
    logit(LOG_DEBUG, "expire: starting");

  while (1) {
    globus_mutex_lock(&ec->mtx);
    while (!(ec->flags & (EF_EXIT|EF_RUN)))
      globus_cond_wait(&ec->cond, &ec->mtx);
    globus_mutex_unlock(&ec->mtx);

    if (loglevel)
      logit(LOG_DEBUG, "expire: running:%s", rli_bloomfilter ? "" :
					     rlih ? " db open" : " db closed");

    globus_mutex_lock(&ec->mtx);
    if (ec->flags & EF_EXIT) {
      ec->flags &= ~EF_RUNNING;
      globus_cond_signal(&ec->cond);
      globus_mutex_unlock(&ec->mtx);
      if (!rli_bloomfilter)
	if (rlih)
	  db_close(rlih);
      pthread_exit(0);
    } else {
      ec->flags &= ~EF_RUN;
      globus_mutex_unlock(&ec->mtx);
    }

    if (rli_bloomfilter)
      now = time(0);

    rls_lock_get(&senderlistlock, readlock);
    for (sender = senderlist; sender && !quitting; sender = sender->nxt) {
      if (!rli_bloomfilter) {
	/* Bloom filters set updateint during updates	*/
	sender->updateinterval = 0;
	if ((r = globus_rls_client_connect(sender->lrcurl,
					   &h)) != GLOBUS_SUCCESS) {
	  globus_rls_client_error_info(r, NULL, errbuf,MAXERRMSG,GLOBUS_FALSE);
	  logit(LOG_WARNING, "expire(%s): %s", sender->lrcurl, errbuf);
	} else {
	  if ((r = globus_rls_client_lrc_rli_info(h, myurl,
					&info)) != GLOBUS_SUCCESS) {
	    globus_rls_client_error_info(r,NULL,errbuf,MAXERRMSG,GLOBUS_FALSE);
	    logit(LOG_WARNING, "expire(%s): %s", sender->lrcurl, errbuf);
	  } else
	    sender->updateinterval = info.updateinterval;
	  globus_rls_client_close(h);
	}
      }
      if (sender->updateinterval)
	si = (int) (sender->updateinterval * 1.2);
      else
	si = rli_expire_stale;
      if (rli_bloomfilter) {
	if (now - sender->lastupdate > si) {
	  bf_free(&sender->bf);
	  if (rli_bloomfilter_dir)
	    bf_unsave(sender->lrcurl);
	}
      } else {
	if (!rlih) {
	  if (db_open(rli_dbname, db_user, db_pwd, 0, &rlih,
		      errbuf) != GLOBUS_RLS_SUCCESS){
	    logit(LOG_WARNING, "expire(%s): %s", rli_dbname, errbuf);
	    rlih = NULL;
	  }
	}
	if (rlih)
	  if (db_rli_expire(rlih, sender->lrcurl, si,
			    errbuf) != GLOBUS_RLS_SUCCESS)
	    logit(LOG_WARNING, "expire(%s): %s", sender->lrcurl, errbuf);
      }
    }
    rls_lock_release(&senderlistlock, readlock);
  }
}

static SENDER *
newsender(char *senderurl, char *lrcurl)

{
  SENDER	*sender;

  if ((sender = globus_libc_malloc(sizeof(SENDER))) == NULL)
    return NULL;
  if ((sender->senderurl = globus_libc_strdup(senderurl)) == NULL) {
    globus_libc_free(sender);
    return NULL;
  }
  if ((sender->lrcurl = globus_libc_strdup(lrcurl)) == NULL) {
    globus_libc_free(sender->senderurl);
    globus_libc_free(sender);
    return NULL;
  }
  sender->updateinterval = 0;
  bf_init(&sender->bf, 0, 0, 0);
  globus_mutex_init(&sender->bf.mtx, GLOBUS_NULL);
  sender->lastupdate = 0;
  return sender;
}

static void
freesender(SENDER *sender)

{
  globus_libc_free(sender->senderurl);
  globus_libc_free(sender->lrcurl);
  if (sender->bf.bfsize)
    bf_free(&sender->bf);
  globus_mutex_destroy(&sender->bf.mtx);
  globus_libc_free(sender);
}

static int
readsender(char *dbname, char *db_user, char *db_pwd)

{
  int	rc;
  void	*rlih;
  char	errbuf[MAXERRMSG];
  char	buf[MAXERRMSG];

  if ((rc = db_open(dbname, db_user, db_pwd, 0, &rlih,
		    errbuf)) != GLOBUS_RLS_SUCCESS) {
    logit(LOG_WARNING, "readsender(%s): %s", dbname, errbuf);
    return rc;
  }
  rls_lock_get(&senderlistlock, writelock);
  freesenderlist();
  if ((rc = db_rli_sender_list(rlih, senderlistcb, NULL,
			       errbuf)) != GLOBUS_SUCCESS)
    if (loglevel)
      logit(LOG_WARNING, "readsender: %s",
	    globus_rls_errmsg(rc, errbuf, buf, MAXERRMSG));
  rls_lock_release(&senderlistlock, writelock);
  db_close(rlih);
  return GLOBUS_RLS_SUCCESS;
}

/*
 * Read saved bloomfilters from disk.
 */
void
readbf()

{
  DIR		*d;
  struct dirent	*de;
  int		len;
  int		i;
  int		fd;
  char		lrc_url[BUFLEN];
  int		bfsize;
  int		numhash;
  SENDER	*sender;
  char		errbuf[MAXERRMSG];
  int		bytes;
  char		fn[BUFLEN];
  int		rc;

  if ((d = opendir(rli_bloomfilter_dir)) == NULL) {
    logit(LOG_WARNING, "readbf(%s): %s", rli_bloomfilter_dir,
	  strerror(errno));
    return;
  }
  rls_lock_get(&senderlistlock, writelock);
  while ((de = readdir(d))) {
    len = strlen(de->d_name);
    if (strcmp(&de->d_name[len-3], ".bf") != 0)
      continue;
    snprintf(fn, BUFLEN, "%s/%s", rli_bloomfilter_dir, de->d_name);
    for (i = 0; i < len - 3; i++)
      if (de->d_name[i] == '%')
	lrc_url[i] = '/';
      else
	lrc_url[i] = de->d_name[i];
    lrc_url[i] = '\0';
    if (loglevel > 1)
      logit(LOG_DEBUG, "readbf: Reading %s", lrc_url);

    if ((fd = open(fn, O_RDONLY)) == -1) {
      logit(LOG_WARNING, "readbf(%s): %s", fn, strerror(errno));
      continue;
    }
    if (read(fd, &bfsize, sizeof(bfsize)) != sizeof(bfsize)) {
      logit(LOG_WARNING, "readbf(%s): %s", fn, strerror(errno));
      goto next;
    }
    if (read(fd, &numhash, sizeof(numhash)) != sizeof(numhash)) {
      logit(LOG_WARNING, "readbf(%s): %s", fn, strerror(errno));
      goto next;
    }
    if ((sender = newsender(lrc_url, lrc_url)) == NULL) {
      logit(LOG_WARNING, "readbf: No memory for %s", lrc_url);
      goto next;
    }
    if ((rc = bf_init(&sender->bf,bfsize,numhash,0)) != GLOBUS_RLS_SUCCESS) {
      logit(LOG_WARNING, "readbf(%s): %s", fn,
	    globus_rls_errmsg(rc, NULL, errbuf, MAXERRMSG));
      freesender(sender);
      goto next;
    }
    bytes = BITS2BYTES(bfsize);
    if (read(fd, sender->bf.bits, bytes) != bytes) {
      logit(LOG_WARNING, "readbf(%s): %s", fn, strerror(errno));
      freesender(sender);
      goto next;
    }
    sender->lastupdate = time(0);
    sender->nxt = senderlist;
    senderlist = sender;

   next:
    close(fd);
  }
  closedir(d);
  rls_lock_release(&senderlistlock, writelock);
}

/*
 * Free senderlist.  senderlistlock should already be write locked.
 */
static void
freesenderlist()

{
  SENDER	*sender;

  if (loglevel > 1)
    logit(LOG_DEBUG, "freesenderlist:");
  while ((sender = senderlist)) {
    senderlist = sender->nxt;
    freesender(sender);
  }
}

static void *
rli_bfupdates(void *a)

{
  EVENTCONTROL	*ec = (EVENTCONTROL *) a;
  RLI		*rli;
  SENDER	*sender;
  int		ui;

  while (1) {
    if (loglevel)
      logit(LOG_DEBUG, "rli_bfupdates:");

    rls_lock_get(&rli_rlilistlock, readlock);
    for (rli = rli_rlilist; rli; rli = rli->nxt) {
      if (!(rli->flags & FR_BLOOMFILTER))
	continue;
      /*
       * Softstatestart and lastupdate will be the same if
       * last update succeeded, hence we update only every
       * update_factor multiples of update_bf_int.
       */
      if (update_immediate && rli->softstatestart == rli->lastupdate)
	ui = update_bf_int * update_factor;
      else
	ui = update_bf_int;
      if (rli->softstatestart && time(0) - rli->lastupdate < ui)
	continue;
      rls_lock_get(&senderlistlock, readlock);
      for (sender = senderlist; sender; sender = sender->nxt)
	update_sendbf(rli, &sender->bf, sender->lrcurl);
      rls_lock_release(&senderlistlock, readlock);
    }
    rls_lock_release(&rli_rlilistlock, readlock);

    rls_lock_get(&senderlistlock, readlock);
    for (sender = senderlist; sender; sender = sender->nxt) {
      globus_mutex_lock(&sender->bf.mtx);
      sender->bf.flags &= ~BF_NEEDUPDATE;
      globus_mutex_unlock(&sender->bf.mtx);
    }
    rls_lock_release(&senderlistlock, readlock);

    globus_mutex_lock(&ec->mtx);
    while (!(ec->flags & (EF_EXIT|EF_RUN)))
      globus_cond_wait(&ec->cond, &ec->mtx);

    if (ec->flags & EF_EXIT) {
      ec->flags &= ~EF_RUNNING;
      globus_cond_signal(&ec->cond);
      globus_mutex_unlock(&ec->mtx);
      pthread_exit(0);
    } else {
      ec->flags &= ~EF_RUN;
      globus_mutex_unlock(&ec->mtx);
    }

    /*
     * Immediate mode may have been enabled since we started, queue
     * immediate bloomfilter updates if it's not running.
     */
    if (update_immediate && !event_exists(rli_bfiupdates))
      event_queue(rli_bfiupdates, &update_buftime);
  }
}

/*
 * Client (RLI) side bloom filter update.  Called every update_buftime seconds
 * when immediate mode is enabled to check if the bloom filter has been
 * modified, if it is it is sent to RLIs.  Note this differs from immediate
 * mode when using LFN lists, in that case only the changed LFNs are
 * sent, here we send entire bloom filter.
 */
static void *
rli_bfiupdates(void *a)

{
  EVENTCONTROL	*ec = (EVENTCONTROL *) a;
  SENDER	*sender;
  RLI		*rli;

  while (1) {
    if (loglevel)
      logit(LOG_DEBUG, "rli_bfiupdates:");

    rls_lock_get(&senderlistlock, readlock);
    for (sender = senderlist; sender; sender = sender->nxt)
      if (sender->bf.flags & BF_NEEDUPDATE) {
	rls_lock_get(&rli_rlilistlock, readlock);
	for (rli = rli_rlilist; rli; rli = rli->nxt)
	  if (rli->flags & FR_BLOOMFILTER)
	    update_sendbf(rli, &sender->bf, sender->lrcurl);
	rls_lock_release(&rli_rlilistlock, readlock);

	globus_mutex_lock(&sender->bf.mtx);
	sender->bf.flags &= ~BF_NEEDUPDATE;
	globus_mutex_unlock(&sender->bf.mtx);
      }
    rls_lock_release(&senderlistlock, readlock);

    globus_mutex_lock(&ec->mtx);
    if (!update_immediate)
      ec->flags |= EF_EXIT;
    while (!(ec->flags & (EF_EXIT|EF_RUN)))
      globus_cond_wait(&ec->cond, &ec->mtx);

    if (ec->flags & EF_EXIT) {
      ec->flags &= ~EF_RUNNING;
      globus_cond_signal(&ec->cond);
      globus_mutex_unlock(&ec->mtx);
      pthread_exit(0);
    } else {
      ec->flags &= ~EF_RUN;
      globus_mutex_unlock(&ec->mtx);
    }
  }
}

/*
 * Read entire lfn table and queue updates to appropriate RLI servers.
 * An RLI is sent the softstate update if we're not in immediate mode,
 * or we're out of sync with the RLI server (cause a previous update failed),
 * or it's been update * update_factor seconds since the last
 * full update.
 */
static void *
rli_llupdates(void *a)

{
  EVENTCONTROL	*ec = (EVENTCONTROL *) a;
  int		rc;
  void		*rlih = NULL;
  char		errbuf[MAXERRMSG];
  char		errbuf2[MAXERRMSG];
  RLI		*rli;
  time_t	now;
  int		unsynced;
  int		ui;

  while (1) {
    if (loglevel)
      logit(LOG_DEBUG, "rli_llupdates: db %s", rlih ? "open" : "closed");

    if (!rlih) {
      if ((rc = db_open(rli_dbname, db_user, db_pwd, 0, &rlih,
			errbuf)) != GLOBUS_RLS_SUCCESS) {
	logit(LOG_WARNING, "rli_llupdates(%s): %s", rli_dbname, errbuf);
	rlih = NULL;
	goto done1;
      }
    }

    rls_lock_get(&rli_rlilistlock, readlock);

    /*
     * Decide which RLIs will get updated now.
     */
    now = time(0);
    unsynced = 0;
    for (rli = rli_rlilist; rli; rli = rli->nxt) {
      if (rli->flags & FR_BLOOMFILTER)
	continue;
      globus_mutex_lock(&rli->mtx);
      if (update_immediate && (rli->flags & FR_SYNCED))
	ui = update_ll_int * update_factor;
      else
	ui = update_ll_int;
      if (now - rli->softstatestart >= ui) {
	rli->softstatestart = now;
	/*
	 * Since we're going to do a full update now flush queue and any
	 * pending output.  Avoids memory leak if RLI is down.
	 */
	rli->flags |= (FR_DOUPDATE|FR_SYNCED|FR_FLUSHPENDING);
	update_flushqueue(rli);
	unsynced++;
      } else
	rli->flags &= ~FR_DOUPDATE;
      globus_mutex_unlock(&rli->mtx);
    }
    if (!unsynced) {
      if (loglevel)
	logit(LOG_WARNING, "rli_llupdates: No unsynced RLI servers");
      goto done;
    }

    if ((rc = db_rli_alllfn(rlih, rli_update_ll_cb, rliop_add,
			    errbuf)) != GLOBUS_RLS_SUCCESS) {
      logit(LOG_WARNING, "rli_llupdates: alllfn: %s",
	    globus_rls_errmsg(rc, errbuf, errbuf2, MAXERRMSG));
      goto done;
    }

    /*
     * Queue magic record so queue processor knows when a softstate update
     * is finished and can log the time it took.
     */
    for (rli = rli_rlilist; rli; rli = rli->nxt)
      if (rli->flags & FR_DOUPDATE)
	update_queue(rli, RU_SOFTSTATEEND);

   done:
    rls_lock_release(&rli_rlilistlock, readlock);

   done1:
    globus_mutex_lock(&ec->mtx);
    while (!(ec->flags & (EF_EXIT|EF_RUN)))
      globus_cond_wait(&ec->cond, &ec->mtx);

    if (ec->flags & EF_EXIT) {
      ec->flags &= ~EF_RUNNING;
      globus_cond_signal(&ec->cond);
      globus_mutex_unlock(&ec->mtx);
      if (rlih)
	db_close(rlih);
      pthread_exit(0);
    } else {
      ec->flags &= ~EF_RUN;
      globus_mutex_unlock(&ec->mtx);
    }
  }
}

static int
rli_update_ll_cb(void *a, char *lfn, char *lrc)

{
  UPDATE	*u;
  RLI		*rli;

  if (loglevel > 4)
    logit(LOG_DEBUG, "rli_update_ll_cb: %s %s", lfn, lrc);

  /* Delay allocating UPDATE until we know we need to send to an RLI	*/
  u = NULL;

  for (rli = rli_rlilist; rli; rli = rli->nxt) {
    if (!(rli->flags & FR_DOUPDATE))
      continue;
    if (update_patternmatch(rli, lfn)) {
      if (!u)		/* Allocate UPDATE if haven't already	*/
	if ((u = update_new(lfn, lrc, (RLIOP) a)) == NULL)
	  return GLOBUS_FALSE;
      update_queue(rli, u);
    }
  }
  return GLOBUS_TRUE;
}
