/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Code to update bloom filter.  Note this code is thread unsafe, callers
 * must protect from simultaneous updates.  LRC server code only
 * makes updates when rlilist.mtx is write locked.
 */

#include <globus_config.h>
#include <globus_io.h>
#include <globus_error_string.h>
#include "globus_rls_client.h"
#include "misc.h"
#include "db.h"
#include "bloom.h"
#include <sys/types.h>
#include <syslog.h>
#include <fcntl.h>
#include <string.h>

static void		bfclearbit(bloomfilter_t *bf, u_int bit);
static void		bfsetbit(bloomfilter_t *bf, u_int bit);
static int		bfhash(char *s, int shift);
static void		url2fn(char *url, char *fn);

extern int		errno;
extern int		loglevel;
extern char		*rli_bloomfilter_dir;

/*
 * Up to 8 bfhash values are supported for each lfn.
 * For each byte of the lfn the bfhash function saves the high order "shift"
 * bits, shifts the value left by "shift" bits, and xors in the byte and high
 * bits.  Seem to get a better distribution with shifts of 3, 5 and 7 bits
 * so use those shifts first.  Following array is the amount of shifting
 * used by each of the bfhashes.
 */
static int		shifts[] = { 3, 5, 7, 1, 2, 4, 6, 8 };

/*
 * Hash lfn and set bits in bloom filter.  bf->mtx should be locked already.
 */
void
bf_addlfn(bloomfilter_t *bf, char *lfn)

{
  int	i;
  u_int	h;

  for (i = 0; i < bf->numhash; i++) {
    h = bfhash(lfn, shifts[i]);
    bfsetbit(bf, h);
  }
  bf->flags |= BF_NEEDUPDATE;
}

/*
 * Hash lfn and clear bits in bloom filter if count is 0.
 * bf->mtx should be locked already.
 */
void
bf_deletelfn(bloomfilter_t *bf, char *lfn)

{
  int	i;
  u_int	h;

  for (i = 0; i < bf->numhash; i++) {
    h = bfhash(lfn, shifts[i]);
    bfclearbit(bf, h);
  }
  bf->flags |= BF_NEEDUPDATE;
}

void
bf_free(bloomfilter_t *bf)

{
  if (bf->bfsize) {
    globus_libc_free(bf->bits);
    if (bf->count)
      globus_libc_free(bf->count);
    bf->bfsize = 0;
  }
}

int
bf_init(bloomfilter_t *bf, int bfsize, int numhash, int countbits)

{
  int	bytes;

  if (bfsize) {
    bytes = BITS2BYTES(bfsize);
    if ((bf->bits = globus_libc_calloc(bytes, 1)) == NULL)
      return GLOBUS_RLS_NOMEMORY;
    if (countbits) {
      if ((bf->count = globus_libc_calloc(bfsize, sizeof(short))) == NULL) {
	globus_libc_free(bf->bits);
	return GLOBUS_RLS_NOMEMORY;
      }
    } else
      bf->count = NULL;
  }
  bf->bfsize = bfsize;
  bf->numhash = numhash;
  bf->flags = 0;
  return GLOBUS_RLS_SUCCESS;
}

/*
 * Write bloom filter to disk so can be read at startup time.
 */
void
bf_save(char *lrc_url, bloomfilter_t *bf)

{
  char	fn[BUFLEN];
  int	fd;

  url2fn(lrc_url, fn);
  if (loglevel > 1)
    logit(LOG_DEBUG, "bf_save: Saving bloom filter: %s", fn);
  if ((fd = open(fn, O_WRONLY|O_CREAT|O_TRUNC, 0644)) == -1) {
    logit(LOG_WARNING, "bf_save(%s): %s: %s", lrc_url, fn, strerror(errno));
    return;
  }
  write(fd, &bf->bfsize, sizeof(bf->bfsize));
  write(fd, &bf->numhash, sizeof(bf->numhash));
  write(fd, bf->bits, BITS2BYTES(bf->bfsize));
  close(fd);
}

/*
 * Delete saved bloom filter (entries have expired).
 */
void
bf_unsave(char *lrc_url)

{
  char	fn[BUFLEN];

  url2fn(lrc_url, fn);
  if (loglevel > 1)
    logit(LOG_DEBUG, "bf_unsave: Removing bloom filter: %s", fn);
  unlink(fn);
}

int
bf_testlfn(bloomfilter_t *bf, char *lfn)

{
  int	i;
  u_int	h;
  u_int	m;
  u_int	idx;
  u_int	offset;

  if (!bf->bfsize)
    return 0;
  for (i = 0; i < bf->numhash; i++) {
    h = bfhash(lfn, shifts[i]);
    m = h % bf->bfsize;
    idx = m / 8;
    offset = m % 8;
    if (!(bf->bits[idx] & (1 << (7 - offset))))
      return 0;
  }
  return 1;
}

static void
bfclearbit(bloomfilter_t *bf, u_int bit)

{
  u_int	m;
  u_int	idx;
  u_int	offset;

  m = bit % bf->bfsize;
  if (--bf->count[m] == 0) {
    idx = m / 8;
    offset = m % 8;
    bf->bits[idx] &= ~(1 << (7 - offset));
  }
}

static void
bfsetbit(bloomfilter_t *bf, u_int bit)

{
  u_int	m;
  u_int	idx;
  u_int	offset;

  m = bit % bf->bfsize;
  if (bf->count[m]++ == 0) {
    idx = m / 8;
    offset = m % 8;
    bf->bits[idx] |= 1 << (7 - offset);
  }
}

static int
bfhash(char *s, int shift)

{
  u_int h = 0;
  u_int high;

  while (*s) {
    high = h >> (sizeof(h) * 8 - shift);
    h = (h << shift) ^ *s ^ high;
    s++;
  }
  return h;
}

/*
 * Map URL to filename for saving a bloomfilter.  Prepends "rli_bloomfilter_dir/"
 * to URL, plus converts slashes (/) in URL to percent signs (%), and appends
 * ".bf".
 */
static void
url2fn(char *url, char *fn)

{
  int	len;
  int	i;

  sprintf(fn, "%s/", rli_bloomfilter_dir);
  len = strlen(fn);
  for (i = 0; url[i]; i++)
    if (url[i] == '/')
      fn[len+i] = '%';
    else
      fn[len+i] = url[i];
  strcpy(&fn[len+i], ".bf");
}
