/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <booh@altern.org>
 */

/******************************
 * User configurable options. *
 ******************************/

#include "gliv.h"

#include <string.h>             /* memcmp() */
#include <gdk/gdkkeysyms.h>     /* GDK_Escape, GDK_c, GDK_C, GDK_o, GDK_O */

extern rt_struct *rt;
extern options_struct *options;
extern gliv_image *current_image;
extern GtkWidget *gl_widget;

static GtkTable *buttons;
static options_struct *new_options;
static GtkWindow *options_win = NULL;

static GtkColorSelection *color_sel;
static gint *col_choice = NULL;

static gboolean save_options = FALSE;

static void toggle_delay(void)
{
    gint events;

    if (options->delay == 0)
        /* Enable. */
        gtk_widget_add_events(gl_widget, GDK_POINTER_MOTION_MASK);
    else if (new_options->delay == 0) {
        /* Disable. */
        events = gtk_widget_get_events(gl_widget) & ~GDK_POINTER_MOTION_MASK;
        gdk_window_set_events(gl_widget->window, events);
    }

    options->delay = new_options->delay;
    schedule_hide_cursor();
}

static void toggle_history(void)
{
    gboolean enable;

    if (options->history_size == 0)
        enable = TRUE;
    else if (new_options->history_size == 0)
        enable = FALSE;
    else
        return;

    options->history_size = new_options->history_size;

    if (enable == TRUE)
        init_history();
    else
        clear_history();
}


static void finish(gboolean apply)
{
    guint what = 0;
    gboolean bg_color_changed;

    if (apply == TRUE) {

        if (options->fullscreen != new_options->fullscreen)
            toggle_fullscreen(new_options->fullscreen);

        if ((options->maximize == FALSE && new_options->maximize == TRUE) ||
            (options->scaledown == FALSE && new_options->scaledown == TRUE)) {

            options->maximize = new_options->maximize;
            options->scaledown = new_options->scaledown;

            matrix_set_max_zoom(-1, -1, TRUE);
            what |= REFRESH_IMAGE | REFRESH_STATUS | APPEND_HISTORY;
        }

        if (options->menu_bar != new_options->menu_bar)
            toggle_menu_bar();

        if (options->status_bar != new_options->status_bar)
            toggle_status_bar();

        if (options->dither != new_options->dither) {
            options->dither = new_options->dither;
            reload_all_images();
            what |= REFRESH_IMAGE;
        }

        if (options->delay != new_options->delay)
            toggle_delay();

        if (options->history_size != new_options->history_size)
            toggle_history();

        bg_color_changed = (memcmp(options->bg_col, new_options->bg_col,
                                   3 * sizeof(gint)) != 0);

        if (memcmp(options->alpha1, new_options->alpha1, 6 * sizeof(gint))) {
            /* 6 : alpha1 and alpha2 */
            rt->alpha_checks_changed = TRUE;
            what |= REFRESH_IMAGE;
        }

        g_free(options);
        options = new_options;

        if (save_options == TRUE)
            save_rc(options);

        if (bg_color_changed == TRUE) {
            update_bg_color(TRUE);
            what |= REFRESH_IMAGE;
        }
    } else
        g_free(new_options);

    gtk_widget_destroy(GTK_WIDGET(options_win));
    options_win = NULL;

    refresh(what);
}

static gboolean toggle_button(GtkToggleButton * button, gboolean * bool)
{
    *bool = gtk_toggle_button_get_active(button);
    return TRUE;
}

static void add_button(gchar * name, guint x, guint y, gboolean * value)
{
    GtkCheckButton *button;

    name = locale_to_utf8(add_mnemonic(name));
    button = GTK_CHECK_BUTTON(gtk_check_button_new_with_mnemonic(name));

    gtk_table_attach_defaults(buttons, GTK_WIDGET(button), x, x + 1, y, y + 1);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), *value);
    g_signal_connect(button, "toggled", G_CALLBACK(toggle_button), value);
}

static gboolean value_changed(GtkAdjustment * widget, gint * val)
{
    *val = (gint) gtk_adjustment_get_value(widget);
    return TRUE;
}

static void add_spin_button(gchar * text, guint pos, gint * value,
                            gdouble minimum)
{
    GtkSpinButton *button;
    GtkAdjustment *adjustment;
    GtkLabel *label;

    adjustment = GTK_ADJUSTMENT(gtk_adjustment_new((gdouble) (*value), minimum,
                                                   1e9, 1.0, 10.0, 10.0));

    g_signal_connect(adjustment, "value_changed",
                     G_CALLBACK(value_changed), value);

    label = GTK_LABEL(gtk_label_new(locale_to_utf8(text)));

    button = GTK_SPIN_BUTTON(gtk_spin_button_new(adjustment, 1.0, 0));
    gtk_spin_button_set_update_policy(button, GTK_UPDATE_IF_VALID);
    gtk_spin_button_set_numeric(button, TRUE);

    gtk_table_attach_defaults(buttons, GTK_WIDGET(label), 0, 1, pos, pos + 1);
    gtk_table_attach_defaults(buttons, GTK_WIDGET(button), 1, 2, pos, pos + 1);
}

static gboolean key_press_event(GtkWidget * unused, GdkEventKey * event)
{
    if (event->keyval == GDK_Escape) {
        finish(FALSE);
        return TRUE;
    }

    return FALSE;
}

/* One of the three radio buttons was activated. */
static void change_color_sel(gint * col)
{
    GdkColor new_color;

    if (col_choice == col)
        return;

    col_choice = col;

    new_color.red = col[0];
    new_color.green = col[1];
    new_color.blue = col[2];

    gtk_color_selection_set_current_color(color_sel, &new_color);
}

static void add_color_radio(gchar * label, gint * color, GtkHBox * container)
{
    GtkRadioButton *radio;
    gpointer parent;

    parent = gtk_container_get_children(GTK_CONTAINER(container));
    if (parent != NULL)
        parent = ((GList *) parent)->data;

    label = locale_to_utf8(add_mnemonic(label));
    radio = GTK_RADIO_BUTTON(gtk_radio_button_new_with_mnemonic_from_widget
                             (parent, label));

    g_signal_connect_swapped(radio, "pressed", G_CALLBACK(change_color_sel),
                             color);

    gtk_box_pack_start_defaults(GTK_BOX(container), GTK_WIDGET(radio));
}

static void add_widget(GtkWidget * widget, guint * pos)
{
    gtk_table_attach_defaults(buttons, widget, 0, 2, *pos, *pos + 1);
    (*pos)++;
}

static void color_changed(void)
{
    GdkColor col;

    gtk_color_selection_get_current_color(color_sel, &col);

    col_choice[0] = col.red;
    col_choice[1] = col.green;
    col_choice[2] = col.blue;
}

static void add_color_selection(guint * pos)
{
    GtkHBox *color_radio;

    color_sel = GTK_COLOR_SELECTION(gtk_color_selection_new());

    /* Hide opacity choice. */
    g_signal_connect(color_sel, "map",
                     G_CALLBACK(gtk_color_selection_set_has_opacity_control),
                     GINT_TO_POINTER(FALSE));

    change_color_sel(new_options->bg_col);
    g_signal_connect(color_sel, "color-changed",
                     G_CALLBACK(color_changed), NULL);

    add_widget(GTK_WIDGET(color_sel), pos);

    color_radio = GTK_HBOX(gtk_hbox_new(FALSE, 10));

    add_color_radio(_("Background"), new_options->bg_col, color_radio);
    add_color_radio(_("Alpha 1"), new_options->alpha1, color_radio);
    add_color_radio(_("Alpha 2"), new_options->alpha2, color_radio);

    add_widget(GTK_WIDGET(color_radio), pos);
}

static void create_buttons(void)
{
    GtkButton *ok_button;
    GtkButton *cancel_button;
    gchar *label;
    guint pos = 0;              /* Vertical position. */

    new_options = g_memdup(options, sizeof(options_struct));

    buttons = GTK_TABLE(gtk_table_new(0, 0, FALSE));
    gtk_table_set_row_spacings(buttons, 5);
    gtk_table_set_col_spacings(buttons, 0);

    add_button(_("Fullscreen mode"), 0, pos, &new_options->fullscreen);
    add_button(_("Zoom centered on pointer"), 1, pos++,
               &new_options->zoom_pointer);
    add_button(_("Menu bar enabled"), 0, pos, &new_options->menu_bar);
    add_button(_("Status bar enabled"), 1, pos++, &new_options->status_bar);
    add_button(_("Scale down large images"), 0, pos, &new_options->scaledown);
    add_button(_("Maximize small images"), 1, pos++, &new_options->maximize);
    add_button(_("Alpha checks"), 0, pos, &new_options->alpha_checks);
    add_button(_("Dithering"), 1, pos++, &new_options->dither);
    add_button(_("Try to load every file"), 0, pos, &new_options->force);
    add_button(_("Build images menus at startup"), 1, pos++,
               &new_options->build_menus);
    add_button(_("Save options in ~/.glivrc"), 0, pos++, &save_options);

    add_spin_button(_("Delay before hiding the cursor\n0 : feature disabled"),
                    pos++, &new_options->delay, 0.0);

    add_spin_button(_("Length of history\n0 : feature disabled\n-1 : infinite"),
                    pos++, &new_options->history_size, -1.0);

    /* The color selection. */


    add_widget(gtk_hseparator_new(), &pos);

    label = locale_to_utf8(_("Background and alpha checks color selection"));
    add_widget(gtk_label_new(label), &pos);

    add_color_selection(&pos);

    /* The OK button. */

    ok_button = GTK_BUTTON(gtk_button_new_from_stock(GTK_STOCK_OK));

    GTK_WIDGET_SET_FLAGS(GTK_WIDGET(ok_button), GTK_CAN_DEFAULT);
    g_signal_connect_swapped(ok_button, "clicked",
                             G_CALLBACK(finish), GINT_TO_POINTER(TRUE));

    gtk_table_attach_defaults(buttons, GTK_WIDGET(ok_button), 0, 1,
                              pos, pos + 1);

    /* The Cancel button. */

    cancel_button = GTK_BUTTON(gtk_button_new_from_stock(GTK_STOCK_CANCEL));

    g_signal_connect_swapped(cancel_button, "clicked",
                             G_CALLBACK(finish), GINT_TO_POINTER(FALSE));

    gtk_table_attach_defaults(buttons, GTK_WIDGET(cancel_button), 1, 2,
                              pos, pos + 1);

    /* Misc. */

    g_signal_connect_swapped(options_win, "delete_event",
                             G_CALLBACK(finish), GINT_TO_POINTER(FALSE));

    g_signal_connect(options_win, "key_press_event",
                     G_CALLBACK(key_press_event), NULL);

    gtk_container_set_border_width(GTK_CONTAINER(options_win), 10);
    gtk_container_add(GTK_CONTAINER(options_win), GTK_WIDGET(buttons));

    gtk_widget_grab_default(GTK_WIDGET(ok_button));
}
void show_options(void)
{
    if (options_win != NULL)
        return;

    options_win = GTK_WINDOW(gtk_window_new(GTK_WINDOW_TOPLEVEL));

    create_buttons();
    show_dialog(GTK_WIDGET(options_win), _("Options"), TRUE);
}
