// $Id: Naming.cc,v 1.26 2001/10/11 14:07:23 christof Exp $
/*  glade--: C++ frontend for glade (Gtk+ User Interface Builder)
 *  Copyright (C) 1998  Christof Petig
 *  Copyright (C) 1999-2000 Adolf Petig GmbH & Co. KG, written by Christof Petig
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#warning copying and substituting in place would be much more efficient

#include "Naming.hh"
#include "Configuration.hh"
#include <cctype>
#include <unistd.h>
// kind of macro misuse, but everything else is non-portable
#define __FILELINE__ __FILE__ << ':' << __LINE__ <<

const std::string Naming::Capitalize(const std::string &s) throw()
{  std::string ret("");
   std::string::const_iterator i=s.begin();
   for (;i!=s.end() && *i=='_';++i) ret+=*i;
   if (i!=s.end()) ret+=toupper((unsigned char)*(i++));
   for (;i!=s.end();++i) ret+=*i;
   return ret;
}

const std::string Naming::TypeName(const std::string &tag,bool glade) const
{  std::string ret(ToCIdentifier(tag));
   if (Configuration.do_case_conversion) ret=Capitalize(ret);
   if (glade) ret+=(Configuration.do_case_conversion?"_Glade":"_glade");
   return ret;
}

const std::string Naming::LowerCase(const std::string &s) throw()
{  std::string ret("");
   std::string::const_iterator i=s.begin();
   for (;i!=s.end();++i) ret+=tolower((unsigned char)*i);
   return ret;
}

const std::string Naming::InstanceName(const std::string &tag) const
{  if (Configuration.do_case_conversion) return LowerCase(ToCIdentifier(tag));
   else return ToCIdentifier(tag);
}

const std::string Naming::FileName(const std::string &tag,File_type tp,int flags) const
{  std::string ret;

   if (!(flags&File_NODIR))
   {  switch(tp)
      {  case File_FOO_GLADE_HH:
         case File_FOO_HH:
         case File_FOO_GLADE_CC:
         case File_FOO_CC:
         case File_MAIN_CC:
         case File_SUPPORT_HH:
         case File_SUPPORT_CC:
         case File_MAKEFILE_AM:
            ret=Configuration.destination_directory
            	+Configuration.source_directory+"/";
            break;
            
         case File_CONFIGURE_IN:
         case File_AUTOGEN_SH:
         case File_README:
         case File_AUTHORS:
         case File_ChangeLog:
         case File_NEWS:
         case File_TODO:
         case File_ACCONFIG_H:
         case File_toplevel_MAKEFILE_AM:
            ret=Configuration.destination_directory;
            if (ret.size() && ret[ret.size()-1]!='/') ret+='/';
            break;
            
         case File_po_ChangeLog:
         case File_po_POTFILES_in:
            ret=Configuration.destination_directory;
            if (ret.size() && ret[ret.size()-1]!='/') ret+='/';
            ret+="po/";
            break;
                                         
         case File_MAKEFILE:
            if (!Configuration.no_autoconf)
            {  ret=Configuration.destination_directory;
            }
            else
            {  ret=Configuration.destination_directory
            	+Configuration.source_directory+"/";
            }
            if (ret.size() && ret[ret.size()-1]!='/') ret+='/';
            break;
         
         default: assert(0); break;
      }
   }
   switch(tp)
   {  case File_FOO_GLADE_HH:
         ret+=FName(tag);
   	 ret+="_glade" + Configuration.header_suffix;
   	 break;
      case File_SUPPORT_HH: 
         ret+="glademm_support" + Configuration.header_suffix;
         break;
      case File_SUPPORT_CC: 
         ret+="glademm_support";
         ret+=flags&File_OBJECT?".o": Configuration.source_suffix;
         break;
      case File_FOO_HH:
      	 ret+=FName(tag);
         ret+= Configuration.header_suffix;
         break;
      case File_FOO_GLADE_CC:
         ret+=FName(tag);
         ret+="_glade";
         ret+=flags&File_OBJECT?".o": Configuration.source_suffix;
         break;
      case File_FOO_CC:
         ret+=FName(tag);
         ret+=flags&File_OBJECT?".o": Configuration.source_suffix;
         break;
      case File_ACCONFIG_H:
      	 ret+="acconfig.h";
      	 break;
      case File_MAKEFILE:
      	 ret+="Makefile";
      	 break;
      case File_toplevel_MAKEFILE_AM:
      case File_MAKEFILE_AM:
      	 ret+="Makefile.am";
      	 break;
      case File_CONFIGURE_IN:
      	 ret+="configure.in";
      	 break;
      case File_AUTOGEN_SH:
      	 ret+="autogen.sh";
      	 break;
      case File_MAIN_CC:
      	 ret+=Configuration.main_filename;
      	 ret+=flags&File_OBJECT?".o": Configuration.source_suffix;
      	 break;
      case File_README: ret+="README"; break;
      case File_AUTHORS: ret+="AUTHORS"; break;
      case File_po_ChangeLog:
      case File_ChangeLog: ret+="ChangeLog"; break;
      case File_NEWS: ret+="NEWS"; break;
      case File_TODO: ret+="TODO"; break;
      case File_po_POTFILES_in: ret+="POTFILES.in"; break;
      default:
         std::cerr<<__FILELINE__" attempt to create unknown file type " << int(tp) << "\n";
         ret+='?';
         break;
   }
   switch (tp)
   {  case File_FOO_GLADE_HH:
      case File_SUPPORT_HH: 
      case File_SUPPORT_CC: 
      case File_FOO_HH:
      case File_FOO_GLADE_CC:
      case File_FOO_CC:
      case File_ACCONFIG_H:
      case File_MAKEFILE:
      case File_toplevel_MAKEFILE_AM:
      case File_MAKEFILE_AM:
      case File_CONFIGURE_IN:
      case File_AUTOGEN_SH:
      case File_MAIN_CC:
         if ((flags&File_GLADE) || ((flags&File_NOREPLACE) && !access(ret.c_str(),F_OK)))
         {  ret+=Configuration.template_postfix;
         }
         break;
      case File_README: 
      case File_AUTHORS: 
      case File_po_ChangeLog:
      case File_ChangeLog: 
      case File_NEWS: 
      case File_TODO: 
      case File_po_POTFILES_in: 
      default:
         break;
   }
   return ret;
}

const std::string Naming::FileDefine(const std::string &tag,File_type tp) const
{  std::string ret(tag);
   switch(tp)
   {  case File_FOO_GLADE_HH:
   	 ret+="_glade" + Configuration.header_suffix;
   	 break;
      case File_FOO_HH:
         ret+= Configuration.header_suffix;
         break;
      default:
         std::cerr << __FILELINE__" attempt to create unknown define type " << int(tp) << "\n";
         break;
   }
   return "_"+DefineName(ret);
}

const std::string Naming::UpperCase(const std::string &s) throw()
{  std::string ret;
   for (std::string::const_iterator i=s.begin();i!=s.end();++i) 
   	ret+=toupper((unsigned char)*i);
   return ret;
}

const std::string Naming::DefineName(const std::string &tag) const throw()
{  std::string ret(ToCIdentifier(tag));
   if (!Configuration.mixedcase_defines) ret=UpperCase(ret);
   return ret;
}

const std::string Naming::CName(const std::string &tag) const throw()
{  return ToCIdentifier(tag);
}

const std::string Naming::ToCIdentifier(const std::string &tag) throw()
{  std::string ret("");
   std::string::const_iterator i=tag.begin();
   // first char [_a-zA-Z]
   if (i!=tag.end()) 
   {  if (!(*i=='_' || isalpha((unsigned char)*i))) ret+='_';
      else ret+=*i;
      ++i;
   }
   // now [_a-zA-Z0-9]
   for (;i!=tag.end();++i) 
   {  if (!(*i=='_' || isalnum((unsigned char)*i))) ret+='_';
      else ret+=*i;
   }
   return ret;
}

const std::string Naming::CString(const std::string &x) const throw()
{  std::string ret("");
   std::string::const_iterator i=x.begin();
   for (;i!=x.end();++i)
   {  switch((unsigned char)*i)
      {  case '"': ret+="\\\"";
            break;
         case '\\': ret+="\\\\";
            break;
         case '\n': ret+="\\n\"\n\t\t\"";
            break;
         default: ret+=*i;
            break;
      }
   }
   return ret;
}

const std::string Naming::static_Translatable(const std::string &s) const
{  if (!Configuration.gettext_support) return CString_WithQuotes(s);
   return "N_("+CString_WithQuotes(s)+")"; }

const std::string Naming::Translatable(const std::string &s) const
{  if (!Configuration.gettext_support) return CString_WithQuotes(s);
   return "_("+CString_WithQuotes(s)+")"; }

const std::string Naming::ToFileName(const std::string &x) throw()
{  std::string ret("");
   std::string::const_iterator i=x.begin();
   for (;i!=x.end();++i)
   {  switch((unsigned char)*i)
      {  case 0 ... 31:
         case 127 ... 159:
      	    ret+="_";
            break;
         default: ret+=*i;
            break;
      }
   }
   return ret;
}

