/*
 * if_linux.c
 *
 * GKrellM LongRun Plugin
 * Copyright (c) 2001-2002 Masaharu FUJITA
 *
 * Initial work by: (c) 2001 Nozomi Sato (nozomi@palette.plala.or.jp)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * You may contact the author by:
 *	e-mail: m@fjts.org
 *	url: http://fjts.org/~m/Soft/GKrelLongRun/ (in Japanese)
 *	     http://fjts.org/~m/Soft/GKrelLongRun/index_e.html (in English)
 *
 */
#include <stdint.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <gkrellm/gkrellm.h>

#include "interface.h"

#define MSR_TMx86_LONGRUN		0x80868010
#define MSR_TMx86_LONGRUN_FLAGS		0x80868011

#define CPUID_TMx86_VENDOR_ID           0x80860000
#define CPUID_TMx86_PROCESSOR_INFO      0x80860001
#define CPUID_TMx86_LONGRUN_STATUS      0x80860007
#define CPUID_TMx86_FEATURE_LONGRUN(x) ((x) & 0x02)

#define LONGRUN_MASK(x) ((x) & 0x0000007f)
#define LONGRUN_RESERVED(x) ((x) & 0xffffff80)
#define LONGRUN_WRITE(x, y) (LONGRUN_RESERVED(x) | LONGRUN_MASK(y))

typedef enum {
	Economy,
	Performance,
} ModeType;

typedef enum {
	Minimum,
	Variableness,
	Maxmum,
} FrequencyType;

static int cpuid_fd = -1;
static int msr_fd = -1; 

static const char *msr_device = "/dev/cpu/0/msr";
static const char *cpuid_device = "/dev/cpu/0/cpuid";
gchar *longrun_mode_label[] = { "Eco.", "Per." };

static int max_frequency = 0;
static int min_frequency = 0;

static int
open_devices()
{
	if ((cpuid_fd = open(cpuid_device, O_RDONLY)) < 0) {
		fprintf(stderr, "gkrellongrun: %s : %s\n",
			strerror(errno), cpuid_device);
		return -1;
	}
	if ((msr_fd = open(msr_device, O_RDWR)) < 0) {
		fprintf(stderr, "gkrellongrun: %s : %s\n",
			strerror(errno), msr_device);
		return -1;
	}
	return 0;
}

static void
close_devices()
{
	if (cpuid_fd > 0)
		close(cpuid_fd); 
	if (msr_fd > 0)
		close(msr_fd); 
	cpuid_fd = msr_fd = -1;
}

static void
read_cpuid( long address, int *eax, int *ebx, int *ecx, int *edx )
{
	uint32_t data[4];

	if ( pread( cpuid_fd, &data, 16, address ) != 16 )
		data[0] = data[1] = data[2] = data[3] = 0;

	if ( eax )
		*eax = data[0];
	if ( ebx )
		*ebx = data[1];
	if ( ecx )
		*ecx = data[2]; 
	if ( edx )
		*edx = data[3]; 
}

static void
read_msr( long address, int *lower, int *upper )
{
	uint32_t data[2];

	if (pread(msr_fd, &data, 8, address) != 8) {
		return;
	}

	if (lower)
		*lower = data[0];
	if (upper)
		*upper = data[1];
}

static void
write_msr(long address, int lower, int upper)
{
	uint32_t data[2];

	data[0] = (uint32_t)lower;
	data[1] = (uint32_t)upper;

	pwrite(msr_fd, &data, 8, address);
}


static void
set_longrun_mode(GtkWidget *widget, gpointer data)
{
	int lower, upper;

	read_msr(MSR_TMx86_LONGRUN_FLAGS, &lower, &upper);
	switch (GPOINTER_TO_INT(data)) {
		case Economy:
			write_msr( MSR_TMx86_LONGRUN_FLAGS,
					lower & 0xfffffffe, upper);
			break;
		case Performance:
			write_msr( MSR_TMx86_LONGRUN_FLAGS,
					(lower & 0xffffffff) | 0x1, upper);
			break;
	}
}

static void
set_longrun_frequency(GtkWidget *widget, gpointer data)
{
	int lower, upper, low, high;

	switch (GPOINTER_TO_INT(data)) {
		case Minimum:
			low = min_frequency;
			high = min_frequency;
			break;
		case Variableness:
			low = min_frequency;
			high = max_frequency;
			break;
		case Maxmum:
			low = max_frequency;
			high = max_frequency;
			break;
	}
	low = ((low - min_frequency) * 100)/(max_frequency - min_frequency);
	high = ((high - min_frequency) * 100)/(max_frequency - min_frequency);

	read_msr(MSR_TMx86_LONGRUN, &lower, &upper);
	write_msr(MSR_TMx86_LONGRUN, LONGRUN_WRITE(lower, low),
			LONGRUN_WRITE(upper, high));
}

/*
 * read_longrun_data
 */
void
read_longrun_data()
{
	int lower, upper;

	read_cpuid( CPUID_TMx86_LONGRUN_STATUS,
			&longrun.data[FrequencyData],
			&longrun.data[VoltageData],
			&longrun.data[PercentageData], NULL );

	read_msr(MSR_TMx86_LONGRUN_FLAGS, &lower, &upper );
	longrun.data[LongRunData] = (lower & 1) ? Performance : Economy;

	set_longrun_label();
}

/*
 * check_cpu
 */
int
check_cpu()
{
	int eax, ebx, ecx, edx;
	int lower, upper;

	if (open_devices()) {
		close_devices();
		return -1;
	}

	/* test for "TransmetaCPU" */
	read_cpuid(CPUID_TMx86_VENDOR_ID, &eax, &ebx, &ecx, &edx);
	if (ebx != 0x6e617254 || ecx != 0x55504361 || edx != 0x74656d73) {
		close_devices();
		return -1;
	}
	/* test for LongRun feature flag */
	read_cpuid(CPUID_TMx86_PROCESSOR_INFO, &eax, &ebx, &ecx, &edx);
	if (!CPUID_TMx86_FEATURE_LONGRUN(edx)) {
		close_devices();
		return -1;
	}

	/* get maximum & minimum frequency */
	read_msr(MSR_TMx86_LONGRUN, &lower, &upper);
	read_cpuid(CPUID_TMx86_PROCESSOR_INFO, 0, 0, &max_frequency, 0);
	write_msr(MSR_TMx86_LONGRUN, LONGRUN_WRITE(lower, 0),
			LONGRUN_WRITE(upper, 0));
	read_cpuid(CPUID_TMx86_LONGRUN_STATUS, &min_frequency, 0, 0, 0);
	write_msr(MSR_TMx86_LONGRUN, lower, upper);

	return 0;
}

GtkItemFactoryEntry gkrellongrun_factory_entry[] =
{
	{"/-",		NULL, NULL, 0,"<Separator>"},
	{"/Mode",	NULL, NULL, 0, "<Item>"},
	{"/-",		NULL, NULL, 0,"<Separator>"},
	{"/Economy",	NULL, set_longrun_mode, Economy, "<Item>"},
	{"/Performance", NULL, set_longrun_mode, Performance, "<Item>"},
	{"/-",		NULL, NULL, 0,"<Separator>"},
	{"/Frequency",	NULL, NULL, 0, "<Item>"},
	{"/-",		NULL, NULL, 0,"<Separator>"},
	{"/Minimum",	NULL, set_longrun_frequency, Minimum, "<Item>"},
	{"/Variableness", NULL, set_longrun_frequency, Variableness, "<Item>"},
	{"/Maxmum",	NULL, set_longrun_frequency, Maxmum, "<Item>"},
	{"/-",		NULL, NULL, 0,"<Separator>"},
};


/*
 * get_gkrellongrun_itemfactory_size
 */
int
get_gkrellongrun_itemfactory_size()
{
	return sizeof(gkrellongrun_factory_entry)/sizeof(GtkItemFactoryEntry);
}

