/* GKrellM
|  Copyright (C) 1999-2002 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.  Version 2 is in the
|  COPYRIGHT file in the top level directory of this distribution.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
|   2/8/2002  Christian Schroeder <chs@baltic-online.de> patch to avoid
|               potential Solaris segfault in get_solaris_vfstab_list()
| 10/29/2001  Darell Tan <dardil@singnet.com.sg> patch adds uid check for
|               "owner" to fstab_user_permission().
|  4/22/2001  Solaris code contributed by Daisuke Yabuki <dxy@acm.org>
| 10/12/2000  NetBSD code contributed by Anthony Mallet
|               <anthony.mallet@useless-ficus.net>
|  2/25/2000  FreeBSD code contributed by Hajimu UMEMOTO ume@mahoroba.org
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"
#include "gkrellm_threads.h"

typedef struct
	{
	gchar	*directory;
	gchar	*device;
	gchar	*type;
	gchar	*options;
	}
	Mount;

typedef struct
	{
	gint		idx;
	Panel		*panel;
	DecalButton	*md_button,
				*eject_button;
	Decal		*mount_decal,
				*eject_decal,
				*text_decal;
	gchar		*label;
	gboolean	label_is_data,
				restore_label,
				mouse_entered;

	Mount		mount;
	gint		enable_mounting;

	Launcher	launch_mount,
				launch_umount;

	/* Monitors in the drawer will not be visible if the drawer is shut
	|  unless the show_if_mounted option is set and the monitor is
	|  mounted.
	*/
	gboolean	in_drawer,
				show_if_mounted,
				is_mounted,
				ejectable,
				is_nfs_fs;
	gchar		*eject_device;
	gint		eject_pending;
	gint		x_eject_button_target;

	GString		*pipe_gstring;		/* output of mount commands */

	gulong		krell_factor;		/* avoid krell math overflow */

	gulong		blocks,
				bfree,
				bavail,
				bsize;
	}
	FSmon;


static GList	*fs_mon_list,
				*proc_mounts_list;
static GList	*fstab_list;

static gint		uid;

void	(*get_mountlist)(),
		(*get_fsusage)(),
		(*get_fstab_list)();

/* If there is a funtion with ioctls to eject a cdrom, assign it to the
|  following function pointer.
*/
void	(*eject_cdrom)();

/* If a system has eject commands, assign them to these:
*/
static gchar	*eject_tray_command,
				*close_tray_command;

/* ====== System dependent interface ====================================== */


/* ----- FreeBSD NetBSD OpenBSD ----------------------------------------- */

#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
#include <sys/mount.h>
#include <sys/cdio.h>
#include <sys/wait.h>

#if defined(__FreeBSD__)
#include <osreldate.h>
#if __FreeBSD_version < 300000
static char	*mnttype[] = INITMOUNTNAMES;
#endif
#endif

static void
get_bsd_mounts_list()
	{
	Mount		*m;
	gchar		*s, *dev, *dir, *type;
	struct statfs	*mntbuf;
	gint		mntsize, i;

	while (proc_mounts_list)
		{
		m = (Mount *) proc_mounts_list->data;
		g_free(m->directory);
		g_free(m->device);
		g_free(m->type);
		g_free(proc_mounts_list->data);
		proc_mounts_list = g_list_remove(proc_mounts_list,
						 proc_mounts_list->data);
		}
	if ((mntsize = getmntinfo(&mntbuf, MNT_NOWAIT)) == 0)
		return;
	for (i = 0; i < mntsize; i++)
		{
#if defined(__FreeBSD__) && __FreeBSD_version < 300000
		type = mnttype[mntbuf[i].f_type];
#else
		type = mntbuf[i].f_fstypename;
#endif
		dir = mntbuf[i].f_mntonname;
		dev = mntbuf[i].f_mntfromname;
		/* Strip trailing / from the directory.
		*/
		s = strrchr(dir, (int) '/');
		if (s && s != dir && *(s+1) == '\0')
			*s = '\0';
		m = g_new0(Mount, 1);
		m->directory = g_strdup(dir);
		m->device = g_strdup(dev);
		m->type = g_strdup(type);

		proc_mounts_list = g_list_append(proc_mounts_list, m);
		}
	}
#endif

#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
static void
eject_bsd_cdrom(gchar *device)
	{
	gint	d;

	if ((d = open(device, O_RDONLY)) >= 0)
		{
		(void) ioctl(d, CDIOCALLOW);
		ioctl(d, CDIOCEJECT);
		close(d);
		}
	}
#endif /* __FreeBSD__ || __NetBSD__ || __OpenBSD__ */


#if defined(__FreeBSD__)
static void
set_eject_command(void)
	{
#if defined(WEXITSTATUS)
	gint	n;

	n = system("cdcontrol quit > /dev/null 2>&1");
	if (WEXITSTATUS(n) == 0)
		{
		eject_tray_command = "cdcontrol -f %s eject";
		close_tray_command = "cdcontrol -f %s close";
		}
#endif
	}
#endif

#if defined(__NetBSD__)
static void
set_eject_command(void)
	{
#if defined(WEXITSTATUS)
	gint	n;

	n = system("eject -n > /dev/null 2>&1");
	if (WEXITSTATUS(n) == 0)
		{
		eject_tray_command = "eject %s";
		close_tray_command = "eject -l %s";
		}
#endif
	}
#endif /* __NetBSD__ */

#if defined(__OpenBSD__)
static void
set_eject_command(void)
	{
	return;
	}
#endif /* __OpenBSD__ */


/* ----- Linux ------------------------------------------------------ */

#if defined(__linux__)
#include <sys/vfs.h>
#include <sys/wait.h>
#include <linux/cdrom.h>
#include <mntent.h>

#define	PROC_MOUNTS_FILE	"/proc/mounts"

  /* A list of mounted file systems can be read from /proc/mounts or
  |  /etc/mtab (getmntent).  Using /proc/mounts eliminates disk accesses,
  |  but for some reason /proc/mounts reports a "." for the mounted
  |  directory for smbfs types.  So I use /proc/mounts with a fallback
  |  to using getmntent().
  */
#if !defined (_PATH_MOUNTED)
#define _PATH_MOUNTED   "/etc/mtab"
#endif

static void
getmntent_fallback(Mount *m)
	{
	FILE			*f;
	struct mntent	*mnt;

	if ((f = setmntent(_PATH_MOUNTED, "r")) == NULL)
		return;
	while ((mnt = getmntent(f)) != NULL)
		{
		if (!strcmp(m->device, mnt->mnt_fsname))
			{
			gkrellm_dup_string(&m->directory, mnt->mnt_dir);
			break;
			}
		}
	endmntent(f);
	}

static void
get_linux_proc_mounts_list(void)
	{
	FILE		*f;
	Mount		*m;
	gchar		*s, buf[128], dev[64], dir[128], type[32];

	while (proc_mounts_list)
		{
		m = (Mount *) proc_mounts_list->data;
		g_free(m->directory);
		g_free(m->device);
		g_free(m->type);
		g_free(proc_mounts_list->data);
		proc_mounts_list =
				g_list_remove(proc_mounts_list, proc_mounts_list->data);
		}
	if ((f = fopen(PROC_MOUNTS_FILE, "r")) == NULL)
		return;
	while (fgets(buf, sizeof(buf), f))
		{
		sscanf(buf, "%63s %127s %31s", dev, dir, type);
		if (   !strcmp(type, "devpts")
			|| !strcmp(type, "proc")
			|| !strcmp(type, "usbdevfs")
		   )
			continue;
		/* Strip trailing / from the directory.
		*/
		s = strrchr(dir, (int) '/');
		if (s && s != dir && *(s+1) == '\0')
			*s = '\0';
		m = g_new0(Mount, 1);
		m->directory = g_strdup(dir);
		m->device = g_strdup(dev);
		m->type = g_strdup(type);
		if (dir[0] == '.')
			getmntent_fallback(m);

		proc_mounts_list = g_list_append(proc_mounts_list, m);
		}
	fclose(f);
	}

static void
eject_linux_cdrom(gchar *device)
	{
#if defined(CDROMEJECT)
	gint	d;

	if ((d = open(device, O_RDONLY|O_NONBLOCK)) >= 0)
		{
		ioctl(d, CDROMEJECT);
		close(d);
		}
#endif
	}

static void
set_eject_command(void)
	{
#if defined(WEXITSTATUS)
	gint	n;

	n = system("eject -d > /dev/null 2>&1");
	if (WEXITSTATUS(n) == 0)
		{
		eject_tray_command = "eject %s";
		close_tray_command = "eject -t %s";
		}
#endif
	}
#endif	/* __linux__ */


/* ----- BSD and Linux ----------------------------------------------- */

#if defined(__linux__) || defined(__FreeBSD__) || defined(__NetBSD__) \
	|| defined(__OpenBSD__)

static void
get_statfs_fsusage(FSmon *fs)
	{
	struct statfs	st;

	if (statfs(fs->mount.directory, &st) == 0)
		{
		fs->blocks = (gulong) st.f_blocks;
		fs->bavail = (gulong) st.f_bavail;
		fs->bfree =  (gulong) st.f_bfree;
		fs->bsize  = (gulong) st.f_bsize;
		}
	else
		{
		fs->blocks = 0;
		fs->bavail = 0;
		fs->bfree =  0;
		fs->bsize  = 0;
		}
	}
#endif

/* ----- Solaris ------------------------------------------------------- */
#if defined(__solaris__)

#include <sys/mnttab.h>
#include <sys/vfstab.h>
#include <sys/statvfs.h>
#include <sys/cdio.h>

static void 
get_solaris_mounts_list(){
    FILE *fp;
    Mount *m;
    struct mnttab mntbuf;

    while (proc_mounts_list) {
        m = (Mount *) proc_mounts_list->data;
        if (m->directory)
            g_free(m->directory);
        if (m->device)
            g_free(m->device);
        if (m->type)
            g_free(m->type);
        if (proc_mounts_list->data)
            g_free(proc_mounts_list->data);
        proc_mounts_list = g_list_remove(proc_mounts_list, proc_mounts_list->data);
    }

    if ((fp = fopen(MNTTAB, "r")) == NULL)
        return;

    while (getmntent(fp, &mntbuf) == 0) {
        if (strcmp(mntbuf.mnt_fstype, "ufs") && 
                                  strcmp(mntbuf.mnt_fstype, "nfs"))
            continue;
        m = g_new0(Mount, 1);
        if (!mntbuf.mnt_special) {
            m->device = g_strdup("");
        } else {
            m->device = g_strdup(mntbuf.mnt_special);
        }
        if (!mntbuf.mnt_mountp) {
            m->directory = g_strdup("");
        } else {
            m->directory = g_strdup(mntbuf.mnt_mountp);
        }
        if (!mntbuf.mnt_fstype) {
            m->type = g_strdup("");
        } else { 
            m->type = g_strdup(mntbuf.mnt_fstype);
        }
        proc_mounts_list = g_list_append(proc_mounts_list, m);
    }
    fclose(fp);
}

static void
get_solaris_statvfs_fsusage(FSmon *fs){
    struct statvfs st;

    if (fs->mount.directory && statvfs(fs->mount.directory, &st) == 0) {
        fs->blocks = (gulong) st.f_blocks;
        fs->bavail = (gulong) st.f_bavail;
        fs->bfree  = (gulong) st.f_bfree;
        fs->bsize  = (gulong) st.f_frsize;
    } else {
        fs->blocks = 0;
        fs->bavail = 0;
        fs->bfree  = 0;
        fs->bsize  = 0;
    } 
}

static void
get_solaris_vfstab_list(){
    Mount *m;
    FILE *fp;
    struct vfstab vfsbuf;

    while (fstab_list) {
        m = (Mount *) fstab_list->data;
        if (m->device)
            g_free(m->device);
        if (m->directory)
            g_free(m->directory);
        if (m->type)
            g_free(m->type);
        if (m->options)
            g_free(m->options);
        if (m)
            g_free(m);
        fstab_list = g_list_remove(fstab_list, fstab_list->data);
    }

    if ((fp = fopen(VFSTAB, "r")) == NULL)
        return;

    while (getvfsent(fp, &vfsbuf) == 0) {
        if (!vfsbuf.vfs_fstype || strcmp(vfsbuf.vfs_fstype, "ufs"))
            continue;
        m = g_new0(Mount, 1);
        if(!vfsbuf.vfs_special) {
            m->device = g_strdup("");
        } else {
            m->device = g_strdup(vfsbuf.vfs_special);
        }
        if(!vfsbuf.vfs_mountp) {
            m->directory = g_strdup("");
        } else {
            m->directory = g_strdup(vfsbuf.vfs_mountp); 
        }
        if(!vfsbuf.vfs_fstype) {
            m->type = g_strdup("");
        } else {
            m->type = g_strdup(vfsbuf.vfs_fstype);
        }
        if(!vfsbuf.vfs_mntopts) { 
            m->options = g_strdup("");
        } else {
            m->options = g_strdup(vfsbuf.vfs_mntopts);
        }
        fstab_list = g_list_append(fstab_list, m);
    }  

    fclose(fp);
}

static void
eject_solaris_cdrom(gchar *device) {
#if defined(CDROMEJECT)
        gint    d;

		if ((d = open(device, O_RDONLY)) >= 0) {
				ioctl(d, CDROMEJECT);
				close(d);
        }
#endif
}

#endif /* __solaris__ */

/* ----- Generic /etc/fstab -------------------------------------------- */
static gchar *
fix_fstab_name(gchar *buf)
	{
	gchar	*rp,
			*wp;

	if (buf == NULL)
		return g_strdup("");
	rp = buf;
	wp = buf;
	do	/* This loop same as in libc6 getmntent()	*/
		if (rp[0] == '\\' && rp[1] == '0' && rp[2] == '4' && rp[3] == '0')
			{
			*wp++ = ' ';		/* \040 is a SPACE.  */
			rp += 3;
			}
		else if (rp[0] == '\\' && rp[1] == '0' && rp[2] == '1' && rp[3] == '2')
			{
			*wp++ = '\t';		/* \012 is a TAB.  */
			rp += 3;
			}
		else if (rp[0] == '\\' && rp[1] == '\\')
			{
			*wp++ = '\\';		/* \\ is a \	*/
			rp += 1;
			}
		else
			*wp++ = *rp;
	while (*rp++ != '\0');

	return g_strdup(buf);
	}

static void
get_etc_fstab_list(void)
	{
	FILE			*f;
	Mount			*m;
	gchar			buf[512], *s;
	gchar			*dev, *dir, *type, *opt;

	while (fstab_list)
		{
		m = (Mount *) fstab_list->data;
		g_free(m->device);
		g_free(m->directory);
		g_free(m->type);
		g_free(m->options);
		g_free(m);
		fstab_list = g_list_remove(fstab_list, fstab_list->data);
		}
	if ((f = fopen("/etc/fstab", "r")) == NULL)
		return;
	while (fgets(buf, sizeof(buf), f))
		{
		s = buf;
		while (*s == ' ' || *s == '\t')
			++s;
		if (*s == '\0' || *s == '#' || *s == '\n')
			continue;
		dev = strtok(buf, " \t\n");
		dir = strtok(NULL, " \t\n");
		type = strtok(NULL, " \t\n");
		opt = strtok(NULL, " \t\n");

		if (   type == NULL
			|| !strcmp(type, "devpts")
			|| !strcmp(type, "swap")
			|| !strcmp(type, "proc")
			|| !strcmp(type, "usbdevfs")
			|| !strcmp(type, "ignore")
		   )
			continue;
		m = g_new0(Mount, 1);
		m->device = fix_fstab_name(dev);
		m->directory = fix_fstab_name(dir);
		m->type = fix_fstab_name(type);
		m->options =  fix_fstab_name(opt);

		fstab_list = g_list_append(fstab_list, m);
		}
	fclose(f);
	}


/* ----- Others -------------------------------------------------------- */
#if defined(USE_LIBGTOP)

#include <glibtop/mountlist.h>
#include <glibtop/fsusage.h>


static void
get_glibtop_mountlist(void)
	{
	glibtop_mountlist	mount_list;
	glibtop_mountentry	*mount_entries;
	Mount				*m;
	gint				i;

	while (proc_mounts_list)
		{
		m = (Mount *) proc_mounts_list->data;
		g_free(m->directory);
		g_free(m->device);
		g_free(m->type);
		g_free(proc_mounts_list->data);
		proc_mounts_list =
				g_list_remove(proc_mounts_list, proc_mounts_list->data);
		}
	mount_entries = glibtop_get_mountlist (&mount_list, 1);
	for (i = 0; i < mount_list.number; ++i)
		{
		m = g_new0(Mount, 1);
		m->directory = g_strdup(mount_entries[i].mountdir);
		m->device = g_strdup(mount_entries[i].devname);
		m->type = g_strdup(mount_entries[i].type);

		proc_mounts_list = g_list_append(proc_mounts_list, m);		
		}
	glibtop_free(mount_entries);
	}

static void
get_glibtop_fsusage(FSmon *fs)
	{
	glibtop_fsusage	fsusage;

	glibtop_get_fsusage(&fsusage, fs->mount.directory);
	fs->blocks = fsusage.blocks;
	fs->bavail = fsusage.bavail;
	fs->bfree = fsusage.bfree;
	fs->bsize  = 512;
	}
#endif


/* ----- Pick a system interface ----------------------------------------- */
static gint
setup_fs_interface(void)
    {
	uid = getuid();	/* only real root is allowed to mount/umount always */

#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
	get_mountlist = get_bsd_mounts_list;
	get_fsusage = get_statfs_fsusage;
	get_fstab_list = get_etc_fstab_list;
	eject_cdrom = eject_bsd_cdrom;
	set_eject_command();
#elif defined(__linux__)
	get_mountlist = get_linux_proc_mounts_list;
	get_fsusage = get_statfs_fsusage;
	get_fstab_list = get_etc_fstab_list;
	eject_cdrom = eject_linux_cdrom;
	set_eject_command();
#elif defined(__solaris__) 
	get_mountlist = get_solaris_mounts_list;
	get_fsusage = get_solaris_statvfs_fsusage;
	get_fstab_list = get_solaris_vfstab_list;
	eject_cdrom = eject_solaris_cdrom;
#else
	get_mountlist = get_glibtop_mountlist;;
	get_fsusage = get_glibtop_fsusage;
	get_fstab_list = get_etc_fstab_list;
#endif
    return TRUE;
    }

/* ======================================================================== */

#define	MT_NONE		0
#define	MT_FSTAB	1
#define	MT_CUSTOM	2

#define	DEFAULT_DATA_FORMAT	_("$t - $f free")

  /* Values for force_fs_check	*/
#define	FORCE_REDRAW	1
#define	FORCE_UPDATE	2


static Monitor		*mon_fs;
static GtkWidget	*fs_main_vbox,
					*fs_drawer_vbox;

static gboolean		fs_check_timeout	= 2,
					nfs_check_timeout	= 16;
static gint			check_tick;

static gint			fs_drawer_is_shown;

static gint			n_fs_monitors;
static gint			force_fs_check;
static FSmon		*fs_in_motion;
static gint			x_fs_motion;
static gint			x_moved;
static gint			x_eject_button_open,
					x_eject_button_closed;

static gint			x_scroll;
static gint			text_decal_width;
static gint			cdrom_auto_eject;
static gint			binary_units;

static gchar		*data_format;

static gint			style_id;

gchar	*remote_fs_types[]	=
	{
	"nfs",
	"smbfs"
	};


static Mount *
in_fstab_list(gchar *s)
	{
	GList	*list;
	Mount	*m;

	for (list = fstab_list; list; list = list->next)
		{
		m = (Mount *)list->data;
		if (strcmp(s, m->directory) == 0)
			return m;
		}
	return NULL;
	}


static gint
fs_is_mounted(FSmon *fs)
	{
	Mount	*m_fs, *m_mounted;
	GList	*list;
	gint	i;

	fs->is_mounted = FALSE;
	m_fs = &fs->mount;
	for (list = proc_mounts_list; list; list = list->next)
		{
		m_mounted = (Mount *) list->data;
		if (strcmp(m_fs->directory, m_mounted->directory) == 0)
			{
			fs->is_mounted = TRUE;
			fs->is_nfs_fs = FALSE;
			for (i = 0; i < (sizeof(remote_fs_types) / sizeof(gchar *)); ++i)
				{
				if (!strcmp(m_mounted->type, remote_fs_types[i]))
					{
					fs->is_nfs_fs = TRUE;
					break;
					}
				}
			}
		}
	return fs->is_mounted;
	}

static size_abbrev_table	fs_decimal_abbrev[] =
	{
	{ MB_SIZE(10),		MB_SIZE(1),		"%.2fM" },
	{ GB_SIZE(1),		MB_SIZE(1),		"%.0fM" },
	{ GB_SIZE(10),		GB_SIZE(1),		"%.2fG" },
	{ GB_SIZE(100),		GB_SIZE(1),		"%.1fG" },
	{ TB_SIZE(1),		GB_SIZE(1),		"%.0fG" },
	{ TB_SIZE(10),		TB_SIZE(1),		"%.2fT" },
	{ TB_SIZE(100),		TB_SIZE(1),		"%.1fT" }
	};

static size_abbrev_table	fs_binary_abbrev[] =
	{
	{ MiB_SIZE(10),		MiB_SIZE(1),	"%.2fM" },
	{ GiB_SIZE(1),		MiB_SIZE(1),	"%.0fM" },
	{ GiB_SIZE(10),		GiB_SIZE(1),	"%.2fG" },
	{ GiB_SIZE(100),	GiB_SIZE(1),	"%.1fG" },
	{ TiB_SIZE(1),		GiB_SIZE(1),	"%.0fG" },
	{ TiB_SIZE(10),		TiB_SIZE(1),	"%.2fT" },
	{ TiB_SIZE(100),	TiB_SIZE(1),	"%.1fT" }
	};

static gint
format_fs_data(FSmon *fs, gchar *buf, gint size)
	{
	gulong	b, u, f;
	gint	len;
	gchar	*s;
	gchar	tbuf[32], ubuf[32], fbuf[32];
	gfloat	bsize;
	size_abbrev_table	*tbl;
	size_t				tbl_size;

	--size;
	*buf = '\0';
	b = fs->blocks;
	u = fs->blocks - fs->bavail;
	f = fs->bavail;
	bsize = (gfloat) fs->bsize;

	tbl = binary_units ? &fs_binary_abbrev[0] : &fs_decimal_abbrev[0];
	tbl_size = binary_units
			? (sizeof(fs_binary_abbrev) / sizeof(size_abbrev_table))
			: (sizeof(fs_decimal_abbrev) / sizeof(size_abbrev_table));

	format_size_abbrev(tbuf, sizeof(tbuf), (gfloat)b * bsize, tbl, tbl_size);
	format_size_abbrev(ubuf, sizeof(ubuf), (gfloat)u * bsize, tbl, tbl_size);
	format_size_abbrev(fbuf, sizeof(fbuf), (gfloat)f * bsize, tbl, tbl_size);

	f = b - u;
	for (s = data_format; *s != '\0' && size > 0; ++s)
		{
		len = 1;
		if (*s == '$' && *(s + 1) != '\0')
			{
			switch(*(s + 1))
				{
				case 'l':
					len = snprintf(buf, size, "%s", fs->label);
					break;
				case 't':
					len = snprintf(buf, size, "%s", tbuf);
					break;
				case 'u':
					len = snprintf(buf, size, "%s", ubuf);
					break;
				case 'U':
					if (b > 0)
						len = snprintf(buf, size, "%ld%%",
							100 * (u >> 8) / (b >> 8));
					break;
				case 'f':
					len = snprintf(buf, size, "%s", fbuf);
					break;
				case 'F':
					if (b > 0)
						len = snprintf(buf, size, "%ld%%",
							100 * (f >> 8) / (b >> 8));
					break;
				default:
					*buf = *s;
					if (size > 1)
						{
						*(buf + 1) = *(s + 1);
						++len;
						}
					break;
				}
			++s;
			}
		else
			*buf = *s;
		size -= len;
		buf += len;
		}
	*buf = '\0';
	return u + 1;
	}

  /* Draw the fs label or toggle the fs total blocks and blocks avail.
  */
static gint
draw_fs_text_decal(FSmon *fs, gint value)
	{
	Decal		*d;
	TextStyle	ts_save;
	gchar		buf[128];
	gint		changed;	/* Decal value, tells whether redraw required */
	gint		w	= 0;

	d = fs->text_decal;

	if (value == 0)
		{
		d->x_off = 0;
		gkrellm_draw_decal_text(fs->panel, d, fs->label, 0);
		}
	else
		{
		ts_save = d->text_style;
		d->text_style = *gkrellm_meter_alt_textstyle(style_id);

		changed = format_fs_data(fs, buf, sizeof(buf));
		w = gdk_string_width(d->text_style.font, buf);
		if (w  > d->w)
			{
			d->x_off = d->w / 3 - x_scroll;
			changed = x_scroll + 1;
			}
		else
			d->x_off = 0;

		/* Draw the decal, actual draws occur only if "changed" is different
		|  from last call.  If scrolling, will be so each time, if not
		|  will be so as format_fs_data() return value changes.
		*/
		gkrellm_draw_decal_text(fs->panel, d, buf, changed);
		d->text_style = ts_save;
		}
	return w;
	}


static void
close_tray(FSmon *fs)
	{
	gchar	buf[512];

	if (close_tray_command)
		{
		snprintf(buf, sizeof(buf), close_tray_command,
			*(fs->eject_device) ? fs->eject_device : fs->mount.directory);
		strncat(buf, " 2>/dev/null &", sizeof(buf));
		gkrellm_system(buf);
		}
	}

static gboolean	eject_thread_busy;

static void
eject_cdrom_thread(void *device)
	{
	(*eject_cdrom)((gchar *) device);
	eject_thread_busy = FALSE;
	}

static void
eject_tray(FSmon *fs)
	{
	Mount			*m;
	static gchar	*eject_target;
	gchar			buf[512];

	eject_target = fs->eject_device;
	if (eject_tray_command)
		{
		snprintf(buf, sizeof(buf), eject_tray_command,
			*eject_target ? eject_target : fs->mount.directory);
		strncat(buf, " 2>/dev/null &", sizeof(buf));
		gkrellm_system(buf);
		}
	else if (eject_cdrom && !eject_thread_busy)
		{
		if (!*eject_target && (m = in_fstab_list(fs->mount.directory)) != NULL)
			eject_target = m->device;
		if (*eject_target)
			{
			eject_thread_busy = TRUE;
#if defined(GKRELLM_THREADS)
			new_thread(eject_cdrom_thread, (void *) eject_target);
#else
			eject_cdrom_thread(eject_target);
#endif
			}
		}
	}

static void
accumulate_pipe_gstring(FSmon *fs)
	{
	gchar	buf[512];
	gint	n;

	n = fread(buf, 1, sizeof(buf) - 1, fs->launch_mount.pipe);
	buf[n] = '\0';
	if (n > 0)
		{
		if (fs->pipe_gstring)
			g_string_append(fs->pipe_gstring, buf);
		else
			fs->pipe_gstring = g_string_new(buf);
		}
	if (feof(fs->launch_mount.pipe))
		{
		pclose(fs->launch_mount.pipe);
		fs->launch_mount.pipe = NULL;
		}
	}

static void
pipe_command(FSmon *fs, gchar *command)
	{
	gchar	buf[512];

	if (fs->launch_mount.pipe)	/* Still running? */
		return;
	snprintf(buf, sizeof(buf), "%s 2>&1", command);
	if ((fs->launch_mount.pipe = popen(buf, "r")) == NULL)
		return;
	fcntl(fileno(fs->launch_mount.pipe), F_SETFL, O_NONBLOCK);
	}

static void
mount_command(FSmon *fs)
	{
	gchar	cmd[CFG_BUFSIZE];

	if (fs->enable_mounting == MT_NONE)
		return;
	if (fs->is_mounted)
		{
		if (fs->enable_mounting == MT_FSTAB)
			snprintf(cmd, sizeof(cmd), "umount %s", fs->mount.directory);
		else	/* MT_CUSTOM */
			snprintf(cmd, sizeof(cmd), "%s", fs->launch_umount.command);
		fs->label_is_data = FALSE;
		draw_fs_text_decal(fs, 0);
		pipe_command(fs, cmd);
		if (cdrom_auto_eject)
			fs->eject_pending = GK.timer_ticks + 5;	/* at least 1/2 sec delay*/
		}
	else
		{
		if (fs->ejectable)
			close_tray(fs);
		if (fs->enable_mounting == MT_FSTAB)
			snprintf(cmd, sizeof(cmd), "mount %s", fs->mount.directory);
		else	/* MT_CUSTOM */
			snprintf(cmd, sizeof(cmd), "%s", fs->launch_mount.command);
		pipe_command(fs, cmd);
		}
	force_fs_check = FORCE_REDRAW;	/* An update triggers when pipe closes */
	}

static void
hide_monitors_in_drawer(void)
	{
	FSmon	*fs;
	GList	*list;

	gkrellm_freeze_side_frame_packing();
	for (list = fs_mon_list; list; list = list->next)
		{
		fs = (FSmon *) list->data;
		if (fs->in_drawer && (!fs_is_mounted(fs) || !fs->show_if_mounted))
			gkrellm_panel_hide(fs->panel);
		}
	gkrellm_thaw_side_frame_packing();
	}

static void
show_monitors_in_drawer(void)
	{
	FSmon	*fs;
	GList	*list;

	gkrellm_freeze_side_frame_packing();
	for (list = fs_mon_list; list; list = list->next)
		{
		fs = (FSmon *) list->data;
		if (fs->in_drawer)
			gkrellm_panel_show(fs->panel);
		}
	gkrellm_thaw_side_frame_packing();
	}

static void
update_fs(void)
	{
	FSmon	*fs;
	Panel	*p;
	GList	*list;
	gulong	l;
	gint	dx, index, w_scroll, w;
	gint	fs_check, nfs_check, next_check;

	if (fs_mon_list == NULL)
		return;

	w = w_scroll = 0;
	for (list = fs_mon_list; list; list = list->next)
		{
		fs = (FSmon *) list->data;
		if (fs->label_is_data && !fs_in_motion)
			{
			w = draw_fs_text_decal(fs, 1);
			if (w > w_scroll)
				w_scroll = w;
			gkrellm_draw_panel_layers(fs->panel);
			}
		}
	if (!fs_in_motion)
		{
		if (w_scroll > text_decal_width)
			x_scroll = (x_scroll + ((gkrellm_update_HZ() < 7) ? 2 : 1))
					% (w_scroll - text_decal_width / 3);
		else
			x_scroll = 0;
		}
	if (GK.second_tick)
		++check_tick;
	fs_check = (check_tick % fs_check_timeout) ? FALSE : TRUE;
	nfs_check = (check_tick % nfs_check_timeout) ? FALSE : TRUE;

	if (!force_fs_check && (!GK.second_tick || (!fs_check && !nfs_check)))
		return;
#if 0
printf("fs update %d nfs %d force %d\n", fs_check, nfs_check, force_fs_check);
#endif

	(*get_mountlist)();

	next_check = 0;

	for (list = fs_mon_list; list; list = list->next)
		{
		fs = (FSmon *) list->data;
		p  = fs->panel;
		KRELL(p)->previous = 0;
		if (fs_is_mounted(fs))
			{
			if (fs->enable_mounting)
				{	/* Blink it while pipe is open.	*/
				if (   fs->launch_mount.pipe
					&& fs->md_button->cur_index == D_MISC_FS_MOUNTED
				   )
					index = D_MISC_FS_UMOUNTED;
				else
					index = D_MISC_FS_MOUNTED;
				gkrellm_set_decal_button_index(fs->md_button, index);
				}
			else
				gkrellm_set_decal_button_index(fs->md_button, D_MISC_LED1);
			if (   force_fs_check == FORCE_UPDATE
				|| (fs_check && !fs->is_nfs_fs)
				|| (nfs_check && fs->is_nfs_fs)
			   )
				{
				(*get_fsusage)(fs);
				fs->krell_factor = fs->blocks > 2097152 ? 1024 : 1;
				}			
			KRELL(p)->full_scale = fs->blocks / fs->krell_factor;
			l = (fs->blocks - fs->bavail) / fs->krell_factor;
			if (   (fs_in_motion && fs->label_is_data && x_moved)
				|| fs->mouse_entered
			   )
				l = 0;
			gkrellm_update_krell(p, KRELL(p), l);
			if (fs->in_drawer && fs->show_if_mounted)
				gkrellm_panel_show(fs->panel);
			if (fs->eject_decal)
				{
				dx = x_eject_button_closed - fs->eject_decal->x;
				if (dx <= 0)
					gkrellm_hide_button(fs->eject_button);
				else if (dx > 0)
					{
					gkrellm_move_decal(p, fs->eject_decal,
						fs->eject_decal->x + 1 + dx / 4, fs->eject_decal->y);
					next_check = FORCE_REDRAW;
					}
				}
			}
		else
			{
			if (fs->enable_mounting)
				{	/* Blink it while pipe is open.	*/
				if (   fs->launch_mount.pipe
					&& fs->md_button->cur_index == D_MISC_FS_UMOUNTED
				   )
					index = D_MISC_FS_MOUNTED;
				else
					index = D_MISC_FS_UMOUNTED;
				gkrellm_set_decal_button_index(fs->md_button, index);
				}
			else
				gkrellm_set_decal_button_index(fs->md_button, D_MISC_LED0);
			KRELL(p)->full_scale = 100;		/* Arbitrary > 0 */
			gkrellm_update_krell(p, KRELL(p), 0);
			if (!fs_drawer_is_shown && fs->in_drawer)
				gkrellm_panel_hide(fs->panel);
			if (fs->eject_decal)
				{
				dx = fs->x_eject_button_target - fs->eject_decal->x;
				if (dx > 0)
					gkrellm_move_decal(p, fs->eject_decal,
						fs->eject_decal->x + 1 + dx / 4, fs->eject_decal->y);
				else if (dx < 0)
					gkrellm_move_decal(p, fs->eject_decal,
						fs->eject_decal->x - 1 + dx / 4, fs->eject_decal->y);
				if (fs->eject_decal->x < x_eject_button_closed)
					gkrellm_show_button(fs->eject_button);
				else
					gkrellm_hide_button(fs->eject_button);
				if (fs->eject_decal->x != fs->x_eject_button_target)
					next_check = FORCE_REDRAW;
				}
			}
		gkrellm_set_button_sensitive(fs->md_button,
						fs->enable_mounting ? TRUE : FALSE);
		if (!fs->label_is_data && fs != fs_in_motion)
			draw_fs_text_decal(fs, 0);

		gkrellm_draw_panel_layers(p);
		if (   fs->ejectable
			&& fs->eject_pending && fs->eject_pending < GK.timer_ticks
		   )
			{
			eject_tray(fs);
			fs->eject_pending = 0;
			}
		if (fs->launch_mount.pipe)
			{
			accumulate_pipe_gstring(fs);
			if (fs->launch_mount.pipe == NULL)	/* Command is done */
				{
				if (fs->pipe_gstring)
					{
					gkrellm_message_window(_("GKrellM Mount Error"),
							fs->pipe_gstring->str, fs->panel->drawing_area);
					g_string_free(fs->pipe_gstring, 1);
					fs->pipe_gstring = NULL;
					}
				next_check = FORCE_UPDATE;
				}
			else
				next_check = FORCE_REDRAW; 	/* Keep it going */
			}
		}
	force_fs_check = next_check;
	}

static gint
fs_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	FSmon	*fs;
	GList	*list;

	for (list = fs_mon_list; list; list = list->next)
		{
		fs = (FSmon *) list->data;
		if (widget == fs->panel->drawing_area)
			{
			gdk_draw_pixmap(widget->window, GK.draw1_GC, fs->panel->pixmap,
					ev->area.x, ev->area.y, ev->area.x, ev->area.y,
					ev->area.width, ev->area.height);
			break;
			}
		}
	return FALSE;
	}

static gint
cb_panel_enter(GtkWidget *w, GdkEventButton *ev, FSmon *fs)
	{
	if (fs->label_is_data)
		{
		fs->mouse_entered = TRUE;
		force_fs_check = FORCE_REDRAW;
		}
	if (fs->ejectable)
		{
		fs->x_eject_button_target = x_eject_button_open;
		force_fs_check = FORCE_REDRAW;
		}
	return TRUE;
	}

static gint
cb_panel_leave(GtkWidget *w, GdkEventButton *ev, FSmon *fs)
	{
	if (fs->mouse_entered)
		force_fs_check = FORCE_REDRAW;
	fs->mouse_entered = FALSE;
	if (fs->ejectable)
		{
		fs->x_eject_button_target = x_eject_button_closed;
		force_fs_check = FORCE_REDRAW;
		}
	return TRUE;
	}


static gint
cb_fs_panel_release(GtkWidget *widget, GdkEventButton *ev)
	{
	if (ev->button != 2)
		return TRUE;
	if (fs_in_motion /* && fs_in_motion->panel.drawing_area == widget */)
		{
		if (fs_in_motion->restore_label)
			{
			if (fs_in_motion->label_is_data)
				gkrellm_config_modified();
			fs_in_motion->label_is_data = FALSE;
			draw_fs_text_decal(fs_in_motion, 0);
			gkrellm_draw_panel_layers(fs_in_motion->panel);
			}
		fs_in_motion->restore_label = TRUE;
		}
	force_fs_check = FORCE_REDRAW;		/* Move krells back */
	fs_in_motion = NULL;
	x_moved = 0;
	return TRUE;
	}

static gint
cb_fs_panel_press(GtkWidget *widget, GdkEventButton *ev, FSmon *fs)
	{
	Decal	*d;

	d = fs->eject_decal ? fs->eject_decal : fs->mount_decal;
	if (ev->button == 3 && ev->x < d->x)
		{
		gkrellm_open_config_window(mon_fs);
		return TRUE;
		}
	if (ev->button == 1 && ev->x < d->x)
		{
		if (fs_drawer_is_shown)
			hide_monitors_in_drawer();
		else
			show_monitors_in_drawer();
		fs_drawer_is_shown = !fs_drawer_is_shown;
		}
	else if (fs->is_mounted && ev->button == 2)
		{
		if (!fs->label_is_data)
			{
			fs->label_is_data = TRUE;
			fs->restore_label = FALSE;
			fs->mouse_entered = TRUE;
			gkrellm_config_modified();
			}
		x_fs_motion = ev->x;
		draw_fs_text_decal(fs, 1);
		gkrellm_draw_panel_layers(fs->panel);
		fs_in_motion = fs;
		x_moved = 0;
		}
	return TRUE;
	}

static gint
cb_fs_panel_motion(GtkWidget *widget, GdkEventButton *ev)
	{
	GdkModifierType	state;
	GList			*list;
	FSmon			*fs;
	Decal			*d;
	GdkFont			*font;
	gchar			buf[128];
	gint			w, x_delta	= 0;

	state = ev->state;
	if (   !fs_in_motion
		|| !(state & (GDK_BUTTON2_MASK | GDK_BUTTON3_MASK))
		|| !fs_in_motion->label_is_data
	   )
		{
		fs_in_motion = NULL;
		return TRUE;
		}
	font = gkrellm_meter_alt_textstyle(style_id)->font;

	d = fs_in_motion->text_decal;
	format_fs_data(fs_in_motion, buf, sizeof(buf));
	w = gdk_string_width(font, buf);
	if (w > d->w)
		{
		x_delta = ev->x - x_fs_motion;
		x_fs_motion = ev->x;
		d->x_off += x_delta;
		if (d->x_off < -w)
			d->x_off = -w;
		if (d->x_off > d->w)
			d->x_off = d->w;
		x_scroll = d->w / 3 - d->x_off;
		for (list = fs_mon_list; list; list = list->next)
			{
			fs = (FSmon *) list->data;
			if (fs->label_is_data)
				{
				draw_fs_text_decal(fs, 1);
				gkrellm_draw_panel_layers(fs->panel);
				}
			}
		if (x_moved > 0)
			fs_in_motion->restore_label = FALSE;
		}
	if (x_moved == 0)
		force_fs_check = FORCE_REDRAW;	/* Move krells out of the way */
	x_moved += (x_delta > 0) ? x_delta : -x_delta;
	return TRUE;
	}

static void
cb_fs_mount_button(DecalButton *button)
	{
	if (button)
		mount_command((FSmon *) button->data);
	}

static void
cb_fs_eject_button(DecalButton *button, FSmon *fs)
	{
	if (button)
		eject_tray(fs);
	}

static void
cb_fs_close_tray(DecalButton *button, FSmon *fs)
	{
	if (button)
		close_tray(fs);
	}

static void
create_fs_monitor(GtkWidget *vbox, FSmon *fs, gint index, gint first_create)
	{
	Style		*style;
	Margin		*m;
	Panel		*p;

	if (first_create)
		fs->panel = gkrellm_panel_new0();
	p = fs->panel;
	fs->idx = index;
	++n_fs_monitors;
	fs->krell_factor = 1;

	style = gkrellm_meter_style(style_id);
	m = gkrellm_get_style_margins(style);
	fs->text_decal = gkrellm_create_decal_text(p, "Afp0",
			gkrellm_meter_textstyle(style_id), style, -1, -1, -1);

	fs->mount_decal = gkrellm_create_decal_pixmap(p, GK.decal_misc_pixmap,
			GK.decal_misc_mask, N_MISC_DECALS, style, -1, -1);
	fs->mount_decal->x =
				gkrellm_chart_width() - fs->mount_decal->w - m->right;

	if (fs->ejectable)
		{
		fs->eject_decal = gkrellm_create_decal_pixmap(p, GK.decal_misc_pixmap,
				GK.decal_misc_mask, N_MISC_DECALS, style, -1, -1);
		x_eject_button_closed = fs->mount_decal->x;
		x_eject_button_open = fs->mount_decal->x - fs->eject_decal->w + 1;
		fs->x_eject_button_target = x_eject_button_closed;
		fs->eject_decal->x = x_eject_button_closed;
		}

	/* Usable width to determine various scrolling parameters.
	*/
	text_decal_width = fs->mount_decal->x - fs->text_decal->x;

	gkrellm_create_krell(p, gkrellm_krell_meter_image(style_id), style);

	gkrellm_panel_configure(p, NULL, style);
	gkrellm_panel_create(vbox, mon_fs, p);

	fs->md_button = gkrellm_make_decal_button(p, fs->mount_decal,
			cb_fs_mount_button, fs, D_MISC_FS_UMOUNTED, D_MISC_FS_PRESSED);
	if (fs->eject_decal)
		{
		fs->eject_button = gkrellm_make_decal_button(p, fs->eject_decal,
			cb_fs_eject_button, fs, D_MISC_BUTTON_OUT, D_MISC_BUTTON_IN);
		gkrellm_hide_button(fs->eject_button);
		if (close_tray_command)
			gkrellm_decal_button_right_connect(fs->eject_button,
						cb_fs_close_tray, fs);
		}
	if (first_create)
		{
		gtk_signal_connect(GTK_OBJECT(p->drawing_area), "expose_event",
				(GtkSignalFunc) fs_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),"button_press_event",
				(GtkSignalFunc) cb_fs_panel_press, fs);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),"button_release_event",
				(GtkSignalFunc) cb_fs_panel_release, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),"motion_notify_event",
				(GtkSignalFunc) cb_fs_panel_motion, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area), "enter_notify_event",
				(GtkSignalFunc) cb_panel_enter, fs);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area), "leave_notify_event",
				(GtkSignalFunc) cb_panel_leave, fs);
		if (   !fs_drawer_is_shown && fs->in_drawer
			&& (!fs_is_mounted(fs) || !fs->show_if_mounted))
			gkrellm_panel_hide(fs->panel);
		}

	draw_fs_text_decal(fs, 0);
	force_fs_check = FORCE_UPDATE;

	if (fs->launch_mount.command == NULL)
		fs->launch_mount.command = g_strdup("");
	if (fs->launch_umount.command == NULL)
		fs->launch_umount.command = g_strdup("");
	}

static void
destroy_fs_monitor(FSmon *fs)
	{
	g_free(fs->label);
	g_free(fs->mount.directory);
	g_free(fs->eject_device);
	gkrellm_destroy_button(fs->md_button);
	gkrellm_destroy_button(fs->eject_button);
	gkrellm_panel_destroy(fs->panel);
	g_free(fs);
	--n_fs_monitors;
	}

static void
create_fs(GtkWidget *vbox, gint first_create)
	{
	GList		*list;
	FSmon		*fs;
	gint		i;

	if (fs_main_vbox == NULL)
		{
		fs_main_vbox = gtk_vbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox), fs_main_vbox, FALSE, FALSE, 0);
		gtk_widget_show(fs_main_vbox);

		fs_drawer_vbox = gtk_vbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox), fs_drawer_vbox, FALSE, FALSE, 0);
		gtk_widget_show(fs_drawer_vbox);
		fs_drawer_is_shown = FALSE;
		}
	n_fs_monitors = 0;
	for (i = 0, list = fs_mon_list; list; ++i, list = list->next)
		{
		fs = (FSmon *)list->data;
		create_fs_monitor(fs->in_drawer ? fs_drawer_vbox : fs_main_vbox,fs,
					i, first_create);
		}
	if (g_list_length(fs_mon_list) == 0)
		hide_spacers(mon_fs);
	}

#define	FS_CONFIG_KEYWORD	"fs"

static void
save_fs_config(FILE *f)
	{
	GList	*list;
	FSmon	*fs;

	for (list = fs_mon_list; list; list = list->next)
		{
		fs = (FSmon *) list->data;
		fprintf(f, "%s \"%s\" %s %d %d %d %d %d\n", FS_CONFIG_KEYWORD,
				fs->label, fs->mount.directory,
				fs->enable_mounting, fs->in_drawer,
				fs->show_if_mounted, fs->label_is_data, fs->ejectable);
		if (*(fs->launch_mount.command))
			fprintf(f, "%s mount_command %s\n", FS_CONFIG_KEYWORD,
					fs->launch_mount.command);
		if (*(fs->launch_umount.command))
			fprintf(f, "%s umount_command %s\n", FS_CONFIG_KEYWORD,
					fs->launch_umount.command);
		if (*(fs->eject_device))
			fprintf(f, "%s eject_device %s\n", FS_CONFIG_KEYWORD,
					fs->eject_device);
		}
	fprintf(f, "%s fs_check_timeout %d\n", FS_CONFIG_KEYWORD,
			fs_check_timeout);
	fprintf(f, "%s nfs_check_timeout %d\n", FS_CONFIG_KEYWORD,
			nfs_check_timeout);
	fprintf(f, "%s auto_eject %d\n", FS_CONFIG_KEYWORD, cdrom_auto_eject);
	fprintf(f, "%s binary_units %d\n", FS_CONFIG_KEYWORD, binary_units);
	fprintf(f, "%s data_format %s\n", FS_CONFIG_KEYWORD, data_format);
	}

static gboolean
fstab_user_permission(Mount *m)
	{
	struct stat my_stat;
	
	stat(m->device, &my_stat);
	if (   strstr(m->options, "user")
		|| (strstr(m->options, "owner") && my_stat.st_uid == uid)
	   )
		return TRUE;
	return FALSE;
	}

static gint
fix_fstab_mountable_changed(FSmon *fs)
	{
	Mount	*m;

	m = in_fstab_list(fs->mount.directory);
	if (   (!m || (!fstab_user_permission(m) && uid != 0))
		&& fs->enable_mounting == MT_FSTAB
	   )
		{
		fs->enable_mounting = MT_NONE;
		return TRUE;
		}		
	return FALSE;
	}

static void
load_fs_config(gchar *arg)
	{
	static FSmon	*fs_prev;
	FSmon			*fs;
	gchar			*label;
	gchar			config[32], item[CFG_BUFSIZE], dir[CFG_BUFSIZE];
	gint 			n;

	if ((n = sscanf(arg, "%31s %[^\n]", config, item)) != 2)
		return;

	if (!strcmp(config, "fs_check_timeout"))
		{
		sscanf(item, "%d", &fs_check_timeout);
		if (fs_check_timeout < 2)
			fs_check_timeout = 2;
		}
	else if (!strcmp(config, "nfs_check_timeout"))
		{
		sscanf(item, "%d", &nfs_check_timeout);
		if (nfs_check_timeout < 5)
			nfs_check_timeout = 5;
		}
	else if (!strcmp(config, "auto_eject"))
		sscanf(item, "%d", &cdrom_auto_eject);
	else if (!strcmp(config, "binary_units"))
		sscanf(item, "%d", &binary_units);
	else if (!strcmp(config, "data_format"))
		gkrellm_dup_string(&data_format, item);
	else if (fs_prev && !strcmp(config, "mount_command"))
		gkrellm_dup_string(&fs_prev->launch_mount.command, item);
	else if (fs_prev && !strcmp(config, "umount_command"))
		gkrellm_dup_string(&fs_prev->launch_umount.command, item);
	else if (fs_prev && !strcmp(config, "eject_device"))
		{
		if (fs_prev->ejectable)
			gkrellm_dup_string(&fs_prev->eject_device, item);
		}
	else
		{
		if ((label = cut_quoted_string(arg, &arg)) != NULL)
			{
			fs = g_new0(FSmon, 1);
			fs->label = g_strdup(label);
			dir[0] = '\0';

			sscanf(arg, "%s %d %d %d %d %d",  dir, &fs->enable_mounting,
					&fs->in_drawer, &fs->show_if_mounted,
					&fs->label_is_data, &fs->ejectable);
			if (!eject_tray_command && !eject_cdrom)
				fs->ejectable = FALSE;
			fs->mount.directory = g_strdup(dir);
			fs->restore_label = fs->label_is_data;

			fix_fstab_mountable_changed(fs);
			fs->krell_factor = 1;
			fs->launch_mount.command = g_strdup("");
			fs->launch_umount.command = g_strdup("");
			fs->eject_device = g_strdup("");
			fs_mon_list = g_list_append(fs_mon_list, fs);
			fs_prev = fs;	/* XXX */
			}
		}
	}


/* --------------------------------------------------------------------- */
typedef struct
	{
	GtkWidget	*clist;
	GtkWidget	*label_entry,
				*dir_combo,
				*mount_entry,
				*umount_entry,
				*mounting_button,
				*ejectable_button,
				*device_entry,
				*show_button;
	gint		selected_row;
	}
	DrawerWidgets;

static DrawerWidgets
				primary_widgets,
				secondary_widgets;

static GtkWidget	*data_format_combo;

static GtkWidget	*fs_check_spin_button,
					*nfs_check_spin_button;

static GtkWidget	*cdrom_auto_eject_button,
					*binary_units_button;

static gint			fs_list_modified;



  /* Watch what is going into the directory combo entry, compare it to
  |  fstab entries and accordingly set sensitivity of the mounting_button.
  */
static void
cb_combo_changed(GtkWidget *widget, DrawerWidgets *drawer)
	{
	Mount		*m;
	gchar		*s;

	s = gkrellm_entry_get_text(&(GTK_COMBO(drawer->dir_combo)->entry));
	m = in_fstab_list(s);
	if (m && (fstab_user_permission(m) || uid == 0))
		{
		gtk_widget_set_sensitive(drawer->mounting_button, TRUE);
		if (GTK_TOGGLE_BUTTON(drawer->mounting_button)->active)
			{
			gtk_entry_set_text(GTK_ENTRY(drawer->mount_entry), "");
			gtk_entry_set_text(GTK_ENTRY(drawer->umount_entry), "");
			gtk_widget_set_sensitive(drawer->mount_entry, FALSE);
			gtk_widget_set_sensitive(drawer->umount_entry, FALSE);
			}
		}
	else
		{
		if (GTK_TOGGLE_BUTTON(drawer->mounting_button)->active)
			gtk_toggle_button_set_active(
						GTK_TOGGLE_BUTTON(drawer->mounting_button), FALSE);
		else
			{
			gtk_widget_set_sensitive(drawer->mount_entry, TRUE);
			gtk_widget_set_sensitive(drawer->umount_entry, TRUE);
			}
		gtk_widget_set_sensitive(drawer->mounting_button, FALSE);
		}
	}

static void
cb_mount_button_clicked(GtkWidget *widget, DrawerWidgets *drawer)
	{
	if (GTK_TOGGLE_BUTTON(drawer->mounting_button)->active)
		{
		gtk_entry_set_text(GTK_ENTRY(drawer->mount_entry), "");
		gtk_entry_set_text(GTK_ENTRY(drawer->umount_entry), "");
		gtk_widget_set_sensitive(drawer->mount_entry, FALSE);
		gtk_widget_set_sensitive(drawer->umount_entry, FALSE);
		if (drawer->device_entry)
			{
			gtk_entry_set_text(GTK_ENTRY(drawer->device_entry), "");
			gtk_widget_set_sensitive(drawer->device_entry, FALSE);
			}
		}
	else
		{
		gtk_widget_set_sensitive(drawer->mount_entry, TRUE);
		gtk_widget_set_sensitive(drawer->umount_entry, TRUE);
		if (   drawer->ejectable_button
			&& GTK_TOGGLE_BUTTON(drawer->ejectable_button)->active
		   )
			gtk_widget_set_sensitive(drawer->device_entry, TRUE);
		}
	}

static void
cb_ejectable_button_clicked(GtkWidget *widget, DrawerWidgets *drawer)
	{
	gboolean	fstab_mounting;

	fstab_mounting = GTK_TOGGLE_BUTTON(drawer->mounting_button)->active;
	if (GTK_TOGGLE_BUTTON(drawer->ejectable_button)->active)
		{
		gtk_widget_set_sensitive(drawer->device_entry, !fstab_mounting);
		}
	else
		{
		gtk_entry_set_text(GTK_ENTRY(drawer->device_entry), "");
		gtk_widget_set_sensitive(drawer->device_entry, FALSE);
		}
	}

static void
reset_fs_entries(DrawerWidgets *drawer)
	{
	gtk_entry_set_text(GTK_ENTRY(drawer->label_entry), "");
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(drawer->dir_combo)->entry), "");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(drawer->mounting_button),
				FALSE);
	if (drawer->show_button)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(drawer->show_button),
					FALSE);
	if (drawer->ejectable_button)
		{
		gtk_toggle_button_set_active(
					GTK_TOGGLE_BUTTON(drawer->ejectable_button), FALSE);
		gtk_entry_set_text(GTK_ENTRY(drawer->device_entry), "");
		}
	gtk_entry_set_text(GTK_ENTRY(drawer->mount_entry), "");
	gtk_entry_set_text(GTK_ENTRY(drawer->umount_entry), "");
	}

  /* Return a pointer to a static FSmon struct initialized with text values
  |  from the config clist.  Caller must dup all text strings to be saved
  |  and must not free any from the static struct.
  */
static FSmon *
get_clist_fs_values(DrawerWidgets *drawer, gint row)
	{
	GtkWidget		*clist;
	static FSmon	fs;
	gchar			*s;
	gint			i;

	clist = drawer->clist;
	i = 0;
	gtk_clist_get_text(GTK_CLIST(clist), row, i++, &fs.label);
	gtk_clist_get_text(GTK_CLIST(clist), row, i++, &fs.mount.directory);

	if (drawer->show_button)
		{
		gtk_clist_get_text(GTK_CLIST(clist), row, i++, &s);
		fs.show_if_mounted = !strcmp(s, "*") ? TRUE : FALSE;
		}
	gtk_clist_get_text(GTK_CLIST(clist), row, i++, &s);
	fs.enable_mounting = !strcmp(s, "*") ? MT_FSTAB : MT_NONE;

	gtk_clist_get_text(GTK_CLIST(clist), row, i++, &fs.launch_mount.command);
	gtk_clist_get_text(GTK_CLIST(clist), row, i++, &fs.launch_umount.command);

	gtk_clist_get_text(GTK_CLIST(clist), row, i++, &s);
	fs.ejectable = *s ? TRUE : FALSE;
	if (fs.ejectable && strcmp(s, "*"))
		fs.eject_device = s;
	else
		fs.eject_device = "";

	return &fs;
	}

static void
cb_fs_clist_selected(GtkWidget *clist, gint row, gint column,
			GdkEventButton *bevent, DrawerWidgets *drawer)
	{
	FSmon		*fs;

	fs = get_clist_fs_values(drawer, row);

	gtk_entry_set_text(GTK_ENTRY(drawer->label_entry), fs->label);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(drawer->dir_combo)->entry),
				fs->mount.directory);
	if (drawer->show_button)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(drawer->show_button),
					fs->show_if_mounted);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(drawer->mounting_button),
				fs->enable_mounting);
	if (drawer->ejectable_button)
		{
		gtk_toggle_button_set_active(
				GTK_TOGGLE_BUTTON(drawer->ejectable_button), fs->ejectable);
		gtk_entry_set_text(GTK_ENTRY(drawer->device_entry), fs->eject_device);
		if (!fs->ejectable)
			gtk_widget_set_sensitive(drawer->device_entry, FALSE);
		}
	gtk_entry_set_text(GTK_ENTRY(drawer->mount_entry),
				fs->launch_mount.command);
	gtk_entry_set_text(GTK_ENTRY(drawer->umount_entry),
				fs->launch_umount.command);
		
	drawer->selected_row = row;
	}

static void
cb_fs_clist_unselected(GtkWidget *clist, gint row, gint column,
			GdkEventButton *bevent, DrawerWidgets *drawer)
	{
	reset_fs_entries(drawer);
	drawer->selected_row = -1;
	}

static void
cb_fs_clist_up(GtkWidget *widget, DrawerWidgets *drawer)
	{
	GtkWidget	*clist;
	gint		row;

	clist = drawer->clist;
	row = drawer->selected_row;
	if (row > 0)
		{
		gtk_clist_row_move(GTK_CLIST(clist), row, row - 1);
		gtk_clist_select_row(GTK_CLIST(clist), row - 1, -1);
		if (gtk_clist_row_is_visible(GTK_CLIST(clist), row - 1)
					!= GTK_VISIBILITY_FULL)
			gtk_clist_moveto(GTK_CLIST(clist), row - 1, -1, 0.0, 0.0);
		drawer->selected_row = row - 1;
		fs_list_modified = TRUE;
		}
	}

static void
cb_fs_clist_down(GtkWidget *widget, DrawerWidgets *drawer)
	{
	GtkWidget	*clist;
	gint		row;

	clist = drawer->clist;
	row = drawer->selected_row;
	if (row >= 0 && row < GTK_CLIST(clist)->rows - 1)
		{
		gtk_clist_row_move(GTK_CLIST(clist), row, row + 1);
		gtk_clist_select_row(GTK_CLIST(clist), row + 1, -1);
		if (gtk_clist_row_is_visible(GTK_CLIST(clist), row + 1)
					!= GTK_VISIBILITY_FULL)
			gtk_clist_moveto(GTK_CLIST(clist), row + 1, -1, 1.0, 0.0);
		drawer->selected_row = row + 1;
		fs_list_modified = TRUE;
		}
	}

static void
cb_enter_fs(GtkWidget *widget, DrawerWidgets *drawer)
	{
	gchar		*buf[8];
	gint		i, n, m;
	gboolean	enable_mounting, ejectable = FALSE;

	i = 0;
	buf[i++] = gkrellm_entry_get_text(&drawer->label_entry);
	buf[i++] = gkrellm_entry_get_text(&(GTK_COMBO(drawer->dir_combo)->entry));
	if (drawer->show_button)
		buf[i++] = GTK_TOGGLE_BUTTON(drawer->show_button)->active ? "*" : "";
	enable_mounting = GTK_TOGGLE_BUTTON(drawer->mounting_button)->active;
	buf[i++] = enable_mounting ? "*" : "";
	n = i;
	buf[i++] = gkrellm_entry_get_text(&drawer->mount_entry);
	buf[i++] = gkrellm_entry_get_text(&drawer->umount_entry);
	m = i;
	if (drawer->ejectable_button)
		{
		ejectable = GTK_TOGGLE_BUTTON(drawer->ejectable_button)->active;
		if (ejectable)
			buf[i++] = enable_mounting ?
						"*" : gkrellm_entry_get_text(&drawer->device_entry);
		else
			buf[i++] = "";
		}
	buf[i] = NULL;
	if (!*(buf[0]) || !*(buf[1]))	/* validate we have input */
		{
		gkrellm_config_message_window(_("Entry Error"),
			_("Both a label and a mount point must be entered."), widget);
		return;
		}
	if ((*(buf[n]) && !*(buf[n+1])) || (!*(buf[n]) && *(buf[n+1])))
		{
		gkrellm_config_message_window(_("Entry Error"),
			_("Both mount and umount commands must be entered."), widget);
		return;
		}
	if (ejectable && !enable_mounting && !*(buf[m]))
		{
		gkrellm_config_message_window(_("Entry Error"),
			_("Missing ejectable device entry."), widget);
		return;
		}
	if (drawer->selected_row >= 0)
		{
		for (i = 0; i < (drawer->show_button ? 7 : 6); ++i)
			gtk_clist_set_text(GTK_CLIST(drawer->clist), drawer->selected_row,
						i, buf[i]);
		gtk_clist_unselect_row(GTK_CLIST(drawer->clist), drawer->selected_row,
						0);
		drawer->selected_row = -1;
		}
	else
		gtk_clist_append(GTK_CLIST(drawer->clist), buf);
	reset_fs_entries(drawer);
	fs_list_modified = TRUE;
	}

static void
cb_delete_fs(GtkWidget *widget, DrawerWidgets *drawer)
	{
	reset_fs_entries(drawer);
	if (drawer->selected_row >= 0)
		{
		gtk_clist_remove(GTK_CLIST(drawer->clist), drawer->selected_row);
		fs_list_modified = TRUE;
		drawer->selected_row = -1;
		}
	}

static void
apply_fs_config(void)
	{
	GList			*list;
	FSmon			*fs, *fs_tmp;
	gchar			*s;
	gint			row;
	GtkSpinButton	*spin;

	if (cdrom_auto_eject_button)
		cdrom_auto_eject = GTK_TOGGLE_BUTTON(cdrom_auto_eject_button)->active;
	binary_units = GTK_TOGGLE_BUTTON(binary_units_button)->active;
	s = gkrellm_entry_get_text(&(GTK_COMBO(data_format_combo)->entry));
	if (gkrellm_dup_string(&data_format, s))
		for (list = fs_mon_list; list; list = list->next)
			((FSmon *) list->data)->text_decal->value = -1;	/* Force redraw */

	spin = GTK_SPIN_BUTTON(fs_check_spin_button);
	fs_check_timeout = gtk_spin_button_get_value_as_int(spin);
	spin = GTK_SPIN_BUTTON(nfs_check_spin_button);
	nfs_check_timeout = gtk_spin_button_get_value_as_int(spin);

	if (!fs_list_modified)
		return;
	for (list = fs_mon_list; list; list = list->next)
		destroy_fs_monitor((FSmon *) list->data);
	g_list_free(fs_mon_list);
	fs_mon_list = NULL;
	n_fs_monitors = 0;

	for (row = 0; row < GTK_CLIST(primary_widgets.clist)->rows; ++row)
		{
		fs = g_new0(FSmon, 1);
		fs_tmp = get_clist_fs_values(&primary_widgets, row);
		gkrellm_dup_string(&fs->label, fs_tmp->label);
		gkrellm_dup_string(&fs->mount.directory, fs_tmp->mount.directory);
		fs->enable_mounting = fs_tmp->enable_mounting;
		fs->ejectable = fs_tmp->ejectable;
		gkrellm_dup_string(&fs->launch_mount.command,
					fs_tmp->launch_mount.command);
		gkrellm_dup_string(&fs->launch_umount.command,
					fs_tmp->launch_umount.command);
		gkrellm_dup_string(&fs->eject_device, fs_tmp->eject_device);

		if (*(fs->launch_umount.command))
			fs->enable_mounting = MT_CUSTOM;

		fs->in_drawer = FALSE;
		fs_mon_list = g_list_append(fs_mon_list, fs);
		create_fs_monitor(fs_main_vbox, fs, n_fs_monitors, TRUE);
		}

	for (row = 0; row < GTK_CLIST(secondary_widgets.clist)->rows; ++row)
		{
		fs = g_new0(FSmon, 1);
		fs_tmp = get_clist_fs_values(&secondary_widgets, row);
		gkrellm_dup_string(&fs->label, fs_tmp->label);
		gkrellm_dup_string(&fs->mount.directory, fs_tmp->mount.directory);
		fs->show_if_mounted = fs_tmp->show_if_mounted;
		fs->enable_mounting = fs_tmp->enable_mounting;
		fs->ejectable = fs_tmp->ejectable;
		gkrellm_dup_string(&fs->launch_mount.command,
					fs_tmp->launch_mount.command);
		gkrellm_dup_string(&fs->launch_umount.command,
					fs_tmp->launch_umount.command);
		gkrellm_dup_string(&fs->eject_device, fs_tmp->eject_device);

		if (*(fs->launch_umount.command))
			fs->enable_mounting = MT_CUSTOM;

		fs->in_drawer = TRUE;
		fs_mon_list = g_list_append(fs_mon_list, fs);
		create_fs_monitor(fs_drawer_vbox, fs, n_fs_monitors, TRUE);
		}
	fs_list_modified = FALSE;
	force_fs_check = FORCE_UPDATE;
	if (g_list_length(fs_mon_list) > 0)
		show_spacers(mon_fs);
	else
		hide_spacers(mon_fs);
	}

static gchar *fs_title1[6] =
	{ N_("Label"), N_("Mount Point"), N_("fstab"),
		N_("mount Command"), N_("umount Command"), N_("Ejectable")};

static gchar *fs_title2[7] =
	{ N_("Label"), N_("Mount Point"), N_("Show"), N_("fstab"),
		N_("mount Command"), N_("umount Command"), N_("Ejectable")};

static void
fill_fs_tab(GtkWidget *vbox, DrawerWidgets *drawer)
	{
	GtkWidget	*clist;
	GtkWidget	*table, *table1;
	GtkWidget	*hbox, *vbox1;
	GtkWidget	*frame;
	GtkWidget	*label;
	GtkWidget	*button;
	GtkWidget	*arrow;
	GtkWidget	*scrolled;
	GList		*list, *combo_list;
	FSmon		*fs;
	gchar		*buf[8];
	gint		row, in_drawer, i, j;

	table = gtk_table_new(4, 2, FALSE);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox, 0, 1, 0, 1,
				GTK_SHRINK, GTK_SHRINK, 0, 0);
	label = gtk_label_new(_("Label"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
	drawer->label_entry = gtk_entry_new_with_max_length(16);
	gtk_widget_set_usize(drawer->label_entry, 100, 0);
	gtk_box_pack_start(GTK_BOX(hbox), drawer->label_entry, FALSE, FALSE, 2);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), hbox, 1, 2, 0, 1);
	label = gtk_label_new(_("Mount Point"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
	drawer->dir_combo = gtk_combo_new();
	gtk_box_pack_start(GTK_BOX(hbox), drawer->dir_combo, TRUE, TRUE, 2);
	combo_list = NULL;
	for (list = fstab_list; list; list = list->next)
		combo_list = g_list_append(combo_list,
					((Mount *)list->data)->directory);
	gtk_combo_set_popdown_strings( GTK_COMBO(drawer->dir_combo), combo_list);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(drawer->dir_combo)->entry), "");
	gtk_signal_connect(GTK_OBJECT(GTK_COMBO(drawer->dir_combo)->entry),
				"changed", GTK_SIGNAL_FUNC (cb_combo_changed), drawer);

	if (drawer == &secondary_widgets)
		{
		vbox1 = gtk_vbox_new(FALSE, 0);
		gtk_table_attach(GTK_TABLE(table), vbox1, 0, 1, 1, 2,
					GTK_SHRINK, GTK_SHRINK, 0, 0);
		gkrellm_check_button_connected(vbox1, &drawer->show_button, 0,
					TRUE, TRUE, 0, NULL, NULL,
					_("Show if mounted"));
		}
	vbox1 = gtk_vbox_new(FALSE, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), vbox1, 1, 2, 1, 2);
	gkrellm_check_button_connected(vbox1, &drawer->mounting_button, 0,
				FALSE, FALSE, 0, cb_mount_button_clicked, drawer,
				_("Enable /etc/fstab mounting"));
	gtk_widget_set_sensitive(drawer->mounting_button, FALSE);

	if (eject_tray_command || eject_cdrom)
		{
		frame = gtk_frame_new(NULL);
		gtk_container_set_border_width(GTK_CONTAINER(frame), 2);
		vbox1 = gtk_vbox_new(FALSE, 0);
		gtk_container_add(GTK_CONTAINER(frame), vbox1);
		gtk_table_attach(GTK_TABLE(table), frame, 0, 1, 2, 3,
					GTK_SHRINK, GTK_SHRINK, 0, 0);
		gkrellm_check_button_connected(vbox1, &drawer->ejectable_button, 0,
					FALSE, FALSE, 0, cb_ejectable_button_clicked, drawer,
					_("Ejectable"));

		hbox = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);
		label = gtk_label_new(_("Device"));
		gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
		drawer->device_entry = gtk_entry_new_with_max_length(64);
		gtk_widget_set_usize(drawer->device_entry, 100, 0);
		gtk_box_pack_start(GTK_BOX(hbox), drawer->device_entry,
				FALSE, FALSE, 2);
		}

	frame = gtk_frame_new(NULL);
	gtk_container_set_border_width(GTK_CONTAINER(frame), 2);
	vbox1 = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox1);
	gtk_table_attach_defaults(GTK_TABLE(table), frame, 1, 2, 2, 3);

	table1 = gkrellm_launcher_table_new(vbox1, 1);	/* Won't have tooltip */
	gkrellm_config_launcher(table1, 0, &drawer->mount_entry,
				NULL, _("mount"), NULL);
	gkrellm_config_launcher(table1, 1, &drawer->umount_entry,
				NULL, _("umount"), NULL);

	vbox1 = gtk_vbox_new(FALSE, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), vbox1, 1, 2, 3, 4);
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 3);

	button = gtk_button_new();
	arrow = gtk_arrow_new(GTK_ARROW_UP, GTK_SHADOW_ETCHED_OUT);
	gtk_container_add(GTK_CONTAINER(button), arrow);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
				(GtkSignalFunc) cb_fs_clist_up, drawer);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 4);

	button = gtk_button_new();
	arrow = gtk_arrow_new(GTK_ARROW_DOWN, GTK_SHADOW_ETCHED_OUT);
	gtk_container_add(GTK_CONTAINER(button), arrow);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
				(GtkSignalFunc) cb_fs_clist_down, drawer);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 4);

	gkrellm_button_connected(hbox, NULL, TRUE, TRUE, 4,
			cb_enter_fs, drawer, _("Enter"));

	gkrellm_button_connected(hbox, NULL, TRUE, TRUE, 4,
			cb_delete_fs, drawer, _("Delete"));

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
	if (drawer->show_button)
		{
	    for(j = 0; j < 7; j++)
			fs_title2[j] = _(fs_title2[j]);
		drawer->clist = gtk_clist_new_with_titles(7, fs_title2);
		}
	else
		{
	    for(j = 0; j < 6; j++)
			fs_title1[j] = _(fs_title1[j]);
		drawer->clist = gtk_clist_new_with_titles(6, fs_title1);
		}
	clist = drawer->clist;
	gtk_clist_set_shadow_type(GTK_CLIST(clist), GTK_SHADOW_OUT);
	gtk_clist_set_column_width(GTK_CLIST(clist), 0, 60);
	gtk_clist_set_column_width(GTK_CLIST(clist), 1, 100);
	gtk_clist_set_column_justification(GTK_CLIST(clist), 2,GTK_JUSTIFY_CENTER);
	gtk_clist_set_column_width(GTK_CLIST(clist), 4, 200);
	if (drawer->show_button)
		{
		gtk_clist_set_column_width(GTK_CLIST(clist), 5, 200);
		gtk_clist_set_column_width(GTK_CLIST(clist), 6, 100);
		gtk_clist_set_column_justification(GTK_CLIST(clist), 3,
					GTK_JUSTIFY_CENTER);
		}
	else
		{
		gtk_clist_set_column_width(GTK_CLIST(clist), 3, 200);
		gtk_clist_set_column_width(GTK_CLIST(clist), 5, 100);
		}
	gtk_signal_connect (GTK_OBJECT(clist), "select_row",
			(GtkSignalFunc) cb_fs_clist_selected, drawer);
	gtk_signal_connect (GTK_OBJECT(clist), "unselect_row",
			(GtkSignalFunc) cb_fs_clist_unselected, drawer);
	gtk_container_add (GTK_CONTAINER (scrolled), clist);
	in_drawer = (drawer == &secondary_widgets) ? TRUE : FALSE;
		
	for (list = fs_mon_list; list; list = list->next)
		{
		fs = (FSmon *) list->data;
		if (fs->in_drawer == in_drawer)
			{
			i = 0;
			buf[i++] = fs->label;
			buf[i++] = fs->mount.directory;
			if (drawer->show_button)
				buf[i++] = fs->show_if_mounted ? "*" : "";
			if (fix_fstab_mountable_changed(fs))
				fs_list_modified = TRUE;
			buf[i++] = (fs->enable_mounting == MT_FSTAB) ? "*" : "";
			buf[i++] = fs->launch_mount.command;
			buf[i++] = fs->launch_umount.command;
			if (fs->ejectable)
				buf[i++] = (fs->enable_mounting == MT_FSTAB)
							? "*" : fs->eject_device;
			else
				buf[i++] = "";
			buf[i] = NULL;
			row = gtk_clist_append(GTK_CLIST(clist), buf);
			gtk_clist_set_row_data(GTK_CLIST(clist), row, fs);
			}
		}
	}


static gchar *fs_info_text[] =
{
N_("Enter file system mount points to monitor and enter a label to describe\n"
"the mount point.  The krell shows the ratio of blocks used to total blocks\n"
"available.  Mounting commands can be enabled for mount points in one\n"
"of two ways:\n\n"),

N_("<b>\t1) Mounting using /etc/fstab: "),

N_("If a mount point is in your\n"
"\t/etc/fstab and you have mount permission then mount and umount\n"
"\tcommands can be enabled and executed for that mount point simply\n"
"\tby checking the \"Enable /etc/fstab mounting\" option.\n"
"\tMount table entries in /etc/fstab need the \"user\" or \"owner\" option\n"
"\tset to grant this permission unless GKrellM is run as root.\n"
"\tFor example, if you run GKrellM as a normal user and you want\n"
"\tto be able to mount your floppy, your /etc/fstab could have\n"
"\teither of:\n"),

N_("<i>\t\t/dev/fd0 /mnt/floppy ext2 user,noauto,rw,exec 0 0\n"),
N_("\tor\n"),
N_("<i>\t\t/dev/fd0 /mnt/floppy ext2 user,defaults 0 0\n\n"),

N_("<b>\t2) Mounting using custom commands: "),
N_("If GKrellM is run as root or if\n"
"\tyou have sudo permission to run the mount commands, then a custom\n"
"\tmount command can be entered into the \"mount command\" entry\n"
"\tbox.  A umount command must also be entered if you choose this\n"
"\tmethod.  Example mount and umount entries using sudo:\n"),

N_("<i>\t\tsudo /bin/mount -t msdos /dev/fd0 /mnt/A\n"),
N_("<i>\t\tsudo /bin/umount /mnt/A\n"),

N_("\tNotes: the mount point specified in a custom mount command\n"
   "\t(/mnt/A in this example) must be the same as entered in the\n"
   "\t\"Mount Point\" entry.  You should have the NOPASSWD option set\n"
   "\tin /etc/sudoers if using sudo.\n\n"),

N_("<b>Primary and Secondary Monitors\n"), /* xgettext:no-c-format */
N_("File system monitors can be created as primary (always visible)\n"
	"or secondary which can be hidden and then shown when they are of\n"
	"interest.  For example, you might make primary file system monitors\n"
	"for root, home, or user so they will be always visible, but make\n"
	"secondary monitors for less frequently used mount points such as\n"
	"floppy, zip, backup partitions, foreign file system types, etc.\n"
	"Secondary FS monitors can also be configured to always be visible if they\n"
	"are mounted by checking the \"Show if mounted\" option.   Using this\n"
	"feature you can show the secondary group, mount a file system, and have\n"
	"that FS monitor remain visible even when the secondary group is hidden.\n"
	"A standard cdrom mount will show as 100% full but a monitor for it\n"
	"could be created with mounting enabled just to have the\n"
	"mount/umount convenience.\n\n"),

N_("<b>Panel Labels\n"),
N_("Substitution variables for the format string for file system panels:\n"),
N_("\t$t    total capacity\n"),
N_("\t$u    used space\n"),
N_("\t$f    free space\n"),
N_("\t$U    used %,\n"),
N_("\t$F    free %\n"),
N_("\t$l    the panel label\n"),
"\n",
N_("<b>Mouse Button Actions:\n"),
N_("<b>\tLeft "),
N_("click on a panel label to show/hide secondary fs monitors.\n"),
N_("<b>\tLeft "),
N_("click on a panel button to apply mount/umount commands (if enabled)\n"
	"\t\tto the mount point being monitored.\n"),
N_("<b>\tMiddle "),
N_("click on a panel to scroll a programmable display\n"),
N_("\t\tof file system capacities (default is total and free space).\n\n"),

N_("At least one primary fs monitor must exist to click on in order to show\n"
"secondary ones.\n")
};

static void
create_fs_tab(GtkWidget *tab_vbox)
	{
	GtkWidget		*tabs;
	GtkWidget		*vbox, *vbox1, *hbox;
	GtkWidget		*text;
	GtkWidget		*label;
	GList			*list;
	gint			i;

	fs_list_modified = FALSE;
	(*get_fstab_list)();

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

	vbox = gkrellm_create_framed_tab(tabs, _("Primary"));
	fill_fs_tab(vbox, &primary_widgets);

	vbox = gkrellm_create_framed_tab(tabs, _("Secondary"));
	fill_fs_tab(vbox, &secondary_widgets);

/* --Setup tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Setup"));

	vbox1 = gkrellm_framed_vbox(vbox, _("Options"), 4, FALSE, 0, 2);
	gkrellm_check_button(vbox1, &binary_units_button, binary_units, FALSE, 0,
	_("Use binary units (MiB, GiG) for reporting disk capacities."));

	if (eject_tray_command || eject_cdrom)
		gkrellm_check_button(vbox1, &cdrom_auto_eject_button, cdrom_auto_eject,
				FALSE, 0,
				_("Auto eject when ejectable devices are unmounted"));

	vbox1 = gkrellm_framed_vbox_end(vbox, NULL, 4, FALSE, 0, 2);
	gkrellm_spin_button(vbox1, &fs_check_spin_button, (gfloat) fs_check_timeout,
		1.0, 15.0, 1.0, 5.0, 0 /*digits*/, 0 /*width*/, NULL, NULL, FALSE,
		_("Seconds between data updates for local mounted file systems"));
	gkrellm_spin_button(vbox1, &nfs_check_spin_button,
		(gfloat) nfs_check_timeout, 5.0, 60.0, 1.0, 5.0, 0, 0, NULL,NULL,FALSE,
		_("Seconds between data updates for remote mounted file systems"));

	vbox1 = gkrellm_framed_vbox_end(vbox,
			_("Format String for Panel Labels"), 4, FALSE, 0, 2);
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 2);
	data_format_combo = gtk_combo_new();
	gtk_box_pack_start(GTK_BOX(hbox), data_format_combo, FALSE, FALSE, 0);
	list = NULL;
	list = g_list_append(list, DEFAULT_DATA_FORMAT);
	list = g_list_append(list, _("$t - $u used"));
	list = g_list_append(list, _("$t - $U"));
	gtk_combo_set_popdown_strings(GTK_COMBO(data_format_combo), list);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(data_format_combo)->entry),
			data_format);
	label = gtk_label_new(_("Format string for FS data display"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 2);

/* --Info tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Info"));
	text = gkrellm_scrolled_text(vbox, NULL,
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	for (i = 0; i < sizeof(fs_info_text)/sizeof(gchar *); ++i)
		gkrellm_add_info_text_string(text, _(fs_info_text[i]));
	}



static Monitor	monitor_fs =
	{
	N_("File System"),		/* Name, for config tab.	*/
	MON_FS,				/* Id,  0 if a plugin		*/
	create_fs,			/* The create function		*/
	update_fs,			/* The update function		*/
	create_fs_tab,		/* The config tab create function	*/
	apply_fs_config,	/* Apply the config function		*/

	save_fs_config,		/* Save user conifg			*/
	load_fs_config,		/* Load user config			*/
	FS_CONFIG_KEYWORD,	/* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_fs_monitor(void)
	{
	monitor_fs.name = _(monitor_fs.name);
	style_id = gkrellm_add_meter_style(&monitor_fs, FS_STYLE_NAME);
	data_format = g_strdup(DEFAULT_DATA_FORMAT);

	primary_widgets.selected_row = -1;
	secondary_widgets.selected_row = -1;

	mon_fs = &monitor_fs;
	if (setup_fs_interface())
		{
		(*get_fstab_list)();
		(*get_mountlist)();
		return &monitor_fs;
		}
	return FALSE;
	}

