/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>

#include "gimageview.h"
#include "gtkutils.h"
#include "menu.h"
#include "prefs.h"
#include "prefs_win.h"
#include "thumbnail_window.h"


GtkWidget *prefs_window;

static Config  *config_changed    = NULL;
static Config  *config_prechanged = NULL;
static gboolean prefs_ok_pressed  = FALSE;

typedef struct _PrefsWin {
   /* image window */
   GtkWidget *image_scale_spin;

   /* thumbnail window */
   GtkWidget *thumbwin_width_spin;
   GtkWidget *thumbwin_height_spin;
   GtkWidget *thumbwin_layout;
   GtkWidget *thumbwin_dirview_toggle;
   GtkWidget *thumbwin_preview_toggle;
   GtkWidget *thumbwin_menubar_toggle;
   GtkWidget *thumbwin_toolbar_toggle;
   GtkWidget *thumbwin_statusbar_toggle;
   GtkWidget *thumbwin_tab_toggle;
   GtkWidget *thumbwin_tab_pos;

   /* Directory View */
   GtkWidget *dirview_show_current_dir;
   GtkWidget *dirview_show_parent_dir;

   /* cache */
   GtkWidget *cache_gimv;
   GtkWidget *cache_picview;
   GtkWidget *write_cache_nautilus;
   GtkWidget *cache_write_menus[16];
   GtkWidget *cache_gimv_size;
   GtkWidget *cache_gqview_size;

   /* external progs */
   GtkWidget *graph_progs_name;
   GtkWidget *graph_progs_command;
   GtkWidget *graph_progs_up;
   GtkWidget *graph_progs_down;
   GtkWidget *graph_progs_new;
   GtkWidget *graph_progs_add;
   GtkWidget *graph_progs_change;
   GtkWidget *graph_progs_delete;
   gint       graph_progs_selected;
   gint       progs_rownum;

   GtkWidget *text_viewer_command;
} PrefsWin;

static PrefsWin prefs_win;

extern GQViewThumbSize gqview_thumb_size [];


/*******************************************************************************
 *
 *   Callback functions for preference window.
 *
 *******************************************************************************/
/*
 *  conf_apply_to_existing_win:
 *     @ Apply changed config to existing window.
 *
 *  src  : source config data.
 *  dest : destination config data.
 */
static void
conf_apply_to_existing_win (Config *src, Config *dest)
{
   ThumbWindow *tw;
   GList *node;

   node = g_list_first (ThumbWinList);
   while (node) {
      tw = node->data;
      gtk_toolbar_set_style (GTK_TOOLBAR(tw->toolbar),
			     dest->thumbwin_toolbar_style);

      gtk_widget_queue_resize (tw->toolbar_handle);

      /*
      gtk_notebook_set_tab_pos (GTK_NOTEBOOK (tw->notebook),
				dest->thumbwin_tab_pos);
      */

      if (tw->dv) {
	 gtk_ctree_set_line_style (GTK_CTREE (tw->dv->dirtree),
				   dest->dirview_line_style);
	 gtk_ctree_set_expander_style (GTK_CTREE (tw->dv->dirtree),
				       dest->dirview_expander_style);
      }

      node = node->next;
   }
}


static void
strings_free (gboolean value_changed)
{
   gint i, j;
   gchar *strings[1][2];
   gint num = sizeof (strings) / sizeof (strings[0]);
   gchar *progs_string1 = NULL, *progs_string2 = NULL;
   gint progs_num = sizeof (conf.progs) / sizeof (conf.progs[0]);
   gboolean not_free = FALSE;

   strings[0][0] = config_prechanged->text_viewer;
   strings[0][1] = config_changed->text_viewer;

   for (i = 0; i < num; i++) {
      if (strings[i][0] == strings[i][1]) continue;

      if (value_changed) {
	 g_free (strings[i][0]);
      } else {
	 g_free (strings[i][1]);
      }
   }

   /* free old external programs string */
   for (i = 0; i < progs_num; i++) {
      for (j = 0; j < progs_num; j++) {
	 if (value_changed) {
	    progs_string1 = config_prechanged->progs[i];
	    progs_string2 = config_changed->progs[j];
	 } else {
	    progs_string1 = config_changed->progs[i];
	    progs_string2 = config_prechanged->progs[j];
	 }
	 if (!progs_string1) {
	    break;
	 } else if (progs_string1 == progs_string2) {
	    not_free = TRUE;
	    break;
	 }
      }
 
      if (!not_free) {
	 g_free (progs_string1);
      }

      not_free = FALSE;
   }
}


static void
cb_close_prefs_window ()
{
  if (prefs_ok_pressed) {
     strings_free (TRUE);
     memcpy (&conf, config_changed, sizeof(Config));
     conf_apply_to_existing_win (config_prechanged, &conf);
  } else {
     strings_free (FALSE);
     memcpy (&conf, config_prechanged, sizeof(Config));
     conf_apply_to_existing_win (config_changed, &conf);
   }
   g_free (config_changed);
   g_free (config_prechanged);
   config_changed = NULL;
   config_prechanged = NULL;
   prefs_window = NULL;
   prefs_ok_pressed = FALSE;
   prefs_save_config ();
}


static void
cb_prefs_ok_button ()
{
   prefs_ok_pressed = TRUE;
   gtk_widget_destroy (prefs_window);
}


static void
cb_prefs_apply_button ()
{
   memcpy (&conf, config_changed, sizeof(Config));
   conf_apply_to_existing_win (config_prechanged, &conf);
}


static void
cb_prefs_cancel_button ()
{
   gtk_widget_destroy (prefs_window);
}


static void
cb_get_data_from_toggle (GtkWidget *toggle, gboolean *data)
{
   g_return_if_fail (data);

   *data = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(toggle));
}


static void
cb_get_data_from_toggle_negative (GtkWidget *toggle, gboolean *data)
{
   g_return_if_fail (data);

   *data = !(gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(toggle)));
}


static void
cb_get_data_from_adjustment_by_int (GtkWidget *widget, gint *data)
{
   g_return_if_fail (data);

   *data = GTK_ADJUSTMENT(widget)->value;
}


static void
cb_get_data_from_adjustment_by_float (GtkWidget *widget, gfloat *data)
{
   g_return_if_fail (data);

   *data = GTK_ADJUSTMENT(widget)->value;
}


static void
cb_get_data_from_menuitem (GtkWidget *widget, gint *data)
{
   g_return_if_fail (data);

   *data = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (widget), "num"));
}


static void
cb_recursive_follow_symlink (GtkWidget *radio, gpointer data)
{
   gint idx = GPOINTER_TO_INT (data);

   if (!GTK_TOGGLE_BUTTON (radio)->active) return;

   switch (idx) {
   case 0:
      config_changed->recursive_follow_link   = TRUE;
      config_changed->recursive_follow_parent = TRUE;
      break;
   case 1:
      config_changed->recursive_follow_link   = FALSE;
      config_changed->recursive_follow_parent = FALSE;
      break;
   case 2:
      config_changed->recursive_follow_link   = TRUE;
      config_changed->recursive_follow_parent = FALSE;
      break;
   default:
      break;
   }
}


static void
cb_fit_image_to_win (GtkWidget *toggle)
{
   config_changed->imgview_fit_image_to_win =
      gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(toggle));
   gtk_widget_set_sensitive (prefs_win.image_scale_spin,
			     !config_changed->imgview_fit_image_to_win);
}


static void
cb_rotate_min_colnum_value (GtkWidget *widget, GtkWidget *spinner)
{
   gint colmin;

   colmin = GTK_ADJUSTMENT(widget)->value;

   if (colmin > config_changed->thumbtable_colnum_max) {
      config_changed->thumbtable_colnum_min = config_changed->thumbtable_colnum_max; 
      gtk_spin_button_set_value (GTK_SPIN_BUTTON(spinner),
				 (gfloat) config_changed->thumbtable_colnum_min);
   } else {
      config_changed->thumbtable_colnum_min = colmin; 
   }
}


static void
cb_rotate_max_colnum_value (GtkWidget *widget, GtkWidget *spinner)
{
   gint colmax;

   colmax = GTK_ADJUSTMENT(widget)->value;

   if (colmax < config_changed->thumbtable_colnum_min) {
      config_changed->thumbtable_colnum_max = config_changed->thumbtable_colnum_min; 
      gtk_spin_button_set_value (GTK_SPIN_BUTTON(spinner),
				 (gfloat) config_changed->thumbtable_colnum_max);
   } else {
      config_changed->thumbtable_colnum_max = colmax; 
   }
}


static void
set_sensitive_thumbwin_state ()
{
   GtkWidget *widgets[] = {
      prefs_win.thumbwin_width_spin,
      prefs_win.thumbwin_height_spin,
      prefs_win.thumbwin_layout,
      prefs_win.thumbwin_dirview_toggle,
      prefs_win.thumbwin_preview_toggle,
      prefs_win.thumbwin_menubar_toggle,
      prefs_win.thumbwin_toolbar_toggle,
      prefs_win.thumbwin_statusbar_toggle,
      prefs_win.thumbwin_tab_toggle,
      prefs_win.thumbwin_tab_pos,
   };
   gint i, num = sizeof (widgets) / sizeof (widgets[0]);

   for (i = 0; i < num; i++)
      gtk_widget_set_sensitive (widgets[i],
				!config_changed->thumbwin_save_win_state);
}


static void
cb_save_thumbwin_state (GtkWidget *toggle)
{
   config_changed->thumbwin_save_win_state =
      gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(toggle));

   set_sensitive_thumbwin_state ();
}


/*
 *  set_sensitive_cache_type:
 *     @ If cache reading is disabled, also same type cache writing will
 *       be diabled.
 */
static void
set_sensitive_cache_type ()
{
   gtk_widget_set_sensitive (GTK_WIDGET (prefs_win.cache_write_menus[1]),
			     config_changed->cache_gimv);
   gtk_widget_set_sensitive (GTK_WIDGET (prefs_win.cache_write_menus[2]),
			     FALSE);
   gtk_widget_set_sensitive (GTK_WIDGET (prefs_win.cache_write_menus[3]),
			     config_changed->cache_konqueror_l);
   gtk_widget_set_sensitive (GTK_WIDGET (prefs_win.cache_write_menus[4]),
			     config_changed->cache_konqueror_m);
   gtk_widget_set_sensitive (GTK_WIDGET (prefs_win.cache_write_menus[5]),
			     config_changed->cache_konqueror_s);
   gtk_widget_set_sensitive (GTK_WIDGET (prefs_win.cache_write_menus[6]),
			     config_changed->cache_gqview);
   gtk_widget_set_sensitive (GTK_WIDGET (prefs_win.cache_write_menus[7]),
			     config_changed->cache_ee_l);
   gtk_widget_set_sensitive (GTK_WIDGET (prefs_win.cache_write_menus[8]),
			     config_changed->cache_ee_m);
   gtk_widget_set_sensitive (GTK_WIDGET (prefs_win.cache_write_menus[9]),
			     config_changed->cache_ee_s);
   gtk_widget_set_sensitive (GTK_WIDGET (prefs_win.cache_write_menus[10]),
			     FALSE);

   /* picview support is only supported under Electric Eyes cache is enabled */
   if (!(config_changed->cache_ee_l || config_changed->cache_ee_m))
      gtk_widget_set_sensitive (GTK_WIDGET (prefs_win.cache_picview), FALSE);
   if (!(!config_changed->cache_ee_l && !config_changed->cache_ee_m))
      gtk_widget_set_sensitive (GTK_WIDGET (prefs_win.cache_picview), TRUE);

   /* GQview thumbnail size */
   gtk_widget_set_sensitive (GTK_WIDGET (prefs_win.cache_gimv_size),
			     config_changed->cache_gimv
			     && config_changed->cache_write_type == GIMAGEVIEW);

   /* GQview thumbnail size */
   gtk_widget_set_sensitive (GTK_WIDGET (prefs_win.cache_gqview_size),
			     config_changed->cache_gqview
			     && config_changed->cache_write_type == GQVIEW);
}


static void
cb_cache_read_type_toggle (GtkWidget *toggle, gboolean *data)
{
   *data = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(toggle));
   set_sensitive_cache_type ();
}


static void
cb_cache_write_type (GtkWidget *widget, gint data)
{
   config_changed->cache_write_type = data;
   set_sensitive_cache_type ();
}


static void
cb_default_disp_mode (GtkWidget *widget)
{
   gchar *tmpstr;
   gint disp_mode;

   disp_mode = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (widget), "num"));
   tmpstr = thumbview_num_to_label (disp_mode);
   if (tmpstr)
      config_changed->thumbwin_disp_mode = tmpstr;
   else
      config_changed->thumbwin_disp_mode = DEFAULT_DISP_MODE;
}


static void
set_sensitive_ext_progs (gboolean text_changed)
{
   gchar *text[2];
   gint num = sizeof (conf.progs) / sizeof (conf.progs[0]);
   gint selected = prefs_win.graph_progs_selected;

   text[0] = gtk_entry_get_text (GTK_ENTRY (prefs_win.graph_progs_name));
   text[1] = gtk_entry_get_text (GTK_ENTRY (prefs_win.graph_progs_command));

   /* add button */
   if (prefs_win.progs_rownum >= num
       || !text[0] || text[0][0] == '\0'
       || !text[1] || text[1][0] == '\0' )
   {
      gtk_widget_set_sensitive (prefs_win.graph_progs_add, FALSE);
   } else {
      gtk_widget_set_sensitive (prefs_win.graph_progs_add, TRUE);
   }

   if (selected < 0) {   /* no item selected */
      gtk_widget_set_sensitive (prefs_win.graph_progs_up, FALSE);
      gtk_widget_set_sensitive (prefs_win.graph_progs_down, FALSE);
      gtk_widget_set_sensitive (prefs_win.graph_progs_change, FALSE);
      gtk_widget_set_sensitive (prefs_win.graph_progs_delete, FALSE);
   } else {                                    /* if an item selected */
      /* change button */
      if (text_changed)
	 gtk_widget_set_sensitive (prefs_win.graph_progs_change, TRUE);
      else
	 gtk_widget_set_sensitive (prefs_win.graph_progs_change, FALSE);

      /* up button */
      if (selected == 0)
	 gtk_widget_set_sensitive (prefs_win.graph_progs_up, FALSE);
      else
	 gtk_widget_set_sensitive (prefs_win.graph_progs_up, TRUE);

      /* down button */
      if (selected < prefs_win.progs_rownum - 1)
	 gtk_widget_set_sensitive (prefs_win.graph_progs_down, TRUE);
      else
	 gtk_widget_set_sensitive (prefs_win.graph_progs_down, FALSE);

      /* delete button */
      gtk_widget_set_sensitive (prefs_win.graph_progs_delete, TRUE);
   }
}


static void
cb_progs_select_row (GtkCList *clist, gint row, gint col,
		     GdkEventButton *event, gpointer data)
{
   gint success;
   gchar *text[2] = {NULL, NULL};

   g_return_if_fail (clist);

   prefs_win.graph_progs_selected = row;

   success = gtk_clist_get_text (clist, row, 0, &text[0]);
   if (success && text[0])
      gtk_entry_set_text (GTK_ENTRY(prefs_win.graph_progs_name), text[0]);

   success = gtk_clist_get_text (clist, row, 1, &text[1]);
   if (success && text[1])
      gtk_entry_set_text (GTK_ENTRY(prefs_win.graph_progs_command), text[1]);

   set_sensitive_ext_progs (FALSE);
}


static void
cb_progs_unselect_row (GtkCList *clist, gint row, gint col,
		       GdkEventButton *event, gpointer data)
{
   g_return_if_fail (clist);

   prefs_win.graph_progs_selected = -1;

   set_sensitive_ext_progs (FALSE);
}


static void
cb_progs_up_button_pressed (GtkButton *button, gpointer data)
{
   GtkCList *clist = data;
   gint selected = prefs_win.graph_progs_selected;
   gint num = sizeof (conf.progs) / sizeof (conf.progs[0]);
   gchar * swap;

   g_return_if_fail (button && clist);

   if (selected < 1 || selected > prefs_win.progs_rownum - 1
       || selected > num - 1)
   {
      return;
   }

   swap = config_changed->progs[selected];
   config_changed->progs[selected] = config_changed->progs[selected - 1];
   config_changed->progs[selected - 1] = swap;
   gtk_clist_swap_rows (clist, selected, selected - 1);
   prefs_win.graph_progs_selected--;

   gtk_clist_moveto (clist, selected - 1, 0, 0, 0);

   set_sensitive_ext_progs (FALSE);
}


static void
cb_progs_down_button_pressed (GtkButton *button, gpointer data)
{
   GtkCList *clist = data;
   gint selected = prefs_win.graph_progs_selected;
   gint num = sizeof (conf.progs) / sizeof (conf.progs[0]);
   gchar * swap;

   g_return_if_fail (button && clist);

   if (selected < 0 || selected > prefs_win.progs_rownum - 2
       || selected > num - 2)
   {
      return;
   }

   swap = config_changed->progs[selected];
   config_changed->progs[selected] = config_changed->progs[selected + 1];
   config_changed->progs[selected + 1] = swap;
   gtk_clist_swap_rows (clist, selected, selected + 1);
   prefs_win.graph_progs_selected++;

   gtk_clist_moveto (clist, selected + 1, 0, 0, 0);

   set_sensitive_ext_progs (FALSE);
}


static void
cb_progs_new_button_pressed (GtkButton *button, gpointer data)
{
   GtkCList *clist = data;

   g_return_if_fail (button && data);

   gtk_clist_unselect_all (clist);
   gtk_entry_set_text (GTK_ENTRY(prefs_win.graph_progs_name), "\0");
   gtk_entry_set_text (GTK_ENTRY(prefs_win.graph_progs_command), "\0");

   set_sensitive_ext_progs (FALSE);
}


static void
cb_progs_add_button_pressed (GtkButton *button, gpointer data)
{
   GtkCList *clist = data;
   gchar *text[2];
   gint num = sizeof (conf.progs) / sizeof (conf.progs[0]);

   g_return_if_fail (clist);

   text[0] = gtk_entry_get_text (GTK_ENTRY (prefs_win.graph_progs_name));
   text[1] = gtk_entry_get_text (GTK_ENTRY (prefs_win.graph_progs_command));
   if (text[0] && text[1] && prefs_win.progs_rownum < num) {
      gtk_clist_append (GTK_CLIST (clist), text);
      config_changed->progs[prefs_win.progs_rownum]
	 = g_strconcat (text[0], ",", text[1], NULL);
      prefs_win.progs_rownum++;
   }

   gtk_entry_set_text (GTK_ENTRY(prefs_win.graph_progs_name), "\0");
   gtk_entry_set_text (GTK_ENTRY(prefs_win.graph_progs_command), "\0");

   set_sensitive_ext_progs (FALSE);
}


static void
cb_progs_change_button_pressed (GtkButton *button, gpointer data)
{
   GtkCList *clist = data;
   gchar *src, *text[2];
   gint num = sizeof (conf.progs) / sizeof (conf.progs[0]);
   gint selected = prefs_win.graph_progs_selected;

   g_return_if_fail (clist);

   if (selected < 0 && selected > num) return;

   text[0] = gtk_entry_get_text (GTK_ENTRY (prefs_win.graph_progs_name));
   text[1] = gtk_entry_get_text (GTK_ENTRY (prefs_win.graph_progs_command));

   if (text[0] && text[1]) {
      src = gtk_clist_get_row_data (clist, selected);
      if (!src) {
	 g_free (config_changed->progs[selected]);
	 gtk_clist_set_row_data (clist, selected, NULL);
      }
      config_changed->progs[selected]
	 = g_strconcat (text[0], ",", text[1], NULL);
      gtk_clist_set_text (clist, selected, 0, text[0]);
      gtk_clist_set_text (clist, selected, 1, text[1]);
   }

   gtk_entry_set_text (GTK_ENTRY(prefs_win.graph_progs_name), "\0");
   gtk_entry_set_text (GTK_ENTRY(prefs_win.graph_progs_command), "\0");

   set_sensitive_ext_progs (FALSE);
}


static void
cb_progs_delete_button_pressed (GtkButton *button, gpointer data)
{
   GtkCList *clist = data;
   gchar *src;
   gint i, num = sizeof (conf.progs) / sizeof (conf.progs[0]);
   gint selected = prefs_win.graph_progs_selected;

   g_return_if_fail (clist);

   if (selected < 0 && selected > num) return;

   src = gtk_clist_get_row_data (clist, selected);

   if (!src)
      g_free (config_changed->progs[selected]);
   config_changed->progs[selected] = NULL;

   for (i = selected; i < num - 1; i++) {
      config_changed->progs[i] = config_changed->progs[i + 1];
   }
   config_changed->progs[num] = NULL;
   prefs_win.progs_rownum--;

   gtk_clist_remove (clist, selected);

   gtk_clist_unselect_all (clist);
   gtk_entry_set_text (GTK_ENTRY(prefs_win.graph_progs_name), "\0");
   gtk_entry_set_text (GTK_ENTRY(prefs_win.graph_progs_command), "\0");

   set_sensitive_ext_progs (FALSE);
}


static void
cb_progs_entry_changed (GtkEditable *entry, gpointer data)
{
   set_sensitive_ext_progs (TRUE);
}


static void
cb_use_internal_text_viewer (GtkWidget *toggle)
{
   config_changed->text_viewer_use_internal =
      gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(toggle));

   gtk_widget_set_sensitive (prefs_win.text_viewer_command,
			     !config_changed->text_viewer_use_internal);
}


static void
cb_text_view_command_changed (GtkEditable *entry, gpointer data)
{
   if (config_changed->text_viewer != config_prechanged->text_viewer)
      g_free (config_changed->text_viewer);

   config_changed->text_viewer
      = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));
}



/*******************************************************************************
 *
 *   Parts of preference window.
 *
 *******************************************************************************/
static const gchar *open_window_items[] = {
   N_("Image View"),
   N_("Thumbnail View"),
   NULL
};


#if HAVE_GDK_PIXBUF
static const gchar *interpolation_items[] = {
   N_("Nearest"),
   N_("Tiles"),
   N_("Bilinear"),
   N_("Hyperbolic"),
   NULL
};
#endif /* HAVE_GDK_PIXBUF */


static const gchar *layout_items[] = {
   N_("Layout0"),
   N_("Layout1"),
   N_("Layout2"),
   N_("Layout3"),
   N_("Layout4"),
   N_("Custom"),
   NULL
};


static const gchar *toolbar_style_items[] = {
   N_("Icon"),
   N_("Text"),
   N_("Both"),
   NULL
};


static const gchar *tab_pos_items[] = {
   N_("Left"),
   N_("Right"),
   N_("Top"),
   N_("Bottom"),
   NULL
};


static const gchar *ctree_line_style_items[] = {
   N_("None"),
   N_("Solid"),
   N_("Dotted"),
   N_("Tabbed"),
   NULL
};


static const gchar *ctree_expander_style_items[] = {
   N_("None"),
   N_("Square"),
   N_("Triangle"),
   N_("Circular"),
   NULL
};


/*
 *  create_prefs_common_page:
 *     @ Create common preference page.
 *
 *  Return : Packed widget (GtkVbox)
 */
GtkWidget *
create_prefs_common_page ()
{
   GtkWidget *main_vbox, *frame, *table, *vbox;
#if HAVE_GDK_PIXBUF
   GtkWidget *hbox;
#endif /* HAVE_GDK_PIXBUF */
   GtkWidget *label;
   GtkWidget *option_menu;
   GtkWidget *toggle, *radio0, *radio1, *radio2;

   main_vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(main_vbox), 5);

   /**********************************************
    * Default Open Window Frame
    **********************************************/
   frame = gtk_frame_new (_("Default Open Window"));
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, TRUE, 0);

   table = gtk_table_new (2, 2, FALSE);
   gtk_container_set_border_width(GTK_CONTAINER(table), 5);
   gtk_container_add (GTK_CONTAINER (frame), table);

   /* File Open Window Selection */
   label = gtk_label_new (_("File Open Window"));
   gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
		     GTK_EXPAND, GTK_FILL, 0, 0);
   option_menu = create_option_menu_simple (open_window_items,
					    conf.default_file_open_window,
					    &config_changed->default_file_open_window);
   gtk_table_attach (GTK_TABLE (table), option_menu, 1, 2, 0, 1,
		     GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);

   /* Directory Open Window Selection */
   label = gtk_label_new (_("Directory Open Window"));
   gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
		     GTK_EXPAND, GTK_FILL, 0, 0);
   option_menu = create_option_menu_simple (open_window_items,
					    conf.default_dir_open_window,
					    &config_changed->default_dir_open_window);
   gtk_table_attach (GTK_TABLE (table), option_menu, 1, 2, 1, 2,
		     GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);

   /**********************************************
    * Scan Directory Frame
    **********************************************/
   frame = gtk_frame_new (_("Directory scan"));
   gtk_container_set_border_width(GTK_CONTAINER(frame), 0);
   gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, TRUE, 0);

   vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
   gtk_container_add (GTK_CONTAINER (frame), vbox);

   /* Ignore filename extension or not */
   toggle = create_check_button (_("Scan directories recursively"),
				 conf.scan_dir_recursive,
				 cb_get_data_from_toggle,
				 &config_changed->scan_dir_recursive);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* follow symbolic link */
   radio0 = gtk_radio_button_new_with_label (NULL, _("Follow symbolic link"));
   gtk_box_pack_start (GTK_BOX (vbox), radio0, FALSE, FALSE, 0);
   gtk_widget_set_uposition (GTK_WIDGET(radio0), 40, -1);

   radio1 = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (radio0),
				_("Do not follow symbolic link"));
   gtk_box_pack_start (GTK_BOX (vbox), radio1, FALSE, FALSE, 0);
   gtk_widget_set_uposition (GTK_WIDGET(radio1), 40, -1);

   radio2 = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (radio1),
                                _("Do not follow link to parent directory"));
   /* not implemented yet
   gtk_box_pack_start (GTK_BOX (vbox), radio2, FALSE, FALSE, 0);
   gtk_widget_set_uposition (GTK_WIDGET(radio2), 40, -1);
   */

   if (conf.recursive_follow_link && conf.recursive_follow_parent)
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio0), TRUE);
   else if (conf.recursive_follow_link && !conf.recursive_follow_parent)
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio2), TRUE);
   else
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio1), TRUE);

   gtk_signal_connect (GTK_OBJECT (radio0), "toggled",
		       GTK_SIGNAL_FUNC (cb_recursive_follow_symlink),
		       GINT_TO_POINTER(0));
   gtk_signal_connect (GTK_OBJECT (radio1), "toggled",
		       GTK_SIGNAL_FUNC (cb_recursive_follow_symlink),
		       GINT_TO_POINTER(1));
   gtk_signal_connect (GTK_OBJECT (radio2), "toggled",
		       GTK_SIGNAL_FUNC (cb_recursive_follow_symlink),
		       GINT_TO_POINTER(2));

   /* Ignore filename extension or not */
   toggle = create_check_button (_("Read dotfiles"),
				 conf.read_dotfile,
				 cb_get_data_from_toggle,
				 &config_changed->read_dotfile);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);


#if HAVE_GDK_PIXBUF
   /********************************************** 
    * Image Frame
    **********************************************/
   frame = gtk_frame_new (_("Image"));
   gtk_container_set_border_width(GTK_CONTAINER (frame), 0);
   gtk_box_pack_start(GTK_BOX (main_vbox), frame, FALSE, TRUE, 0);
   vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
   gtk_container_add (GTK_CONTAINER (frame), vbox);

   /* Interpolation type */
   hbox = gtk_hbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER (hbox), 5);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
   label = gtk_label_new (_("Interpolation type for scaling"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 5);
   option_menu = create_option_menu_simple (interpolation_items,
					    conf.interpolation,
					    (gint *) &config_changed->interpolation);
   gtk_box_pack_start (GTK_BOX (hbox), option_menu, FALSE, FALSE, 5);
#endif /* HAVE_GDK_PIXBUF */

   /********************************************** 
    * Start up Frame
    **********************************************/
   frame = gtk_frame_new (_("Start up"));
   gtk_container_set_border_width(GTK_CONTAINER (frame), 0);
   gtk_box_pack_start(GTK_BOX (main_vbox), frame, FALSE, TRUE, 0);
   vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
   gtk_container_add (GTK_CONTAINER (frame), vbox);

   /* Open thumbnail window or not */
   toggle = create_check_button (_("Open thumbnail window"),
				  conf.startup_open_thumbwin,
				  cb_get_data_from_toggle,
				  &config_changed->startup_open_thumbwin);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* Scan (or not) Directory */
   toggle = create_check_button (_("Ignore directory"), !(conf.startup_read_dir),
				 cb_get_data_from_toggle_negative,
				 &config_changed->startup_read_dir);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* No warning */
   toggle = create_check_button (_("No warning"), conf.startup_no_warning,
				 cb_get_data_from_toggle,
				 &config_changed->startup_no_warning);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);


   /* Ignore filename extension or not */
   toggle = create_check_button (_("Ignore file name extension"),
				 !(conf.detect_filetype_by_ext),
				 cb_get_data_from_toggle_negative,
				 &config_changed->detect_filetype_by_ext);
   gtk_box_pack_start (GTK_BOX (main_vbox), toggle, FALSE, FALSE, 0);

   return main_vbox;
}


/*
 *  create_prefs_imageview_page:
 *     @ Create Image View preference page.
 *
 *  Return : Packed widget (GtkVbox)
 */
GtkWidget *
create_prefs_imageview_page ()
{
   GtkWidget *main_vbox;
   GtkWidget *frame;
   GtkWidget *hbox, *vbox;
   GtkWidget *label;
   GtkAdjustment *adj;
   GtkWidget *spinner;
   GtkWidget *toggle;

   main_vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(main_vbox), 5);

   /**********************************************
    * Window Frame
    **********************************************/
   frame = gtk_frame_new (_("Window"));
   gtk_container_set_border_width(GTK_CONTAINER (frame), 5);
   gtk_box_pack_start(GTK_BOX (main_vbox), frame, FALSE, TRUE, 0);
   vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
   gtk_container_add (GTK_CONTAINER (frame), vbox);

   /* Window width and height spinner */
   hbox = gtk_hbox_new (FALSE, 10);
   gtk_container_set_border_width (GTK_CONTAINER(hbox), 5);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

   label = gtk_label_new (_("Initial window size: "));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

   label = gtk_label_new (_("width"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
   adj = (GtkAdjustment *) gtk_adjustment_new (conf.imgwin_width,
					       1.0, 10000.0, 1.0, 5.0, 0.0);
   spinner = gtk_spin_button_new (adj, 0, 0);
   gtk_widget_set_usize(spinner, 50, -1);
   gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
   gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner),
				    GTK_SHADOW_OUT);
   gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		       GTK_SIGNAL_FUNC (cb_get_data_from_adjustment_by_int),
		       &config_changed->imgwin_width);
   gtk_box_pack_start (GTK_BOX (hbox), spinner, FALSE, FALSE, 0);

   label = gtk_label_new (_("height"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
   adj = (GtkAdjustment *) gtk_adjustment_new (conf.imgwin_height,
					       1.0, 10000.0, 1.0, 5.0, 0.0);
   spinner = gtk_spin_button_new (adj, 0, 0);
   gtk_widget_set_usize(spinner, 50, -1);
   gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
   gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner),
				    GTK_SHADOW_OUT);
   gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		       GTK_SIGNAL_FUNC (cb_get_data_from_adjustment_by_int),
		       &config_changed->imgwin_height);
   gtk_box_pack_start (GTK_BOX (hbox), spinner, FALSE, FALSE, 0);

   /* Auto resize window to image size */
   toggle = create_check_button (_("Auto resize to image size"),
				 conf.imgwin_fit_to_image,
				 cb_get_data_from_toggle,
				 &config_changed->imgwin_fit_to_image);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* Open New Window or not */
   toggle = create_check_button (_("Open each file in separate window"),
				 conf.imgwin_open_new_win,
				 cb_get_data_from_toggle,
				 &config_changed->imgwin_open_new_win);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* Raise window or not */
   toggle = create_check_button (_("Raise window when open image in shared window"),
				 conf.imgwin_raise_window,
				 cb_get_data_from_toggle,
				 &config_changed->imgwin_raise_window);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /********************************************** 
    * Show/Hide Frame
    **********************************************/
   frame = gtk_frame_new (_("Show/Hide"));
   gtk_container_set_border_width(GTK_CONTAINER (frame), 5);
   gtk_box_pack_start(GTK_BOX (main_vbox), frame, FALSE, TRUE, 0);
   vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
   gtk_container_add (GTK_CONTAINER (frame), vbox);

   /* Show Menubar or not */
   toggle = create_check_button (_("Show menubar"), conf.imgwin_show_menubar,
				 cb_get_data_from_toggle,
				 &config_changed->imgwin_show_menubar);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* Show Toolbar or not */
   toggle = create_check_button (_("Show toolbar"), conf.imgwin_show_toolbar,
				 cb_get_data_from_toggle,
				 &config_changed->imgwin_show_toolbar);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* Show Statusbar or not */
   toggle = create_check_button (_("Show statusbar"), conf.imgwin_show_statusbar,
				 cb_get_data_from_toggle,
				 &config_changed->imgwin_show_statusbar);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /********************************************** 
    * Image Frame
    **********************************************/
   frame = gtk_frame_new (_("Image"));
   gtk_container_set_border_width(GTK_CONTAINER (frame), 5);
   gtk_box_pack_start(GTK_BOX (main_vbox), frame, FALSE, TRUE, 0);
   vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
   gtk_container_add (GTK_CONTAINER (frame), vbox);

   /* Auto resize image to window size */
   toggle = create_check_button (_("Auto resize to window size when opening"),
				 conf.imgview_fit_image_to_win,
				 cb_fit_image_to_win, NULL);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* Keep Aspect Ratio */
   toggle = create_check_button (_("Keep aspect rario"), conf.imgview_keep_aspect,
				 cb_get_data_from_toggle,
				 &config_changed->imgview_keep_aspect);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* Default Image Scale Spinner */
   hbox = gtk_hbox_new (FALSE, 5);
   gtk_container_set_border_width (GTK_CONTAINER(hbox), 5);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
   label = gtk_label_new (_("Default Image Scale"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
   adj = (GtkAdjustment *) gtk_adjustment_new (conf.imgview_scale,
					       1.0, 10000.0, 1.0, 5.0, 0.0);
   spinner = gtk_spin_button_new (adj, 0, 0);
   gtk_widget_set_usize(spinner, 50, -1);
   prefs_win.image_scale_spin = spinner;
   gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
   gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner),
				    GTK_SHADOW_OUT);
   gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		       GTK_SIGNAL_FUNC (cb_get_data_from_adjustment_by_int),
		       &config_changed->imgview_scale);
   gtk_box_pack_start (GTK_BOX (hbox), spinner, FALSE, FALSE, 0);
   label = gtk_label_new (_("%"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
   gtk_widget_set_sensitive (prefs_win.image_scale_spin, !conf.imgview_fit_image_to_win);

   /* keep original image on memory or not */
   toggle = create_check_button (_("Keep original image on memory"),
				 conf.imgview_buffer,
				 cb_get_data_from_toggle,
				 &config_changed->imgview_buffer);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   return main_vbox;
}


/*
 *  create_prefs_thumbview_page:
 *     @ Create Thumbnail View preference page.
 *
 *  Return : Packed widget (GtkVbox)
 */
GtkWidget *
create_prefs_thumbview_page ()
{
   GtkWidget *main_vbox;
   GtkWidget *frame, *vbox, *vbox2, *hbox, *hbox2;
   GtkWidget *label;
   GtkAdjustment *adj;
   GtkWidget *spinner;
   GtkWidget *option_menu;
   GtkWidget *toggle;
   gint num, disp_mode;

   main_vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(main_vbox), 5);

   /**********************************************
    * Window Frame
    **********************************************/
   frame = gtk_frame_new (_("Window"));
   gtk_container_set_border_width(GTK_CONTAINER (frame), 5);
   gtk_box_pack_start(GTK_BOX (main_vbox), frame, FALSE, TRUE, 0);
   vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
   gtk_container_add (GTK_CONTAINER (frame), vbox);

   /* Save window state on exit */
   toggle = create_check_button (_("Save window state on exit"),
				 conf.thumbwin_save_win_state,
                                 cb_save_thumbwin_state,
                                 &config_changed->thumbwin_save_win_state);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* Window width and height spinner */
   hbox = gtk_hbox_new (FALSE, 10);
   gtk_container_set_border_width (GTK_CONTAINER(hbox), 5);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
   label = gtk_label_new (_("width"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
   adj = (GtkAdjustment *) gtk_adjustment_new (conf.thumbwin_width,
					       1.0, 10000.0, 1.0, 5.0, 0.0);
   spinner = prefs_win.thumbwin_width_spin = gtk_spin_button_new (adj, 0, 0);
   gtk_widget_set_usize(spinner, 50, -1);
   gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
   gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner),
				    GTK_SHADOW_OUT);
   gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		       GTK_SIGNAL_FUNC (cb_get_data_from_adjustment_by_int),
		       &config_changed->thumbwin_width);
   gtk_box_pack_start (GTK_BOX (hbox), spinner, FALSE, FALSE, 0);

   label = gtk_label_new (_("height"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
   adj = (GtkAdjustment *) gtk_adjustment_new (conf.thumbwin_height,
					       1.0, 10000.0, 1.0, 5.0, 0.0);
   spinner = prefs_win.thumbwin_height_spin = gtk_spin_button_new (adj, 0, 0);
   gtk_widget_set_usize(spinner, 50, -1);
   gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
   gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner),
				    GTK_SHADOW_OUT);
   gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		       GTK_SIGNAL_FUNC (cb_get_data_from_adjustment_by_int),
		       &config_changed->thumbwin_height);
   gtk_box_pack_start (GTK_BOX (hbox), spinner, FALSE, FALSE, 0);

   /* Toolbar Style */
   label = gtk_label_new (_("Default Layout"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 5);

   option_menu = create_option_menu_simple (layout_items,
					    conf.thumbwin_layout_type,
					    (gint *) &config_changed->thumbwin_layout_type);
   prefs_win.thumbwin_layout = option_menu;
   gtk_box_pack_start (GTK_BOX (hbox), option_menu, FALSE, FALSE, 0);

   /* Show/Hide frame */
   frame = gtk_frame_new (_("Show/Hide"));
   gtk_container_set_border_width(GTK_CONTAINER (frame), 5);
   gtk_box_pack_start(GTK_BOX (vbox), frame, FALSE, TRUE, 0);
   vbox2 = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
   gtk_container_add (GTK_CONTAINER (frame), vbox2);

   hbox = gtk_hbox_new (FALSE, 10);
   gtk_container_set_border_width (GTK_CONTAINER(hbox), 5);
   gtk_box_pack_start (GTK_BOX (vbox2), hbox, FALSE, FALSE, 0);

   hbox2 = gtk_hbox_new (FALSE, 10);
   gtk_container_set_border_width (GTK_CONTAINER(hbox), 5);
   gtk_box_pack_start (GTK_BOX (vbox2), hbox2, FALSE, FALSE, 0);

   /* Show Directory or not */
   toggle = create_check_button (_("Directory view"), conf.thumbwin_show_dir_view,
                                 cb_get_data_from_toggle,
                                 &config_changed->thumbwin_show_dir_view);
   prefs_win.thumbwin_dirview_toggle = toggle;
   gtk_box_pack_start (GTK_BOX (hbox), toggle, FALSE, FALSE, 0);

   /* Show Preview or not */
   toggle = create_check_button (_("Preview"), conf.thumbwin_show_preview,
                                 cb_get_data_from_toggle,
                                 &config_changed->thumbwin_show_preview);
   prefs_win.thumbwin_preview_toggle = toggle;
   gtk_box_pack_start (GTK_BOX (hbox), toggle, FALSE, FALSE, 0);

   /* Show Menubar or not */
   toggle = create_check_button (_("Menubar"), conf.thumbwin_show_menubar,
                                 cb_get_data_from_toggle,
                                 &config_changed->thumbwin_show_menubar);
   prefs_win.thumbwin_menubar_toggle = toggle;
   gtk_box_pack_start (GTK_BOX (hbox2), toggle, FALSE, FALSE, 0);

   /* Show Toolbar or not */
   toggle = create_check_button (_("Toolbar"), conf.thumbwin_show_toolbar,
				 cb_get_data_from_toggle,
				 &config_changed->thumbwin_show_toolbar);
   prefs_win.thumbwin_toolbar_toggle = toggle;
   gtk_box_pack_start (GTK_BOX (hbox2), toggle, FALSE, FALSE, 0);

   /* Show Statusbar or not */
   toggle = create_check_button (_("Statusbar"), conf.thumbwin_show_statusbar,
				 cb_get_data_from_toggle,
				 &config_changed->thumbwin_show_statusbar);
   prefs_win.thumbwin_statusbar_toggle = toggle;
   gtk_box_pack_start (GTK_BOX (hbox2), toggle, FALSE, FALSE, 0);

   /* Show Tab or not */
   toggle = create_check_button (_("Tab"), conf.thumbwin_show_tab,
				 cb_get_data_from_toggle,
				 &config_changed->thumbwin_show_tab);
   prefs_win.thumbwin_tab_toggle = toggle;
   gtk_box_pack_start (GTK_BOX (hbox2), toggle, FALSE, FALSE, 0);

   /* Raise window or not */
   toggle = create_check_button (_("Raise window when opening thumbnails"),
				 conf.thumbwin_raise_window,
				 cb_get_data_from_toggle,
				 &config_changed->thumbwin_raise_window);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   hbox = gtk_hbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

   /* Toolbar Style */
   hbox = gtk_hbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

   label = gtk_label_new (_("Toolbar Style"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 5);

   option_menu = create_option_menu_simple (toolbar_style_items,
					    conf.thumbwin_toolbar_style,
					    (gint *) &config_changed->thumbwin_toolbar_style);
   gtk_box_pack_start (GTK_BOX (hbox), option_menu, FALSE, FALSE, 5);

   /* Default display mode */
   hbox = gtk_hbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

   if (!disp_mode_labels)
      disp_mode_labels = thumbview_get_disp_mode_labels (&num);

   label = gtk_label_new (_("Default display mode"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 5);

   disp_mode = thumbview_label_to_num (conf.thumbwin_disp_mode);
   if (disp_mode < 0)
      disp_mode = DEFAULT_DISP_MODE_NUM;
   option_menu = create_option_menu ((const gchar **)disp_mode_labels,
				     disp_mode,
				     cb_default_disp_mode,
				     NULL);
   gtk_box_pack_start (GTK_BOX (hbox), option_menu, FALSE, FALSE, 5);

   /**********************************************
    * Tab Frame
    **********************************************/
   frame = gtk_frame_new (_("Tab"));
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, TRUE, 0);
   vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_container_add (GTK_CONTAINER (frame), vbox);

   /* Tab Position Selection */
   hbox = gtk_hbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
   label = gtk_label_new (_("Tab Position"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 5);

   option_menu = create_option_menu_simple (tab_pos_items,
					    conf.thumbwin_tab_pos,
					    (gint *) &config_changed->thumbwin_tab_pos);
   gtk_box_pack_start (GTK_BOX (hbox), option_menu, FALSE, FALSE, 5);

   hbox = gtk_hbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
   prefs_win.thumbwin_tab_pos = option_menu;

   /* Move to new tab automatically or not */
   toggle = create_check_button (_("Move to new tab automatically"),
				 conf.thumbwin_move_to_newtab,
                                 cb_get_data_from_toggle,
                                 &config_changed->thumbwin_move_to_newtab);
   gtk_box_pack_start (GTK_BOX (hbox), toggle, FALSE, FALSE, 5);

   /* show tab close button */
   toggle = create_check_button (_("Show close button"),
				 conf.thumbwin_show_tab_close,
                                 cb_get_data_from_toggle,
                                 &config_changed->thumbwin_show_tab_close);
   gtk_box_pack_start (GTK_BOX (hbox), toggle, FALSE, FALSE, 5);

   hbox = gtk_hbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

   /* show full path in tab */
   toggle = create_check_button (_("Show full path"),
				 conf.thumbwin_tab_fullpath,
                                 cb_get_data_from_toggle,
                                 &config_changed->thumbwin_tab_fullpath);
   gtk_box_pack_start (GTK_BOX (hbox), toggle, FALSE, FALSE, 5);

   hbox = gtk_hbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

   /* force open tab */
   toggle = create_check_button (_("Open new tab whether image is exist or not "
				   "in the directory"),
				 conf.thumbwin_force_open_tab,
                                 cb_get_data_from_toggle,
                                 &config_changed->thumbwin_force_open_tab);
   gtk_box_pack_start (GTK_BOX (hbox), toggle, FALSE, FALSE, 5);


   /* Thumbnail size spinner */
   hbox = gtk_hbox_new (FALSE, 10);
   gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);
   gtk_box_pack_start (GTK_BOX (main_vbox), hbox, FALSE, FALSE, 0);

   label = gtk_label_new (_("Thumnail Size"));
   gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 0);

   adj = (GtkAdjustment *) gtk_adjustment_new (conf.thumbwin_thumb_size,
					       MIN_THUMB_SIZE, MAX_THUMB_SIZE,
					       1.0, 5.0, 0.0);
   spinner = gtk_spin_button_new (adj, 0, 0);
   gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
   gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner),
				    GTK_SHADOW_OUT);
   gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		       GTK_SIGNAL_FUNC (cb_get_data_from_adjustment_by_int),
		       &config_changed->thumbwin_thumb_size);
   gtk_box_pack_start(GTK_BOX(hbox), spinner, FALSE, TRUE, 0);

   /* GUI redraw interval */
   hbox = gtk_hbox_new (FALSE, 10);
   gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);
   gtk_box_pack_start (GTK_BOX (main_vbox), hbox, FALSE, FALSE, 0);

   label = gtk_label_new (_("GUI redraw interval while loading: Every"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
   adj = (GtkAdjustment *) gtk_adjustment_new (conf.thumbwin_redraw_interval,
					       1.0, 1000.0, 1.0, 5.0, 0.0);
   spinner = gtk_spin_button_new (adj, 0, 0);
   gtk_widget_set_usize(spinner, 50, -1);
   gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
   gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner),
				    GTK_SHADOW_OUT);
   gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		       GTK_SIGNAL_FUNC (cb_get_data_from_adjustment_by_int),
		       &config_changed->thumbwin_redraw_interval);
   gtk_box_pack_start (GTK_BOX (hbox), spinner, FALSE, FALSE, 0);

   label = gtk_label_new (_("files"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

   /* set sensitive */
   set_sensitive_thumbwin_state ();

   return main_vbox;
}


/*
 *  create_prefs_thumbtable_page:
 *     @ Create Thumbnail Table preference page.
 *
 *  Return : Packed widget (GtkVbox)
 */
GtkWidget *
create_prefs_thumbtable_page ()
{
   GtkWidget *main_vbox;
   GtkWidget *frame, *table, *vbox, *hbox;
   GtkWidget *label;
   GtkAdjustment *adj;
   GtkWidget *spinner;

   main_vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(main_vbox), 5);

   /********************************************** 
    * Thumbnail Table Frame
    **********************************************/
   frame = gtk_frame_new (_("Thumbnail Table"));
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, TRUE, 0);

   table = gtk_table_new (2, 4, FALSE);
   gtk_container_set_border_width(GTK_CONTAINER(table), 5);
   gtk_container_add (GTK_CONTAINER (frame), table);

   /* Minimum column number spinner */
   label = gtk_label_new (_("Minimum Columns"));
   gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
		     GTK_EXPAND, GTK_FILL, 0, 0);

   adj = (GtkAdjustment *) gtk_adjustment_new (conf.thumbtable_colnum_min,
					       1.0, 256.0, 1.0, 5.0, 0.0);
   spinner = gtk_spin_button_new (adj, 0, 0);
   gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
   gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner),
				    GTK_SHADOW_OUT);
   gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		       GTK_SIGNAL_FUNC (cb_rotate_min_colnum_value), spinner);
   gtk_table_attach (GTK_TABLE (table), spinner, 1, 2, 0, 1,
		     GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);

   /* Maximum column number spinner */
   label = gtk_label_new (_("Maximum Columns"));
   gtk_table_attach (GTK_TABLE (table), label, 2, 3, 0, 1,
		     GTK_EXPAND, GTK_FILL, 0, 0);

   adj = (GtkAdjustment *) gtk_adjustment_new (conf.thumbtable_colnum_max,
					       1.0, 256.0, 1.0, 5.0, 0.0);
   spinner = gtk_spin_button_new (adj, 0, 0);
   gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
   gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner),
				    GTK_SHADOW_OUT);
   gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		       GTK_SIGNAL_FUNC (cb_rotate_max_colnum_value), spinner);
   gtk_table_attach (GTK_TABLE (table), spinner, 3, 4, 0, 1,
		     GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);

   /* Row Spacing spinner */
   label = gtk_label_new (_("Row Spacing"));
   gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
		     GTK_EXPAND, GTK_FILL, 0, 0);

   adj = (GtkAdjustment *) gtk_adjustment_new (conf.thumbtable_row_space,
					       0.0, 255.0, 1.0, 5.0, 0.0);
   spinner = gtk_spin_button_new (adj, 0, 0);
   gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
   gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner),
				    GTK_SHADOW_OUT);
   gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		       GTK_SIGNAL_FUNC (cb_get_data_from_adjustment_by_int),
		       &config_changed->thumbtable_row_space);
   gtk_table_attach (GTK_TABLE (table), spinner, 1, 2, 1, 2,
		     GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);

   /* Column Spacing spinner */
   label = gtk_label_new (_("Column Spacing"));
   gtk_table_attach (GTK_TABLE (table), label, 2, 3, 1, 2,
		     GTK_EXPAND, GTK_FILL, 0, 0);

   adj = (GtkAdjustment *) gtk_adjustment_new (conf.thumbtable_col_space,
					       0.0, 255.0, 1.0, 5.0, 0.0);
   spinner = gtk_spin_button_new (adj, 0, 0);
   gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
   gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner),
				    GTK_SHADOW_OUT);
   gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		       GTK_SIGNAL_FUNC (cb_get_data_from_adjustment_by_int),
		       &config_changed->thumbtable_col_space);
   gtk_table_attach (GTK_TABLE (table), spinner, 3, 4, 1, 2,
		     GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);

   /**********************************************
    * Thumbnail Button Frame
    **********************************************/
   frame = gtk_frame_new (_("Thumbnail Button"));
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, TRUE, 0);

   vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_container_add (GTK_CONTAINER (frame), vbox);

   hbox = gtk_hbox_new (FALSE, 10);
   gtk_container_set_border_width (GTK_CONTAINER(hbox), 5);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

   /* Minimum border width spinner */
   label = gtk_label_new (_("Minimum Border Width"));
   gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 0);

   adj = (GtkAdjustment *) gtk_adjustment_new (conf.thumbbutton_border_width,
					       0.0, 32.0, 1.0, 5.0, 0.0);
   spinner = gtk_spin_button_new (adj, 0, 0);
   gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
   gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner),
				    GTK_SHADOW_OUT);
   gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		       GTK_SIGNAL_FUNC (cb_get_data_from_adjustment_by_int),
		       &config_changed->thumbbutton_border_width);
   gtk_box_pack_start(GTK_BOX(hbox), spinner, FALSE, TRUE, 0);

   return main_vbox;
}


/*
 *  create_prefs_dirview_page:
 *     @ Create Directory View preference page.
 *
 *  Return : Packed widget (GtkVbox)
 */
GtkWidget *
create_prefs_dirview_page ()
{
   GtkWidget *main_vbox;
   GtkWidget *frame, *vbox, *hbox;
   GtkWidget *toggle, *label, *option_menu, *spinner;
   GtkAdjustment *adj;

   main_vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(main_vbox), 5);


   /**********************************************
    * Shoe/Hide Frame
    **********************************************/
   frame = gtk_frame_new (_("Show/Hide"));
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, TRUE, 0);
   vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_container_add (GTK_CONTAINER (frame), vbox);

   /* show/hide toolbar */
   toggle = create_check_button (_("Show toolbar"),
				 conf.dirview_show_toolbar,
                                 cb_get_data_from_toggle,
                                 &config_changed->dirview_show_toolbar);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* show/hide dot file */
   toggle = create_check_button (_("Show dot file"),
				 conf.dirview_show_dotfile,
                                 cb_get_data_from_toggle,
                                 &config_changed->dirview_show_dotfile);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* show/hide dot file */
   toggle = create_check_button (_("Show \".\" directory whether hide dotfile or not."),
				 conf.dirview_show_current_dir,
                                 cb_get_data_from_toggle,
                                 &config_changed->dirview_show_current_dir);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* show/hide dot file */
   toggle = create_check_button (_("Show \"..\" directory whether hide dotfile or not."),
				 conf.dirview_show_parent_dir,
                                 cb_get_data_from_toggle,
                                 &config_changed->dirview_show_parent_dir);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);


   /**********************************************
    * Style Frame
    **********************************************/
   frame = gtk_frame_new (_("Style"));
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, TRUE, 0);
   vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_container_add (GTK_CONTAINER (frame), vbox);

   /* tree line style */
   hbox = gtk_hbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER (hbox), 0);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
   label = gtk_label_new (_("Tree line style"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 5);
   option_menu = create_option_menu_simple (ctree_line_style_items,
					    conf.dirview_line_style,
					    (gint *) &config_changed->dirview_line_style);
   gtk_widget_set_usize (option_menu, 100, -1);
   gtk_box_pack_start (GTK_BOX (hbox), option_menu, FALSE, FALSE, 5);

   /* tree expander style */
   hbox = gtk_hbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER (hbox), 0);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
   label = gtk_label_new (_("Tree expander style"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 5);
   option_menu = create_option_menu_simple (ctree_expander_style_items,
					    conf.dirview_expander_style,
					    (gint *) &config_changed->dirview_expander_style);
   gtk_widget_set_usize (option_menu, 100, -1);
   gtk_box_pack_start (GTK_BOX (hbox), option_menu, FALSE, FALSE, 5);


   /**********************************************
    * Drag and Drop Frame
    **********************************************/
   frame = gtk_frame_new (_("Drag and Drop"));
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, TRUE, 0);
   vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_container_add (GTK_CONTAINER (frame), vbox);

   /* auto scroll */
   toggle = create_check_button (_("Enable auto scroll."),
				 conf.dirview_auto_scroll,
                                 cb_get_data_from_toggle,
                                 &config_changed->dirview_auto_scroll);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* auto scroll interval */
   hbox = gtk_hbox_new (FALSE, 5);
   gtk_container_set_border_width (GTK_CONTAINER(hbox), 5);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
   label = gtk_label_new (_("Auto scroll interval"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
   adj = (GtkAdjustment *) gtk_adjustment_new (conf.dirview_auto_scroll_time,
					       0, 1000,
					       5.0, 10.0, 0.0);
   spinner = gtk_spin_button_new (adj, 0, 0);
   gtk_widget_set_usize(spinner, 70, -1);
   gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
   gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner),
				    GTK_SHADOW_OUT);
   gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		       GTK_SIGNAL_FUNC (cb_get_data_from_adjustment_by_int),
		       &config_changed->dirview_auto_scroll_time);
   gtk_box_pack_start (GTK_BOX (hbox), spinner, FALSE, FALSE, 0);

   label = gtk_label_new (_("[ms]"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

   /* auto expand */
   toggle = create_check_button (_("Enable auto expand directory tree."),
				 conf.dirview_auto_expand,
                                 cb_get_data_from_toggle,
                                 &config_changed->dirview_auto_expand);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* auto expand time */
   hbox = gtk_hbox_new (FALSE, 5);
   gtk_container_set_border_width (GTK_CONTAINER(hbox), 5);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
   label = gtk_label_new (_("Lag time for auto expand"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
   adj = (GtkAdjustment *) gtk_adjustment_new (conf.dirview_auto_expand_time,
					       0, 60000,
					       100.0, 500.0, 0.0);
   spinner = gtk_spin_button_new (adj, 0, 0);
   gtk_widget_set_usize(spinner, 70, -1);
   gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
   gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner),
				    GTK_SHADOW_OUT);
   gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		       GTK_SIGNAL_FUNC (cb_get_data_from_adjustment_by_int),
		       &config_changed->dirview_auto_expand_time);
   gtk_box_pack_start (GTK_BOX (hbox), spinner, FALSE, FALSE, 0);

   label = gtk_label_new (_("[ms]"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

   return main_vbox;
}


/*
 *  create_prefs_cache_page:
 *     @ Create cache preference page.
 *
 *  Return : Packed widget (GtkVbox)
 */
GtkWidget *
create_prefs_cache_page ()
{
   GtkWidget *main_vbox;
   GtkWidget *frame, *vbox, *hbox;
   GtkWidget *label;
   GtkWidget *toggle;
   GtkWidget *option_menu, *menu, *menu_item;
   GtkWidget *spinner;
   GtkAdjustment *adj;
   gint i;

   main_vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(main_vbox), 5);

   /**********************************************
    *  Cache reading frame
    **********************************************/
   frame = gtk_frame_new (_("Cache reading"));
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, TRUE, 0);
   vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_container_add (GTK_CONTAINER (frame), vbox);

   /* GImageView */
   toggle = create_check_button (_("Use GImageView cache."), conf.cache_gimv,
				  cb_cache_read_type_toggle,
				  &config_changed->cache_gimv);
   prefs_win.cache_gimv = toggle;
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* Nautilus */
   toggle = create_check_button (_("Use Nautilus cache."), conf.cache_nautilus,
				  cb_cache_read_type_toggle,
				  &config_changed->cache_nautilus);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* Konqueror (large) */
   toggle = create_check_button (_("Use Konqueror cache (Large)"),
				  conf.cache_konqueror_l,
				  cb_cache_read_type_toggle,
				  &config_changed->cache_konqueror_l);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* Konqueror (medium) */
   toggle = create_check_button (_("Use Konqueror cache (Medium)"),
				  conf.cache_konqueror_m,
				  cb_cache_read_type_toggle,
				  &config_changed->cache_konqueror_m);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* Konqueror (small) */
   toggle = create_check_button (_("Use Konqueror cache (Small)"),
				  conf.cache_konqueror_s,
				  cb_cache_read_type_toggle,
				  &config_changed->cache_konqueror_s);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* Picview fix */
   toggle = create_check_button (_("Enable KDE-2.2 (or later) fix"),
				 conf.cache_kde_2_2,
				 cb_cache_read_type_toggle,
				 &config_changed->cache_kde_2_2);
   prefs_win.cache_picview = toggle;
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
   gtk_widget_set_uposition (GTK_WIDGET(toggle), 40, -1);

   /* GQview */
   toggle = create_check_button (_("Use GQview cache"), conf.cache_gqview,
				 cb_cache_read_type_toggle,
				 &config_changed->cache_gqview);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* Electric Eyes (large) */
   toggle = create_check_button (_("Use Electric Eyes cache (Preview)"),
				 conf.cache_ee_l,
				 cb_cache_read_type_toggle,
				 &config_changed->cache_ee_l);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* Electric Eyes (middle) */
   toggle = create_check_button (_("Use Electric Eyes cache (Icon)"),
				 conf.cache_ee_m,
				 cb_cache_read_type_toggle,
				 &config_changed->cache_ee_m);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* Electric Eyes (small) */
   toggle = create_check_button (_("Use Electric Eyes cache (Mini)"),
				 conf.cache_ee_s,
				 cb_cache_read_type_toggle,
				 &config_changed->cache_ee_s);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* Picview fix */
   toggle = create_check_button (_("Enable Picview fix"),
				 conf.cache_picview,
				 cb_cache_read_type_toggle,
				 &config_changed->cache_picview);
   prefs_win.cache_picview = toggle;
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
   gtk_widget_set_uposition (GTK_WIDGET(toggle), 40, -1);

   /* .xvpics */
   toggle = create_check_button (_("Use XV thumbnail"), conf.cache_xvpics,
				 cb_get_data_from_toggle,
				 &config_changed->cache_xvpics);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /**********************************************
    *  Cache writing frame
    **********************************************/
   frame = gtk_frame_new (_("Cache writing"));
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, TRUE, 0);
   vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_container_add (GTK_CONTAINER (frame), vbox);

   /* Cache type menu */
   hbox = gtk_hbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

   label = gtk_label_new (_("Cache type for save"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 5);
   option_menu = gtk_option_menu_new();
   menu = gtk_menu_new();
   for (i = 0; cache_write_items[i]; i++) {
      menu_item = gtk_menu_item_new_with_label (_(cache_write_items[i]));
      prefs_win.cache_write_menus[i] = menu_item;
      gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			 GTK_SIGNAL_FUNC(cb_cache_write_type), GINT_TO_POINTER (i));
      gtk_menu_append (GTK_MENU(menu), menu_item);
      gtk_widget_show (menu_item);
   }
   gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), menu);
   gtk_option_menu_set_history (GTK_OPTION_MENU (option_menu),
				conf.cache_write_type);
   gtk_box_pack_start (GTK_BOX (hbox), option_menu, FALSE, FALSE, 5);

   /* GImageView thumbnail size */
   hbox = gtk_hbox_new (FALSE, 5);
   gtk_container_set_border_width (GTK_CONTAINER(hbox), 5);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
   label = gtk_label_new (_("GImageVIew thumbnail size"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
   adj = (GtkAdjustment *) gtk_adjustment_new (conf.cache_gimv_write_size,
					       CACHE_GIMV_MIN_SIZE, CACHE_GIMV_MAX_SIZE,
					       1.0, 5.0, 0.0);
   spinner = gtk_spin_button_new (adj, 0, 0);
   prefs_win.cache_gimv_size = spinner;
   gtk_widget_set_usize(spinner, 70, -1);
   //gtk_spin_button_set_digits (GTK_SPIN_BUTTON (spinner), 1);
   gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
   gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner),
				    GTK_SHADOW_OUT);
   gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		       GTK_SIGNAL_FUNC (cb_get_data_from_adjustment_by_int),
		       &config_changed->cache_gimv_write_size);
   gtk_box_pack_start (GTK_BOX (hbox), spinner, FALSE, FALSE, 0);

   /* GQview thumbnail size */
   hbox = gtk_hbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
   label = gtk_label_new (_("GQview thumbnail size"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 5);
   option_menu = gtk_option_menu_new();
   prefs_win.cache_gqview_size = option_menu;
   menu = gtk_menu_new();
   for (i = 0; gqview_thumb_size[i].width; i++) {
      gchar buf [BUF_SIZE];

      g_snprintf (buf, BUF_SIZE, "%d x %d",
		  gqview_thumb_size[i].width, gqview_thumb_size[i].height);
      menu_item = gtk_menu_item_new_with_label (buf);
      gtk_object_set_data (GTK_OBJECT (menu_item), "num", GINT_TO_POINTER(i));
      gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			 GTK_SIGNAL_FUNC(cb_get_data_from_menuitem),
			 &config_changed->cache_gqview_write_size);
      gtk_menu_append (GTK_MENU(menu), menu_item);
      gtk_widget_show (menu_item);
   }
   gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), menu);
   gtk_option_menu_set_history (GTK_OPTION_MENU (option_menu),
				conf.cache_gqview_write_size);
   gtk_box_pack_start (GTK_BOX (hbox), option_menu, FALSE, FALSE, 5);

   /* set sensitive */
   set_sensitive_cache_type ();

   return main_vbox;
}


/*
 *  create_prefs_slideshow_page:
 *     @ Create slide show preference page.
 *
 *  Return : Packed widget (GtkVbox)
 */
GtkWidget *
create_prefs_slideshow_page ()
{
   GtkWidget *main_vbox;
   GtkWidget *hbox;
   GtkWidget *label;
   GtkWidget *spinner, *toggle;
   GtkAdjustment *adj;

   main_vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(main_vbox), 5);

   /* Default Image Scale Spinner */
   hbox = gtk_hbox_new (FALSE, 5);
   gtk_container_set_border_width (GTK_CONTAINER(hbox), 5);
   gtk_box_pack_start (GTK_BOX (main_vbox), hbox, FALSE, FALSE, 0);
   label = gtk_label_new (_("Image change interval"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
   adj = (GtkAdjustment *) gtk_adjustment_new (conf.slideshow_interval,
					       0.0, 7200.0, 0.01, 0.1, 0.0);
   spinner = gtk_spin_button_new (adj, 0, 0);
   gtk_widget_set_usize(spinner, 70, -1);
   gtk_spin_button_set_digits (GTK_SPIN_BUTTON (spinner), 2);
   gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
   gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner),
				    GTK_SHADOW_OUT);
   gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		       GTK_SIGNAL_FUNC (cb_get_data_from_adjustment_by_float),
		       &config_changed->slideshow_interval);
   gtk_box_pack_start (GTK_BOX (hbox), spinner, FALSE, FALSE, 0);

   label = gtk_label_new (_("[sec]"));
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

   /* repeat */
   toggle = create_check_button (_("Repeat slide show"), conf.slideshow_repeat,
				 cb_get_data_from_toggle,
				 &config_changed->slideshow_repeat);
   gtk_box_pack_start (GTK_BOX (main_vbox), toggle, FALSE, FALSE, 0);

   /* fit to window */
   toggle = create_check_button (_("Auto resize to window size"), conf.slideshow_fit_to_win,
				 cb_get_data_from_toggle,
				 &config_changed->slideshow_fit_to_win);
   gtk_box_pack_start (GTK_BOX (main_vbox), toggle, FALSE, FALSE, 0);

   return main_vbox;
}


/*
 *  create_prefs_dnd_page:
 *     @ Create Drag and Drop preference page.
 *
 *  Return : Packed widget (GtkVbox)
 */
GtkWidget *
create_prefs_dnd_page ()
{
   GtkWidget *main_vbox;
   GtkWidget *frame, *vbox;
   GtkWidget *toggle;

   main_vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(main_vbox), 5);

   /**********************************************
    *  File Operation frame
    **********************************************/
   frame = gtk_frame_new (_("File operation"));
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, TRUE, 0);
   vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_container_add (GTK_CONTAINER (frame), vbox);

   /* Drag and Drop to external proccess */
   toggle = create_check_button (_("Enable DnD to external proccess (Experimental)"),
				 conf.dnd_enable_to_external,
				 cb_get_data_from_toggle,
				 &config_changed->dnd_enable_to_external);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* Drag and Drop from external proccess */
   toggle = create_check_button (_("Enable DnD from extenal proccess (Experimental)"),
				 conf.dnd_enable_from_external,
				 cb_get_data_from_toggle,
				 &config_changed->dnd_enable_from_external);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   /* Always refresh list when DnD end */
   toggle = create_check_button (_("Always refresh list when DnD end"),
				 conf.dnd_refresh_list_always,
				 cb_get_data_from_toggle,
				 &config_changed->dnd_refresh_list_always);
   gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);

   return main_vbox;
}


GtkWidget *
create_prefs_progs_page ()
{
   gchar *titles[] = {
      N_("Program Name"),
      N_("Command"),
   };

   GtkWidget *main_vbox, *frame, *toggle;
   GtkWidget *frame_vbox, *vbox, *hbox, *hbox1, *scrollwin;
   GtkWidget *clist, *label, *entry, *button;
   gchar *data[2], **pair;
   gint i, j, num;

   main_vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(main_vbox), 5);

   /**********************************************
    *  Viewer/Editor frame
    **********************************************/
   frame = gtk_frame_new (_("Graphic Viewer/Editor"));
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, TRUE, 0);
   frame_vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(frame_vbox), 5);
   gtk_container_add (GTK_CONTAINER (frame), frame_vbox);

   /* clist */
   hbox = gtk_hbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(hbox), 0);
   gtk_box_pack_start (GTK_BOX (frame_vbox), hbox, TRUE, TRUE, 0);

   scrollwin = gtk_scrolled_window_new (NULL, NULL);
   gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrollwin),
				   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
   gtk_container_set_border_width(GTK_CONTAINER(scrollwin), 5);
   gtk_box_pack_start (GTK_BOX (hbox), scrollwin, TRUE, TRUE, 0);
   gtk_widget_set_usize (scrollwin, -1, 120);

   clist = gtk_clist_new (2);
   num = sizeof (titles)/ sizeof (titles[0]);
   for (i = 0; i < num; i++)
      gtk_clist_set_column_title (GTK_CLIST (clist), i, _(titles[i]));
   gtk_clist_column_titles_show (GTK_CLIST (clist));
   gtk_clist_set_selection_mode (GTK_CLIST (clist), GTK_SELECTION_SINGLE);
   gtk_container_add (GTK_CONTAINER (scrollwin), clist);

   gtk_signal_connect (GTK_OBJECT (clist),"select_row",
		       GTK_SIGNAL_FUNC (cb_progs_select_row), NULL);

   gtk_signal_connect (GTK_OBJECT (clist),"unselect_row",
		       GTK_SIGNAL_FUNC (cb_progs_unselect_row), NULL);

   /* set default values */
   num = sizeof (conf.progs) / sizeof (conf.progs[0]);
   j = 0;
   prefs_win.progs_rownum = 0;
   for (i = 0; i < num; i++) {
      if (config_changed->progs[i]) {
	 pair = g_strsplit (conf.progs[i], ",", 2);
	 if (pair[0] && pair[1]) {
	    data[0] = pair[0]; data[1] = pair[1];
	    gtk_clist_append (GTK_CLIST (clist), data);
	    gtk_clist_set_row_data (GTK_CLIST (clist), j,
				    config_prechanged->progs[i]);
	    config_changed->progs[j] = config_changed->progs[i];
	    j++;
	    prefs_win.progs_rownum++;
	 }
	 g_strfreev (pair);
      }
   }

   prefs_win.graph_progs_selected = -1;

   /* move buttons */
   vbox = gtk_vbox_new (TRUE, 0);
   gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);

   button = gtk_button_new_with_label (_("Up"));
   prefs_win.graph_progs_up = button;
   gtk_widget_set_usize (button, 50, -1);
   gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 2);
   gtk_signal_connect (GTK_OBJECT (button),"pressed",
		       GTK_SIGNAL_FUNC (cb_progs_up_button_pressed), clist);

   button = gtk_button_new_with_label (_("Down"));
   prefs_win.graph_progs_down = button;
   gtk_widget_set_usize (button, 50, -1);
   gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 2);
   gtk_signal_connect (GTK_OBJECT (button),"pressed",
		       GTK_SIGNAL_FUNC (cb_progs_down_button_pressed), clist);

   /* entry area */
   hbox = gtk_hbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
   gtk_box_pack_start (GTK_BOX (frame_vbox), hbox, TRUE, TRUE, 0);

   vbox = gtk_vbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);
   hbox1 = gtk_hbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (vbox), hbox1, TRUE, TRUE, 0);
   label = gtk_label_new (_("Program Name: "));
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
   gtk_box_pack_start (GTK_BOX (hbox1), label, FALSE, FALSE, 0);
   prefs_win.graph_progs_name = entry = gtk_entry_new ();
   gtk_widget_set_usize (entry, 100, -1);
   gtk_box_pack_start (GTK_BOX (vbox), entry, FALSE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (entry),"changed",
		       GTK_SIGNAL_FUNC (cb_progs_entry_changed), NULL);

   vbox = gtk_vbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);
   hbox1 = gtk_hbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (vbox), hbox1, TRUE, TRUE, 0);
   label = gtk_label_new (_("Command: "));
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
   gtk_box_pack_start (GTK_BOX (hbox1), label, FALSE, FALSE, 0);
   prefs_win.graph_progs_command = entry = gtk_entry_new ();
   gtk_box_pack_start (GTK_BOX (vbox), entry, FALSE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (entry),"changed",
		       GTK_SIGNAL_FUNC (cb_progs_entry_changed), NULL);

   /* edit buttons */
   hbox = gtk_hbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
   gtk_box_pack_start (GTK_BOX (frame_vbox), hbox, TRUE, TRUE, 0);

   hbox1 = gtk_hbox_new (FALSE, 0);
   gtk_box_pack_end (GTK_BOX (hbox), hbox1, FALSE, TRUE, 0);
   gtk_box_set_homogeneous (GTK_BOX(hbox1), TRUE);

   button = gtk_button_new_with_label (_("New"));
   prefs_win.graph_progs_new = button;
   gtk_widget_set_usize (button, 70, -1);
   gtk_box_pack_start (GTK_BOX (hbox1), button, FALSE, TRUE, 2);
   gtk_signal_connect (GTK_OBJECT (button),"pressed",
		       GTK_SIGNAL_FUNC (cb_progs_new_button_pressed), clist);

   button = gtk_button_new_with_label (_("Add"));
   prefs_win.graph_progs_add = button;
   gtk_box_pack_start (GTK_BOX (hbox1), button, FALSE, TRUE, 2);
   gtk_signal_connect (GTK_OBJECT (button),"pressed",
		       GTK_SIGNAL_FUNC (cb_progs_add_button_pressed), clist);

   button = gtk_button_new_with_label (_("Change"));
   prefs_win.graph_progs_change = button;
   gtk_box_pack_start (GTK_BOX (hbox1), button, FALSE, TRUE, 2);
   gtk_signal_connect (GTK_OBJECT (button),"pressed",
		       GTK_SIGNAL_FUNC (cb_progs_change_button_pressed), clist);

   button = gtk_button_new_with_label (_("Delete"));
   prefs_win.graph_progs_delete = button;
   gtk_box_pack_start (GTK_BOX (hbox1), button, FALSE, TRUE, 2);
   gtk_signal_connect (GTK_OBJECT (button),"pressed",
		       GTK_SIGNAL_FUNC (cb_progs_delete_button_pressed), clist);

   set_sensitive_ext_progs (FALSE);

   /**********************************************
    *  Web Browser frame
    **********************************************/
   frame = gtk_frame_new (_("Web Browser (Not implemented yet)"));
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, TRUE, 0);
   frame_vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(frame_vbox), 5);
   gtk_container_add (GTK_CONTAINER (frame), frame_vbox);

   /* web browser command */
   hbox = gtk_hbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (frame_vbox), hbox, TRUE, TRUE, 0);
   label = gtk_label_new (_("Command: "));
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
   entry = gtk_entry_new ();
   gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);

   gtk_widget_set_sensitive (entry, FALSE);

   /**********************************************
    *  Text Viewer frame
    **********************************************/
   frame = gtk_frame_new (_("Text Viewer"));
   gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
   gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, TRUE, 0);
   frame_vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(frame_vbox), 5);
   gtk_container_add (GTK_CONTAINER (frame), frame_vbox);

   /* use internal text viewer or not */
   toggle = create_check_button (_("Use internal viewer"),
				 conf.text_viewer_use_internal,
				 cb_use_internal_text_viewer,
				 &config_changed->text_viewer_use_internal);
   gtk_box_pack_start (GTK_BOX (frame_vbox), toggle, FALSE, FALSE, 0);

   /* text viewer command */
   hbox = gtk_hbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (frame_vbox), hbox, TRUE, TRUE, 0);
   label = gtk_label_new (_("Command: "));
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
   prefs_win.text_viewer_command = entry = gtk_entry_new ();
   gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);
   gtk_entry_set_text (GTK_ENTRY (entry), conf.text_viewer);
   gtk_signal_connect (GTK_OBJECT (entry),"changed",
		       GTK_SIGNAL_FUNC (cb_text_view_command_changed), NULL);

   gtk_widget_set_sensitive (prefs_win.text_viewer_command,
			     !config_changed->text_viewer_use_internal);

   return main_vbox;
}


/*******************************************************************************
 *
 *   prefs_open_window:
 *      @ Create preference window. If already opened, raise it and return.
 *
 *   GtkWidget *prefs_window (global valiable):
 *      Pointer to preference window.
 *
 *   Static PrefsWin prefs_win (local valiable):
 *      store pointer to eache widget;
 *
 *******************************************************************************/
void
prefs_open_window ()
{
   GtkWidget *notebook;
   GtkWidget *vbox;
   GtkWidget *label, *button;

   /* if preference window is alredy opend, raise it and return */
   if (prefs_window) {
      gdk_window_raise (prefs_window->window);
      return;
   }

   /* allocate buffer memory for new config */
   config_prechanged = g_memdup (&conf, sizeof(Config));
   config_changed = g_memdup (&conf, sizeof(Config));

   /* create config window and notebook */
   prefs_window = gtk_dialog_new ();
   gtk_window_set_default_size (GTK_WINDOW(prefs_window), 300, 400);
   gtk_window_set_title (GTK_WINDOW (prefs_window), _("Preference")); 
   gtk_signal_connect (GTK_OBJECT(prefs_window), "destroy",
		       GTK_SIGNAL_FUNC(cb_close_prefs_window), NULL);

   notebook = gtk_notebook_new ();
   gtk_container_set_border_width(GTK_CONTAINER(notebook), 5);
   gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);
   gtk_notebook_set_scrollable (GTK_NOTEBOOK (notebook), TRUE);
   gtk_notebook_popup_enable (GTK_NOTEBOOK (notebook));
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (prefs_window)->vbox), 
		       notebook, TRUE, TRUE, 0);

   /* Common Preference Page */
   vbox = create_prefs_common_page ();
   label = gtk_label_new (_("Common"));
   gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox, label);

   /* Image View Preference Page */
   vbox = create_prefs_imageview_page ();
   label = gtk_label_new (_("Image View"));
   gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox, label);

   /* Thumbnail View Preference Page */
   vbox = create_prefs_thumbview_page ();
   label = gtk_label_new (_("Thumbnail View"));
   gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox, label);

   /* Thumbnail Table Preference Page */
   vbox = create_prefs_thumbtable_page ();
   label = gtk_label_new (_("Thumbnail Table"));
   gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox, label);

   /* Directory View Preference Page */
   vbox = create_prefs_dirview_page ();
   label = gtk_label_new (_("Directory View"));
   gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox, label);

   /* Thumbnail Cache Preference Page */
   vbox = create_prefs_cache_page ();
   label = gtk_label_new (_("Thumbnail Cache"));
   gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox, label);

   /* Slide Show Preference Page */
   vbox = create_prefs_slideshow_page ();
   label = gtk_label_new (_("Slide Show"));
   gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox, label);

   /* Drag and Drop Preference Page */
   vbox = create_prefs_dnd_page ();
   label = gtk_label_new (_("Drag and Drop"));
   gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox, label);

   /* External Program Page */
   vbox = create_prefs_progs_page ();
   label = gtk_label_new (_("External Program"));
   gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox, label);

   /* dialog buttons */
   button = gtk_button_new_with_label (_("OK"));
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (prefs_window)->action_area), 
		       button, TRUE, TRUE, 0);
   gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			      GTK_SIGNAL_FUNC (cb_prefs_ok_button),
			      GTK_OBJECT (prefs_window));

   gtk_widget_grab_focus (button);

   button = gtk_button_new_with_label (_("Apply"));
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (prefs_window)->action_area), 
		       button, TRUE, TRUE, 0);
   gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			      GTK_SIGNAL_FUNC (cb_prefs_apply_button),
			      GTK_OBJECT (prefs_window));

   button = gtk_button_new_with_label (_("Cancel"));
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (prefs_window)->action_area), 
		       button, TRUE, TRUE, 0);
   gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			      GTK_SIGNAL_FUNC (cb_prefs_cancel_button),
			      GTK_OBJECT (prefs_window));

   gtk_widget_show_all (prefs_window);
}
