/*
 * $Id: tx_link.c,v 1.3 2003/09/17 19:13:59 hipnod Exp $
 *
 * Copyright (C) 2003 giFT project (gift.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#include "gt_gnutella.h"

#include "tx_stack.h"
#include "tx_layer.h"

#include "io_buf.h"

/*****************************************************************************/

struct tx_link
{
	input_id  id;
	List     *queue;     /* TODO: use something besides a list */
};

#define TX_LINK(layer)       ((struct tx_link *)((layer)->udata))

/*****************************************************************************/

static void deactivate_queue    (struct tx_layer *tx);
static void activate_queue      (struct tx_layer *tx);

/*****************************************************************************/

static BOOL tx_link_init (struct tx_layer *tx)
{
	struct tx_link *tx_link;

	if (!(tx_link = NEW (struct tx_link)))
		return FALSE;

	/* store our layer-specific info in the toplevel layer */
	tx->udata = tx_link;

	return TRUE;
}

static BOOL free_io_buf (struct io_buf *io_buf, void *udata)
{
	io_buf_free (io_buf);
	return FALSE;
}

static void tx_link_destroy (struct tx_layer *tx)
{
	struct tx_link *tx_link = TX_LINK(tx);

	input_remove (tx_link->id);
	tx_link->id = 0;

	list_foreach (tx_link->queue, (ListForeachFunc)free_io_buf, NULL);
	list_free (tx_link->queue);

	FREE (tx_link);
}

static BOOL link_has_data_pending (struct tx_link *tx_link)
{
	if (tx_link->queue != NULL)
		return TRUE;

	return FALSE;
}

/*****************************************************************************/

static void tx_link_send_data (int fd, input_id id, struct tx_layer *tx)
{
	struct tx_link *tx_link = TX_LINK(tx);
	struct io_buf  *io_buf;
	List           *link;
	uint8_t        *ptr;
	size_t          len;
	int             n;

	assert (tx_link->queue != NULL);

	link   = list_nth (tx_link->queue, 0);
	io_buf = link->data;

	ptr = io_buf_read_ptr   (io_buf);
	len = io_buf_read_avail (io_buf);

	/*
	 * gt_tx_stack_send() calls tcp_send() to send the data on the
	 * connection. This is done because no interface for passing parameters
	 * like a TCPC is exposed anywhere to users of GtTxStack.
	 */
	if ((n = gt_tx_stack_send (tx->stack, ptr, len)) <= 0)
	{
		gt_tx_stack_abort (tx->stack);
		return;
	}

	/*
	 * Pop whatever bytes were written off the buffer. This may be less than
	 * the the whole buffer in the case of a short write. In that rare case we
	 * don't remove the buffer from the list.
	 */
	io_buf_pop (io_buf, n);

	if (io_buf_read_avail (io_buf) > 0)
	{
		assert (io_buf_read_avail (io_buf) < len);
		return;
	}

	io_buf_free (io_buf);

	tx_link->queue = list_remove_link (tx_link->queue, link);

	/* check if the queue should still be running */
	if (!link_has_data_pending (tx_link))
		deactivate_queue (tx);
}

static void activate_queue (struct tx_layer *tx)
{
	struct tx_link *tx_link = TX_LINK(tx);

	/* skip if input already active */
	if (tx_link->id)
		return;

	/* skip if there's no data to write */
	if (!link_has_data_pending (tx_link))
		return;

	tx_link->id = gt_tx_stack_input_add (tx->stack, 
	                                     (InputCallback)tx_link_send_data, tx);
}

static void deactivate_queue (struct tx_layer *tx)
{
	struct tx_link *tx_link = TX_LINK(tx);

	if (!tx_link->id)
		return;

	input_remove (tx_link->id);
	tx_link->id = 0;
}

/*****************************************************************************/

static void tx_link_enable (struct tx_layer *tx)
{
	activate_queue (tx);
}

static void tx_link_disable (struct tx_layer *tx)
{
	deactivate_queue (tx);
}

static void tx_link_send (struct tx_layer *tx, struct io_buf *io_buf)
{
	struct tx_link *tx_link = TX_LINK(tx);

	tx_link->queue = list_append (tx_link->queue, io_buf);

	activate_queue (tx);
}

/*****************************************************************************/

struct tx_layer_ops tx_link_ops =
{
	tx_link_init,
	tx_link_destroy,
	tx_link_enable,
	tx_link_disable,
	tx_link_send,
};
