// MM2ALG.CPP

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "mm2alg.h"

/*################################################################################################*/

mm2_geomopt::mm2_geomopt(mm2_eng * p1, i32s p2, f64 p3) : conjugate_gradient(p2, p3)
{
	eng = p1;
	
	for (i32u n1 = 0;n1 < eng->index_vector.size();n1++)
	{
		AddVar(& eng->crd[n1][0], & eng->d1[n1][0]);
		AddVar(& eng->crd[n1][1], & eng->d1[n1][1]);
		AddVar(& eng->crd[n1][2], & eng->d1[n1][2]);
	}
}

mm2_geomopt::~mm2_geomopt(void)
{
}

f64 mm2_geomopt::GetValue(void)
{
	eng->Compute(0);	// request energy
	return eng->energy;
}

f64 mm2_geomopt::GetGradient(void)
{
	eng->Compute(1);	// request energy and gradient
	return eng->energy;
}

/*################################################################################################*/

mm2_moldyn::mm2_moldyn(mm2_eng * p1, f64 p2, f64 p3)
{
	eng = p1;
	
	vel = new f64_a3[eng->index_vector.size()];		// [1.0E+3 m/s]
	acc = new f64_a3[eng->index_vector.size()];		// [1.0E+12 m/s^2]
	
	// so far we set the initial velocities to zero...
	
	for (i32u n1 = 0;n1 < eng->index_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < 3;n2++)
		{
			vel[n1][n2] = 0.0;
			acc[n1][n2] = 0.0;
		}
	}
	
	temp = p2; step1 = p3;
	step2 = step1 * step1;		// [1.0E-15 s]
}

mm2_moldyn::~mm2_moldyn(void)
{
	delete[] vel;
	delete[] acc;
}

// Allen MP, Tildesley DJ : "Computer Simulation of Liquids", Clarendon Press, 1987

// this is a "velocity-Verlet"-type integrator...

void mm2_moldyn::TakeMDStep(void)
{
	for (i32u n1 = 0;n1 < eng->index_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < 3;n2++)
		{
			f64 tmp1 = step1 * vel[n1][n2] * 1.0e-3;
			f64 tmp2 = step2 * acc[n1][n2] * 0.5e-9;
			
			eng->crd[n1][n2] += tmp1 + tmp2;
			vel[n1][n2] += step1 * acc[n1][n2] * 0.5e-6;
		}
	}
	
	eng->Compute(1);
	pot = eng->energy;
	
	for (i32u n1 = 0;n1 < eng->index_vector.size();n1++)
	{
		f64 mass1 = eng->mass[n1];
		f64 mass2 = mass1 * 1.6605402e-27 * 6.0221367e+23;
		
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			acc[n1][n2] = -eng->d1[n1][n2] / mass2;
			vel[n1][n2] += step1 * acc[n1][n2] * 0.5e-6;
		}
	}
	
	kin = KineticEnergy();
}

f64 mm2_moldyn::KineticEnergy(void)
{
	f64 energy = 0.0;
	
	for (i32u n1 = 0;n1 < eng->index_vector.size();n1++)
	{
		f64 mass1 = eng->mass[n1];
		f64 mass2 = mass1 * 1.6605402e-27 * 6.0221367e+23;
		
		f64 sum = 0.0;
		for (i32u n2 = 0;n2 < 3;n2++)
		{
			f64 tmp = vel[n1][n2];
			sum += tmp * tmp;
		}
		
		energy += 500.0 * mass2 * sum;
	}
	
	return energy;
}

// these conversions are pretty much the same as in mm1-algorithms... make common???
// these conversions are pretty much the same as in mm1-algorithms... make common???
// these conversions are pretty much the same as in mm1-algorithms... make common???

f64 mm2_moldyn::ConvTempEKin(f64 p1)
{
	return (3.0 / 2.0) * eng->index_vector.size() * 8.314510 * p1 / 1000.0;
}

f64 mm2_moldyn::ConvEKinTemp(f64 p1)
{
	return (2.0 / 3.0) * p1 * 1000.0 / (eng->index_vector.size() * 8.314510);
}

void mm2_moldyn::SetEKin(f64 p1)
{
	f64 tmp1 = p1 / KineticEnergy();
	f64 tmp2 = (tmp1 < 0.0 ? 0.0 : sqrt(tmp1));
	
	for (i32u n1 = 0;n1 < eng->index_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < 3;n2++)
		{
			vel[n1][n2] *= tmp2;
		}
	}
}

/*################################################################################################*/

// eof
