{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2001  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to section 3.8 (Texturing) of the OpenGL 1.2.1 specs.
-}

module GL_Texturing (
   ProxyFlag(..), LOD(..), InternalFormat(..),
   marshalInternalFormat, unmarshalInternalFormat,     -- internal use only
   TextureTarget(..),
   marshalTextureTarget,                               -- internal use only
   BorderFlag(..),
   texImage3D,                                         -- @GL_1_2@
   texImage2D, texImage1D,
   copyTexImage2D, copyTexImage1D,
   texSubImage3D,                                      -- @GL_1_2@
   texSubImage2D, texSubImage1D,
   copyTexSubImage3D,                                  -- @GL_1_2@
   copyTexSubImage2D, copyTexSubImage1D,
   TextureFilter(..),
   unmarshalTextureFilter,                             -- internal use only
   Wrapping(..),
   unmarshalWrapping,                                  -- internal use only
   TextureParameter(..), texParameter,
   TextureName(..),                                    -- constructor for internal use only
   defaultTexture, bindTexture, deleteTextures, genTextures,
   areTexturesResident, prioritizeTextures,
   TextureFunction(..),
   unmarshalTextureFunction,                           -- internal use only
   TexEnv(..), texEnv
) where

import Foreign          (Ptr, Storable(..), withObject,
                         withArray, allocaArray, peekArray)
import List             (genericLength)

import GL_Constants     (gl_TEXTURE_3D, gl_PROXY_TEXTURE_3D,
                         gl_TEXTURE_2D, gl_PROXY_TEXTURE_2D,
                         gl_TEXTURE_1D, gl_PROXY_TEXTURE_1D,
                         gl_ALPHA, gl_LUMINANCE, gl_LUMINANCE_ALPHA,
                         gl_INTENSITY, gl_RGB, gl_RGBA, gl_ALPHA4, gl_ALPHA8,
                         gl_ALPHA12, gl_ALPHA16, gl_LUMINANCE4, gl_LUMINANCE8,
                         gl_LUMINANCE12, gl_LUMINANCE16, gl_LUMINANCE4_ALPHA4,
                         gl_LUMINANCE6_ALPHA2, gl_LUMINANCE8_ALPHA8,
                         gl_LUMINANCE12_ALPHA4, gl_LUMINANCE12_ALPHA12,
                         gl_LUMINANCE16_ALPHA16, gl_INTENSITY4, gl_INTENSITY8,
                         gl_INTENSITY12, gl_INTENSITY16, gl_R3_G3_B2, gl_RGB4,
                         gl_RGB5, gl_RGB8, gl_RGB10, gl_RGB12, gl_RGB16,
                         gl_RGBA2, gl_RGBA4, gl_RGB5_A1, gl_RGBA8, gl_RGB10_A2,
                         gl_RGBA12, gl_RGBA16, gl_NEAREST, gl_LINEAR,
                         gl_NEAREST_MIPMAP_NEAREST, gl_NEAREST_MIPMAP_LINEAR,
                         gl_LINEAR_MIPMAP_NEAREST, gl_LINEAR_MIPMAP_LINEAR,
                         gl_CLAMP, gl_CLAMP_TO_EDGE, gl_REPEAT, gl_TEXTURE_WRAP_S,
                         gl_TEXTURE_WRAP_T, gl_TEXTURE_WRAP_R,
                         gl_TEXTURE_MIN_FILTER, gl_TEXTURE_MAG_FILTER,
                         gl_TEXTURE_BORDER_COLOR, gl_TEXTURE_PRIORITY,
                         gl_TEXTURE_MIN_LOD, gl_TEXTURE_MAX_LOD,
                         gl_TEXTURE_BASE_LEVEL, gl_TEXTURE_MAX_LEVEL,
                         gl_REPLACE, gl_MODULATE, gl_DECAL, gl_BLEND,
                         gl_TEXTURE_ENV_MODE, gl_TEXTURE_ENV, gl_TEXTURE_ENV_COLOR)
import GL_BasicTypes    (GLboolean_, unmarshalGLboolean, GLenum, GLint, GLuint,
                         GLsizei, GLfloat, GLclampf, Capability(..), peek1, poke1)
import GL_CoordTrans    (TextureCoordinate(..))
import GL_VertexSpec    (Color4(..))
import GL_VertexArray   (marshalType)
import GL_PixelRect     (PixelDescriptor(..), marshalPixelFormat)

---------------------------------------------------------------------------
-- Section 3.8.1 (Texture Image Specification)

data ProxyFlag = NoProxy | Proxy deriving (Eq,Ord)

marshalProxyFlag :: ProxyFlag -> TextureTarget -> GLenum
marshalProxyFlag NoProxy Texture3d = gl_TEXTURE_3D
marshalProxyFlag Proxy   Texture3d = gl_PROXY_TEXTURE_3D
marshalProxyFlag NoProxy Texture2d = gl_TEXTURE_2D
marshalProxyFlag Proxy   Texture2d = gl_PROXY_TEXTURE_2D
marshalProxyFlag NoProxy Texture1d = gl_TEXTURE_1D
marshalProxyFlag Proxy   Texture1d = gl_PROXY_TEXTURE_1D

newtype LOD = LOD GLint deriving (Eq,Ord)

-- NOTE: The people at Silicon Graphics must probably have been drunk or
-- something similar when they decided to use GLint for the internal
-- format in glTexImage{1,2,3}D, but GLenum everywhere else...

-- GL_ALPHA           : Collision with GL_PixelRect.Alpha          (resolved here)
-- GL_LUMINANCE       : Collision with GL_PixelRect.Luminance      (resolved here)
-- GL_LUMINANCE_ALPHA : Collision with GL_PixelRect.LuminanceAlpha (resolved here)
-- GL_RGB             : Collision with GL_PixelRect.Rgb            (resolved here)
-- GL_RGBA            : Collision with GL_PixelRect.Rgba           (resolved here)
data InternalFormat =
     Alpha'
   | Luminance'
   | LuminanceAlpha'
   | Intensity
   | Rgb'
   | Rgba'
   | Alpha4
   | Alpha8
   | Alpha12
   | Alpha16
   | Luminance4
   | Luminance8
   | Luminance12
   | Luminance16
   | Luminance4Alpha4
   | Luminance6Alpha2
   | Luminance8Alpha8
   | Luminance12Alpha4
   | Luminance12Alpha12
   | Luminance16Alpha16
   | Intensity4
   | Intensity8
   | Intensity12
   | Intensity16
   | R3G3B2
   | Rgb4
   | Rgb5
   | Rgb8
   | Rgb10
   | Rgb12
   | Rgb16
   | Rgba2
   | Rgba4
   | Rgb5A1
   | Rgba8
   | Rgb10A2
   | Rgba12
   | Rgba16
   deriving (Eq,Ord)

marshalInternalFormat :: InternalFormat -> GLenum
marshalInternalFormat Alpha'             = gl_ALPHA
marshalInternalFormat Luminance'         = gl_LUMINANCE
marshalInternalFormat LuminanceAlpha'    = gl_LUMINANCE_ALPHA
marshalInternalFormat Intensity          = gl_INTENSITY
marshalInternalFormat Rgb'               = gl_RGB
marshalInternalFormat Rgba'              = gl_RGBA
marshalInternalFormat Alpha4             = gl_ALPHA4
marshalInternalFormat Alpha8             = gl_ALPHA8
marshalInternalFormat Alpha12            = gl_ALPHA12
marshalInternalFormat Alpha16            = gl_ALPHA16
marshalInternalFormat Luminance4         = gl_LUMINANCE4
marshalInternalFormat Luminance8         = gl_LUMINANCE8
marshalInternalFormat Luminance12        = gl_LUMINANCE12
marshalInternalFormat Luminance16        = gl_LUMINANCE16
marshalInternalFormat Luminance4Alpha4   = gl_LUMINANCE4_ALPHA4
marshalInternalFormat Luminance6Alpha2   = gl_LUMINANCE6_ALPHA2
marshalInternalFormat Luminance8Alpha8   = gl_LUMINANCE8_ALPHA8
marshalInternalFormat Luminance12Alpha4  = gl_LUMINANCE12_ALPHA4
marshalInternalFormat Luminance12Alpha12 = gl_LUMINANCE12_ALPHA12
marshalInternalFormat Luminance16Alpha16 = gl_LUMINANCE16_ALPHA16
marshalInternalFormat Intensity4         = gl_INTENSITY4
marshalInternalFormat Intensity8         = gl_INTENSITY8
marshalInternalFormat Intensity12        = gl_INTENSITY12
marshalInternalFormat Intensity16        = gl_INTENSITY16
marshalInternalFormat R3G3B2             = gl_R3_G3_B2
marshalInternalFormat Rgb4               = gl_RGB4
marshalInternalFormat Rgb5               = gl_RGB5
marshalInternalFormat Rgb8               = gl_RGB8
marshalInternalFormat Rgb10              = gl_RGB10
marshalInternalFormat Rgb12              = gl_RGB12
marshalInternalFormat Rgb16              = gl_RGB16
marshalInternalFormat Rgba2              = gl_RGBA2
marshalInternalFormat Rgba4              = gl_RGBA4
marshalInternalFormat Rgb5A1             = gl_RGB5_A1
marshalInternalFormat Rgba8              = gl_RGBA8
marshalInternalFormat Rgb10A2            = gl_RGB10_A2
marshalInternalFormat Rgba12             = gl_RGBA12
marshalInternalFormat Rgba16             = gl_RGBA16

unmarshalInternalFormat :: GLenum -> InternalFormat
unmarshalInternalFormat f
   | f == gl_ALPHA               = Alpha'
   | f == gl_LUMINANCE           = Luminance'
   | f == gl_LUMINANCE_ALPHA     = LuminanceAlpha'
   | f == gl_INTENSITY           = Intensity
   | f == gl_RGB                 = Rgb'
   | f == gl_RGBA                = Rgba'
   | f == gl_ALPHA4              = Alpha4
   | f == gl_ALPHA8              = Alpha8
   | f == gl_ALPHA12             = Alpha12
   | f == gl_ALPHA16             = Alpha16
   | f == gl_LUMINANCE4          = Luminance4
   | f == gl_LUMINANCE8          = Luminance8
   | f == gl_LUMINANCE12         = Luminance12
   | f == gl_LUMINANCE16         = Luminance16
   | f == gl_LUMINANCE4_ALPHA4   = Luminance4Alpha4
   | f == gl_LUMINANCE6_ALPHA2   = Luminance6Alpha2
   | f == gl_LUMINANCE8_ALPHA8   = Luminance8Alpha8
   | f == gl_LUMINANCE12_ALPHA4  = Luminance12Alpha4
   | f == gl_LUMINANCE12_ALPHA12 = Luminance12Alpha12
   | f == gl_LUMINANCE16_ALPHA16 = Luminance16Alpha16
   | f == gl_INTENSITY4          = Intensity4
   | f == gl_INTENSITY8          = Intensity8
   | f == gl_INTENSITY12         = Intensity12
   | f == gl_INTENSITY16         = Intensity16
   | f == gl_R3_G3_B2            = R3G3B2
   | f == gl_RGB4                = Rgb4
   | f == gl_RGB5                = Rgb5
   | f == gl_RGB8                = Rgb8
   | f == gl_RGB10               = Rgb10
   | f == gl_RGB12               = Rgb12
   | f == gl_RGB16               = Rgb16
   | f == gl_RGBA2               = Rgba2
   | f == gl_RGBA4               = Rgba4
   | f == gl_RGB5_A1             = Rgb5A1
   | f == gl_RGBA8               = Rgba8
   | f == gl_RGB10_A2            = Rgb10A2
   | f == gl_RGBA12              = Rgba12
   | f == gl_RGBA16              = Rgba16
   | otherwise                   = error "unmarshalInternalFormat"

-- GL_TEXTURE_3D: Collision with TextureCapability (resolved there)
-- GL_TEXTURE_2D: Collision with TextureCapability (resolved there)
-- GL_TEXTURE_1D: Collision with TextureCapability (resolved there)
data TextureTarget =
     Texture3d        -- @GL_1_2@
   | Texture2d
   | Texture1d
   deriving (Eq,Ord)

marshalTextureTarget :: TextureTarget -> GLenum
marshalTextureTarget Texture3d = gl_TEXTURE_3D
marshalTextureTarget Texture2d = gl_TEXTURE_2D
marshalTextureTarget Texture1d = gl_TEXTURE_1D

data BorderFlag = NoBorder | Border deriving (Eq,Ord)

marshalBorderFlag :: BorderFlag -> GLint
marshalBorderFlag NoBorder = 0
marshalBorderFlag Border   = 1

-- @GL_1_2@
texImage3D :: ProxyFlag -> LOD -> InternalFormat
           -> GLsizei -> GLsizei -> GLsizei
           -> BorderFlag -> PixelDescriptor -> IO ()
texImage3D proxyFlag level internalFormat width height depth borderFlag (PixelDescriptor f t p) =
   glTexImage3D (marshalProxyFlag proxyFlag Texture3d) level (fromIntegral (marshalInternalFormat internalFormat))
                width height depth (marshalBorderFlag borderFlag) (marshalPixelFormat f) (marshalType t) p

-- @GL_1_2@
foreign import "glTexImage3D" unsafe glTexImage3D ::
   GLenum -> LOD -> GLint -> GLsizei -> GLsizei -> GLsizei -> GLint -> GLenum -> GLenum -> Ptr a -> IO ()

texImage2D :: ProxyFlag -> LOD -> InternalFormat
           -> GLsizei -> GLsizei
           -> BorderFlag -> PixelDescriptor -> IO ()
texImage2D proxyFlag level internalFormat width height borderFlag (PixelDescriptor f t p) =
   glTexImage2D (marshalProxyFlag proxyFlag Texture2d) level (fromIntegral (marshalInternalFormat internalFormat))
                width height (marshalBorderFlag borderFlag) (marshalPixelFormat f) (marshalType t) p

foreign import "glTexImage2D" unsafe glTexImage2D ::
   GLenum -> LOD -> GLint -> GLsizei -> GLsizei -> GLint -> GLenum -> GLenum -> Ptr a -> IO ()

texImage1D :: ProxyFlag -> LOD -> InternalFormat
           -> GLsizei
           -> BorderFlag -> PixelDescriptor -> IO ()
texImage1D proxyFlag level internalFormat width borderFlag (PixelDescriptor f t p) =
   glTexImage1D (marshalProxyFlag proxyFlag Texture1d)
                level (fromIntegral (marshalInternalFormat internalFormat))
                width (marshalBorderFlag borderFlag) (marshalPixelFormat f) (marshalType t) p

foreign import "glTexImage1D" unsafe glTexImage1D :: 
   GLenum -> LOD -> GLint -> GLsizei -> GLint -> GLenum -> GLenum -> Ptr a -> IO ()

---------------------------------------------------------------------------
-- Section 3.8.2 (Alternate Texture Image Specification Commands)

copyTexImage2D :: LOD -> InternalFormat
               -> GLint   -> GLint
               -> GLsizei -> GLsizei
               -> BorderFlag -> IO ()
copyTexImage2D level internalFormat x y width height borderFlag =
   glCopyTexImage2D gl_TEXTURE_2D level (marshalInternalFormat internalFormat)
                    x y width height (marshalBorderFlag borderFlag)

foreign import "glCopyTexImage2D" unsafe glCopyTexImage2D ::
   GLenum -> LOD -> GLenum -> GLint -> GLint -> GLsizei -> GLsizei -> GLint -> IO ()

copyTexImage1D :: LOD -> InternalFormat
               -> GLint -> GLint
               -> GLsizei
               -> BorderFlag -> IO ()
copyTexImage1D level internalFormat x y width borderFlag =
   glCopyTexImage1D gl_TEXTURE_1D level (marshalInternalFormat internalFormat)
                     x y width (marshalBorderFlag borderFlag)

foreign import "glCopyTexImage1D" unsafe  glCopyTexImage1D ::
   GLenum -> LOD -> GLenum -> GLint -> GLint -> GLsizei -> GLint -> IO ()

-- @GL_1_2@
texSubImage3D :: LOD
              -> GLint   -> GLint   -> GLint
              -> GLsizei -> GLsizei -> GLsizei
              -> PixelDescriptor -> IO ()
texSubImage3D level xOffset yOffset zOffset width height depth (PixelDescriptor f t p) =
   glTexSubImage3D gl_TEXTURE_3D level xOffset yOffset zOffset
                   width height depth (marshalPixelFormat f) (marshalType t) p

foreign import "glTexSubImage3D" unsafe glTexSubImage3D ::
   GLenum -> LOD -> GLint -> GLint -> GLint -> GLsizei -> GLsizei -> GLsizei -> GLenum -> GLenum -> Ptr a -> IO ()

texSubImage2D :: LOD
              -> GLint   -> GLint
              -> GLsizei -> GLsizei
              -> PixelDescriptor -> IO ()
texSubImage2D level xOffset yOffset width height (PixelDescriptor f t p) =
   glTexSubImage2D gl_TEXTURE_2D level xOffset yOffset
                   width height (marshalPixelFormat f) (marshalType t) p

foreign import "glTexSubImage2D" unsafe glTexSubImage2D ::
   GLenum -> LOD -> GLint -> GLint -> GLsizei -> GLsizei -> GLenum -> GLenum -> Ptr a -> IO ()

texSubImage1D :: LOD -> GLint -> GLsizei -> PixelDescriptor -> IO ()
texSubImage1D level xOffset width (PixelDescriptor f t p) =
   glTexSubImage1D gl_TEXTURE_1D level xOffset
                   width (marshalPixelFormat f) (marshalType t) p

foreign import "glTexSubImage1D" unsafe glTexSubImage1D ::
   GLenum -> LOD -> GLint -> GLsizei -> GLenum -> GLenum -> Ptr a -> IO ()

-- @GL_1_2@
copyTexSubImage3D :: LOD
                  -> GLint   -> GLint -> GLint
                  -> GLint   -> GLint
                  -> GLsizei -> GLsizei -> IO ()
copyTexSubImage3D = glCopyTexSubImage3D gl_TEXTURE_3D

foreign import "glCopyTexSubImage3D" unsafe glCopyTexSubImage3D ::
   GLenum -> LOD -> GLint -> GLint -> GLint -> GLint -> GLint -> GLsizei -> GLsizei -> IO ()

copyTexSubImage2D :: LOD
                  -> GLint   -> GLint
                  -> GLint   -> GLint
                  -> GLsizei -> GLsizei -> IO ()
copyTexSubImage2D = glCopyTexSubImage2D gl_TEXTURE_2D

foreign import "glCopyTexSubImage2D" unsafe glCopyTexSubImage2D ::
   GLenum -> LOD -> GLint -> GLint -> GLint -> GLint -> GLsizei -> GLsizei -> IO ()

copyTexSubImage1D :: LOD
                  -> GLint
                  -> GLint -> GLint
                  -> GLsizei -> IO ()
copyTexSubImage1D = glCopyTexSubImage1D gl_TEXTURE_1D

foreign import "glCopyTexSubImage1D" unsafe glCopyTexSubImage1D ::
   GLenum -> LOD -> GLint -> GLint -> GLint -> GLsizei -> IO ()

---------------------------------------------------------------------------
-- Section 3.8.3 (Texture Parameters)

-- GL_LINEAR: Collision with GL_Texturing.TextureFilter (resolved there)
data TextureFilter =
     Nearest
   | Linear
   | NearestMipmapNearest
   | NearestMipmapLinear
   | LinearMipmapNearest
   | LinearMipmapLinear
   deriving (Eq,Ord)

marshalTextureFilter :: TextureFilter -> GLint
marshalTextureFilter Nearest              = gl_NEAREST
marshalTextureFilter Linear               = gl_LINEAR
marshalTextureFilter NearestMipmapNearest = gl_NEAREST_MIPMAP_NEAREST
marshalTextureFilter NearestMipmapLinear  = gl_NEAREST_MIPMAP_LINEAR
marshalTextureFilter LinearMipmapNearest  = gl_LINEAR_MIPMAP_NEAREST
marshalTextureFilter LinearMipmapLinear   = gl_LINEAR_MIPMAP_LINEAR

unmarshalTextureFilter :: GLint -> TextureFilter
unmarshalTextureFilter f
   | f == gl_NEAREST                =  Nearest
   | f == gl_LINEAR                 =  Linear
   | f == gl_NEAREST_MIPMAP_NEAREST =  NearestMipmapNearest
   | f == gl_NEAREST_MIPMAP_LINEAR  =  NearestMipmapLinear
   | f == gl_LINEAR_MIPMAP_NEAREST  =  LinearMipmapNearest
   | f == gl_LINEAR_MIPMAP_LINEAR   =  LinearMipmapLinear
   | otherwise                      = error "unmarshalTextureFilter"

data Wrapping =
     Clamp
   | ClampToEdge   -- @GL_1_2@
   | Repeat

marshalWrapping :: Wrapping -> GLint
marshalWrapping Clamp       = gl_CLAMP
marshalWrapping ClampToEdge = gl_CLAMP_TO_EDGE
marshalWrapping Repeat      = gl_REPEAT

unmarshalWrapping :: GLint -> Wrapping
unmarshalWrapping w
   | w == gl_CLAMP         = Clamp
   | w == gl_CLAMP_TO_EDGE = ClampToEdge
   | w == gl_REPEAT        = Repeat
   | otherwise             = error "unmarshalWrapping"

data TextureParameter =
     TextureWrap TextureCoordinate Wrapping
   | TextureFilters TextureFilter TextureFilter
   | TextureBorderColor (Color4 GLfloat)
   | TexturePriority GLfloat
   | TextureLod GLfloat GLfloat   -- @GL_1_2@
   | TextureLevel GLint GLint     -- @GL_1_2@

texParameter :: TextureTarget -> TextureParameter -> IO ()
texParameter target (TextureWrap coord wrap) =
   let w = case coord of
              S -> gl_TEXTURE_WRAP_S
              T -> gl_TEXTURE_WRAP_T
              R -> gl_TEXTURE_WRAP_R
              Q -> error "texParam: Q coordinate is illegal"
   in glTexParameteri (marshalTextureTarget target) w (marshalWrapping wrap)
texParameter target (TextureFilters mini mag) = do
   let t = marshalTextureTarget target
   glTexParameteri t gl_TEXTURE_MIN_FILTER (marshalTextureFilter mini)
   glTexParameteri t gl_TEXTURE_MAG_FILTER (marshalTextureFilter mag)
texParameter target (TextureBorderColor c) =
   withObject c $ glTexParameterfv (marshalTextureTarget target) gl_TEXTURE_BORDER_COLOR
texParameter target (TexturePriority p) =
   glTexParameterf (marshalTextureTarget target) gl_TEXTURE_PRIORITY p
texParameter target (TextureLod mini maxi) = do
   let t = marshalTextureTarget target
   glTexParameterf t gl_TEXTURE_MIN_LOD mini
   glTexParameterf t gl_TEXTURE_MAX_LOD maxi
texParameter target (TextureLevel base maxi) = do
   let t = marshalTextureTarget target
   glTexParameteri t gl_TEXTURE_BASE_LEVEL base
   glTexParameteri t gl_TEXTURE_MAX_LEVEL maxi

foreign import "glTexParameteri"  unsafe glTexParameteri  :: GLenum -> GLenum -> GLint                -> IO ()
foreign import "glTexParameterf"  unsafe glTexParameterf  :: GLenum -> GLenum -> GLfloat              -> IO ()
foreign import "glTexParameterfv" unsafe glTexParameterfv :: GLenum -> GLenum -> Ptr (Color4 GLfloat) -> IO ()

---------------------------------------------------------------------------
-- Section 3.8.8 (Texture Objects)

newtype TextureName = TextureName GLuint

instance Storable TextureName where
   sizeOf    ~(TextureName n) = sizeOf n
   alignment ~(TextureName n) = alignment n
   peek                       = peek1 TextureName
   poke ptr   (TextureName n) = poke1 ptr n

defaultTexture :: TextureName
defaultTexture = TextureName 0

bindTexture :: TextureTarget -> TextureName -> IO ()
bindTexture = glBindTexture . marshalTextureTarget

foreign import "glBindTexture" unsafe glBindTexture :: GLenum -> TextureName -> IO ()

deleteTextures :: [TextureName] -> IO ()
deleteTextures names = withArray names $ glDeleteTextures (genericLength names)

foreign import "glDeleteTextures" unsafe glDeleteTextures :: GLsizei -> Ptr TextureName -> IO ()

genTextures :: Int -> IO [TextureName]
genTextures n =
   allocaArray n $ \buf -> do
   glGenTextures (fromIntegral n) buf
   peekArray n buf

foreign import "glGenTextures" unsafe glGenTextures :: GLsizei -> Ptr TextureName -> IO ()

areTexturesResident :: [TextureName] -> IO (Maybe [TextureName])
areTexturesResident textureNames =
   let len = length textureNames
   in  withArray textureNames $ \nBuf ->
       allocaArray len $ \rBuf -> do
       allResident <- glAreTexturesResident (fromIntegral len) nBuf rBuf
       if unmarshalGLboolean allResident
          then return Nothing   -- TODO: "Nothing" sounds a little bit strange here...
          else do flags <- peekArray len rBuf
                  return $ Just [ n | (f,n) <- zip flags textureNames, unmarshalGLboolean f ]

foreign import "glAreTexturesResident" unsafe glAreTexturesResident ::
   GLsizei -> Ptr TextureName -> Ptr GLboolean_ -> IO GLboolean_

prioritizeTextures :: [(TextureName,GLclampf)] -> IO ()
prioritizeTextures texPrios =
   let (textureNames, priorities) = unzip texPrios
   in withArray textureNames $ \nameBuf ->
      withArray priorities   $ \prioBuf ->
      glPrioritizeTextures (genericLength priorities) nameBuf prioBuf

foreign import "glPrioritizeTextures" unsafe glPrioritizeTextures ::
   GLsizei -> Ptr TextureName -> Ptr GLclampf -> IO ()

---------------------------------------------------------------------------
-- Section 3.8.9 (Texture Environments and Texture Functions)

-- GL_REPLACE: Collision with GL_PerFragment.StencilAction (resolved there)
-- GL_BLEND:   Collision with GL_PerFragment.Blend (resolved there)
data TextureFunction =
     Replace
   | Modulate
   | Decal
   | Blend
   deriving (Eq,Ord)

marshalTextureFunction :: TextureFunction -> GLenum
marshalTextureFunction Replace  = gl_REPLACE
marshalTextureFunction Modulate = gl_MODULATE
marshalTextureFunction Decal    = gl_DECAL
marshalTextureFunction Blend    = gl_BLEND

unmarshalTextureFunction ::  GLenum -> TextureFunction
unmarshalTextureFunction f
   | f == gl_REPLACE  = Replace
   | f == gl_MODULATE = Modulate
   | f == gl_DECAL    = Decal
   | f == gl_BLEND    = Blend
   | otherwise        = error "unmarshalTextureFunction"

data TexEnv =
     TextureEnvMode TextureFunction
   | TextureEnvColor (Color4 GLfloat)

texEnv :: TexEnv -> IO ()
texEnv (TextureEnvMode texFunc) =
   glTexEnvi gl_TEXTURE_ENV gl_TEXTURE_ENV_MODE (fromIntegral (marshalTextureFunction texFunc))
texEnv (TextureEnvColor c) = withObject c $ glTexEnvfv gl_TEXTURE_ENV gl_TEXTURE_ENV_COLOR

foreign import "glTexEnvi"  unsafe glTexEnvi  :: GLenum -> GLenum -> GLint                -> IO ()
foreign import "glTexEnvfv" unsafe glTexEnvfv :: GLenum -> GLenum -> Ptr (Color4 GLfloat) -> IO ()

---------------------------------------------------------------------------
-- Section 3.8.10 (Texture Application)

instance Capability TextureTarget where
   marshalCapability = marshalTextureTarget
